package Sim::OPT::Modish;
# This is Modish.pl, a program for calculating the modification of diffuse solar shading
# to be specified in the ISH (shading and insolation) files of ESP-r building simulation suite
# to take into account the effect of the reflections due to the shadings.  
# That effect is calculated:
# 1) by calculating the difference between (a) the amount (measured as irradiance) of solar radiation 
# (including direct and diffuse) impinging a surface in the case that obstructions 
# have their true reflectivity and all other surfaces with the exception of the ground are completely black;
# b) the amount of solar radiation impinging the external side of that surface in the case that solar obstructions 
# and all other surfaces in the scene with the exception of the ground are completely black.
# 2) by calculating the ratio of: (a) the result of the calculation at point 1; and b) the amount (measured as irradiance)
# of diffuse solar radiation impinging the surface. 
# The value given by 1) minus the result of 2) gives the diffuse shading factor to be put in the ISH file for that surface.
# When the reflection by the shadings increases the amount of radiation entering, that diffuse shading factor is negative.
# Written since April to June 2015 by Gian Luca Brunetti, Politecnico di Milano - gianluca.brunetti@polimi.it.
# This is free software.  You can redistribute it and/or modify it under the terms of the GNU General Public License, version 2 or newer,
# as published by the Free Software Foundation.
# All rights reserved.

use v5.14;
use Exporter;
use parent 'Exporter'; # imports and subclasses Exporter
use vars qw( $VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS );
use Math::Trig;
use List::Util qw[ min max reduce shuffle];
use List::MoreUtils qw(uniq);
use List::AllUtils qw(sum);
use Data::Dump qw(dump);
use Data::Dumper;
use Vector::Object3D::Polygon;
use Math::Polygon::Tree;
$Data::Dumper::Terse  = 1;
$Data::Dumper::Purity  = 1;
use feature 'say';    

no strict; 
no warnings;
our @ISA = qw(Exporter); # our @adamkISA = qw(Exporter);

@EXPORT = qw( modish ); # our @EXPORT = qw( );
$VERSION = '0.003';
$ABSTRACT = 'This is "Sim::OPT::Modish", a program for calculating the increase of diffuse solar radiation impinging transparent envelope surfaces of building thermal zones in the ESP-r building performance optimization platform due to the effect of reflective solar obstructions positioned outside of the zones.';

#########################################################################################

my ( @zoneshds, @winsdata );
my ( %surfslist, %shdfileslist, %obsinfo );

my %days_inmonths = ( Jan => 15, Feb => 14, Mar => 15, Apr => 15, May => 15, Jun => 15, Jul => 15, Aug => 15, Sep => 15, Oct => 15, Nov => 15, Dec => 15 );
my %monthsnum = ( Jan => 1, Feb => 2, Mar => 3, Apr => 4, May => 5, Jun => 6, Jul => 7, Aug => 8, Sep => 9, Oct => 10, Nov => 11, Dec => 12 );


sub getmonthname
{
	my $monthnum = shift;
	
	my %monthsnames = ( 1 => "Jan" , 2 => "Feb", 3 => "Mar", 4 => "Apr", 5 => "May", 6 => "Jun", 7 => "Jul", 8 => "Aug", 9 => "Sep", 10 => "Oct", 11 => "Nov", 12 => "Dec" );
	my $monthname = $monthsnames{ "$monthnum" };
	return $monthname;
}

sub getmonthnum
{
	my $monthname = shift;
	my %monthsnums = ( Jan => 1, Feb => 2, Mar => 3, Apr => 4, May => 5, Jun => 6, Jul => 7, Aug => 8, Sep => 9, Oct => 10, Nov => 11, Dec => 12 );
	my $monthnum = $monthsnums{ "$monthname" };
	return $monthnum;
}

sub _mean_ { return @_ ? sum(@_) / @_ : 0 }

sub getconffilenames
{	# THIS GETS THE CONSTRUCTION AND MATERIALS FILES FROM THE CFG FILE. IT IS CALLED BY sub createfictitious
	my ($conffile, $path) = @_; #say "\$conffile: " . dump($conffile);
	open (CONFFILE, "$conffile") or die;
	my @lines = <CONFFILE>;
	close CONFFILE;
	my ($constrdbfile, $matdbfile);
	my @zonedata;
	my $countzone = 0;
	foreach my $line (@lines)
	{
		my ($geofile, $constrfile, $shdfile, $zonenum_cfg );
		my @row = split(/\s+|,/, $line); 
		if ($row[0] eq "*mlc")
		{
			$constrdbfile = $row[1];
			$constrdbfile =~ s/..//; 
			$constrdbfile = $path . $constrdbfile;
		}
		elsif ($row[0] eq "*mat")
		{
			$matdbfile = $row[1];
			$matdbfile =~ s/..//; 
			$matdbfile = $path . $matdbfile;
		}
		elsif ($row[0] eq "*zon")
		{
			push ( @{$zonedata[$countzone]}, ( $countzone + 1 ) );
			push ( @{$zonedata[$countzone -1]}, $row[1] );
			$countzone++;
		}
		elsif ($row[0] eq "*geo")
		{
			$geofile = $row[1];
			$geofile =~ s/..//; 
			$geofile = $path . $geofile;
			push ( @{$zonedata[ $countzone -1 ]}, $geofile );
		}
		elsif ($row[0] eq "*con")
		{
			$constrfile = $row[1];
			$constrfile =~ s/..//; 
			$constrfile = $path . $constrfile;
			push ( @{$zonedata[ $countzone - 1 ]}, $constrfile );

		}
		elsif ( $row[0] eq "*isi")
		{
			$shdfile = $row[1];
			$shdfile =~ s/..//; 
			$shdfile = $path . $shdfile;
			push ( @{$zonedata[ $countzone - 1 ]}, $shdfile );
		}

	} #say "ZONEDATA! " . dump(@zonedata);
	return ( $constrdbfile, $matdbfile, \@zonedata, \@lines );
}

sub createfictitiousfiles
{
	#THIS CREATE FILES FOR THE MODELS FEATURING FICTITIOUS QUALITIES AIMED TO THE MAIN modish PROGRAM AND MODIFIES MATERIALS DB AS REQUESTED
	# _AND_ PREPARES THE CONFIGURATION FILES FOR THE FICTITIOUS MODELS ACCORDINGLY
	my ($conffile, $path, $zonenum) = @_; #say "\$conffile: " . dump($conffile); say "\$path " . dump($path); say "\$zonenum " . dump($zonenum);
	my $conffile_fict1 = $conffile;
	my ($flaggeo, $flagconstrdb, $flagmatdb, $flagconstr);
	$conffile_fict1 =~ s/\.cfg/\_fict1\.cfg/; #say "\$conffile_fict1: " . dump($conffile_fict1);
	my $conffile_fict2 = $conffile;
	$conffile_fict2 =~ s/\.cfg/\_fict2\.cfg/; #say "\$conffile_fict2: " . dump($conffile_fict2);
	my $conffile_fict3 = $conffile;
	$conffile_fict3 =~ s/\.cfg/\_fict3\.cfg/; #say "\$conffile_fict3: " . dump($conffile_fict3);
	#say "cp -R -f $conffile $conffile_fict1\n";
	`cp -R -f $conffile $conffile_fict1\n`;
	`cp -R -f $conffile $conffile_fict2\n`;
	`cp -R -f $conffile $conffile_fict3\n`;
		
	my ($constrdbfile, $matdbfile, $zonedataref, $conflinesref) = getconffilenames($conffile, $path); #say "\$constrdbfile " . dump($constrdbfile); say "\$matdbfile " . dump($matdbfile);  say "\$zonedataref " . dump($zonedataref); #say "\$linesref: " . dump($linesref);
	my @zonedata = @$zonedataref; #say "ZONEDATA: " . dump(@zonedata);
	my @conflines = @$conflinesref;  #say "\@conflines: " . dump(@conflines);
	my (@originals, @fictitia1, @fictitia2, @fictitia3 ); # CONTAINERS.
	
	push ( @originals, $constrdbfile);
	
	my $constrdbfile_fict = $constrdbfile; #say "--\$constrdbfile_fict: " . dump($constrdbfile_fict);
	$constrdbfile_fict = $constrdbfile . "_fict" ;  #say "--\$constrdbfile_fict: " . dump($constrdbfile_fict);
	push ( @fictitia1, $constrdbfile_fict);
	push ( @fictitia2, $constrdbfile_fict);
	push ( @fictitia3, $constrdbfile_fict);
	
	#say "cp -R -f $constrdbfile $constrdbfile_fict\n";
	#unless (-e $constrdbfile_fict) { `cp -R -f $constrdbfile $constrdbfile_fict\n`; $flagconstrdb = "y"; }
	
	push ( @originals, $matdbfile);
	
	my $matdbfile_fict1 = $matdbfile; 
	$matdbfile_fict1 = $matdbfile . "_fict1";  #say "\$matdbfile_fict1: " . dump($matdbfile_fict1);
	push ( @fictitia1, $matdbfile_fict1 );
	
	my $matdbfile_fict2 = $matdbfile; 
	$matdbfile_fict2 = $matdbfile . "_fict2";  #say "\$matdbfile_fict2: " . dump($matdbfile_fict2);
	push ( @fictitia2, $matdbfile_fict2 );
	
	push ( @fictitia3, $matdbfile_fict1 );
	
	#unless (-e $matdbfile_fict1) { `cp -R -f $matdbfile $matdbfile_fict1\n`; $flagmatdb = "y"; }
	#unless (-e $matdbfile_fict2) { `cp -R -f $matdbfile $matdbfile_fict2\n`; $flagmatdb = "y"; }
	
	foreach my $datum (@zonedata)
	{
		my ( @tempbox_original, @tempbox_fictitia1, @tempbox_fictitia2, @tempbox_fictitia3 );
		my @array = @$datum;  #say "\@array\: " . dump(@array);
		#my $zonenum_ = $array[0]; say "ZONENUM! $szonenum, ZONENUM! $zonenum";
		#if ( $zonenum == $zonenum )
		{
			my $geofile = $array[2];
			push ( @tempbox_originals, $geofile );
			
			my $geofile_fict = $geofile; 
			$geofile_fict =~ s/.geo/_fict.geo/;  #say "\$geofile_fict: " . dump($geofile_fict);
			
			my $geofile_fict3 = $geofile;
			$geofile_fict3 =~ s/\.geo/\_fict3.geo/;

			push ( @tempbox_fictitia1, $geofile_fict);
			push ( @tempbox_fictitia2, $geofile_fict);
			push ( @tempbox_fictitia3, $geofile_fict3);
			#unless (-e $geofile_fict) { `cp -R -f $geofile $geofile_fict\n`; $flaggeo = "y"; }
			
			my $constrfile = $array[3];
			push ( @tempbox_originals, $constrfile);
			
			my $constrfile_fict1 = $constrfile; 
			$constrfile_fict1 =~ s/.con/_fict1.con/;  #say "\$constrfile_fict1: " . dump($constrfile_fict1);
			push ( @tempbox_fictitia1, $constrfile_fict1);
			
			my $constrfile_fict2 = $constrfile; 
			$constrfile_fict2 =~ s/.con/_fict2.con/;  #say "\$constrfile_fict2: " . dump($constrfile_fict2);
			push ( @tempbox_fictitia2, $constrfile_fict2);
			
			push ( @tempbox_fictitia3, $constrfile_fict1);
						
			unless (-e $constrfile_fict1) { `cp -R -f $constrfile $constrfile_fict1\n`; $flagconstr = "y"; }
			unless (-e $constrfile_fict2) { `cp -R -f $constrfile $constrfile_fict2\n`; $flagconstr = "y"; }
			
			my $shdfile = $array[4];
			push ( @tempbox_originals, $shdfile);
			push ( @tempbox_fictitia1, $shdfile);
			push ( @tempbox_fictitia2, $shdfile);
			push ( @tempbox_fictitia3, $shdfile);
			
			my $zonenum_cfg = $array[1];
			push ( @tempbox_originals, $zonenum_cfg);
			push ( @tempbox_fictitia1, $zonenum_cfg);
			push ( @tempbox_fictitia2, $zonenum_cfg);
			push ( @tempbox_fictitia3, $zonenum_cfg);
			
			push ( @originals, [ @tempbox_originals ] );
			push ( @fictitia1, [ @tempbox_fictitia1 ] );
			push ( @fictitia2, [ @tempbox_fictitia2 ] );
			push ( @fictitia3, [ @tempbox_fictitia3 ] );
		}	
	}
		
	open (CONFFILE_FICT1, ">$conffile_fict1");	
	my @conflines2 = @conflines;
	
	foreach my $line (@conflines)
	{
		my $counter = 0;
		foreach my $elt ( @fictitia1 )
		{
			if ( not ( ref($elt) ) )
			{
				
				my $original = $originals[$counter]; #say "ORIGINAL: $original, ELT: $elt";
				$elt =~ s/$path//;
				$original =~ s/$path//;
				$line =~ s/$original/$elt/;
				#my $eltwithpath = $path . $elt;
				#$line =~ s/$elt/$eltwithpath/;
			}
			else
			{
				my @elts = @$elt;
				my @originalelts = @{$originals[$counter]}; #say "ORIGINAL: $original, ELT: $elt";
				my $count = 0;
				foreach my $el ( @elts )
				{
					my $original = $originalelts[$count];
					$el =~ s/$path//;
					$original =~ s/$path//;
					$line =~ s/$original/$el/;
					#my $eltwithpath = $path . $el;
					#$line =~ s/$el/$eltwithpath/;
					$count++;
				}
			}
			$counter++;
		}
		print CONFFILE_FICT1 $line;
	}
	close CONFFILE_FICT1;
	
	open (CONFFILE_FICT2, ">$conffile_fict2");
	foreach my $line (@conflines2)
	{
		my $counter = 0;
		foreach my $elt ( @fictitia2 )
		{
			if ( not ( ref($elt) ) )
			{
				my $original = $originals[$counter]; #say "ORIGINAL: $original, ELT: $elt";
				$elt =~ s/$path//;
				$original =~ s/$path//;
				$line =~ s/$original/$elt/;
				#my $eltwithpath = $path . $elt;
				#$line =~ s/$elt/$eltwithpath/;
			}
			else
			{
				my @elts = @$elt;
				my @originalelts = @{$originals[$counter]}; #say "ORIGINAL: $original, ELT: $elt";
				my $count = 0;
				foreach my $el ( @elts )
				{
					my $original = $originalelts[$count];
					$el =~ s/$path//;
					$original =~ s/$path//;
					$line =~ s/$original/$el/;
					#my $eltwithpath = $path . $el;
					#$line =~ s/$el/$eltwithpath/;
					$count++;
				}
			}
			$counter++;
		}
		print CONFFILE_FICT2 $line;
	}
	close CONFFILE_FICT2;
	
	open (CONFFILE_FICT1, "$conffile_fict1");
	my @conflines3 = <CONFFILE_FICT1>;
	close CONFFILE_FICT1;
	
	open (CONFFILE_FICT3, ">$conffile_fict3");
	my $count = 0;
	foreach my $line (@conflines3)
	{
		my $redflag;
		if ( $$conflines3[ $count + 1] =~ /^# block entities:/ )
		{
			$redflag = "yes";
		}
		
		unless ( $redflag eq "yes" )
		{
			unless ( $line =~ /^\*shad_calc,/ )
			{
				print CONFFILE_FICT3 $line;
			}
			else
			{
				print CONFFILE_FICT3 "*shad_calc,none  # no temporal shading requested\n";
			}
		}
		$count++;
	}
	
	close CONFFILE_FICT3;
	
	return ($conffile, $conffile_fict1, $conffile_fict2, $conffile_fict3, $constrdbfile, $constrdbfile_fict,
	$matdbfile, $matdbfile_fict1, $matdbfile_fict2, $flagconstrdb, $flagmatdb, $flaggeo, $flagconstr, [ @originals ], [ @fictitia1], [ @fictitia2 ] , [ @fictitia3 ] );
}

sub definepath
{
	# THIS DEFINES THE PATH STARTING FROM THE CONFIGURATION FILE PATH. IT IS CALLED FROM sub modish
	my $path = shift;
	$path =~ s/\/cfg(.+)$//;
	return ($path);
}

sub readgeofile
{	# THIS READS A GEO FILE TO GET THE DATA OF THE REQUESTED SURFACES
	my $geofile = $_[0]; #say "geofile: $geofile";
	my @transpsurfs = @{$_[1]}; #say "transpsurfs: " . dump(@transpsurfs);
	my $zonenum = $_[2]; #say "$zonenum $zonenum";
	open (GEOFILE, "$geofile") or die;
	my @lines = <GEOFILE>;
	close GEOFILE;
	my ( @geofilestruct, @transpelts, @obs );
	my %datalist;
	my $countsurf = 0;
	foreach my $surfnum ( @transpsurfs )
	{
		foreach my $line (@lines)
		{
			my @elts = split(/\s+|,/, $line); #say "elts: " . dump(@elts);
			if ( $elts[0] eq "\*surf")
			{
				my $surfname = $elts[1]; say REPORT "HERE READGEO surfname " . dump($surfname);
				my $surfnum = $elts[12]; say REPORT "HERE surfnum " . dump($surfnum);
				#unless ( $elts[3] eq "-" ) 
				#{ 
					my $parent = $elts[3]; 
				#} say REPORT "HERE \$parent " . dump($parent);
				my $constr = $elts[6]; say REPORT "HERE \$constr " . dump($constr);
				# THIS POPULATES THE VARIABLE %surfslist (HASH - DICTIONARY - ASSOCIATIVE ARRAY) LINKING ZONES, SURFACES NAMES AND SURFACE NUMBER:

				$surfslist{ $zonenum }{ $surfnum } = $surfname;
				$surfslist{ $zonenum }{ $surfname } = $surfnum;
				$datalist{ $zonenum }{ $surfnum }{ surfname } = $surfname;
				$datalist{ $zonenum }{ $surfnum }{ parent } = $parent;
				$datalist{ $zonenum }{ $surfnum }{ constr } = $constr;
				$datalist{ $zonenum }{ $surfnum }{ surfnum } = $surfnum;	
				$datalist{ $zonenum }{ $surfnum }{ geofile } = $geofile;
				unless ( $parent eq "-" ) 
				{ 
					my $parentnum = $surfslist{ $zonenum }{ $parent }; #say REPORT "HERE FOCUS \$parentnum " . dump($parentnum);
					push ( @{ $datalist{ $zonenum }{ children }{ $parentnum } }, $surfnum );  #say REPORT "HERE FOCUS \$datalist " . dump($datalist);
					@{ $datalist{ $zonenum }{ children }{ $parentnum } } = uniq( @{ $datalist{ $zonenum }{ children }{ $parentnum } } );
				}
				
			}
			
			if ( $elts[0] eq "\*vertex")
			{
				my $x =  $elts[1];
				my $y =  $elts[2];
				my $z =  $elts[3];
				my $vertnum =  $elts[5];
				$datalist{ $zonenum }{ vertex }{ $vertnum } = [ $x, $y, $z ];
			}
			
			if ( $elts[0] eq "*edges")
			{
				my $surfnum = $elts[ $#surfnum ];
				my @vertnums = @elts[ 1 .. ( $#surfnum - 1 ) ]; say REPORT "\@vertnums " . dump(@vertnums);
				$datalist{ $zonenum }{ $surfnum }{ edges }{ $surfnum } = [ @vertnums ]; 
			}
			
			if ( ($elts[0] eq "\*surf") and ( $surfnum == $elts[12] ) )
			{
				my $surfname = $elts[1]; #say "surfname " . dump($surfname);
				my $parent = $elts[3]; #say "parent " . dump($parent);
				my $constr = $elts[6]; #say "constr " . dump($constr);
				my $surfnum = $elts[12]; #say "surfnum " . dump($surfnum);
				push (@transpelts, [ $surfname, $parent, $constr, $surfnum, $geofile, $zonenum ] ); #say "transpelts " . dump(@transpelts);
			}
			
			if ($elts[0] eq "\*obs")
			{
				my $obsconstr = $elts[10]; #say "\$obsconstr " . dump($obsconstr);
				my $obsname = $elts[9]; #say "\$obsname " . dump($obsname);
				my $obsnum = $elts[13]; #say "\$obsnum " . dump($obsnum);
				push (@obs, [ $obsname, $obsconstr, $obsnum ] ); #say "\@obs " . dump(@obs);
			}
			if ( $countsurf == 0 )
			{
				push ( @geofilestruct, [ @elts ] );
			}
		}
		$countsurf++;
	}
	
	foreach ( @geofilestruct )
	{
		my $obsmaterial = $_->[10];
		my $obsnumber = $_->[13];
		if ( ( $_->[0] eq "#" ) and ( $_->[1] eq "*obs" ) )
		{
			$semaphore = 1;
		}
		if ( ( $semaphore == 1) and ( $_->[0] eq "*obs" ) )
		{
			push ( @obsconstr,$obsmaterial = $_->[10] );
		}
		$obsinfo{ $obsnumber } = $obsnumber;
		$obsinfo{ $obsmaterial } = $obsmaterial;
	}
	my @obsconstrset = uniq( @obsconstr );
		
	return ( \@transpelts, \@geofilestruct, \%surfslist, \@obs, \@obsconstrset, \%datalist );
}

sub readverts
{
	# THIS READS THE VERTEX NUMBERS OF THE REQUESTED SURFACES IN A GEO FILE
	my @transpelts = @{$_[0]}; #say "transpelts ". dump(@transpelts);
	my $geofile = $_[1]; #say "geofile: " . dump($geofile);
	my @geodata = @{$_[2]}; #say "geodata " . dump(@geodata);
	my %datalist = %{$_[3]}; #say "%surfslist " . dump(%surfslist); 
	my @winselts;
	foreach my $transpelt (@transpelts)
	{
		#say "\$transpelt ". dump($transpelt);
		my $surfname = $transpelt->[0]; #say "$surfname ". dump($surfname);
		my $parent = $transpelt->[1]; #say "$parent ". dump($parent);
		my $constr = $transpelt->[2]; #say "$constr ". dump($constr);
		my $surfnum = $transpelt->[3]; #say "$surfnum ". dump($surfnum);
		my $geofile = $transpelt->[4]; #say "$geofile ". dump($geofile);
		my $zonenum = $transpelt->[5]; #say "$zonenum ". dump($zonenum);
		my @winelts;
		foreach my $datum (@geodata)
		{
			my @data = @$datum; # say "\@data ". dump(@data);
			if ( ($data[0] eq "*edges") and ( $data[$#data] == $surfnum ) )
			{
				push ( @winelts, [ [ @data[ 2..( $#data - 2 ) ] ], $surfnum ] ); 
				my @surfverts = @data[ 2..( $#data - 2 ) ]; say REPORT "\@surfverts ". dump(@surfverts);
				$datalist{ $zonenum }{ $surfnum }{vertnums} = [ @surfverts ];
			}
			#say "\@winelts ". dump(@winelts);
		}
		push ( @winselts, [ @winelts ] );
	}
	#say "winselts: " . dump(@winselts);
	return ( \@winselts, \%datalist );
}


sub readcoords
{
	# THIS READS THE COORDINATES OF THE REQUESTED VERTEX NUMBERS
	my ( $winseltsref, $geofile, $geodataref, $datalistref, $transpeltsref ) = @_; #say "\$geofile " . dump($geofile);
	my @winselts = @$winseltsref; #say REPORT "IN READCOORDS winselts: " . dump(@winselts);
	my @geodata = @$geodataref; #say REPORT "IN READCOORDS \@geodata " . dump(@geodata);
	my %datalist = %$datalistref; #say REPORT "IN READCOORDS \%datalist " . dump(%datalist);
	my @transpelts = @$transpeltsref; say REPORT "IN READCOORDS \@transpelts " . dump(@transpelts);
	my @allcoords;
	my $count = 1;
	foreach my $winseltref (@winselts)
	{
		my @transpelt = @{ $transpelts[ $count -1 ] }; say REPORT "SAYHERE \@transpelt ". dump(@transpelt);
		my $zonenum = $transpelt[5]; say REPORT "SAYHERE \$zonenum ". dump($zonenum);

		my @winselt = @$winseltref;
		my @vertnums = @{ $winselt[0][0] }; #say "SAY \@vertnums " . dump(@vertnums);
		my $surfnum = $winselt[0][1]; #say "SAY \$surfnum " . dump($surfnum);
		my @coords;
		foreach my $num (@vertnums)
		{
			
			foreach my $datum (@geodata)
			{
				my @data = @$datum;
				if ( ($data[0] eq "*vertex") and ( $data[5] == $num ) )
				{
					push ( @coords, [ [ @data[ 1..3 ] ], $num ] ); say REPORT "CONTROL \@coords: " . dump(@coords);
					$datalist{ $zonenum }{ $num }{vertcoords} = [ @data[ 1..3 ] ];
				}
			}
		}
		push ( @allcoords, [ @coords ] ); #say "\@allcoords: " . dump(@allcoords);
		$count++;
	}
	return (\@allcoords, \%datalist );
}

sub getcorners
{
	# THIS PACKS INTO SUBARRAYS THE X, Y, AND Z COORDINATES OF THE vertices OF THE REQUESTED SURFACES
	my ( $winscoordsref, $winseltsref ) = @_;
	my @winscoords = @$winscoordsref; #say "\@winscoords: " . dump(@winscoords);
	my @winselts = @$winseltsref; #say "\@winselts: " . dump(@winselts);
	my @packsurfsdata;
	my $countsurf = 0;
	foreach $surfcoordsref ( @winscoords )
	{
		my @surfcoords = @$surfcoordsref; #say "\@surfcoords: " . dump(@surfcoords);
		my ( @xdata, @ydata, @zdata );
		my @packsurfdata;
		my $surfnum = $winselts[$countsurf][0][1]; #say REPORT "HERE IMPORTANT \$surfnum " . dump($surfnum);
		foreach my $coordsetref (@surfcoords)
		{
			my @coordset = @$coordsetref; #say "\@coordset: " . dump(@coordset);
			push (@xdata, $coordset[0][0]);
			push (@ydata, $coordset[0][1]);
			push (@zdata, $coordset[0][2]);
		}
		#say "xdata: " . dump(@xdata); say "ydata: " . dump(@ydata); say "zdata: " . dump(@zdata);
		push (@packsurfdata, [ @xdata ], [ @ydata ], [ @zdata ], $surfnum  );
		push ( @packsurfsdata, [ @packsurfdata ] ); #say "\@packsurfsdata: " . dump(@packsurfsdata);
		$countsurf++;
	}
	say REPORT "HERE IMPORTANT \@packsurfsdata: " . dump(@packsurfsdata);
	return ( @packsurfsdata ); 
}

sub findextremes
{
	# THIS FINDS THE MAXIMA AND THE MINIMA FOR EACH COORDINATE FOR REQUESTED SURFACE
	my @xyzcoords = @_; say REPORT "HERE FINDEXTREMES \@xyzcoords: " . dump(@xyzcoords);
	my @surfsdata;
	foreach my $coordsdataref ( @xyzcoords )
	{
		my @coordsdata = @$coordsdataref; say REPORT "HERE FINDEXTREMES \@coordsdata: " . dump(@coordsdata);
		my $count = 0;
		my @surfdata;
		foreach $coordstyperef (@coordsdata)
		{
			if ($count < 3)
			{
				my @coordstype = @$coordstyperef; say REPORT "HERE FINDEXTREMES \@coordstype: " . dump(@coordstype);
				my $extreme1 = max(@coordstype); say REPORT "HERE FINDEXTREMES \$extreme1: " . dump($extreme1);
				my $extreme2 = min(@coordstype); say REPORT "HERE FINDEXTREMES \$extreme2: " . dump($extreme2);			
				my $countpos = 0;
				my (@extreme1positions, @extreme2positions);
				foreach my $elt ( @coordstype )
				{
					if ( $elt ~~ $extreme1 )
					{
						push ( @extreme1positions, $countpos );
					}
					if ( $elt ~~ $extreme2 )
					{
						push ( @extreme2positions, $countpos );
					}
					$countpos++;
				}
				
				push ( @surfdata, [ [ $extreme1, [ @extreme1positions ] ], [ $extreme2, [ @extreme2positions ] ] ] ); #say "\@surfdata: " . dump(@surfdata);
				$count++;				
			}
			else
			{	#say "\$surfdata[0][0][1]: " . dump( $surfdata[0][0][1] ); #say "\$surfdata[1][1][1]: " . dump( $surfdata[1][1][1] ); #say "\$surfdata[1][0]: " . dump( $surfdata[1][0] ); #say "\$surfdata[1][1]: " . dump( $surfdata[1][1] );
				if ( $surfdata[0][0][1] ~~ $surfdata[1][1][1] )
				{
					my $swap = $surfdata[1][1];
					$surfdata[1][1] = $surfdata[1][0];
					$surfdata[1][0] = $swap;
				}
				
				my $surfnum = $coordstyperef; #say "\$surfnum: " . dump($surfnum);
				push ( @surfdata, $surfnum ); #say "\@surfdata: " . dump(@surfdata);
			}
		}
		push (@surfsdata, [ @surfdata ] ); #say "\@surfsdata: " . dump(@surfsdata);
	}
	return ( @surfsdata ); #say "\@surfsdata: " . dump(@surfsdata);
}

sub makecoordsgrid
{
	# THIS FORMS A GRID OVER EACH REQUESTED SURFACE
	my ($extremesref, $resolutionsref, $dirsvectorsrefsref) = @_;
	my @extremesdata = @$extremesref; #say "\@extremesdata: " . dump(@extremesdata);
	my @resolutions = @$resolutionsref; #say "\@resolutions: " . dump(@resolutions);
	my @dirsvectorsrefs = @$dirsvectorsrefsref; #say "\@dirsvectorsrefs: " . dump(@dirsvectorsrefs);
	my @wholegrid;
	my $countsurf = 0;
	foreach my $surfcase ( @extremesdata )
	{
		my $dirsvectorsref = $dirsvectorsrefs[$countsurf]; #say "\$dirsvectorsref: " . dump($dirsvectorsref);
		my @surfdata = @$surfcase; #say "\@surfdata: " . dump(@surfdata);
		my $surf = pop @surfdata; #say "\$surf: " . dump($surf);
		my @coordspoints;
		my $count = 0;
		foreach ( @surfdata )
		{
			my $extreme1 = $_->[0][0]; #say "\$extreme1: " . dump($extreme1);
			my $extreme2 = $_->[1][0]; #say "\$extreme2: " . dump($extreme2);
			my @extreme1positions = @{$_->[0][1]}; #say "\@extreme1positions: " . dump(@extreme1positions);
			my @extreme2positions = @{$_->[1][1]}; #say "\@extreme2positions: " . dump(@extreme2positions);
			my $resolution = $resolutions[$counter]; #say "\$resolution: " . dump($resolution);
			my $diffextremes = ( $extreme1 - $extreme2 ); #say "\$diffextremes: " . dump($diffextremes);
			my $variation = ( $diffextremes / ( $resolution + 1) ); #say "\$variation: " . dump($variation);
			my @coordpoints;
			my $othercount = 1;
			while ( $othercount < ( $resolution +1 ) )
			{
				my $compoundvariation = ( $variation * $othercount ); #say "\$compoundvariation: " . dump($compoundvariation);
				my $coordvalue = ( $extreme2 + $compoundvariation ); #say "\$coordvalue: " . dump($coordvalue);
				push ( @coordpoints, $coordvalue ); #say "\@coordpoints: " . dump(@coordpoints);
				$othercount++;
			}
			push ( @coordspoints, [ @coordpoints ] ); #say "\@coordspoints: " . dump(@coordspoints);
			$count++;
		}
		push ( @coordspoints, $surf, $dirsvectorsref ); #say "\@coordspoints: " . dump(@coordspoints);
		push ( @wholegrid, [ @coordspoints ] ); #say "\@wholegrid: " . dump(@wholegrid);
		$countsurf++;
	}
	return(@wholegrid);
}

sub makegrid
{
	# THIS CONVERTS THE GRID DATA IN VERTEX FORM
	my @gridcoords = @_; #say REPORT "HERE IN MAKE GRID \@gridcoords:" . dump(@gridcoords);
	my @gridsvertices; 
	foreach my $surfdataref ( @gridcoords )
	{
		my @xyzcoords;
		my @surfdata = @$surfdataref; #say "\@surfdata:" . dump(@surfdata);
		my @xdata = @{$surfdata[0]}; #say "\@xdata:" . dump(@xdata);
		my @ydata = @{$surfdata[1]}; #say "\@ydata:" . dump(@ydata);
		my @zdata = @{$surfdata[2]}; #say "\@zdata:" . dump(@zdata);
		my $surf = $surfdata[3]; #say "\$surf:" . dump($surf);
		my $dirvectorsref = $surfdata[4]; #say "\$dirvectorsref:" . dump($dirvectorsref);
		my $counter = 0;
		my @gridvertices;
		my ( @groups, @xyzdata );
		foreach my $xdatum (@xdata)
		{
			my $ydatum = $ydata[$counter]; #say "\$ydatum:" . dump($ydatum);
			push ( @xyzdata, [ $xdatum, $ydatum ] ); #say "\@xyzdata:" . dump(@xyzdata);
			$counter++;
		}
		foreach my $elt (@xyzdata)
		{
			foreach my $zdatum ( @zdata )
			{
				my @group = @$elt;
				push ( @group, $zdatum ); #say "\@group: " . dump(@group);
				push ( @groups, [ @group ] ); #say "\@groups" . dump(@groups);
			}
		}
		push ( @gridvertices, [ @groups ], $surf, $dirvectorsref ); #say "\@gridvertices:" . dump(@gridvertices);
		push ( @gridsvertices, [ @gridvertices ] ); #say "\@gridsvertices:" . dump(@gridsvertices);
	}
	return ( @gridsvertices );
}

sub adjust_dirvector
{	# IT SCALES THE DIRECTION VECTORS TO EASE THE MANIPULATION OF THE GRIDS IN DETACHING THEM FROM SURFACES.
	# IT IS CALLED FROM sub adjustgrid.	
	my ( $vectorref, $distgrid ) = @_;
	my @vector = @$vectorref;
	my $denominator = ( 1 / $distgrid );
	my @adjusted_vector;
	foreach my $elt ( @vector )
	{
		my $adjusted_component = ( $elt / $denominator );
		$adjusted_component = sprintf ( "%.3f", $adjusted_component );
		push ( @adjusted_vector, $adjusted_component );
	}
	return ( @adjusted_vector );
}

sub adjustgrid
{	# IT ADJUSTS THE GRIDS OF POINTS OVER THE REQUESTED SURFACES BY DETACHING THEM OF ABOUT 1 CM TOWARDS THE OUTSIDE.
	# IT IS CALLED FROM THE MAIN PROGRAM.
	my ( $griddataref, $distgrid )  = @_;
	my @griddata = @$griddataref;
	my @adjustedsurfs;
	foreach my $elt ( @griddata )
	{
		my @surfdata = @$elt; #say "\@surfdata " . dump(@surfdata);
		my @vertexdatarefs = @{$surfdata[0]}; #say "\@vertexdatarefs " . dump(@vertexdatarefs);
		my $surfnum = $surfdata[1]; #say "\$surfnum " . dump($surfnum);
		my @dirvector = @{$surfdata[2]}; #say "\@dirvector " . dump(@dirvector);
		my @adjusted_dirvector = adjust_dirvector( \@dirvector, $distgrid ); #say "\@adjusted_dirvector: " . dump(@adjusted_dirvector);
		my @adjustedsurfs;		
		foreach my $vertexref ( @vertexdatarefs )
		{
			my @vertexcoords = @$vertexref;
			my @adjustedvertex;
			$countcomp = 0;
			foreach my $el ( @vertexcoords )
			{
				my $component = $adjusted_dirvector[$countcomp]; #say "\$component " . dump($component);
				my $newel = ( $el + $component ); #say "\$el " . dump($el); say "\$newel " . dump($newel);
				push ( @adjustedvertex, $newel ); #say "\$countcomp " . dump($countcomp);
				$countcomp++; 
			}
			push ( @adjustedsurfs, [ @adjustedvertex ] );

		}
		push ( @adjusteddata, [ [ @adjustedsurfs ], $surfnum, [ @dirvector ] ] );
	}
	return ( @adjusteddata );
}

sub treatshdfile
{
	#say "GOT!";
	my @lines = @_;
	my @newlines;
	my $count = 0;
	foreach my $line ( @lines )
	{
		my $lineafter = $lines[ $count + 1 ]; #say "\$lineafter: $lineafter";
		my $linebefore = $lines[ $count - 1 ]; #say "\$linebefore: $linebefore";
		my $linecontrol = $line;
		if ( ( $lineafter =~ /# direct - surface/ ) or ( $lineafter =~ /# diffuse - surface/ ) )
		{
			$linecontrol = "";
		}
		elsif ( ( $line =~ /# direct - surface/ ) or ( $line =~ /# diffuse - surface/ ) )
		{
			chomp $linebefore; #say "\$linebefore__: $linebefore"; say "\$line: $line";
			$line = "$linebefore" . " " . "$line" ; #say "\$joinline $line"; 
		}
				
		unless ( $linecontrol eq "" )
		{
			push ( @newlines, $line );
		}
		$count++;
	}
	return ( @newlines );
}
		

sub readshdfile
{
	my ( $shdfile ) = @_; #say "\$shdfile " . dump($shdfile);
	my $shdafile = $shdfile . "a"; #say "\$shdafile " . dump($shdafile);
	my $tempfile = $shdafile; #say "\$tempfile " . dump($tempfile);
	$tempfile =~ s/\.shda/\.temp\.shda/ ; #say "\$tempfile " . dump($tempfile);
	open (SHDAFILE, "$shdafile") or die;
	my @shdalines = <SHDAFILE>;
	close SHDAFILE,
	
	my (@filearray, @rawlines, @months);
	foreach my $line ( @shdalines )
	{
		push ( @rawlines, $line );
	}
	my @treatedlines = treatshdfile ( @rawlines ); #say "\@treatedlines " . dump(@treatedlines);
	
	foreach my $line ( @treatedlines )
	{
		my @elts = split(/\s+|,/, $line);
		if ( $line =~ /\* month:/ )
		{
			my $month = $elts[2]; #say "\$month " . dump($month);
			####$month =~ s/`//g;
			push ( @months, $month );
		}
		push ( @filearray, [ @elts ] );
	} #say "\@months " . dump(@months);
	
	open ( TEMP , ">$tempfile" ) or die;
	foreach my $line ( @treatedlines )
	{
		print TEMP $line;
	}
	close TEMP;
	return ( \@treatedlines, \@filearray, \@months );
}

sub tellsurfnames
{
	my ( $transpsurfsref, $geodataref ) = @_;
	my @transpsurfs = @$transpsurfsref; say REPORT "HERE \@transpsurfs " . dump(@transpsurfs);
	my @geodata = @$geodataref; #say REPORT "\@geodata " . dump(@geodata);
	my ( @containernums, @containernames, @nums, @names );
	my $count = 0;
	foreach my $surf ( @transpsurfs )
	{
		foreach my $rowref ( @geodata )
		{
			my @row = @$rowref;
			if ( ( $surf eq $row[12] ) and ( $row[0] eq "*surf" ) )
			{
				push ( @nums, $surf );
				push ( @names, $row[1] );
			}
			$count++;
		}
	}
	return ( \@nums, \@names );
}

sub getsurfshd
{
	my ( $shdfilearrayref, $monthsref, $surfnumsref, $surfnamesref ) = @_;
	my @shdfilearray = @$shdfilearrayref; #say "\@shdfilearray " . dump(@shdfilearray);
	my @months = @$monthsref; #say "\@months " . dump(@months);
	my @surfnums = @$surfnumsref; say REPORT "HERE IN GETSURFSSHD \@surfnums " . dump(@surfnums);
	my @surfnames = @$surfnamesref; say REPORT"HERE IN GETSURFSSHD \@surfnames " . dump(@surfnames);

	my @yearbag;
	foreach my $month ( @months )
	{
		my $semaphore = 0;
		my @monthbag;
		foreach my $rowref ( @shdfilearray )
		{
			my @row = @$rowref; #say "\@row " . dump(@row);
			if ( ( $row[0] eq "*") and ( $row[1] eq "month:" ) and ( $row[2] eq "$month" ) )
			{
				$semaphore = 1; 
			}
			elsif ( ( $row[0] eq "*") and ( $row[1] eq "month:" ) and ( not ( $row[2] eq "$month" ) ) )
			{
				$semaphore = 0; 
			}
			foreach my $surfname ( @surfnames )
			{
				if ( ( $row[25] eq "diffuse") and ( $row[27] eq "surface") and ( $row[28] eq "$surfname" ) and ( $semaphore == 1 ) )
				{
					push ( @monthbag, [ [ @row[0..23] ], $surfname ] ); #say "\@monthbag " . dump(@monthbag);
				}
			}
		}
		push ( @yearbag, [ [ @monthbag ], $month ] ); 
	}
	#say "\@yearbag " . dump(@yearbag);
	return ( @yearbag );		
}

sub checklight
{
	my ( $shdfilearrayref, $monthsref ) = @_;
	my @shdfilearray = @$shdfilearrayref; #say "\@shdfilearray " . dump(@shdfilearray);
	my @months = @$monthsref; #say "\@months " . dump(@months);

	my @yearbag;
	foreach my $month ( @months )
	{
		my $semaphore = 0;
		my @monthbag;
		foreach my $rowref ( @shdfilearray )
		{
			my @row = @$rowref; #say "\@row " . dump(@row);
			if ( ( $row[0] eq "*") and ( $row[1] eq "month:" ) and ( $row[2] eq "$month" ) )
			{
				$semaphore = 1; 
			}
			elsif ( ( $row[0] eq "*") and ( $row[1] eq "month:" ) and ( not ( $row[2] eq "$month" ) ) )
			{
				$semaphore = 0;
			}
			if ( ( $row[25] eq "diffuse") and ( $row[27] eq "surface") and ( $semaphore == 1 ) )
			{
				push ( @monthbag, [ @row[0..23] ] ); #say "\@monthbag " . dump(@monthbag);
			}
		}
		push ( @yearbag, [ [ @monthbag ], $month ] ); 
	}
	#say "\@yearbag " . dump(@yearbag);
	
	my @yearbox;
	foreach my $monthsref ( @yearbag )
	{
		my @monthstrans = @$monthsref;
		my @months = @{$monthstrans[0]}; #say "\@months " . dump(@months);
		my $month = $monthstrans[1];
		my @montharray = ( 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
		1.0000, 1.0000, 1.0000, 1.0000, 1.0000,	1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000); 
		my @monthbox;
		foreach my $monthref ( @months )
		{
			my @elts = @$monthref; #say "\@elts " . dump(@elts);
			my $counthour = 0;
			foreach my $elt ( @elts )
			{
				if ( ( $montharray[$counthour] == 1.0000 ) and ( $elt < 1 ) )
				{
					$montharray[$counthour] = 0.0000;
				}
				$counthour++;
			}
		}
		push ( @monthbox, [ @montharray ] ); #say "\@monthbox " . dump(@monthbox);
		push ( @yearbox, [ [ @monthbox ], $month ] ); #say "\@yearbox " . dump(@yearbox);#say "\@yearbox " . dump(@yearbox);
	} 	
	return ( @yearbox ); 	
}

sub tellradfilenames
{
	my ( $path, $conffile_fict1, $conffile_fict2, $conffile_fict3 ) = @_;
	my @confs = ( $conffile_fict1, $conffile_fict2, $conffile_fict3 );
	my @result;
	foreach my $conf ( @confs )
	{
		my $confstripped = $conf;
		$confstripped =~ s/$path\/cfg\///;
		$confstripped =~ s/.cfg//;
		my $radoct = "$confstripped" . "_Extern.oct";
		my $rcffile = "$confstripped" . ".rcf" ;
		push ( @result, [ $conf, $radoct, $rcffile ] );
	}
	return( @result );
}

sub setrad
{
	#print REPORT "SETRAD\n"; print "SETRAD\n";
	my ( $conffile, $radoctfile, $rcffile, $path, $monthnum, $day, $hour ) = @_;
print REPORT "cd $path/rad/
e2r -file $conffile -mode text<<YYY
b
$rcffile
d

d
$day $monthnum $hour
g
-
-
YYY
";

`cd $path/rad/
e2r -file $conffile -mode text<<YYY
b
$rcffile
d

d
$day $monthnum $hour
g
-
-
YYY
`; #print REPORT "END SETRAD\n"; print "END SETRAD\n";
}

sub createrad
{
	my ( $conffile, $radoctfile, $rcffile, $path ) = @_;
	my $pathrcffile = "$path/rad/$rcffile";	
	
	unless ( -e $pathrcffile ) 
	{
		#say REPORT "CREATERAD\n"; say "CREATERAD\n";
print REPORT "cd $path/rad/
e2r -file $conffile -mode text<<YYY
b
$rcffile
a
a
a
n
h
c
a
d
a
f
c
h
y
>

-
-
YYY
";

`cd $path/rad/
e2r -file $conffile -mode text<<YYY
b
$rcffile
a
a
a
n
h
c
a
d
a
f
c
h
y
>

-
-
YYY
`;
	}
	#else 
	#{
	#	die "THE RCF DIRECTORY IS NOT CLEAN!";
	#}
	#say REPORT "END CREATERAD\n"; say "END CREATERAD\n";
}

sub setroot
{
	#say REPORT "SETROOT\n"; say "SETROOT\n";
	my ( $conffile, $path ) = @_;
	my $rootname = $conffile;
	$rootname =~ s/$path\/cfg\///;
	$rootname =~ s/\.cfg//;
print REPORT "cd $path/cfg
prj -file $conffile -mode text<<YYY
s

$rootname

-
YYY
";

`cd $path/cfg
prj -file $conffile -mode text<<YYY
s

$rootname

m
c
b
#
y
-
-
-
-

YYY
`; #say REPORT "END SETROOT\n"; say "END SETROOT\n";
}

sub populatelight
{
	my @daylighthoursarr = @_;
	my %daylighthours;
	my $count = 0;
	foreach my $monthref ( @daylighthoursarr )
	{
		my @monthdata = @$monthref;
		my $month = $monthdata[1];
		$month =~ s/`//g;
		my @lithours = @{$monthdata[0][0]};
		$daylighthours{$month} = [ @lithours ] ;
		$count++;
	} #say "\%daylighthours\ " . dump(%daylighthours);
	return ( %daylighthours );
}

sub rotate2d
{
    my ( $x, $y, $angle ) = @_;
    my $x_new = cos($angle)*$x - sin($angle)*$y;
    my $y_new = sin($angle)*$x + cos($angle)*$y;
  return ( $x_new, $y_new);
}

sub getdirvectors
{
	 ( $basevectorsref, $dirvectorref, $pointcoordsref ) = @_;
	 my @basevectors = @$basevectorsref; #say "\@basevectors\: " . dump(@basevectors); 
	 my @dirvector = @$dirvectorref; #say "\@dirvector\: " . dump(@dirvector); 
	 my @topcoords = @{$basevectors[0]}; #say "\@topcoords\: " . dump(@topcoords); 
	 my @newdirvectors;
	 my $xbase = $topcoords[0]; #say "\$xbase\: " . dump($xbase); 
	 my $ybase = $topcoords[1]; #say "\$ybase\: " . dump($ybase); 
	 my $zbase = $topcoords[2]; #say "\$zbase\: " . dump($zbase); 
	 my $xnew = $dirvector[0]; #say "\$xnew\: " . dump($xnew); 
	 my $ynew = $dirvector[1]; #say "\$ynew\: " . dump($ynew); 
	 my $znew = $dirvector[2]; #say "\$znew\: " . dump($znew); 
	 my $anglebasexz = acos($xbase); #say "\$anglebasexz\: " . dump($anglebasexz); $anglebasexz_deg = rad2deg($anglebasexz);say "\$anglebasexz_deg\: " . dump($anglebasexz_deg);
	 my $anglebaseyz = acos($zbase); #say "\$anglebaseyz\: " . dump($anglebaseyz); $anglebaseyz_deg = rad2deg($anglebaseyz); say "\$anglebaseyz_deg\: " . dump($anglebaseyz_deg);
	 my $anglenewxz = acos($xnew); #say "\$anglenewxz\: " . dump($anglenewxz); $anglenewxz_deg = rad2deg($anglenewxz); say "\$anglenewxz_deg\: " . dump($anglenewxz_deg); 
	 my $anglenewyz = acos($znew); #say "\$anglenewyz\: " . dump($anglenewyz); $anglenewyz_deg = rad2deg($anglenewyz); say "\$anglenewyz_deg\: " . dump($anglenewyz_deg);
	 my $anglediffxz = ( $anglenewxz - $anglebasexz ); #say "\$anglediffxz\: " . dump($anglediffxz); 
	 my $anglediffyz = ( $anglenewyz - $anglebaseyz ); #say "\$anglediffyz\: " . dump($anglediffyz); 
	 foreach my $eltsref ( @basevectors )
	 {
	 	my @elts = @$eltsref;
	 	my ( $x, $y, $z ) = @elts ;
	 	my ( $x_ok, $tempy ) = rotate2d( $x, $y, $anglediffxz );
	 	my ( $y_ok, $z_ok ) = rotate2d( $tempy, $z, $anglediffyz );
	 	$x_ok = sprintf ( "%.3f", $x_ok );
	 	$y_ok = sprintf ( "%.3f", $y_ok );
	 	$z_ok = sprintf ( "%.3f", $z_ok );
	 	push ( @newdirvectors, [ $x_ok, $y_ok, $z_ok ] );
	 }
	 #say "\@newdirvectors\: " . dump(@newdirvectors); 
	 return ( @newdirvectors );
}

sub pursue
{	# RADIANCE EXAMPLE: echo 1 -0.01 2 0 -1 0 | rtrace  -I -ab 2 -h /home/luca/boxform/rad/boxform_fict1_Extern.oct | rcalc -e '$1=179*(.265*$1+.670*$2+.065*$3)'
	my $exportconstrref = shift, 
	my $exportreflref = shift; 
	my $conffile = shift;
	my $basevectorsref = shift;
	my @basevectors = @$basevectorsref;
	my $path = shift; #say "\$path\ " . dump($path); 
	my $fict1ref = shift; #say "\$fict1ref\ " . dump($fict1ref); 
	my $fict2ref = shift; #say "\$fict2ref\ " . dump($fict2ref); 
	my $fict3ref = shift; #say "\$fict2ref\ " . dump($fict2ref); 
	my $bounceambnum = shift; #say "\$bounceambnum\: " . dump($bounceambnum); 
	my $bouncemaxnum = shift; #say "\$bouncemaxnum\: " . dump($bouncemaxnum); 
	my ( $conffile_fict1, $radoctfile_fict1, $rcffile_fict1 ) = @$fict1ref;
	my ( $conffile_fict2, $radoctfile_fict2, $rcffile_fict2 ) = @$fict2ref;
	#my ( $conffile_fict3, $radoctfile_fict3, $rcffile_fict3 ) = @$fict3ref;
	my @conffiles = ( $conffile_fict1, $conffile_fict2, $conffile_fict3 ); #say "\@conffiles\ " . dump(@conffiles); 
	my @radoctfiles = ( $radoctfile_fict1, $radoctfile_fict2  ); #say "HERE \@radoctfiles\ " . dump(@radoctfiles); my @radoctfiles = ( $radoctfile_fict1, $radoctfile_fict2, $radoctfile_fict3  );
	my @rcffiles = ( $rcffile_fict1, $rcffile_fict2 ); #say "HERE \@rcffiles\ " . dump(@rcffiles); my @rcffiles = ( $rcffile_fict1, $rcffile_fict2, $rcffile_fict3 );
	
	my @zonedata = @_; say REPORT "HERE PURSUE \@zonedata\ " . dump(@zonedata); 
	my $fileroot = $radoctfile_fict1;
	$fileroot =~ s/_fict1_Extern.oct//; #say REPORT "FILEROOT $fileroot";
	my ( %irrs, %irrs_dir, %irrs_amb);
	my $setoldfiles = "on";
	#say REPORT "\@zonedata\ " . dump(@zonedata) . " \$countzone $countzone, \$countrad $countrad, \$countmonth $countmonth, \$countsurf $countsurf, \$counthour $counthour, \$countpoint $countpoint ";
		

	my ( $casedataref , $gridpointsref, $shdsurfdataref, $daylighthoursref, $shdfilearrayref  ) = @{ $zonedata[0] }; #say "\$daylighthoursref\ " . dump($daylighthoursref);
	my ( $zonenum, $geofile, $constrfile, $shdfile ) = @$casedataref; say "\$zonenum\ " . dump($zonenum); say "\$geofile\ " . dump($geofile); say "\$constrfile\ " . dump($constrfile); say "\$shdfile\ " . dump($shdfile);
	my @gridpoints = @$gridpointsref; say REPORT "HERE PURSUE \@gridpoints\ " . dump(@gridpoints);
	my @shdsurfdata = @$shdsurfdataref; #say "\@shdsurfdata\ " . dump(@shdsurfdata);
	my @daylighthoursarr = @$daylighthoursref; #say "\@daylighthoursarr\ " . dump(@daylighthoursarr);
	my @shdfilearray = @$shdfilearrayref; #say "\@shdfilearray\: " . dump(@shdfilearray);
	#say REPORT "\@radoctfiles\ " . dump(@radoctfiles) . " \$countzone $countzone, \$countrad $countrad, \$countmonth $countmonth, \$countsurf $countsurf, \$counthour $counthour, \$countpoint $countpoint "; 
	my $countrad = 0;
	foreach my $radoctfile ( @radoctfiles )
	{	#say "COUNTRAD! $countrad";
		#say REPORT "\@daylighthoursarr\ " . dump(@daylighthoursarr); 
		my $countmonth = 0;
		foreach my $monthsdataref ( @daylighthoursarr )
		{ 	#say REPORT "\$monthsdataref " . Dumper($monthsdataref);
			my @monthsdata = @$monthsdataref; #say REPORT "\@monthsdata\: " . dump(@monthsdata);
			my @hourslight = @{$monthsdata[0][0]}; #say REPORT "\@hourslight!\: " . dump(@hourslight) . " \$countzone $countzone, \$countrad $countrad, \$countmonth $countmonth, \$countsurf $countsurf, \$counthour $counthour, \$countpoint $countpoint ";
			my $month = $monthsdata[1]; 
			$month =~ s/`//g; #say "\$month\: " . dump($month);
			my $monthnum = getmonthnum( $month );					
			my $day = $days_inmonths{ $month }; #say "\$day\: " . dump($day);
			my ( @surftests, @surftests_dir, @surftests_amb );
			my $countsurf = 0;
			
			#say REPORT "\@gridpoints\ " . dump(@gridpoints) . " \$countzone $countzone, \$countrad $countrad, \$countmonth $countmonth, \$countsurf $countsurf, \$counthour $counthour, \$countpoint $countpoint ";
			foreach my $surfref ( @gridpoints )
			{	#say "COUNTSURF! $countsurf";
				#say REPORT "\$surfref " . Dumper($surfref);	
				my @surfdata = @$surfref;
				my @pointrefs = @{ $surfdata[0] }; #say "\@pointrefs\: " . dump(@pointrefs);
				my $surfnum = $surfdata[1]; say REPORT " HERE PURSUE \$surfnum\: " . dump($surfnum);
				my @dirvector = @{ $surfdata[2] }; say REPORT " HERE PURSUE \@dirvector\: " . dump(@dirvector);
				my ( $dirvectorx, $dirvectory, $dirvectorz ) = @dirvector;					
				my $conffile = $conffiles[$countrad]; say REPORT " HERE PURSUE \$conffile\: " . dump($conffile);
				my $radoctfile = $radoctfiles[$countrad]; say REPORT " HERE PURSUE \$radoctfile\: " . dump($radoctfile);
				my $rcffile = $rcffiles[$countrad]; say REPORT " HERE PURSUE \$rcffile\: " . dump($rcffile);

				my $counthour = 0; 
				foreach my $hourlight ( @hourslight )
				{	#say "COUNTHOUR! $counthour";	
					if ( $hourlight != 1 )
					{	#say REPORT "HEREOUT \@gridpoints\ " . dump(@gridpoints) . " \$countzone $countzone, \$countrad $countrad, \$countmonth $countmonth, \$countsurf $countsurf, \$counthour $counthour, \$countpoint $countpoint ";
						my $hour = ( $counthour + 1) ; 
						
						my $countpoint = 0;
						if ( ( $countmonth == 0 ) and ( $countsurf == 0 ) and ( $countpoint == 0 ) and ( $countpoint == 0 ) and ( $setoldfiles = "on" ) )
						{	setroot( $conffile, $path);
							createrad( $conffile, $radoctfile, $rcffile, $path );
							if ( $countrad == 1)
							{
								adjust_radmatfile( $exportconstrref, $exportreflref, $conffile, $path );
							}
							$setoldfiles = "off";
						}
						setrad( $conffile, $radoctfile, $rcffile, $path, $monthnum, $day, $hour ); #say REPORT "\$fictitious model number: " . ($countrad + 1) . "\$conffile: $conffile, \$radoctfile: $radoctfile, \$rcffile: $rcffile, $path, ". ", \$surface_number: $surfnum, \$month: $month, \$day: $day, \$hour: $hour";	
						
						#say REPORT "\@pointrefs " . Dumper(@pointrefs);	
						foreach my $pointref ( @pointrefs )
						{	#say REPORT "\$pointref " . Dumper($pointref);	
							my @pointcoords = @$pointref; #say REPORT "\@pointcoords\: " . dump(@pointcoords);
							my ( $xcoord, $ycoord, $zcoord ) = @pointcoords;
							my $raddir = "$path/rad/";
							my $cfgpath = "$path/cfg/";
							my @dirvgroup = getdirvectors ( \@basevectors, \@dirvector );
							my $countdirvec = 0;
							foreach my $dirvector ( @dirvgroup )
							{ 
								my ( $valstring, $x, $y, $z, $irr );
								if ( ( $countrad == 0 ) or ( $countrad == 1 ) )
								{
									$valstring = `cd $raddir \n echo $xcoord $ycoord $zcoord $dirvectorx $dirvectory $dirvectorz | rtrace  -I -ab $bounceambnum -lr $bouncemaxnum -h $radoctfile`; say REPORT "TO SHELL: cd $raddir \n echo $xcoord $ycoord $zcoord $dirvectorx $dirvectory $dirvectorz | rtrace  -I -ab $bounceambnum -lr $bouncemaxnum -h $radoctfile";
									( $x, $y, $z ) = ( $valstring =~ m/(.+)\t(.+)\t(.+)\t/ ); say REPORT "VALSTRING: " . dump( $valstring ); say REPORT "x: $x, y: $y, z: $z"; 
									$irr = ( 179 * ( ( .265 * $x ) + ( .670 * $y ) + ( .065 * $z ) ) ); say REPORT "\$irr\: $irr, ";
								}
								#elsif ( ( $countrad == 2 ) and ( $countpoint == 0) )
								#{
								#	$valstring = `cd $raddir \n echo $xcoord $ycoord $zcoord $dirvectorx $dirvectory $dirvectorz | rtrace  -I -ab 0 -h $radoctfile`; say REPORT "TO SHELL: cd $raddir \n echo $xcoord $ycoord $zcoord $dirvectorx $dirvectory $dirvectorz | rtrace  -I -ab 0 -h $radoctfile";
								#	( $x, $y, $z ) = ( $valstring =~ m/(.+)\t(.+)\t(.+)\t/ ); say REPORT "VALSTRING: " . dump( $valstring ); say REPORT "x: $x, y: $y, z: $z"; 
								#	$irr = ( 179 * ( ( .265 * $x ) + ( .670 * $y ) + ( .065 * $z ) ) ); say REPORT "\$irr\: $irr, ";
								#}

								push ( @surftests, $irr );

								$countdirvec++;
							}
							$countpoint++;
						} #say "\@surftests\: " . dump(@surftests);
						my $meanvaluesurf;
						if ( @surftests )
						{ 
							$meanvaluesurf = _mean_( @surftests ); say "Calculating irradiance: $meanvaluesurf, for surface $surfnum, zone $zonenum, month $monthnum, day $day, hour $hour, octree $radoctfile.\n" ;

						} 
													
						if ( $meanvaluesurf and $surfnum and $hour )
						{
							$irrs{ $zonenum }{ $countrad + 1 }{ $monthnum }{ $surfnum }{ $hour }{ meanirr } = $meanvaluesurf;
						}
					}
					$counthour++;					
				}
				$countsurf++;
			}
			$countmonth++;
		}
		$countrad++;
	} say REPORT " HERE PURSUE \%irrs\: " . dump(%irrs);
	return ( \%irrs ); #return ( \%irrs, \%irrs_dir, \%irrs_amb ); 
}


sub cleanblanks
{
	my @arr = @_;
	my @box;
	my $count = 0;
	foreach my $el ( @arr )
	{
		unless ( $el eq "" )
		{
			push ( @box, $el );
		}
		$count++;
	}
	return( @box );
}

sub createconstrdbfile
{
	my ( $constrdbfile, $constrdbfile_fict, $obsconstrsetref ) = @_; #say REPORT "BEGINNING: \$constrdbfile, $constrdbfile";
	my @obsconstrset = @$obsconstrsetref; #say REPORT "\@obsconstrset " . dump(@obsconstrset);
	my ( @bigcopy, @updatedlines );
	open ( DBFILE, "$constrdbfile" ) or die; 
	my @lines = <DBFILE>; #say REPORT "\@lines " . dump(@lines);
	close DBFILE;	
	
	my $topline;
	my $countline = 0;
	foreach my $line ( @lines )
	{
		my @row = split( /\s+|,/ , $line); #say REPORT "countline: $countline, row: " . dump( @row );
		@row = cleanblanks( @row ); # say REPORT "countline: $countline, row: " . dump( @row );
		my ( $oldnumber, $newnumber );
		if ( $countline == 2 )
		{
			$oldnumber = $row[0]; #say REPORT "oldnumber: $oldnumber";
			$newnumber = ( $oldnumber + scalar( @obsconstrset ) ); #say REPORT "newnumber: $newnumber";
			$line =~ s/$oldnumber/$newnumber/;
			push ( @updatedlines, $line );
		}
		else 
		{
			push ( @updatedlines, $line );
		}
		$countline++;
	} #say REPORT "updatedlines " . dump(@updatedlines);
	
	foreach my $el ( @obsconstrset )
	{
		my @copy;
		my $semaphore = 0;
		$countel = 0;
		$countline = 0;
		foreach my $line ( @updatedlines )
		{
			my @row = split( /\s+|,/ , $line); #say REPORT "countline: $countline, row: " . dump( @row );
			@row = cleanblanks( @row ); # say REPORT "countline: $countline, row: " . dump( @row );
			if ( $countline > 2 )
			{
				if ( $el eq $row[1] )
				{
					$semaphore = 1; #say REPORT "HIT. semaphore on";
				}
					
				if ( ( $semaphore == 1 ) and ( $countel == 0) )
				{
					push ( @copy, "# layers  description   optics name   symmetry tag\n" );
					push ( @copy, $line ); #say REPORT "pushing first, $countel"; say REPORT "\@copy " . dump(@copy);
					$countel++;
				}
				elsif ( ( $semaphore == 1 ) and ( $countel > 0) )
				{
					push ( @copy, $line ); #say REPORT "pushing other, $countel"; say REPORT "\@copy " . dump(@copy);
					if (  ( $row[0] eq "#" ) and ( $row[1] eq "layers" ) and ( $row[2] eq "description" )  )
					{
						pop(@copy);
						$semaphore = 0; #say REPORT "semaphore off";
					}
					$countel++;
				}
				
			}
			$countline++;
		}
		push ( @bigcopy, [ @copy ] ); #say REPORT "\@bigcopy " . dump(@bigcopy);
	} #say REPORT "\@bigcopy" . dump(@bigcopy);
	
	my ( @materials, @newmaterials, @newcopy, @newbigcopy );
	my ( $newmatinssurf, $newmatextsurf );
	my %exportconstr;
	foreach my $copyref ( @bigcopy )
	{	#say REPORT "DOING 1";
		my @constrlines = @$copyref; 
		my $firstrow = $constrlines[1]; #say REPORT "\$firstrow " . dump($firstrow);
		my @row = split ( /\s+|,/ , $firstrow );
		@row = cleanblanks( @row );
		my $constrname = $row[1]; #say REPORT "\$constrname " . dump($constrname);
		my $newconstrname = $constrname;
		$newconstrname =~ s/\w\b// ;  
		$newconstrname =~ s/\w\b// ;  
		$newconstrname = "f_" . "$newconstrname"; #say REPORT "\$newconstrname " . dump($newconstrname);
		
		my $intlayer = $constrlines[3]; #say REPORT "\$intlayer " . dump($intlayer);
		my @row = split ( /\s+|,/ , $intlayer );
		@row = cleanblanks( @row ); 
		my $matintlayer = $row[2]; #say REPORT "\$matintlayer " . dump($matintlayer);
		my $newmatintlayer = $matintlayer;
		$newmatintlayer =~ s/\w\b// ;  
		$newmatintlayer =~ s/\w\b// ;  
		$newmatintlayer = "f_" . "$newmatintlayer"; #say REPORT "\$newmatintlayer " . dump($newmatintlayer);
		my $extlayer = $constrlines[$#constrlines];  #say REPORT "\$extlayer " . dump($extlayer);
		my @row = split ( /\s+|,/ , $extlayer );
		@row = cleanblanks( @row ); 
		my $matextlayer = $row[2]; #say REPORT "\$matextlayer " . dump($matextlayer);
		my $newmatextlayer = $matextlayer;
		$newmatextlayer =~ s/\w\b// ; 
		$newmatextlayer =~ s/\w\b// ;  
		$newmatextlayer = "f_" . "$newmatextlayer"; #say REPORT "\$newmatextlayer $newmatextlayer";
		push ( @materials, $matintlayer, $matextlayer ); 
		push ( @newmaterials, $newmatintlayer, $newmatextlayer );
		
		#say REPORT "\$constrlines[3] " . dump( $constrlines[3] );
		$constrlines[1] =~ s/$constrname/$newconstrname/g; #say REPORT "\$constrlines[3] " . dump( $constrlines[3] );
		$constrlines[3] =~ s/$matintlayer/$newmatintlayer/g; #say REPORT "\$constrlines[3] " . dump( $constrlines[3] ); 
		$constrlines[$#constrlines] =~ s/$matextlayer/$newmatextlayer/g; #say REPORT "\$constrlines[\$#constrlines] " . dump($constrlines[$#constrlines]);
		foreach my $line ( @constrlines )
		{
			push ( @newcopy, $line );
		}
		push ( @newbigcopy, [ @newcopy ] );
		$exportconstr{ $newconstrname }{ extlayer } = $newmatextlayer;
		$exportconstr{ $newconstrname }{ intlayer } = $newmatintlayer;
	} #say REPORT "DONE 1";
	
	@materials = uniq( @materials ); #say REPORT "materials: " . dump( @materials );
	@newmaterials = uniq( @newmaterials ); #say REPORT "newmaterials: " . dump( @newmaterials );
	#say REPORT "newbigcopy: " . dump( @newbigcopy );
	
	my %newmatnums;
	my $countmat = 1;
	foreach ( @newmaterials )
	{
		$newmatnums{ $_ } = $countmat;
		$countmat++;
	} #say REPORT "\%newmatnums " . Dumper( %newmatnums );
	
	my %matnums;
	$countmat = 1;
	foreach ( @materials )
	{
		$matnums{ $_ } = $countmat ;
		$countmat++;
	} #say REPORT "\%matnums " . Dumper( %matnums );
	
	my ( @lastbigcopy, @lastcopy );
	$countmat = 1;
	foreach my $copyref ( @newbigcopy )
	{	#say REPORT "DOING 2";
		my @constrlines = @$copyref; 
		my $intlayer = $constrlines[3]; #say REPORT "\$intlayer " . dump($intlayer);
		my @row = split ( /\s+|,/ , $intlayer );
		@row = cleanblanks( @row ); 
		my $matintlayernum = $row[0];  #say REPORT "\$matintlayernum " . dump($matintlayernum);
		my $matintlayer = $row[2]; #say REPORT "\$matintlayer " . dump($matintlayer);

		my $extlayer = $constrlines[$#constrlines];  #say REPORT "\$extlayer " . dump($extlayer);
		my @row = split ( /\s+|,/ , $extlayer );
		@row = cleanblanks( @row ); 
		my $matextlayernum = $row[0];
		my $matextlayer = $row[2]; #say REPORT "\$matextlayer " . dump($matextlayer);
		
		
		my $newmatnumint = $newmatnums{ $matintlayer }; #say REPORT "\$newmatnumint " . dump($newmatnumint); say " had " . dump ($newmatnums{ $matintlayer });
		my $newmatnumext = $newmatnums{ $matextlayer }; #say REPORT "\$newmatnumext " . dump($newmatnumext); say " had " . dump ($newmatnums{ $matextlayer });
		$constrlines[3] =~ s/$matintlayernum/$newmatnumint/g; #say REPORT "\$constrlines[3] " . dump( $constrlines[3] ); 
		$constrlines[$#constrlines] =~ s/$matextlayernum/$newmatnumext/g; #say REPORT "\$constrlines[\$#constrlines] " . dump($constrlines[$#constrlines]);
		foreach my $line ( @constrlines )
		{
			push ( @lastcopy, $line );
		}
		push ( @lastbigcopy, [ @lastcopy ] );
	} #say REPORT "DONE 2";
	#say REPORT "lastbigcopy: " . dump( @lastbigcopy );
	
	foreach ( @lastbigcopy )
	{
		foreach ( @lastcopy )
		{
			push ( @updatedlines, $_ )
		}
	}
	
	open ( CONSTRDBFILE_FICT, ">$constrdbfile_fict" ) or die;
	foreach ( @updatedlines )
	{
		
		print CONSTRDBFILE_FICT $_;
	} #say "\@updatedlines " . dump( @updatedlines );
	close CONSTRDBFILE_FICT;
	#say "\%exportconstr: " . dump( %exportconstr );
	return ( \@materials, \@newmaterials, \%matnums, \%newmatnums, \%exportconstr );		
}



sub compareirrs
{ 
	my ( $zonefilelistsref, $irrsref ) = @_;  
	my %zonefilelists = %$zonefilelistsref; say  REPORT "HERE IN COMPAREIRRS \%zonefilelists\: " . Dumper(%zonefilelists);
	my %irrs = %$irrsref; say  REPORT "HERE IN COMPAREIRRS \%irrs\: " . Dumper(%irrs);
	my %irrvars;
	foreach my $zonenum ( sort {$a <=> $b} ( keys %irrs ) )
	{	#say REPORT "IN COMPAREIRRS - FOREACH zonenum\: $zonenum" ;
		my $shdfile = $zonefilelists{ $zonenum }{ shdfile }; #say REPORT "IN COMPAREIRRS \$shdfile\: " . Dumper($shdfile);
		foreach my $monthnum ( sort {$a <=> $b} ( keys %{ $irrs{ $zonenum }{ 1 } } ) )
		{	#say REPORT "IN COMPAREIRRS - FOREACH MONTHNUM\: $monthnum" ;
			foreach my $surfnum ( sort {$a <=> $b} ( keys %{ $irrs{ $zonenum }{ 1 }{ $monthnum } } ) )
			{	#say REPORT "IN COMPAREIRRS - FOREACH surfnum\: $surfnum" ;
				foreach my $hour ( sort {$a <=> $b} ( keys %{ $irrs{ $zonenum }{ 1 }{ $monthnum }{ $surfnum } } ) )
				{	#say REPORT "IN COMPAREIRRS - FOREACH hour\: $hour" ;
					my $surfirr = $irrs{ $zonenum }{ 1 }{ $monthnum }{ $surfnum }{ $hour }{ meanirr }; say REPORT "\$surfirr " . dump($surfirr); say REPORT "\$zonenum " . dump($zonenum); say REPORT "\$monthnum " . dump($monthnum);say REPORT "\$surfnum " . dump($surfnum);say REPORT "\$hour " . dump($hour);
					my $whitesurfirr = $irrs{ $zonenum }{ 2 }{ $monthnum }{ $surfnum }{ $hour }{ meanirr }; say REPORT "\$whitesurfirr " . dump( $whitesurfirr );say REPORT "\$zonenum " . dump($zonenum);say REPORT "\$monthnum " . dump($monthnum);say REPORT "\$surfnum " . dump($surfnum);say REPORT "\$hour " . dump($hour);
					#my $surfirr_amb = $irrs{ $zonenum }{ 3 }{ $monthnum }{ $surfnum }{ $hour }{ meanirr }; say REPORT "\$surfirr_amb " . dump( $surfirr_amb );say REPORT "\$zonenum " . dump($zonenum);say REPORT "\$monthnum " . dump($monthnum);say REPORT "\$surfnum " . dump($surfnum);say REPORT "\$hour " . dump($hour);
					my $irrratio = ( $whitesurfirr / $surfirr ); say REPORT "HERE IN COMPAREIRRS \$irrratio " . dump( $irrratio );say REPORT "HERE IN COMPAREIRRS \$zonenum " . dump($zonenum);say REPORT "\$monthnum " . dump($monthnum);say REPORT "\$surfnum " . dump($surfnum);say REPORT "\$hour " . dump($hour);
				
					$irrvars{ $zonenum }{ $monthnum }{ $surfnum }{ $hour }{ irrvar } = $irrratio;
					#$irrvars{ $zonenum }{ $monthnum }{ $surfnum }{ $hour }{ irramb } = $surfirr_amb;
					$countsurf++;
				}					
				$counthour++;
			}
			$countmonth++;
		}
		$countzone++;
	} say REPORT  "HERE IN COMPAREIRRS \%irrvars " . dump(%irrvars);
	return ( \%irrvars ); 
}



sub fillhours
{
	my ( $newhourvalsref, $monthname, $daylighthoursref ) = @_;  #say REPORT "CALLED IN FILLHOURS \$monthname\: " . Dumper($monthname);
	my @hadhours = @$newhourvalsref; #say REPORT "\@hadhours\: " . Dumper(@hadhours);
	my %lithours = %$daylighthoursref; #say REPORT "\%lithours\: " . Dumper(%lithours);
	my @monthhours = @{ $lithours{ $monthname } }; #say REPORT "\@monthhours\: " . Dumper(@monthhours); 
	my @values;
	
	my $sunhoursnum = 0;
	foreach my $lightcond ( @monthhours )
	{
		unless ( $lightcond == 1 )
		{
			$sunhoursnum++;
		}
	}
	
	#if ( $sunhoursnum > scalar( @hadhours ) )
	#{
	#	say REPORT "NOT SUFFICIENT, BECAUSE hadhours are " . scalar( @hadhours ) . " instead of $sunhoursnum in month $monthname.";
	#}
		
	if ( $sunhoursnum == scalar( @hadhours ) )
	{
		my $counthr = 1;
		my $countlit = 0;
		foreach my $lightcond ( @monthhours )
		{
			if ( $lightcond == 1 )
			{
				push ( @values, "1.0000" ); #say REPORT "IN FILLHOURS \$counthr: $counthr, \$countlit $countlit";
			}
			else
			{
				push ( @values, $hadhours[ $countlit ] ); #say REPORT "IN FILLHOURS \$counthr: $counthr, \$countlit $countlit \$lightcond $lightcond, \$hadhours[ \$countlit ]: $hadhours[ $countlit ]"; 
				$countlit++;
			}
			$counthr++;
		} #say REPORT "VALUES @values";
		return ( @values );
	}
}


sub modifyshda
{
	my ( $comparedirrsref, $surfslistref, $zonefilelistsref, $shdfileslistref, $daylighthoursref, $irrvarsref ) = @_;
	my %surfslist = %$surfslistref;
	my %zonefilelists = %$zonefilelistsref; #say "\%zonefilelists\: " . dump(%zonefilelists);
	my %shdfileslist = %$shdfileslistref; #print "\%shdfileslist\: " . Dumper(%shdfileslist);
	my %daylighthours = %$daylighthoursref; #print "\%daylighthours\: " . Dumper(%daylighthours);
	my %irrvars = %$irrvarsref; say REPORT "HERE IN MODIFISHDA \%irrvars\: " . dump(%irrvars);
	my ( @printcontainer, @monthnames, @pushmodline, @pushreportline, @mainbag, @mainoriginal );
	foreach my $zonenum ( sort {$a <=> $b} ( keys %irrvars ) )
	{	#say REPORT "FROM MODIFYSHDA: - FOREACH \$zonenum\: $zonenum" ;
		my $shdfile = $zonefilelists{ $zonenum }{ shdfile }; #say "\$shdfile\: " . dump($shdfile);
		my $shdafile = "$shdfile" . "a";  #say "\$shdafile\: " . dump($shdafile);
		my $shdafilemod = $shdafile;
		$shdafilemod =~ s/.shda/.mod.shda/;
		my $inlinesref = $shdfileslist{ $zonenum }; 
		my @inlines = @$inlinesref; #say "\@inlines\: " . dump(@inlines);
		open ( SHDAMOD, ">$shdafilemod" ) or die; #say REPORT "I OPEN $shdafilemod";
		my $shdafilereport = $shdafile;
		$shdafilereport =~ s/.shda/.report.shda/; #say "\$shdafilereport\: " . dump($shdafilereport);
		`cp -R -f $shdafile $shdafilereport`;
		open ( SHDAREPORT, ">>$shdafilereport" ) or die;
		print SHDAREPORT "# FOLLOWING, THE VERIFIED VARIATIONS (EXPRESSED IN RATIOS) OF IRRADIANCES DUE TO REFLECTIONS BY OBSTRUCTIONS.\n";
		my $semaphore = 0;
		my ( $readmonthname, $readmonthnum );
		#say "\THE MODIFIED SHADE FILE DERIVING FROM THE .shda FILE WILL SAVEED AS .mod.shda FILE.			
		my $countline = 1;

		foreach my $line ( @inlines )
		{
			my $line2;
			my @row = split( /\s+|,/ , $line); #say "\$line\: " . dump($line);
			my ( $readsurfname, $readsurfnum );
			if ( ( $row[0] eq "*" ) and ( $row[1] eq "month:" ) )
			{
				$semaphore = 1; 
				$readmonthname = $row[2];
				$readmonthname =~ s/`//g; #say "\$readmonthname\: " . dump($readmonthname);
				$readmonthnum = getmonthnum( $readmonthname );  #say "\$readmonthnum\: " . dump($readmonthnum);
			}
			if ( ( ( $row[0] eq "24" ) and ( $row[1] eq "hour" ) and ( $row[1] eq "surface" ) ) or ( $row[0] eq "*end" ) )
			{
				$semaphore = 0; #say "\$semaphore\: " . dump($semaphore);
			}

			my ( @newhourvals, @newhourvals2);
			foreach my $monthnum ( sort {$a <=> $b} ( keys %{ $irrvars{ $zonenum } } ) )
			{	#say REPORT " %{ FROM MODIFYSHDA: FOREACH MONTHNUM: \$irrvars{ \$zonenum }{ \$monthnum } }; \$irrvars{ $zonenum }{ $monthnum } } : " . dump ( %{ $irrvars{ $zonenum }{ $monthnum } } ) . " keys %{ \$irrvars{ \$zonenum }{ \$monthnum } }; keys %{ \$irrvars{ $zonenum }{ $monthnum } } : " . dump ( keys %{ $irrvars{ $zonenum }{ $monthnum } } );
				my $monthname = getmonthname( $monthnum ); #say "\$monthname\: " . dump($monthname);
				push ( @monthnames, $monthname );
				push ( @monthnames2, $monthname );
				
				say REPORT "HERE IN MODIFISHDA ( sort {$a <=> $b}( keys %{ \$irrvars{ \$zonenum }{ \$monthnum } } ) ) " . dump( sort {$a <=> $b}( keys %{ $irrvars{ $zonenum }{ $monthnum } } ) );
				foreach my $surfnum ( sort {$a <=> $b} ( keys %{ $irrvars{ $zonenum }{ $monthnum } } ) )
				{	#say REPORT " %{ FROM MODIFYSHDA: FOREACH SURFNUM \$irrvars{ \$zonenum }{ \$monthnum }{ \$surfnum } }; \$irrvars{ $zonenum }{ $monthnum }{ $surfnum } } : " . dump ( %{ $irrvars{ $zonenum }{ $monthnum }{ $surfnum } } ) . " keys %{ \$irrvars{ \$zonenum }{ \$monthnum }{ \$surfnum } }; keys %{ \$irrvars{ $zonenum }{ $monthnum }{ $surfnum } } : " . dump ( keys %{ $irrvars{ $zonenum }{ $monthnum }{ $surfnum } } );
					my $surfname = $surfslist{$zonenum}{$surfnum}; #say "\$surfname\: " . dump($surfname);
					foreach my $hour ( sort {$a <=> $b} ( keys %{ $irrvars{ $zonenum }{ $monthnum }{ $surfnum } } ) )
					{	#say REPORT " %{ FROM MODIFYSHDA: FOREACH HOUR \$irrvars{ \$zonenum }{ \$monthnum }{ \$surfnum }{ \$hour } }; \$irrvars{ $zonenum }{ $monthnum }{ $surfnum }{ $hour } } : " . dump ( %{ $irrvars{ $zonenum }{ $monthnum }{ $surfnum }{ $hour } } ) . " keys %{ \$irrvars{ \$zonenum }{ \$monthnum }{ \$surfnum }{ \$hour } }; keys %{ \$irrvars{ $zonenum }{ $monthnum }{ $surfnum }{ $hour } } : " . dump ( keys %{ $irrvars{ $zonenum }{ $monthnum }{ $surfnum }{ $hour } } );
						my $irrvariation = $irrvars{ $zonenum }{ $monthnum }{ $surfnum }{ $hour }{ irrvar }; #say REPORT "\$irrvariation\: " . dump($irrvariation);
						#my $ambbase = $irrvars{ $zonenum }{ $monthnum }{ $surfnum }{ $hour }{ irramb }; #say REPORT "\$ambbase\: " . dump($ambbase);
						
						if ( ( $zonenum ) and ( $monthnum ) and ( $hour ) and ( $surfnum ) and ( $irrvariation ) and ( $surfname ) and ( $monthname ) ) # IF ALL THE NEEDED DATA EXIST
						{	#say "\$zonenum  $zonenum  \$monthnum  $monthnum  \$hour  $hour  \$surfnum  $surfnum  \$irrvariation  $irrvariation  \$surfname  $surfname  \$monthname  $monthname " ;
							if ( $semaphore == 1 )
							{
								if ( $row[27] eq "surface" )
								{
									$readsurfname = $row[28]; #say "\$readsurfname\: " . dump($readsurfname);
									$readsurfnum = $surfslist{$zonenum}{$readsurfname}; #say "\$readsurfnum\: " . dump($readsurfnum);
									my @filledhourvals;
									if ( ( $row[25] eq "diffuse" ) and ( $readsurfname eq $surfname ) )
									{
										my @hourvals = ( @row[ 0..23 ] ); #say "\@hourvals\: " . dump(@hourvals);
										my $counthour = 1;
										foreach my $el ( @hourvals )
										{	# $el IS THE DIFFUSE SHADING FACTOR IN THE ORIGINAL SHDA FILE.
											# %irrvariation IS THE IRRADIANCE DIFFERENCE BETWEEN THE "WHITE" MODEL AND THE "BLACK" MODEL.
											# $ambase IS THE AMBIENT RADIATION WITHOUT SHADINGS
											#say "EL!: $el, \$counthour, $counthour  \$hour $hour \$readmonthname $readmonthname \$monthname $monthname \$readsurfnum $readsurfnum \$surfnum $surfnum";
											
											my ( $calcamount, $improvedguess, $newshadingvalue);
											if ( $irrvariation > 1 )
											{ 
												$calcamount = ( 1 - $el ); # THIS IS THE RATIO OF NON-SHADED IRRADIATION AS CALCULATED BY THE ESP-r's ISH MODULE
												$improvedguess = ( $calcamount * $irrvariation ); # THIS IS THE RATIO ABOVE CORRECTED BY MULTIPLYING IT BY THE IRRADIANCE RATIO TO TAKE REFLECTIONS INTO ACCOUNT.
												$newshadingvalue = ( 1 - $improvedguess ); # AS THE NAME SAYS, THIS IS THE NEW SHADING VALUE.
											}
											else
											{
												$newshadingvalue = $el; # IF THE IRRADIANCE RATIO IS < 1 DON'T CHANCE THE ORIGINAL DIFFUSE SHADING FACTOR.
											}
											
											if ( ( $counthour == $hour ) and ( $readmonthname eq $monthname ) and ( $readsurfnum == $surfnum ) ) # IT MEANS: IF THIS LINE IS THE RIGHT ONE...
											{	#say REPORT "GOT!: $el, \$counthour, $counthour  \$hour $hour \$readmonthname $readmonthname \$monthname $monthname \$readsurfnum $readsurfnum \$surfnum $surfnum";								
	
												if ( $newshadingvalue == 1 ) 
												{ 
													$newshadingvalue = "1.0000"; 
												}
												if ( ( $newshadingvalue > 0 ) and ( $newshadingvalue < 1 ) ) # IF THE VARIATION OF IRRADIANCE FROM MODEL A AND MODEL B IS NEGATIVE...
												{	# ...INCREASE THE SHADING FACTOR ACCORDINGLY. IMPORTANT NOTE: THIS IS A SIMPLIFICATION, 
													# BECAUSE FOR BEST PRECISION THE WHOLE AMOUNT OF RADIATION ENTERING SHOULD BE REDUCED, NOT JUST THE DIFFUSE PART OF IT, 
													# GOVERNED BY THE DIFFUSE SHADING FACTOR, AS IT IS BEING DONE HERE.
													$newshadingvalue = sprintf ( "%.4f", $newshadingvalue ); # FORMAT THE NUMBER SO THAT IT HAS FOUR DECIMALS
												}
												if ( ( $newshadingvalue < 10 ) and ( $newshadingvalue < 0 ) )
												{
													$newshadingvalue = sprintf ( "%.3f", $newshadingvalue ); # IT THE NUMBER IS COMPRISED BETWEEN -10 AND = 0 FORMAT IT SO THAT IT HAS 3 DECIMALS
												}
												if ( ( $newshadingvalue < 100 ) and ( $newshadingvalue <= -10 ) )
												{
													$newshadingvalue = sprintf ( "%.2f", $newshadingvalue ); # IT THE NUMBER IS COMPRISED BETWEEN -100 AND = -10 FORMAT IT SO THAT IT HAS 2 DECIMALS
												}
												
		
												say REPORT "HERE IN MODIFISHDA value $el , due to \$newshadingvalue $newshadingvalue and \$irrvariation $irrvariation, in monthnum: $monthnum, monthname: $monthname, surfnum: $surfnum, hour: $hour " ;
														
												my $irrvariation = sprintf ( "%.4f", $irrvariation ); # say REPORT "SPRINTING $irrvariation!";
												push ( @newhourvals, $newshadingvalue); say REPORT "PUSHING $newshadingvalue! in newhourvals";
												push ( @newhourvals2, $irrvariation ); say REPORT "PUSHING $irrvariation! in newhourvals2";		
											}
											$counthour++;
										}
										#say REPORT "HERE IN MODIFISHDA PRODUCED \@newhourvals " . dump(@newhourvals) . ", in monthnum: $monthnum, monthname: $monthname, surfnum: $surfnum, hour: $hour";
										#say REPORT "HERE IN MODIFISHDA PRODUCED \@newhourvals2 " . dump(@newhourvals2) . ", in monthnum: $monthnum, monthname: $monthname, surfnum: $surfnum, hour: $hour";
										my @filledhourvals = fillhours( \@newhourvals, $monthname, \%daylighthours ); say REPORT "FOR MONTH $monthname, CALLING filledhourvals " . Dumper( @filledhourvals ); say REPORT "\scalar(\@filledhourvals) " . scalar( @filledhourvals) ;
										my @filledhourvals2 = fillhours( \@newhourvals2, $monthname, \%daylighthours ); say REPORT "FOR MONTH $monthname, CALLING filledhourvals2 " . Dumper( @filledhourvals2 ); say REPORT "\scalar(\@filledhourvals2) " . scalar( @filledhourvals2) ;
										
										if ( ( scalar ( @filledhourvals ) == 24 ) and ( $monthname eq $monthnames[0] ) )
										{	say REPORT "GOT. WRITING FOR SHDA $monthname! ";
											say REPORT "HERE IN MODIFISHDA PRODUCED \@newhourvals " . dump(@newhourvals) . ", in monthnum: $monthnum, monthname: $monthname, surfnum: $surfnum, hour: $hour";
											shift @monthnames;
											my @firstarr = @filledhourvals[ 0..11 ]; say REPORT "\@firstarr " . dump(@firstarr);
											my @secondarr = @filledhourvals[ 12..$#filledhourvals ]; say REPORT "\@secondarr " . dump(@secondarr);
											my $joinedfirst = join ( ' ' , @firstarr ); say REPORT "\$joinedfirst " . dump($joinedfirst);
											my $joinedsecond = join ( ' ' , @secondarr ); say REPORT "\$joinedsecond " . dump($joinedsecond);
											my $newline = "$joinedfirst\n" . "$joinedsecond" . " # diffuse - surface " . "$readsurfname\n"; 
											#$line = $newline;  say REPORT "\$line\: " . dump($line); say REPORT "HERE IN MODIFISHDA DERIVED \$newline\: " . dump($newline);
											push ( @pushmodline, $newline );
										}
										
										if ( ( scalar ( @filledhourvals2 ) == 24 ) and ( $monthname eq $monthnames2[0] ) )
										{	say REPORT "GOT. WRITING FOR REPORT $monthname! ";
											say REPORT "HERE IN MODIFISHDA PRODUCED \@newhourvals2 " . dump(@newhourvals2) . ", in monthnum: $monthnum, monthname: $monthname, surfnum: $surfnum, hour: $hour";
											shift @monthnames2;
											my @firstarr2 = @filledhourvals2[ 0..11 ]; say REPORT "\@firstarr2 " . dump(@firstarr2);
											my @secondarr2 = @filledhourvals2[ 12..$#filledhourvals2 ]; say REPORT "\@secondarr2 " . dump(@secondarr2);
											my $joinedfirst2 = join ( ' ' , @firstarr2 ); say REPORT "\$joinedfirst2 " . dump($joinedfirst2);
											my $joinedsecond2 = join ( ' ' , @secondarr2 ); say REPORT "\$joinedsecond2 " . dump($joinedsecond2);
											my $newline2 = "$joinedfirst2\n" . "$joinedsecond2" . " # for surface " . "$readsurfname in $monthname\n"; 
											#$line2 = $newline2;  say REPORT "HERE IN MODIFISHDA DERIVED \$line2\: " . dump($line2); say REPORT"\$newline2\: " . dump($newline2);
											push ( @pushreportline, $newline2 );
										}
									}
									else
									{
										my @firstarr = @row[ 0..11 ];
										my @secondarr = @row[ 12..$#row ];
										my $joinedfirst = join ( ' ' , @firstarr ); say REPORT "\$joinedfirst " . dump($joinedfirst);
										my $joinedsecond = join ( ' ' , @secondarr ); say REPORT "\$joinedsecond " . dump($joinedsecond);
										my $newline = "$joinedfirst\n" . "$joinedsecond\n"; say REPORT "\$newline\: " . dump($newline);
										$line = $newline;
										$line2 = undef;
									}
								}
							}
						}
						$countref++;
					} 
				} 
			}
			push ( @mainbag, $line );
			my @mainoriginal = @mainbag;
		}
		my ( $lin, $l );
		my @lastbag;
		foreach my $lin ( @mainbag )
		{
			my @rowmain = split(/\s+|,/, $lin); 
			foreach my $li ( @pushmodline )
			{
				my @rowmod = split(/\s+|,/, $li); 
				if ( @rowmain[ 24..$#rowmod ] ~~ @rowmod[ 24..$#rowmod ] )
				{
					$lin = $li;
				}
			}
			print SHDAMOD $lin;
		}
		
		push ( @mainoriginal, @pushreportline );
		foreach my $li ( @mainoriginal )
		{
			print SHDAREPORT $li;
		}
			
		close SHDAMOD;
		close SHDAREPORT;
	}
}		


sub getbasevectors
{
	# PRE-COMPUTED EVENTLY DISTRIBUTED N POINTS ON THE SURFACE OF A HEMISPHERE.
	my ( $dirvectorsnum ) = @_;
	my @basevectors;
	if ( $dirvectorsnum == 1 )
	{
		@basevectors = ( #[ 0, 0, 0 ] , # origin, base point of direction vector
				[ 0, 0, 1 ], # direction vector of high, central, vertical point
				 ); # lowest vertices
	}
	if ( $dirvectorsnum == 5 )
	{
		@basevectors = ( #[ 0, 0, 0 ] , # origin, base point of direction vector
				[ 0, 0, 1 ], # direction vector of high, central, vertical point
				[ 1, 0, 0 ] , [ 0, 1, 0 ], [ -1, 0, 0 ], [ 0, -1, 0 ] ); # lowest vertices
	}
	elsif ( $dirvectorsnum == 9 )
	{
		@basevectors = ( #[ 0, 0, 0 ] , # origin, base point of direction vector
				[ 0, 0, 1 ], # direction vector of high, central, vertical point
				[ 0.7071, -0.7071, 0.4472 ] , [ 0.7071, 0.7071, 0.4472 ], [ -0.7071, 0.7071, 0.4472 ], [ -0.7071, -0.7071, 0.4472 ], # lower vectexes
				[ 1, 0, 0 ] , [ 0, 1, 0 ], [ -1, 0, 0 ], [ 0, -1, 0 ] ); # lowest vertices
	}
	return ( @basevectors );
}

sub createfictgeofile
{	# IT MANAGES THE MODIFICATION OF THE FICTITIOUS GEO FILES FOR THE ZONE BY ADJUSTING THE OBSTRUCTIONS CONSTRUCTIONS TO FICTITIOUS EQUIVALENT
	my ( $geofile, $obsconstrsetref, $geofile_fict ) = @_;
	my @obsconstrset = @$obsconstrsetref;
	open ( GEOFILE, "$geofile" ) or die;
	my @lines = <GEOFILE>;
	close GEOFILE;
	
	#unless ( -e $geofile_fict )
	#{
		open ( GEOFILE_FICT, ">$geofile_fict" ) or die;
	#}
	
	foreach my $line ( @lines )
	{
		if ( $line =~ /^\*obs/ )
		{
			foreach my $obsconstr ( @obsconstrset )
			{
				my $newobsconstr = $obsconstr; #say "\$obsconstr $obsconstr"; 
				$newobsconstr =~ s/\w\b// ;  #say "\$newobsconstr $newobsconstr"; 
				$newobsconstr =~ s/\w\b// ;  #say "\$newobsconstr $newobsconstr"; 
				$newobsconstr = "f_" . $newobsconstr; #say REPORT "\$newobsconstr $newobsconstr"; say REPORT "\$oldline $line"; 
				$line =~ s/$obsconstr/$newobsconstr/; #say REPORT "\$newline $line"; 
			}
			print GEOFILE_FICT $line;
		}
		else
		{
			print GEOFILE_FICT $line;
		}
	}
	close ( GEOFILE_FICT);
}

sub creatematdbfiles
{	# IT MANAGES THE CREATION OF THE TWO FICTITIOUS MATERIALS DATABASES
	my ( $materialsref, $newmaterialsref, $matnumsref, 
	$newmatnumsref, $matdbfile, $matdbfile_fict1, $matdbfile_fict2 ) = @_;
	my @materials = @$materialsref; say REPORT "\@materials:\ " . dump( @materials );
	my @newmaterials = @$newmaterialsref; say REPORT "\@newmaterials\ " . dump( @newmaterials );
	my $newmattotnum = scalar( @newmaterials ); #say REPORT "\$newmattotnum\ " . dump( $newmattotnum );
	my %matnums = %$matnumsref; say REPORT "\%matnums\ " . dump( %matnums );
	my %newmatnums = %$newmatnumsref; say REPORT "\%newmatnums\ " . dump( %newmatnums );
	
	open ( MATDBFILE, "$matdbfile" ) or die; #say REPORT "\$matdbfile\ " . dump( $matdbfile );
	my @lines = <MATDBFILE>;
	close MATDBFILE;
	
	my @bag;
	foreach ( @lines )
	{
		$_ =~ /^\*item,(.+),(.+),(.+),(.+)$/ ; #say REPORT "\$_\ " . dump( $_ );
		push ( @bag, $2 );
	}
	my $mattotnum = max( @bag ); #say REPORT "\$mattotnum\ " . dump( $mattotnum );
	
	my ( $classnum, $newclassnum, $newclasstotnum );
	
	my ( @firstloop, @copy );
	my $countline = 0;
	foreach my $line ( @lines )
	{
		if ( $line =~ /# number of classifications/ )
		{
			my @row = split( /\s+|,/ , $line );
			my $classtotnum = $row[0]; #say REPORT "\$classtotnum\ " . dump( $classtotnum );
			$newclasstotnum = ( $classtotnum + 1 ); #say REPORT "\$newclasstotnum\ " . dump( $newclasstotnum );
			$line =~ s/$classtotnum/$newclasstotnum/; #say REPORT "\$line\ " . dump( $line );
			my $extractvalue = $line; 
			$line = $extractvalue;
		}

		if ( $line =~ /\*item/ )
		{
			my @row = split( /,/ , $line );
			my $item = $row[1]; #say REPORT "\$item\ " . dump( $item );
			if ( $item ~~ @materials )
			{
				my $newitem = $item;
				$newitem =~ s/\w\b// ;
				$newitem =~ s/\w\b// ;
				$newitem = "f_" . $newitem; #say REPORT "\$newitem\ " . dump( $newitem );
				my $newline = $line;
				$newline =~ s/$item/$newitem/; #say REPORT "\$newline\ " . dump( $newline );
				my $afterline = $lines[ $countline + 1 ]; #say REPORT "\$afterline\ " . dump( $afterline );
				push ( @copy, [ $newline, $afterline ] ); #say REPORT "\@copy\ " . dump( @copy );
			}
		}
		push ( @firstloop, $line );
		$countline++;
	}
	pop ( @firstloop ); #say REPORT "\@firstloop\ " . dump( @firstloop );
	say REPORT "\@copy \n\n" . dump ( @copy );
	
	my @newcopy;
	my %exportrefl;
	my $newmatstotnum = scalar ( @copy );
	push ( @newcopy, "*class,$newclasstotnum, 6,Fictitious\n" );
	push ( @newcopy, "Category ($newclasstotnum) holds fictitious materials for shading and insolation calculations\n" );
	my $countnew = 1;
	foreach ( @copy )
	{
		my $counttwo = 1;
		my ( $namematobs, $absout, $absin );
		my @box;
		foreach ( @$_ )
		{
			if ( $counttwo == 1 )
			{
				my $newmattotnum = ( $mattotnum + $countnew ); #say REPORT "\$newmattotnum\ " . dump( $newmattotnum );
				$_ =~ /^\*item,(.+),(.+),(.+),(.+)$/ ; #say REPORT "\$_\ " . dump( $_ );
				#say REPORT "\$1: $1, \$2: $2, \$3: $3, \$4: $4 ";
				$namematobs = $1; say REPORT "\$namematobs\n" . dump ( $namematobs );
				$_ = "*item,$1, $newmattotnum, $countnew,$4\n"; #say REPORT "\$_\ " . dump( $_ );
			}
			$_ =~ s/\s//;
			$_ =~ s/\s//;
			if ( $counttwo == 2 )
			{				
				$_ =~ /(.+),(.+),(.+),(.+),(.+),(.+),(.+),(.+),(.+),-/ ;
				$absout = $6; say REPORT "\$absout\n" . dump ( $absout );
				$absin = $7; say REPORT "\$absin\n" . dump ( $absin );
			}
			push ( @newcopy, $_ );
			$exportrefl{ $namematobs }{ absout } =  $absout;
			$exportrefl{ $namematobs }{ absin } = $absin;
			$counttwo++;
		}
		$countnew++;
	} #say REPORT "\@newcopy\ " . dump( @newcopy );
	
	my @blackcopy;
	my @tempcopy = @newcopy;
	foreach my $line ( @tempcopy )
	{
		if ( $line =~ /(.+),(.+),(.+),(.+),(.+),(.+),(.+),(.+),(.+),-/ )
		{
			my $absout = "1.000";
			my $absin = "1.000";
			$line = "$1,$2,$3,$4,$5,$absout,$absin,$8,$9,-\n";
		}
		push ( @blackcopy, $line );
	}
	say REPORT "\@blackcopy\ " . dump( @blackcopy );
	say REPORT "\@newcopy\ " . dump( @newcopy );
	
	my @blackloop;
	my @secondtemploop = @firstloop;
	foreach ( @secondtemploop )
	{
		if ( $_ =~ /(.+),(.+),(.+),(.+),(.+),(.+),(.+),(.+),(.+),-/ )
		{
			my $absout = "1.000";
			my $absin = "1.000";
			$_ = "$1,$2,$3,$4,$5,$absout,$absin,$8,$9,-\n";
		}
		push ( @blackloop, $_ );
	} #say REPORT "\@blackloop " . dump( @blackloop );	
	
	my @whiteloop = @blackloop;
	push ( @blackloop, @blackcopy, "*end\n" ); #say REPORT "FINAL\@blackloop " . dump( @blackloop );	
	push ( @whiteloop, @newcopy, "*end\n" ); #say REPORT "FINAL\@whiteloop " . dump( @whiteloop );	
	
	open (MATDBFILE_FICT1, ">$matdbfile_fict1" ) or die;
	foreach ( @blackloop )
	{
		print MATDBFILE_FICT1 $_;
	}
	close MATDBFILE_FICT1;
	
	open (MATDBFILE_FICT2, ">$matdbfile_fict2" ) or die;
	foreach ( @whiteloop )
	{
		print MATDBFILE_FICT2 $_;
	}
	close MATDBFILE_FICT2;
	say REPORT "IN FUNCTION \%exportrefl: " . dump( %exportrefl );
	return ( \%exportrefl );
}

sub adjust_radmatfile
{
	my ( $exportconstrref, $exportreflref, $conffile, $path ) = @_;
	my %exportconstr = %$exportconstrref; #say REPORT "\%exportconstr\ " . dump( %exportconstr );
	my %exportrefl = %$exportreflref; #say REPORT "\%exportrefl\ " . dump( %exportrefl );
	my $radmat_fict2 = $conffile; #say "\$radmat_fict2\ " . dump( $radmat_fict2 );
	$radmat_fict2 =~ s/$path\/cfg\///;
	$radmat_fict2 =~ s/.cfg//;
	$radmat_fict2 = $radmat_fict2 . "_Extern.mat";
	$radmat_fict2 = "$path/rad/$radmat_fict2"; #say REPORT"\$radmat_fict2\ " . dump( $radmat_fict2 );
	my $radmattemp = $radmat_fict2 . ".temp"; #say "\$radmattemp\ " . dump( $radmattemp );
	`mv -f $radmat_fict2 $radmattemp`;
	open( RADMATTEMP, "$radmattemp" ) or die;
	my @lines = <RADMATTEMP>;
	close RADMATTEMP;
	open( RADMAT_FICT2, ">$radmat_fict2" ) or die;
	my $count = 0;
	my @constrs = keys %exportconstr; #say REPORT "\@constrs\ " . dump( @constrs );
	foreach ( @lines )
	{
		if ( $_ =~ /# External MLC Colours.../ )
		{
			foreach ( @constrs )
			{
				my $constrout = "rc_ex_" . $_; #say REPORT "\$constrout\ " . dump( $constrout );
				if ( $lines[ $count + 1] =~ /$constrout/ )
				{
					my $extmat = $exportconstr{ $_ }{ extlayer }; say REPORT "\$extmat\ " . dump( $extmat );
					my $absextext = $exportrefl{ $extmat }{ absout }; say REPORT "\$absextext\ " . dump( $absextext );
					my $refl = 1 - $absextext; say REPORT "\$refl\ " . dump( $refl );
					$refl = sprintf( "%.2f", $refl ); say REPORT "\$refl\ " . dump( $refl );
					$lines[ $count + 4 ] =~ /5  (\d+) (\d+) (\d+) (\d+) (\d+)/ ;
					$lines[ $count + 4 ] = "5  $refl $refl $refl 0 0\n"; say REPORT "\$lines[ \$count + 4 ]\ " . $lines[ $count + 4 ];
				}
			}
		}
		if ( $_ =~ /# Internal MLC Colours.../ )
		{
			foreach ( @constrs )
			{
				my $constrin = "rc_in_" . $_; #say REPORT "\$constrin\ " . dump( $constrin );
				if ( $lines[ $count + 1] =~ /$constrin/ )
				{
					my $intmat = $exportconstr{ $_ }{ intlayer }; say REPORT "\$intmat\ " . dump( $intmat );
					my $absintint = $exportrefl{ $intmat }{ absin }; say REPORT "\$absintint\ " . dump( $absintint );
					my $refl = 1 - $absintint; say REPORT "\$refl\ " . dump( $refl );
					$refl = sprintf( "%.2f", $refl ); say REPORT "\$refl\ " . dump( $refl );
					$lines[ $count + 4 ] =~ /5  (\d+) (\d+) (\d+) (\d+) (\d+)/ ;
					$lines[ $count + 4 ] = "5  $refl $refl $refl 0 0\n"; say REPORT "\$lines[ \$count + 4 ]\ " . $lines[ $count + 4 ];
				}
			}
		}
		print RADMAT_FICT2 $lines[ $count ];
		$count++;
	}
	close RADMAT_FICT2;
}

sub calcdirvectors
{	
	my @winscoords = @_; #say "\@winscoords \: " . dump(@winscoords);
	my ( @groupbag );
	foreach my $surf ( @winscoords )
	{
		my ( @surfbag );
		foreach my $v ( @$surf )
		{
			my @fields = @$v; #say "\@fields \: " . dump(@fields);
			my $coordsref = $fields[0]; #say "\$coordsref \: " . dump($coordsref);
			my @coords = @$coordsref;
			my $vertex = Vector::Object3D::Point->new( x => $coords[0], y => $coords[1], z => $coords[2] ); #say "\$vertex \: " . dump($vertex);
			push ( @surfbag, $vertex); #say "\@surfbag \: " . dump(@surfbag);
		}
		#say "\@surfbag \: " . dump(@surfbag);
		my $polygon = Vector::Object3D::Polygon->new(vertices => [ @surfbag ]); #say "\$polygon \: " . dump($polygon);
		my $normal_vector = $polygon->get_normal_vector; #say "\$normal_vector \: " . dump($normal_vector);
		my ($x_, $y_, $z_) = $normal_vector->array; #say "\$x_, \$y_, \$z_ \: $x_, $y_, $z_" ;
		my ( $x, $y, $z ) = ( -$x_, -$y_, -$z_ ); #say "\$x, \$y, \$z \: $x, $y, $z" ;
		my $max = max( abs($x), abs($y), abs($z) ); #say "\$max\: $max" ;
		my @dirvector;
		unless ( $max == 0 )
		{
			$x = ( $x / $max );
			$y = ( $y / $max );
			$z = ( $z / $max );
			@dirvector =  ( $x, $y, $z );
		}
		else 
		{
			@dirvector =  ( $x, $y, $z );
		}
		#say "DIRVECTOR \$x_, \$y_, \$z_ \: $x, $y, $z" ;
		push ( @groupbag, [ @dirvector ] )
	}
	return ( @groupbag);
}

sub prunepoints
{
	my ( $gridpoints_transitionalref, $xyzcoordsref ) = @_;
	my @gridpoints = @$gridpoints_transitionalref; say REPORT "HERE LOOK  \@gridpoints \: " . dump(@gridpoints);
	my @vertstaken = @$xyzcoordsref; say REPORT "HERE LOOK INSIDE \@vertstaken \: " . dump(@vertstaken);
	
	my @verts;
	foreach ( @vertstaken )
	{
		my @fields = @$_; #say "\@fields \: " . dump(@fields);
		my @xs = @{ $fields[0] }; say REPORT "\@xs \: " . dump(@xs);
		my @ys = @{ $fields[1] }; say REPORT "\@ys \: " . dump(@ys);
		my @zs = @{ $fields[2] }; say REPORT "\@zs \: " . dump(@zs);
		my $i = 0;
		my @bag;
		foreach ( @xs )
		{
			push ( @bag, [ $xs[ $i ], $ys[ $i ], $zs[ $i ] ] );
			$i++;
		}
		push ( @verts, [ @bag ] );
	} say REPORT "HERE LOOK INSIDE \@verts \: " . dump(@verts);
	
	my ( @coords, @prunegridpoints );

	foreach my $v ( @gridpoints )
	{
		my @fields = @$v; say REPORT "\@fields \: " . dump(@fields);
		my $coordsref = $fields[0]; #say "\$coordsref \: " . dump($coordsref);
		push( @coords, [ @$coordsref ] );
	} say REPORT "\@coords \: " . dump(@coords);
	
	
	my ( @boxpointxy, @boxpointxz, @boxpointyz );
	foreach my $gridpoint ( @coords )
	{
		my ( @point_xys, @point_xzs, @point_yzs );
		foreach ( @$gridpoint )
		{
			push ( @point_xys, [ $_->[0], $_->[1] ] );
			push ( @point_xzs, [ $_->[0], $_->[2] ] );
			push ( @point_yzs, [ $_->[1], $_->[2] ] );
		}
		push ( @boxpointxy, [ @point_xys ] );
		push ( @boxpointxz, [ @point_xzs ] );
		push ( @boxpointyz, [ @point_yzs ] );
	} say REPORT "\@boxpointxy \: " . dump(@boxpointxy); say REPORT "\@boxpointxz \: " . dump(@boxpointxz); say REPORT "\@boxpointyz \: " . dump(@boxpointyz);
	
	my ( @boxvertxy, @boxvertxz, @boxvertyz );
	foreach my $surf ( @verts )
	{
		my ( @vert_xys, @vert_xzs, @vert_yzs );
		foreach my $vert ( @$surf )
		{
			push ( @vert_xys, [ $vert->[0], $vert->[1] ] );
			push ( @vert_xzs, [ $vert->[0], $vert->[2] ] );
			push ( @vert_yzs, [ $vert->[1], $vert->[2] ] );
		}
		push ( @boxvertxy, [ @vert_xys ] );
		push ( @boxvertxz, [ @vert_xzs ] );
		push ( @boxvertyz, [ @vert_yzs ] );
	} say REPORT "\@boxvertxy \: " . dump(@boxvertxy);say REPORT "\@boxvertxz \: " . dump(@boxvertxz);say REPORT "\@boxvertyz \: " . dump(@boxvertyz);
	
	my $count = 0;
	my ( $vert_xys, $vert_xzs, $vert_yzs, $polyxy, $polyxz, $polyyz );
	my ( @verts_xys, @verts_xzs, @verts_yzs);
	foreach my $case ( @boxvertxy )
	{
		$vert_xys = $boxvertxy[$count]; say REPORT "\$vert_xys \: " . dump($vert_xys); #$vert_xys = Math::Geometry::Planar->new;
		$vert_xzs = $boxvertxz[$count]; say REPORT "\$vert_xzs \: " . dump($vert_xzs); #$vert_xzs = Math::Geometry::Planar->new;
		$vert_yzs = $boxvertyz[$count]; say REPORT "\$vert_yzs \: " . dump($vert_yzs); #$vert_yzs = Math::Geometry::Planar->new;
		$polyxy = Math::Polygon::Tree->new( $vert_xys ); say REPORT "\$polyxy \: " . dump($polyxy);
		$polyxz = Math::Polygon::Tree->new( $vert_xzs ); say REPORT "\$polyxz \: " . dump($polyxz);
		$polyyz = Math::Polygon::Tree->new( $vert_yzs ); say REPORT "\$polyyz \: " . dump($polyyz);
		$count++;
	}
	
	my $count = 0;
	my @newbox;
	foreach my $caseref ( @gridpoints )
	{
		my @case = @$caseref; say REPORT "\$xyref \: " . dump($xyref);
		my $surfnum = $case[ 1 ];
		my $dirvector = $case[ 2 ];
		my @bag;
		foreach my $vert ( @{ $case[ 0 ] } )
		
		{	#say REPORT "HERE LAST WATCH \$vert \: " . dump($vert);
			my $xyref = $boxpointxy[ $count ][ 0 ]; say REPORT "\$xyref \: " . dump($xyref);
			my $xzref = $boxpointxz[ $count ][ 0 ]; say REPORT "\$xzref \: " . dump($xzref);
			my $yzref = $boxpointyz[ $count ][ 0 ]; say REPORT "\$yzref \: " . dump($yzref);
			unless ( ( $polyxy->contains( $xyref ) ) == 0 ) { say REPORT "\$polyxy contains \$xyref\. "; } else { say REPORT "\$polyxy doesn't contain \$xyref\. " } say REPORT "\$polyxy \: " . dump($polyxy); say REPORT "\$xyref \: " . dump($xyref);  
			unless ( ( $polyxz->contains( $xzref ) ) == 0 ) { say REPORT "\$polyxz contains \$xzref\. "; } else { say REPORT "\$polyxz doesn't contain \$xzref\. " } say REPORT "\$polyxz \: " . dump($polyxz); say REPORT "\$xzref \: " . dump($xzref); 
			unless ( ( $polyyz->contains( $yzref ) ) == 0 ) { say REPORT "\$polyyz contains \$yzref\. "; } else { say REPORT "\$polyyz doesn't contain \$yzref\. " } say REPORT "\$polyyz \: " . dump($polyyz); say REPORT "\$yzref \: " . dump($yzref); 
			unless ( ( ( $polyxy->contains( $xyref ) ) == 0 ) and ( ( $polyxz->contains( $xzref ) ) == 0 ) and ( ( $polyyz->contains( $yzref ) ) == 0 ) )
			{
				push( @bag, $vert ); 
			}
		} #say REPORT "\@bag \: " . dump(@bag);
		push ( @newbox, [ [ @bag ], $surfnum, $dirvector ] );
		$count++;
	} say REPORT "\@newbox \: " . dump(@newbox);
	return ( @newbox );
}


sub checkchildren
{
	my ( $gridpoints_newtransitionalref, $surfnumsref, $datalistref, $zonenum ) = @_;
	my @gridpoints = @$gridpoints_newtransitionalref; say REPORT "HERE REALLY LAST INSIDE \@gridpoints \: " . dump(@gridpoints);
	my @surfnums = @$surfnumsref; say REPORT "HERE REALLY LAST INSIDE \@surfnums \: " . dump(@surfnums);
	my %datalist = %$datalistref; say REPORT "HERE REALLY LAST INSIDE \%datalist \: " . dump(%datalist);
	say REPORT "HERE REALLY LAST INSIDE \$zonenum \: " . dump($zonenum);
	
	foreach my $surfnum ( @surfnums )
	{
		say REPORT "HERE REALLY \$surfnum \: " . dump($surfnum);
		my $childrenref = $datalist{ $zonenum }{ children }{ $surfnum }; say REPORT "HERE REALLY \$childrenref \: " . dump($childrenref);
		my @children = @{ $datalist{ $zonenum }{ children }{ $surfnum } }; say REPORT "HERE REALLY \@children \: " . dump(@children);
		my @verts;
		
		foreach my $child ( @children )
		{
			my $parent = $datalist{ $zonenum }{ $child }{ parent };
			my @vertnums = @{ $datalist{ $zonenum }{ $child }{vertnums} }; say REPORT "HERE REALLY \@vertnums \: " . dump(@vertnums);
			foreach my $vertnum ( @vertnums )
			{
				say REPORT "HERE REALLY \$vertnum \: " . dump($vertnum);
				say REPORT "HERE REALLY LAST: \$datalist{ \$zonenum }{ \$vertnum }{vertcoords} " . dump( $datalist{ $zonenum }{ $vertnum }{vertcoords} );
				push ( @verts, $datalist{ $zonenum }{ $vertnum }{ vertcoords } ); say REPORT " HERE REALLY \$datalist{ \$zonenum }{ \$vertnum }{vertcoords} : " . dump( $datalist{ $zonenum }{ $vertnum }{ vertcoords } );
			}
			say REPORT "HERE REALLY \@verts \: " . dump(@verts);
			
			my ( @coords );
			foreach my $v ( @gridpoints )
			{
				my @fields = @$v; say REPORT "REALLY \@fields \: " . dump(@fields);
				my $coordsref = $fields[0]; #say "\$coordsref \: " . dump($coordsref);
				my $surfnum = $fields[1]; say "\$surfnum \: " . dump($surfnum);
				if ( $surfnum ~~ $parent )
				{
					push( @coords, [ @$coordsref ] );
				}
			} say REPORT "REALLY \@coords \: " . dump(@coords);
		
			my ( @boxpointxy, @boxpointxz, @boxpointyz );
			foreach my $gridpoint ( @coords )
			{
				my ( @point_xys, @point_xzs, @point_yzs );
				foreach ( @$gridpoint )
				{
					push ( @point_xys, [ $_->[0], $_->[1] ] );
					push ( @point_xzs, [ $_->[0], $_->[2] ] );
					push ( @point_yzs, [ $_->[1], $_->[2] ] );
				}
				push ( @boxpointxy, [ @point_xys ] );
				push ( @boxpointxz, [ @point_xzs ] );
				push ( @boxpointyz, [ @point_yzs ] );
			} say REPORT "REALLY \@boxpointxy \: " . dump(@boxpointxy); say REPORT "\@boxpointxz \: " . dump(@boxpointxz); say REPORT "\@boxpointyz \: " . dump(@boxpointyz);
			
			my ( @boxvertxy, @boxvertxz, @boxvertyz );
			say REPORT "REALLY \@verts! \: " . dump(@verts);
			foreach my $vert ( @verts )
			{
				push ( @boxvertxy, [ $vert->[0], $vert->[1] ] );
				push ( @boxvertxz, [ $vert->[0], $vert->[2] ] );
				push ( @boxvertyz, [ $vert->[1], $vert->[2] ] );
			} say REPORT "REALLY \@boxvertxy \: " . dump(@boxvertxy);say REPORT "\@boxvertxz \: " . dump(@boxvertxz);say REPORT "\@boxvertyz \: " . dump(@boxvertyz);
	
			my $count = 0;
			my ( $vert_xys, $vert_xzs, $vert_yzs, $polyxy, $polyxz, $polyyz );
			my ( @verts_xys, @verts_xzs, @verts_yzs);
			foreach my $case ( @boxvertxy )
			{
				$vert_xys = $boxvertxy[$count]; say REPORT "REALLY \$vert_xys \: " . dump($vert_xys); #$vert_xys = Math::Geometry::Planar->new;
				$vert_xzs = $boxvertxz[$count]; say REPORT "REALLY \$vert_xzs \: " . dump($vert_xzs); #$vert_xzs = Math::Geometry::Planar->new;
				$vert_yzs = $boxvertyz[$count]; say REPORT "REALLY \$vert_yzs \: " . dump($vert_yzs); #$vert_yzs = Math::Geometry::Planar->new;
				#$polyxy = Math::Polygon::Tree->new( $vert_xys ); say REPORT "REALLY \$polyxy \: " . dump($polyxy);
				#$polyxz = Math::Polygon::Tree->new( $vert_xzs ); say REPORT "REALLY \$polyxz \: " . dump($polyxz);
				#$polyyz = Math::Polygon::Tree->new( $vert_yzs ); say REPORT "REALLY \$polyyz \: " . dump($polyyz);
				$count++;
			}
	
			foreach my $caseref ( @gridpoints )
			{
				my @case = @$caseref; say REPORT "REALLY \$xyref \: " . dump($xyref);
				my $surfnum = $case[ 1 ];
				my $dirvector = $case[ 2 ];
				my @bag;
				foreach my $vert ( @{ $case[ 0 ] } )
				
				{	say REPORT "HERE REALLY LAST WATCH \$vert \: " . dump($vert);
					my $xyref = $boxpointxy[ $count ][ 0 ]; say REPORT "REALLY \$xyref \: " . dump($xyref);
					my $xzref = $boxpointxz[ $count ][ 0 ]; say REPORT "REALLY \$xzref \: " . dump($xzref);
					my $yzref = $boxpointyz[ $count ][ 0 ]; say REPORT "REALLY \$yzref \: " . dump($yzref);
					#unless ( ( $polyxy->contains( $xyref ) ) == 0 ) { say REPORT "REALLY \$polyxy contains \$xyref\. "; } else { say REPORT "REALLY \$polyxy doesn't contain \$xyref\. " } say REPORT "\$polyxy \: " . dump($polyxy); say REPORT "\$xyref \: " . dump($xyref);  
					#unless ( ( $polyxz->contains( $xzref ) ) == 0 ) { say REPORT "REALLY \$polyxz contains \$xzref\. "; } else { say REPORT "REALLY \$polyxz doesn't contain \$xzref\. " } say REPORT "\$polyxz \: " . dump($polyxz); say REPORT "\$xzref \: " . dump($xzref); 
					#unless ( ( $polyyz->contains( $yzref ) ) == 0 ) { say REPORT "REALLY \$polyyz contains \$yzref\. "; } else { say REPORT "REALLY \$polyyz doesn't contain \$yzref\. " } say REPORT "\$polyyz \: " . dump($polyyz); say REPORT "\$yzref \: " . dump($yzref); 
					#unless ( ( ( $polyxy->contains( $xyref ) ) == 0 ) and ( ( $polyxz->contains( $xzref ) ) == 0 ) and ( ( $polyyz->contains( $yzref ) ) == 0 ) )
					{
						push( @bag, $vert ); 
					}
				}
				say REPORT "REALLY \@bag \: " . dump(@bag);
				push ( @newbox, [ [ @bag ], $surfnum, $dirvector ] );
				$count++;
			} say REPORT "REALLY \@newbox \: " . dump(@newbox);
		}	
	}
	return ( @newbox );
}


sub modish
{	# MAIN PROGRAM
	my $launchfile = shift;
	my @restpars = @_; #REMAINING PARAMETERS GIVEN TO THE SHELL
	my ( $conffile, $path, $zonenum, $dirvectorsnum, $bounceambnum, $bouncemaxnum, $distgrid );
	my ( @transpdata, @surfaces, @dirvectorsrefs, @transpsurfs, @resolutions );
	
	if ( scalar( @restpars ) == 0 ) 
	{ 	# IF modish.pl IS CALLED THROUGH A CONFIGURATION FILE...
		require $launchfile; #say "\@launchdata: " . dump(@launchdata);
		$conffile = $launchdata[0]; #say "\$conffile: " . dump($conffile);
		$path = definepath($conffile); #say "\$path:" . dump($path);
		$zonenum = $launchdata[1]; #say "\@transpdata " . dump(@transpdata);
		@transpsurfs = @launchdata[ 2..$#launchdata ];
	}
	else 
	{	# IF modish.pl IS CALLED ENTIRELY THROUGH THE COMMAND LINE, ASSUME THOSE DEFAULTS: RESOLUTION: 2, 2; DIRECTION VECTORS: 1; diffuse reflections: 1; direct reflections: 7; distance of the calculation point from the surface: 0.01 meter.
		$conffile = $launchfile; #say "\$conffile" . dump($conffile);
		$path = definepath($conffile); #say "\$path:" . dump($path);
		$zonenum = shift( @restpars ); #say "\$zonenum " . dump($zonenum);
		@transpsurfs = @restpars; #say "\@transpsurfs " . dump(@transpsurfs);
	}
	if ( -e "./modish_defaults.pl" )
	{ 
		require "./modish_defaults.pl";
	}
	@resolutions = @{ $defaults[0] };
	$dirvectorsnum = $defaults[1];
	$bounceambnum = $defaults[2];
	$bouncemaxnum = $defaults[3];
	$distgrid = $defaults[4];
	
	open ( REPORT, ">$path/writefile.txt" ) or die;
	open ( REPORT2, ">$path/report2.txt" ) or die;
	
	#say "caseref: " . dump($caseref);
	
	if ( not ( @resolutions ) ) { @resolutions = ( 2, 2 ); };
	@resolutions = ( $resolutions[0], @resolutions ); #say "resolutions: " . dump(@resolutions);
	if ( not ( $dirvectorsnum ) ) { $dirvectorsnum = 1; }; #say "\$dirvectorsnum " . dump($dirvectorsnum); 
	if ( not ( $bounceambnum ) ) { $bounceambnum = 1; }; #say "\$bounceambnum " . dump($bounceambnum); 
	if ( not ( $bouncemaxnum ) ) { $bouncemaxnum = 7; }; #say "\$bouncemaxnum " . dump($bouncemaxnum);
	if ( not ( $distgrid ) ) { $distgrid = 0.01; };
	
	my ($conffile, $conffile_fict1, $conffile_fict2, $conffile_fict3, $constrdbfile, $constrdbfile_fict,
	$matdbfile, $matdbfile_fict1, $matdbfile_fict2, $flagconstrdb, $flagmatdb, $flaggeo, $flagconstr, $originalsref, 
	$fictitia1ref, $fictitia2ref, $fictitia3ref ) = createfictitiousfiles( $conffile, $path, $zonenum ); 	
	#say "\$conffile, $conffile, \$conffile_fict1, $conffile_fict1, \$conffile_fict2, $conffile_fict2, \$constrdbfile, $constrdbfile, \$constrdbfile_fict1, $constrdbfile_fict,\$matdbfile, $matdbfile, \$matdbfile_fict1, $matdbfile_fict1, \$matdbfile_fict2, $matdbfile_fict2, \$flagconstrdb, $flagconstrdb, \$flagmatdb, $flagmatdb, \$flaggeo, $flaggeo, \$flagconstr, $flagconstr, \$originalsref: " . dump($originalsref) . ", \$fictitia1ref: " . dump($fictitia1ref) . ", \$fictitia2ref: " . dump($fictitia2ref); 
	
	my @basevectors = getbasevectors( $dirvectorsnum );
			
	my @originals = @$originalsref; #say "\@originals " . dump(@originals);
	my @fictitia1 = @$fictitia1ref; #say "\@fictitia1 " . dump(@fictitia1);
	my @fictitia2 = @$fictitia2ref; #say "\@fictitia2 " . dump(@fictitia2);
	my @fictitia3 = @$fictitia3ref; #say "\@fictitia3 " . dump(@fictitia3);
	
	my ( @actiondata, @daylighthours);
	my @zoneoriginals = @originals; 
	shift(@zoneoriginals); shift(@zoneoriginals);
	my @zonefictitia1 = @fictitia1; # "BLACK" MODEL
	shift(@zonefictitia1); shift(@zonefictitia1);
	my @zonefictitia2 = @fictitia2; # "WHITE" MODEL
	shift(@zonefictitia2); shift(@zonefictitia2);
	my @zonefictitia3 = @fictitia3; # "BLACK" MODEL WITHOUT SHADINGS
	shift(@zonefictitia3); shift(@zonefictitia3);
	
	my ( %zonefilelists, %fict1filelists, %fict2filelists, %fict3filelists );
	my @daylighthoursarr;
	my %daylighthours;
	my ( $exportreflref__, $exportconstrref__ );
	my $countzone = 1;
	foreach my $elt (@zoneoriginals)
	{
		my @zonefiles = @$elt; #say "\@zonefiles " . dump(@zonefiles);
		my @fict1files = @{ $zonefictitia1[ $countzone - 1 ] }; #say "\@fict1files " . dump(@fict1files);
		my @fict2files = @{ $zonefictitia2[ $countzone - 1 ] }; #say "\@fict2files " . dump(@fict2files);
		my @fict3files = @{ $zonefictitia3[ $countzone - 1 ] }; #say "\@fict3files " . dump(@fict3files);
		my $geofile = $zonefiles[0]; #say "GEOFILE: $geofile";
		my $constrfile = $zonefiles[1]; #say "\$constrfile $constrfile";
		my $shdfile = $zonefiles[2]; #say "\$shdfile $shdfile";
		my $zonenum_cfg = $zonefiles[3]; #say "\$zonenum_cfg $zonenum_cfg";
		my $geofile_fict = $fict1files[0]; #say "\$geofile_fict $geofile_fict";
		my $constrfile_fict = $fict1files[1]; #say "\$constrfile_fict $constrfile_fict";
		$zonefilelists{ $zonenum }{ geofile } = $geofile;
		$zonefilelists{ $zonenum }{ geofile_fict } = $geofile_fict;
		$zonefilelists{ $zonenum }{ constrfile } = $constrfile;
		$zonefilelists{ $zonenum }{ constrfile_fict } = $constrfile_fict;
		$zonefilelists{ $zonenum }{ shdfile } = $shdfile;
		
		my ( $transpeltsref, $geofilestructref, $surfslistref, $obsref, $obsconstrsetref, $datalistref ) = readgeofile( $geofile, \@transpsurfs, $zonenum ); #say REPORT "HERE winsdata: " . dump(@winsdata);
		
		my @transpelts = @$transpeltsref; #say REPORT "\@transpelts " . dump(@transpelts); # ( [ $surfname, $parent, $constr, $surfnum, $geofile ] ) ;
		my @geodata = @$geofilestructref; #say REPORT "\@geodata " . dump(@geodata);	
		my %surfslist = %$surfslistref; #say REPORT "HERE EX GEOFILE \%surfslist " . dump ( %surfslist );
		my @obsdata = @$obsref; #say REPORT "\@obsdata " . dump(@obsdata);	# $obsname, $obsconstr, $obsnum
		my @obsconstrset = @$obsconstrsetref; #say REPORT "\@obsconstrset: " . dump ( @obsconstrset );
		my %datalist = %$datalistref; say REPORT "HERE WATCH EX GEOFILE \%datalist " . dump ( %datalist );
		
		createfictgeofile( $geofile, \@obsconstrset, $geofile_fict );

		my ( $materialsref, $newmaterialsref, $matnumsref, $newmatnumsref, $exportconstrref ) = 
		createconstrdbfile( $constrdbfile, $constrdbfile_fict, \@obsconstrset ); #say REPORT "\$exportconstrref " . dump($exportconstrref);
		#say REPORT " materialsref " . dump($materialsref) . " newmaterialsref " . dump($newmaterialsref) . " matnumsref " . dump($matnumsref) . " newmatnumsref " . dump($newmatnumsref) ;
		$exportconstrref__ = $exportconstrref;
		
		my $exportreflref = creatematdbfiles( $materialsref, $newmaterialsref, $matnumsref, $newmatnumsref, 
		$matdbfile, $matdbfile_fict1, $matdbfile_fict2 ); #print REPORT "\$materialsref, " . Dumper($materialsref) . "\$newmaterialsref, " . Dumper( $newmaterialsref) . "\$matnumsref, " . Dumper( $matnumsref) . "\$newmatnumsref, "  . Dumper( $newmatnumsref ) .
		#"\$matdbfile, " . Dumper( $matdbfile ) . "\$matdbfile_fict1, " . Dumper( $matdbfile_fict1 ) . "\$matdbfile_fict2" . Dumper( $matdbfile_fict2 ); say REPORT "\RECEIVED__ $exportreflref " . Dumper($exportreflref); 	
		$exportreflref__ = $exportreflref;
		
		my ( $surfnumsref, $surfnamesref ) = tellsurfnames( \@transpsurfs, \@geodata ); 
		my @surfnums = @$surfnumsref; #say REPORT "HERE \@surfnums " . dump(@surfnums);
		my @surfnames = @$surfnamesref; #say REPORT "HERE \@surfnames " . dump(@surfnames);
		my ( $winseltsref, $datalistref ) = readverts( \@transpelts, $geofile, \@geodata, \%datalist ); #say "winselts: " . dump(@winselts);
		my @winselts = @$winseltsref; #say REPORT "HERE winselts: " . dump(@winselts);
		my %datalist = %$datalistref; #say REPORT "HERE WATCH EX READVERTS \%datalist " . dump(%datalist);
		my ( $winscoordsref, $datalistref ) = readcoords( \@winselts, $geofile, \@geodata, \%datalist, \@transpelts ); 
		my @winscoords = @$winscoordsref; #say REPORT "HERE \@winscoords " . dump(@winscoords);
		my %datalist = %$datalistref; #say REPORT "HERE WATCH EX READCOORDS \%datalist " . dump(%datalist);
		my @dirvectorsrefs = calcdirvectors( @winscoords ); #say REPORT "HERE \@dirvectorsrefs " . dump(@dirvectorsrefs);
		my @xyzcoords = getcorners( \@winscoords, \@winselts ); #say REPORT "HERE NOW \@xyzcoords " . dump(@xyzcoords);
		my @extremes = findextremes( @xyzcoords ); #say REPORT "HERE \@extremes: " . dump(@extremes);
		my @gridcoords = makecoordsgrid( \@extremes, \@resolutions, \@dirvectorsrefs ); #say REPORT "HERE \@gridcoords " . dump(@gridcoords);
		my @gridpoints_transitional = makegrid( @gridcoords ); #say REPORT "HERE NOW THIS \@gridpoints_transitional " . dump(@gridpoints_transitional);
		my @gridpoints_newtransitional = prunepoints( \@gridpoints_transitional, \@xyzcoords ); #say REPORT "HERE NOW THIS \@gridpoints_newtransitional " . dump(@gridpoints_newtransitional);
		my @gridpoints = adjustgrid( \@gridpoints_newtransitional, $distgrid ); #say REPORT  "HERE BEFORE \@gridpoints " . dump(@gridpoints);
		
		my ( $treatedlinesref, $filearrayref, $monthsref ) = readshdfile( $shdfile ); 
		my @treatedlines = @$treatedlinesref;
		my @shdfilearray = @$filearrayref; #say "\@shdfilearray " . dump(@shdfilearray);
		my @months = @$monthsref; #say "\@months " . dump(@months);
		my @shdsurfdata = getsurfshd( \@shdfilearray, \@months, \@surfnums, \@surfnames ); #say "\@shdsurfdata " . dump(@shdsurfdata);
		@daylighthoursarr = checklight( \@shdfilearray, \@months ); #say "\@daylighthoursarr " . dump(@daylighthoursarr);		
		%daylighthours = populatelight( @daylighthoursarr ); # say REPORT "\%daylighthours " . dump(%daylighthours);
		push ( @actiondata, [ [ $zonenum, $geofile, $constrfile, $shdfile ], [ @gridpoints ], [ @shdsurfdata ], [ @daylighthoursarr ], [ @shdfilearray ] ] ); 
		$shdfileslist{ $zonenum } = \@treatedlines; #say "\$shdfileslist " . dump($shdfileslist);
		$countzone++;
	}
	my @radfilesrefs = tellradfilenames( $path, $conffile_fict1, $conffile_fict2, $conffile_fict3 ); # say "\@radfilesrefs " . dump( @radfilesrefs );
	unshift( @actiondata, $path, @radfilesrefs, $bounceambnum, $bouncemaxnum ); #say "OBTAINED \@actiondata " . dump( @actiondata );
	my ( $hashirrsref ) = pursue( $exportconstrref__, $exportreflref__, $conffile, \@basevectors, @actiondata ); #say REPORT "OBTAINED \$hashirrsref\: " . Dumper( $hashirrsref ); say REPORT "\$hashirrsref\: " . Dumper( $hashirrsref );
	my ( $irrvarsref ) = compareirrs( \%zonefilelists, $hashirrsref ); #say REPORT "OBTAINED \%irrvars\: " . Dumper(%irrvars); 				
	modifyshda( \@comparedirrs, \%surfslist, \%zonefilelists, \%shdfileslist, \%daylighthours, $irrvarsref ); 
	# @treatedlines IS A PRE-TREATED SHD FILE PUT IN MEMORY§
}

modish(@ARGV);

1;

__END__

=head1 NAME

Sim::OPT::Modish.

=head1 SYNOPSIS

  use Sim::OPT::Modish;
  modish(Esp-r_configuration file);

=head1 DESCRIPTION

Sim::OPT::Modish is a program for calculating the modification of diffuse solar shading to be specified in the ISH (shading and insolation) files of ESP-r building simulation suite to take into account the effect of the reflections due to the shadings.  
That effect is calculated: 
1) by calculating the difference between:
(a) the amount (measured as irradiance) of solar radiation (including direct and diffuse) impinging a surface in the case that obstructions have their true reflectivity and all other surfaces with the exception of the ground are completely black;
b) the amount of solar radiation impinging the external side of that surface in the case that solar obstructions and all other surfaces in the scene with the exception of the ground are completely black.
2) by calculating the ratio of: 
(a) the result of the calculation at point 1; and 
b) the amount (measured as irradiance) of diffuse solar radiation impinging the surface. 
The value given by 1) minus the result of 2) gives the diffuse shading factor to be put in the ISH file for that surface.
When the reflections due to the shadings increase the amount of radiation arriving on the surface, that diffuse shading factor is negative.

Gian Luca Brunetti, Politecnico di Milano, 2015.
gianluca.brunetti@polimi.it

=head2 EXPORT

"opt".

=head1 SEE ALSO

The available examples are collected in the "example" directory in this distribution.

=head1 AUTHOR

Gian Luca Brunetti, E<lt>gianluca.brunetti@polimi.itE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2015 by Gian Luca Brunetti and Politecnico di Milano. This is free software.  You can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, version 2 or later.


=cut
