package Unicode::CharName;

require Exporter;
@ISA=qw(Exporter);
@EXPORT_OK = qw(uname ublock);

use strict;
use vars qw(%JAMO_SHORT_NAME %NAMES @BLOCKS);

sub uname {
    my $code = shift;
    if ($code >= 0x4E00) {
	if ($code <= 0x9FFF || ($code >= 0xF900 && $code <= 0xFAFF)) {
	    # CJK Ideographs
	    return sprintf "CJK UNIFIED IDEOGRAPH %04X", $code;
	} elsif ($code >= 0xD800 && $code <= 0xF8FF) {
	    # Surrogate and private
	    if ($code <= 0xDFFF) {
		return "<surrogate>";
	    } else {
		return "<private>";
	    }
	} elsif ($code >= 0xAC00 && $code <= 0xD7A3) {
	    # Hangul Syllables
	    my $sindex = $code - 0xAC00;
	    my $l = 0x1100 + int($sindex / (21*28));
	    my $v = 0x1161 + int(($sindex % (21*28)) / 28);
	    my $t = 0x11A7 + $sindex % 28;
	    my @s = ($l, $v, $t);
	    pop(@s) if $t == 0x11A7;
	    @s = map {
		$_ = sprintf("%04X", $_);
		$JAMO_SHORT_NAME{$_} || "U+$_";
            } @s;
	    return "HANGUL SYLLABLE @s";
	} elsif ($code > 0xFFFF) {
	    return undef;  # outside Unicode range
	}
    }
    _init_names() unless defined %NAMES;
    $NAMES{sprintf("%04X",$code)}
}

sub ublock
{
    my $code = shift;
    # XXX: could use a binary search, but I am too lazy today...
    my $block;
    for $block (@BLOCKS) {
	return $block->[2] if $block->[0] <= $code && $block->[1] >= $code;
    }
    undef;
}

@BLOCKS = (
#  start   end        block name
  [0x0000, 0x007F => 'Basic Latin'],
  [0x0080, 0x00FF => 'Latin-1 Supplement'],
  [0x0100, 0x017F => 'Latin Extended-A'],
  [0x0180, 0x024F => 'Latin Extended-B'],
  [0x0250, 0x02AF => 'IPA Extensions'],
  [0x02B0, 0x02FF => 'Spacing Modifier Letters'],
  [0x0300, 0x036F => 'Combining Diacritical Marks'],
  [0x0370, 0x03FF => 'Greek'],
  [0x0400, 0x04FF => 'Cyrillic'],
  [0x0530, 0x058F => 'Armenian'],
  [0x0590, 0x05FF => 'Hebrew'],
  [0x0600, 0x06FF => 'Arabic'],
  [0x0900, 0x097F => 'Devanagari'],
  [0x0980, 0x09FF => 'Bengali'],
  [0x0A00, 0x0A7F => 'Gurmukhi'],
  [0x0A80, 0x0AFF => 'Gujarati'],
  [0x0B00, 0x0B7F => 'Oriya'],
  [0x0B80, 0x0BFF => 'Tamil'],
  [0x0C00, 0x0C7F => 'Telugu'],
  [0x0C80, 0x0CFF => 'Kannada'],
  [0x0D00, 0x0D7F => 'Malayalam'],
  [0x0E00, 0x0E7F => 'Thai'],
  [0x0E80, 0x0EFF => 'Lao'],
  [0x0F00, 0x0FBF => 'Tibetan'],
  [0x10A0, 0x10FF => 'Georgian'],
  [0x1100, 0x11FF => 'Hangul Jamo'],
  [0x1E00, 0x1EFF => 'Latin Extended Additional'],
  [0x1F00, 0x1FFF => 'Greek Extended'],
  [0x2000, 0x206F => 'General Punctuation'],
  [0x2070, 0x209F => 'Superscripts and Subscripts'],
  [0x20A0, 0x20CF => 'Currency Symbols'],
  [0x20D0, 0x20FF => 'Combining Marks for Symbols'],
  [0x2100, 0x214F => 'Letterlike Symbols'],
  [0x2150, 0x218F => 'Number Forms'],
  [0x2190, 0x21FF => 'Arrows'],
  [0x2200, 0x22FF => 'Mathematical Operators'],
  [0x2300, 0x23FF => 'Miscellaneous Technical'],
  [0x2400, 0x243F => 'Control Pictures'],
  [0x2440, 0x245F => 'Optical Character Recognition'],
  [0x2460, 0x24FF => 'Enclosed Alphanumerics'],
  [0x2500, 0x257F => 'Box Drawing'],
  [0x2580, 0x259F => 'Block Elements'],
  [0x25A0, 0x25FF => 'Geometric Shapes'],
  [0x2600, 0x26FF => 'Miscellaneous Symbols'],
  [0x2700, 0x27BF => 'Dingbats'],
  [0x3000, 0x303F => 'CJK Symbols and Punctuation'],
  [0x3040, 0x309F => 'Hiragana'],
  [0x30A0, 0x30FF => 'Katakana'],
  [0x3100, 0x312F => 'Bopomofo'],
  [0x3130, 0x318F => 'Hangul Compatibility Jamo'],
  [0x3190, 0x319F => 'Kanbun'],
  [0x3200, 0x32FF => 'Enclosed CJK Letters and Months'],
  [0x3300, 0x33FF => 'CJK Compatibility'],
  [0x4E00, 0x9FFF => 'CJK Unified Ideographs'],
  [0xAC00, 0xD7A3 => 'Hangul Syllables'],
  [0xD800, 0xDB7F => 'High Surrogates'],
  [0xDB80, 0xDBFF => 'High Private Use Surrogates'],
  [0xDC00, 0xDFFF => 'Low Surrogates'],
  [0xE000, 0xF8FF => 'Private Use'],
  [0xF900, 0xFAFF => 'CJK Compatibility Ideographs'],
  [0xFB00, 0xFB4F => 'Alphabetic Presentation Forms'],
  [0xFB50, 0xFDFF => 'Arabic Presentation Forms-A'],
  [0xFE20, 0xFE2F => 'Combining Half Marks'],
  [0xFE30, 0xFE4F => 'CJK Compatibility Forms'],
  [0xFE50, 0xFE6F => 'Small Form Variants'],
  [0xFE70, 0xFEFF => 'Arabic Presentation Forms-B'],
  [0xFF00, 0xFFEF => 'Halfwidth and Fullwidth Forms'],
  [0xFEFF, 0xFEFF => 'Specials'],
  [0xFFF0, 0xFFFF => 'Specials'],
);

%JAMO_SHORT_NAME = (
    '1100' => 'G',
    '1101' => 'GG',
    '1102' => 'N',
    '1103' => 'D',
    '1104' => 'DD',
    '1105' => 'L',
    '1106' => 'M',
    '1107' => 'B',
    '1108' => 'BB',
    '1109' => 'S',
    '110A' => 'SS',
    '110B' => '',
    '110C' => 'J',
    '110D' => 'JJ',
    '110E' => 'C',
    '110F' => 'K',
    '1110' => 'T',
    '1111' => 'P',
    '1112' => 'H',
    '1161' => 'A',
    '1162' => 'AE',
    '1163' => 'YA',
    '1164' => 'YAE',
    '1165' => 'EO',
    '1166' => 'E',
    '1167' => 'YEO',
    '1168' => 'YE',
    '1169' => 'O',
    '116A' => 'WA',
    '116B' => 'WAE',
    '116C' => 'OE',
    '116D' => 'YO',
    '116E' => 'U',
    '116F' => 'WEO',
    '1170' => 'WE',
    '1171' => 'WI',
    '1172' => 'YU',
    '1173' => 'EU',
    '1174' => 'YI',
    '1175' => 'I',
    '11A8' => 'G',
    '11A9' => 'GG',
    '11AA' => 'GS',
    '11AB' => 'N',
    '11AC' => 'NJ',
    '11AD' => 'NH',
    '11AE' => 'D',
    '11AF' => 'L',
    '11B0' => 'LG',
    '11B1' => 'LM',
    '11B2' => 'LB',
    '11B3' => 'LS',
    '11B4' => 'LT',
    '11B5' => 'LP',
    '11B6' => 'LH',
    '11B7' => 'M',
    '11B8' => 'B',
    '11B9' => 'BS',
    '11BA' => 'S',
    '11BB' => 'SS',
    '11BC' => 'NG',
    '11BD' => 'J',
    '11BE' => 'C',
    '11BF' => 'K',
    '11C0' => 'T',
    '11C1' => 'P',
    '11C2' => 'H',
);

sub _init_names
{
    keys %NAMES = 6588;  # preextent
    local($_);
    while (<DATA>) {
	chop;
	my($code, $name) = split(' ', $_, 2);
	$NAMES{$code} = $name;
    }
    close(DATA);
}

1;


__DATA__
0000 <control>
0001 <control>
0002 <control>
0003 <control>
0004 <control>
0005 <control>
0006 <control>
0007 <control>
0008 <control>
0009 <control>
000A <control>
000B <control>
000C <control>
000D <control>
000E <control>
000F <control>
0010 <control>
0011 <control>
0012 <control>
0013 <control>
0014 <control>
0015 <control>
0016 <control>
0017 <control>
0018 <control>
0019 <control>
001A <control>
001B <control>
001C <control>
001D <control>
001E <control>
001F <control>
0020 SPACE
0021 EXCLAMATION MARK
0022 QUOTATION MARK
0023 NUMBER SIGN
0024 DOLLAR SIGN
0025 PERCENT SIGN
0026 AMPERSAND
0027 APOSTROPHE
0028 LEFT PARENTHESIS
0029 RIGHT PARENTHESIS
002A ASTERISK
002B PLUS SIGN
002C COMMA
002D HYPHEN-MINUS
002E FULL STOP
002F SOLIDUS
0030 DIGIT ZERO
0031 DIGIT ONE
0032 DIGIT TWO
0033 DIGIT THREE
0034 DIGIT FOUR
0035 DIGIT FIVE
0036 DIGIT SIX
0037 DIGIT SEVEN
0038 DIGIT EIGHT
0039 DIGIT NINE
003A COLON
003B SEMICOLON
003C LESS-THAN SIGN
003D EQUALS SIGN
003E GREATER-THAN SIGN
003F QUESTION MARK
0040 COMMERCIAL AT
0041 LATIN CAPITAL LETTER A
0042 LATIN CAPITAL LETTER B
0043 LATIN CAPITAL LETTER C
0044 LATIN CAPITAL LETTER D
0045 LATIN CAPITAL LETTER E
0046 LATIN CAPITAL LETTER F
0047 LATIN CAPITAL LETTER G
0048 LATIN CAPITAL LETTER H
0049 LATIN CAPITAL LETTER I
004A LATIN CAPITAL LETTER J
004B LATIN CAPITAL LETTER K
004C LATIN CAPITAL LETTER L
004D LATIN CAPITAL LETTER M
004E LATIN CAPITAL LETTER N
004F LATIN CAPITAL LETTER O
0050 LATIN CAPITAL LETTER P
0051 LATIN CAPITAL LETTER Q
0052 LATIN CAPITAL LETTER R
0053 LATIN CAPITAL LETTER S
0054 LATIN CAPITAL LETTER T
0055 LATIN CAPITAL LETTER U
0056 LATIN CAPITAL LETTER V
0057 LATIN CAPITAL LETTER W
0058 LATIN CAPITAL LETTER X
0059 LATIN CAPITAL LETTER Y
005A LATIN CAPITAL LETTER Z
005B LEFT SQUARE BRACKET
005C REVERSE SOLIDUS
005D RIGHT SQUARE BRACKET
005E CIRCUMFLEX ACCENT
005F LOW LINE
0060 GRAVE ACCENT
0061 LATIN SMALL LETTER A
0062 LATIN SMALL LETTER B
0063 LATIN SMALL LETTER C
0064 LATIN SMALL LETTER D
0065 LATIN SMALL LETTER E
0066 LATIN SMALL LETTER F
0067 LATIN SMALL LETTER G
0068 LATIN SMALL LETTER H
0069 LATIN SMALL LETTER I
006A LATIN SMALL LETTER J
006B LATIN SMALL LETTER K
006C LATIN SMALL LETTER L
006D LATIN SMALL LETTER M
006E LATIN SMALL LETTER N
006F LATIN SMALL LETTER O
0070 LATIN SMALL LETTER P
0071 LATIN SMALL LETTER Q
0072 LATIN SMALL LETTER R
0073 LATIN SMALL LETTER S
0074 LATIN SMALL LETTER T
0075 LATIN SMALL LETTER U
0076 LATIN SMALL LETTER V
0077 LATIN SMALL LETTER W
0078 LATIN SMALL LETTER X
0079 LATIN SMALL LETTER Y
007A LATIN SMALL LETTER Z
007B LEFT CURLY BRACKET
007C VERTICAL LINE
007D RIGHT CURLY BRACKET
007E TILDE
007F <control>
0080 <control>
0081 <control>
0082 <control>
0083 <control>
0084 <control>
0085 <control>
0086 <control>
0087 <control>
0088 <control>
0089 <control>
008A <control>
008B <control>
008C <control>
008D <control>
008E <control>
008F <control>
0090 <control>
0091 <control>
0092 <control>
0093 <control>
0094 <control>
0095 <control>
0096 <control>
0097 <control>
0098 <control>
0099 <control>
009A <control>
009B <control>
009C <control>
009D <control>
009E <control>
009F <control>
00A0 NO-BREAK SPACE
00A1 INVERTED EXCLAMATION MARK
00A2 CENT SIGN
00A3 POUND SIGN
00A4 CURRENCY SIGN
00A5 YEN SIGN
00A6 BROKEN BAR
00A7 SECTION SIGN
00A8 DIAERESIS
00A9 COPYRIGHT SIGN
00AA FEMININE ORDINAL INDICATOR
00AB LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
00AC NOT SIGN
00AD SOFT HYPHEN
00AE REGISTERED SIGN
00AF MACRON
00B0 DEGREE SIGN
00B1 PLUS-MINUS SIGN
00B2 SUPERSCRIPT TWO
00B3 SUPERSCRIPT THREE
00B4 ACUTE ACCENT
00B5 MICRO SIGN
00B6 PILCROW SIGN
00B7 MIDDLE DOT
00B8 CEDILLA
00B9 SUPERSCRIPT ONE
00BA MASCULINE ORDINAL INDICATOR
00BB RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
00BC VULGAR FRACTION ONE QUARTER
00BD VULGAR FRACTION ONE HALF
00BE VULGAR FRACTION THREE QUARTERS
00BF INVERTED QUESTION MARK
00C0 LATIN CAPITAL LETTER A WITH GRAVE
00C1 LATIN CAPITAL LETTER A WITH ACUTE
00C2 LATIN CAPITAL LETTER A WITH CIRCUMFLEX
00C3 LATIN CAPITAL LETTER A WITH TILDE
00C4 LATIN CAPITAL LETTER A WITH DIAERESIS
00C5 LATIN CAPITAL LETTER A WITH RING ABOVE
00C6 LATIN CAPITAL LETTER AE
00C7 LATIN CAPITAL LETTER C WITH CEDILLA
00C8 LATIN CAPITAL LETTER E WITH GRAVE
00C9 LATIN CAPITAL LETTER E WITH ACUTE
00CA LATIN CAPITAL LETTER E WITH CIRCUMFLEX
00CB LATIN CAPITAL LETTER E WITH DIAERESIS
00CC LATIN CAPITAL LETTER I WITH GRAVE
00CD LATIN CAPITAL LETTER I WITH ACUTE
00CE LATIN CAPITAL LETTER I WITH CIRCUMFLEX
00CF LATIN CAPITAL LETTER I WITH DIAERESIS
00D0 LATIN CAPITAL LETTER ETH
00D1 LATIN CAPITAL LETTER N WITH TILDE
00D2 LATIN CAPITAL LETTER O WITH GRAVE
00D3 LATIN CAPITAL LETTER O WITH ACUTE
00D4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX
00D5 LATIN CAPITAL LETTER O WITH TILDE
00D6 LATIN CAPITAL LETTER O WITH DIAERESIS
00D7 MULTIPLICATION SIGN
00D8 LATIN CAPITAL LETTER O WITH STROKE
00D9 LATIN CAPITAL LETTER U WITH GRAVE
00DA LATIN CAPITAL LETTER U WITH ACUTE
00DB LATIN CAPITAL LETTER U WITH CIRCUMFLEX
00DC LATIN CAPITAL LETTER U WITH DIAERESIS
00DD LATIN CAPITAL LETTER Y WITH ACUTE
00DE LATIN CAPITAL LETTER THORN
00DF LATIN SMALL LETTER SHARP S
00E0 LATIN SMALL LETTER A WITH GRAVE
00E1 LATIN SMALL LETTER A WITH ACUTE
00E2 LATIN SMALL LETTER A WITH CIRCUMFLEX
00E3 LATIN SMALL LETTER A WITH TILDE
00E4 LATIN SMALL LETTER A WITH DIAERESIS
00E5 LATIN SMALL LETTER A WITH RING ABOVE
00E6 LATIN SMALL LETTER AE
00E7 LATIN SMALL LETTER C WITH CEDILLA
00E8 LATIN SMALL LETTER E WITH GRAVE
00E9 LATIN SMALL LETTER E WITH ACUTE
00EA LATIN SMALL LETTER E WITH CIRCUMFLEX
00EB LATIN SMALL LETTER E WITH DIAERESIS
00EC LATIN SMALL LETTER I WITH GRAVE
00ED LATIN SMALL LETTER I WITH ACUTE
00EE LATIN SMALL LETTER I WITH CIRCUMFLEX
00EF LATIN SMALL LETTER I WITH DIAERESIS
00F0 LATIN SMALL LETTER ETH
00F1 LATIN SMALL LETTER N WITH TILDE
00F2 LATIN SMALL LETTER O WITH GRAVE
00F3 LATIN SMALL LETTER O WITH ACUTE
00F4 LATIN SMALL LETTER O WITH CIRCUMFLEX
00F5 LATIN SMALL LETTER O WITH TILDE
00F6 LATIN SMALL LETTER O WITH DIAERESIS
00F7 DIVISION SIGN
00F8 LATIN SMALL LETTER O WITH STROKE
00F9 LATIN SMALL LETTER U WITH GRAVE
00FA LATIN SMALL LETTER U WITH ACUTE
00FB LATIN SMALL LETTER U WITH CIRCUMFLEX
00FC LATIN SMALL LETTER U WITH DIAERESIS
00FD LATIN SMALL LETTER Y WITH ACUTE
00FE LATIN SMALL LETTER THORN
00FF LATIN SMALL LETTER Y WITH DIAERESIS
0100 LATIN CAPITAL LETTER A WITH MACRON
0101 LATIN SMALL LETTER A WITH MACRON
0102 LATIN CAPITAL LETTER A WITH BREVE
0103 LATIN SMALL LETTER A WITH BREVE
0104 LATIN CAPITAL LETTER A WITH OGONEK
0105 LATIN SMALL LETTER A WITH OGONEK
0106 LATIN CAPITAL LETTER C WITH ACUTE
0107 LATIN SMALL LETTER C WITH ACUTE
0108 LATIN CAPITAL LETTER C WITH CIRCUMFLEX
0109 LATIN SMALL LETTER C WITH CIRCUMFLEX
010A LATIN CAPITAL LETTER C WITH DOT ABOVE
010B LATIN SMALL LETTER C WITH DOT ABOVE
010C LATIN CAPITAL LETTER C WITH CARON
010D LATIN SMALL LETTER C WITH CARON
010E LATIN CAPITAL LETTER D WITH CARON
010F LATIN SMALL LETTER D WITH CARON
0110 LATIN CAPITAL LETTER D WITH STROKE
0111 LATIN SMALL LETTER D WITH STROKE
0112 LATIN CAPITAL LETTER E WITH MACRON
0113 LATIN SMALL LETTER E WITH MACRON
0114 LATIN CAPITAL LETTER E WITH BREVE
0115 LATIN SMALL LETTER E WITH BREVE
0116 LATIN CAPITAL LETTER E WITH DOT ABOVE
0117 LATIN SMALL LETTER E WITH DOT ABOVE
0118 LATIN CAPITAL LETTER E WITH OGONEK
0119 LATIN SMALL LETTER E WITH OGONEK
011A LATIN CAPITAL LETTER E WITH CARON
011B LATIN SMALL LETTER E WITH CARON
011C LATIN CAPITAL LETTER G WITH CIRCUMFLEX
011D LATIN SMALL LETTER G WITH CIRCUMFLEX
011E LATIN CAPITAL LETTER G WITH BREVE
011F LATIN SMALL LETTER G WITH BREVE
0120 LATIN CAPITAL LETTER G WITH DOT ABOVE
0121 LATIN SMALL LETTER G WITH DOT ABOVE
0122 LATIN CAPITAL LETTER G WITH CEDILLA
0123 LATIN SMALL LETTER G WITH CEDILLA
0124 LATIN CAPITAL LETTER H WITH CIRCUMFLEX
0125 LATIN SMALL LETTER H WITH CIRCUMFLEX
0126 LATIN CAPITAL LETTER H WITH STROKE
0127 LATIN SMALL LETTER H WITH STROKE
0128 LATIN CAPITAL LETTER I WITH TILDE
0129 LATIN SMALL LETTER I WITH TILDE
012A LATIN CAPITAL LETTER I WITH MACRON
012B LATIN SMALL LETTER I WITH MACRON
012C LATIN CAPITAL LETTER I WITH BREVE
012D LATIN SMALL LETTER I WITH BREVE
012E LATIN CAPITAL LETTER I WITH OGONEK
012F LATIN SMALL LETTER I WITH OGONEK
0130 LATIN CAPITAL LETTER I WITH DOT ABOVE
0131 LATIN SMALL LETTER DOTLESS I
0132 LATIN CAPITAL LIGATURE IJ
0133 LATIN SMALL LIGATURE IJ
0134 LATIN CAPITAL LETTER J WITH CIRCUMFLEX
0135 LATIN SMALL LETTER J WITH CIRCUMFLEX
0136 LATIN CAPITAL LETTER K WITH CEDILLA
0137 LATIN SMALL LETTER K WITH CEDILLA
0138 LATIN SMALL LETTER KRA
0139 LATIN CAPITAL LETTER L WITH ACUTE
013A LATIN SMALL LETTER L WITH ACUTE
013B LATIN CAPITAL LETTER L WITH CEDILLA
013C LATIN SMALL LETTER L WITH CEDILLA
013D LATIN CAPITAL LETTER L WITH CARON
013E LATIN SMALL LETTER L WITH CARON
013F LATIN CAPITAL LETTER L WITH MIDDLE DOT
0140 LATIN SMALL LETTER L WITH MIDDLE DOT
0141 LATIN CAPITAL LETTER L WITH STROKE
0142 LATIN SMALL LETTER L WITH STROKE
0143 LATIN CAPITAL LETTER N WITH ACUTE
0144 LATIN SMALL LETTER N WITH ACUTE
0145 LATIN CAPITAL LETTER N WITH CEDILLA
0146 LATIN SMALL LETTER N WITH CEDILLA
0147 LATIN CAPITAL LETTER N WITH CARON
0148 LATIN SMALL LETTER N WITH CARON
0149 LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
014A LATIN CAPITAL LETTER ENG
014B LATIN SMALL LETTER ENG
014C LATIN CAPITAL LETTER O WITH MACRON
014D LATIN SMALL LETTER O WITH MACRON
014E LATIN CAPITAL LETTER O WITH BREVE
014F LATIN SMALL LETTER O WITH BREVE
0150 LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
0151 LATIN SMALL LETTER O WITH DOUBLE ACUTE
0152 LATIN CAPITAL LIGATURE OE
0153 LATIN SMALL LIGATURE OE
0154 LATIN CAPITAL LETTER R WITH ACUTE
0155 LATIN SMALL LETTER R WITH ACUTE
0156 LATIN CAPITAL LETTER R WITH CEDILLA
0157 LATIN SMALL LETTER R WITH CEDILLA
0158 LATIN CAPITAL LETTER R WITH CARON
0159 LATIN SMALL LETTER R WITH CARON
015A LATIN CAPITAL LETTER S WITH ACUTE
015B LATIN SMALL LETTER S WITH ACUTE
015C LATIN CAPITAL LETTER S WITH CIRCUMFLEX
015D LATIN SMALL LETTER S WITH CIRCUMFLEX
015E LATIN CAPITAL LETTER S WITH CEDILLA
015F LATIN SMALL LETTER S WITH CEDILLA
0160 LATIN CAPITAL LETTER S WITH CARON
0161 LATIN SMALL LETTER S WITH CARON
0162 LATIN CAPITAL LETTER T WITH CEDILLA
0163 LATIN SMALL LETTER T WITH CEDILLA
0164 LATIN CAPITAL LETTER T WITH CARON
0165 LATIN SMALL LETTER T WITH CARON
0166 LATIN CAPITAL LETTER T WITH STROKE
0167 LATIN SMALL LETTER T WITH STROKE
0168 LATIN CAPITAL LETTER U WITH TILDE
0169 LATIN SMALL LETTER U WITH TILDE
016A LATIN CAPITAL LETTER U WITH MACRON
016B LATIN SMALL LETTER U WITH MACRON
016C LATIN CAPITAL LETTER U WITH BREVE
016D LATIN SMALL LETTER U WITH BREVE
016E LATIN CAPITAL LETTER U WITH RING ABOVE
016F LATIN SMALL LETTER U WITH RING ABOVE
0170 LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
0171 LATIN SMALL LETTER U WITH DOUBLE ACUTE
0172 LATIN CAPITAL LETTER U WITH OGONEK
0173 LATIN SMALL LETTER U WITH OGONEK
0174 LATIN CAPITAL LETTER W WITH CIRCUMFLEX
0175 LATIN SMALL LETTER W WITH CIRCUMFLEX
0176 LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
0177 LATIN SMALL LETTER Y WITH CIRCUMFLEX
0178 LATIN CAPITAL LETTER Y WITH DIAERESIS
0179 LATIN CAPITAL LETTER Z WITH ACUTE
017A LATIN SMALL LETTER Z WITH ACUTE
017B LATIN CAPITAL LETTER Z WITH DOT ABOVE
017C LATIN SMALL LETTER Z WITH DOT ABOVE
017D LATIN CAPITAL LETTER Z WITH CARON
017E LATIN SMALL LETTER Z WITH CARON
017F LATIN SMALL LETTER LONG S
0180 LATIN SMALL LETTER B WITH STROKE
0181 LATIN CAPITAL LETTER B WITH HOOK
0182 LATIN CAPITAL LETTER B WITH TOPBAR
0183 LATIN SMALL LETTER B WITH TOPBAR
0184 LATIN CAPITAL LETTER TONE SIX
0185 LATIN SMALL LETTER TONE SIX
0186 LATIN CAPITAL LETTER OPEN O
0187 LATIN CAPITAL LETTER C WITH HOOK
0188 LATIN SMALL LETTER C WITH HOOK
0189 LATIN CAPITAL LETTER AFRICAN D
018A LATIN CAPITAL LETTER D WITH HOOK
018B LATIN CAPITAL LETTER D WITH TOPBAR
018C LATIN SMALL LETTER D WITH TOPBAR
018D LATIN SMALL LETTER TURNED DELTA
018E LATIN CAPITAL LETTER REVERSED E
018F LATIN CAPITAL LETTER SCHWA
0190 LATIN CAPITAL LETTER OPEN E
0191 LATIN CAPITAL LETTER F WITH HOOK
0192 LATIN SMALL LETTER F WITH HOOK
0193 LATIN CAPITAL LETTER G WITH HOOK
0194 LATIN CAPITAL LETTER GAMMA
0195 LATIN SMALL LETTER HV
0196 LATIN CAPITAL LETTER IOTA
0197 LATIN CAPITAL LETTER I WITH STROKE
0198 LATIN CAPITAL LETTER K WITH HOOK
0199 LATIN SMALL LETTER K WITH HOOK
019A LATIN SMALL LETTER L WITH BAR
019B LATIN SMALL LETTER LAMBDA WITH STROKE
019C LATIN CAPITAL LETTER TURNED M
019D LATIN CAPITAL LETTER N WITH LEFT HOOK
019E LATIN SMALL LETTER N WITH LONG RIGHT LEG
019F LATIN CAPITAL LETTER O WITH MIDDLE TILDE
01A0 LATIN CAPITAL LETTER O WITH HORN
01A1 LATIN SMALL LETTER O WITH HORN
01A2 LATIN CAPITAL LETTER OI
01A3 LATIN SMALL LETTER OI
01A4 LATIN CAPITAL LETTER P WITH HOOK
01A5 LATIN SMALL LETTER P WITH HOOK
01A6 LATIN LETTER YR
01A7 LATIN CAPITAL LETTER TONE TWO
01A8 LATIN SMALL LETTER TONE TWO
01A9 LATIN CAPITAL LETTER ESH
01AA LATIN LETTER REVERSED ESH LOOP
01AB LATIN SMALL LETTER T WITH PALATAL HOOK
01AC LATIN CAPITAL LETTER T WITH HOOK
01AD LATIN SMALL LETTER T WITH HOOK
01AE LATIN CAPITAL LETTER T WITH RETROFLEX HOOK
01AF LATIN CAPITAL LETTER U WITH HORN
01B0 LATIN SMALL LETTER U WITH HORN
01B1 LATIN CAPITAL LETTER UPSILON
01B2 LATIN CAPITAL LETTER V WITH HOOK
01B3 LATIN CAPITAL LETTER Y WITH HOOK
01B4 LATIN SMALL LETTER Y WITH HOOK
01B5 LATIN CAPITAL LETTER Z WITH STROKE
01B6 LATIN SMALL LETTER Z WITH STROKE
01B7 LATIN CAPITAL LETTER EZH
01B8 LATIN CAPITAL LETTER EZH REVERSED
01B9 LATIN SMALL LETTER EZH REVERSED
01BA LATIN SMALL LETTER EZH WITH TAIL
01BB LATIN LETTER TWO WITH STROKE
01BC LATIN CAPITAL LETTER TONE FIVE
01BD LATIN SMALL LETTER TONE FIVE
01BE LATIN LETTER INVERTED GLOTTAL STOP WITH STROKE
01BF LATIN LETTER WYNN
01C0 LATIN LETTER DENTAL CLICK
01C1 LATIN LETTER LATERAL CLICK
01C2 LATIN LETTER ALVEOLAR CLICK
01C3 LATIN LETTER RETROFLEX CLICK
01C4 LATIN CAPITAL LETTER DZ WITH CARON
01C5 LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON
01C6 LATIN SMALL LETTER DZ WITH CARON
01C7 LATIN CAPITAL LETTER LJ
01C8 LATIN CAPITAL LETTER L WITH SMALL LETTER J
01C9 LATIN SMALL LETTER LJ
01CA LATIN CAPITAL LETTER NJ
01CB LATIN CAPITAL LETTER N WITH SMALL LETTER J
01CC LATIN SMALL LETTER NJ
01CD LATIN CAPITAL LETTER A WITH CARON
01CE LATIN SMALL LETTER A WITH CARON
01CF LATIN CAPITAL LETTER I WITH CARON
01D0 LATIN SMALL LETTER I WITH CARON
01D1 LATIN CAPITAL LETTER O WITH CARON
01D2 LATIN SMALL LETTER O WITH CARON
01D3 LATIN CAPITAL LETTER U WITH CARON
01D4 LATIN SMALL LETTER U WITH CARON
01D5 LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
01D6 LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
01D7 LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
01D8 LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
01D9 LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
01DA LATIN SMALL LETTER U WITH DIAERESIS AND CARON
01DB LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
01DC LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE
01DD LATIN SMALL LETTER TURNED E
01DE LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
01DF LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
01E0 LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
01E1 LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
01E2 LATIN CAPITAL LETTER AE WITH MACRON
01E3 LATIN SMALL LETTER AE WITH MACRON
01E4 LATIN CAPITAL LETTER G WITH STROKE
01E5 LATIN SMALL LETTER G WITH STROKE
01E6 LATIN CAPITAL LETTER G WITH CARON
01E7 LATIN SMALL LETTER G WITH CARON
01E8 LATIN CAPITAL LETTER K WITH CARON
01E9 LATIN SMALL LETTER K WITH CARON
01EA LATIN CAPITAL LETTER O WITH OGONEK
01EB LATIN SMALL LETTER O WITH OGONEK
01EC LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
01ED LATIN SMALL LETTER O WITH OGONEK AND MACRON
01EE LATIN CAPITAL LETTER EZH WITH CARON
01EF LATIN SMALL LETTER EZH WITH CARON
01F0 LATIN SMALL LETTER J WITH CARON
01F1 LATIN CAPITAL LETTER DZ
01F2 LATIN CAPITAL LETTER D WITH SMALL LETTER Z
01F3 LATIN SMALL LETTER DZ
01F4 LATIN CAPITAL LETTER G WITH ACUTE
01F5 LATIN SMALL LETTER G WITH ACUTE
01FA LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
01FB LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE
01FC LATIN CAPITAL LETTER AE WITH ACUTE
01FD LATIN SMALL LETTER AE WITH ACUTE
01FE LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
01FF LATIN SMALL LETTER O WITH STROKE AND ACUTE
0200 LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
0201 LATIN SMALL LETTER A WITH DOUBLE GRAVE
0202 LATIN CAPITAL LETTER A WITH INVERTED BREVE
0203 LATIN SMALL LETTER A WITH INVERTED BREVE
0204 LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
0205 LATIN SMALL LETTER E WITH DOUBLE GRAVE
0206 LATIN CAPITAL LETTER E WITH INVERTED BREVE
0207 LATIN SMALL LETTER E WITH INVERTED BREVE
0208 LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
0209 LATIN SMALL LETTER I WITH DOUBLE GRAVE
020A LATIN CAPITAL LETTER I WITH INVERTED BREVE
020B LATIN SMALL LETTER I WITH INVERTED BREVE
020C LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
020D LATIN SMALL LETTER O WITH DOUBLE GRAVE
020E LATIN CAPITAL LETTER O WITH INVERTED BREVE
020F LATIN SMALL LETTER O WITH INVERTED BREVE
0210 LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
0211 LATIN SMALL LETTER R WITH DOUBLE GRAVE
0212 LATIN CAPITAL LETTER R WITH INVERTED BREVE
0213 LATIN SMALL LETTER R WITH INVERTED BREVE
0214 LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
0215 LATIN SMALL LETTER U WITH DOUBLE GRAVE
0216 LATIN CAPITAL LETTER U WITH INVERTED BREVE
0217 LATIN SMALL LETTER U WITH INVERTED BREVE
0250 LATIN SMALL LETTER TURNED A
0251 LATIN SMALL LETTER ALPHA
0252 LATIN SMALL LETTER TURNED ALPHA
0253 LATIN SMALL LETTER B WITH HOOK
0254 LATIN SMALL LETTER OPEN O
0255 LATIN SMALL LETTER C WITH CURL
0256 LATIN SMALL LETTER D WITH TAIL
0257 LATIN SMALL LETTER D WITH HOOK
0258 LATIN SMALL LETTER REVERSED E
0259 LATIN SMALL LETTER SCHWA
025A LATIN SMALL LETTER SCHWA WITH HOOK
025B LATIN SMALL LETTER OPEN E
025C LATIN SMALL LETTER REVERSED OPEN E
025D LATIN SMALL LETTER REVERSED OPEN E WITH HOOK
025E LATIN SMALL LETTER CLOSED REVERSED OPEN E
025F LATIN SMALL LETTER DOTLESS J WITH STROKE
0260 LATIN SMALL LETTER G WITH HOOK
0261 LATIN SMALL LETTER SCRIPT G
0262 LATIN LETTER SMALL CAPITAL G
0263 LATIN SMALL LETTER GAMMA
0264 LATIN SMALL LETTER RAMS HORN
0265 LATIN SMALL LETTER TURNED H
0266 LATIN SMALL LETTER H WITH HOOK
0267 LATIN SMALL LETTER HENG WITH HOOK
0268 LATIN SMALL LETTER I WITH STROKE
0269 LATIN SMALL LETTER IOTA
026A LATIN LETTER SMALL CAPITAL I
026B LATIN SMALL LETTER L WITH MIDDLE TILDE
026C LATIN SMALL LETTER L WITH BELT
026D LATIN SMALL LETTER L WITH RETROFLEX HOOK
026E LATIN SMALL LETTER LEZH
026F LATIN SMALL LETTER TURNED M
0270 LATIN SMALL LETTER TURNED M WITH LONG LEG
0271 LATIN SMALL LETTER M WITH HOOK
0272 LATIN SMALL LETTER N WITH LEFT HOOK
0273 LATIN SMALL LETTER N WITH RETROFLEX HOOK
0274 LATIN LETTER SMALL CAPITAL N
0275 LATIN SMALL LETTER BARRED O
0276 LATIN LETTER SMALL CAPITAL OE
0277 LATIN SMALL LETTER CLOSED OMEGA
0278 LATIN SMALL LETTER PHI
0279 LATIN SMALL LETTER TURNED R
027A LATIN SMALL LETTER TURNED R WITH LONG LEG
027B LATIN SMALL LETTER TURNED R WITH HOOK
027C LATIN SMALL LETTER R WITH LONG LEG
027D LATIN SMALL LETTER R WITH TAIL
027E LATIN SMALL LETTER R WITH FISHHOOK
027F LATIN SMALL LETTER REVERSED R WITH FISHHOOK
0280 LATIN LETTER SMALL CAPITAL R
0281 LATIN LETTER SMALL CAPITAL INVERTED R
0282 LATIN SMALL LETTER S WITH HOOK
0283 LATIN SMALL LETTER ESH
0284 LATIN SMALL LETTER DOTLESS J WITH STROKE AND HOOK
0285 LATIN SMALL LETTER SQUAT REVERSED ESH
0286 LATIN SMALL LETTER ESH WITH CURL
0287 LATIN SMALL LETTER TURNED T
0288 LATIN SMALL LETTER T WITH RETROFLEX HOOK
0289 LATIN SMALL LETTER U BAR
028A LATIN SMALL LETTER UPSILON
028B LATIN SMALL LETTER V WITH HOOK
028C LATIN SMALL LETTER TURNED V
028D LATIN SMALL LETTER TURNED W
028E LATIN SMALL LETTER TURNED Y
028F LATIN LETTER SMALL CAPITAL Y
0290 LATIN SMALL LETTER Z WITH RETROFLEX HOOK
0291 LATIN SMALL LETTER Z WITH CURL
0292 LATIN SMALL LETTER EZH
0293 LATIN SMALL LETTER EZH WITH CURL
0294 LATIN LETTER GLOTTAL STOP
0295 LATIN LETTER PHARYNGEAL VOICED FRICATIVE
0296 LATIN LETTER INVERTED GLOTTAL STOP
0297 LATIN LETTER STRETCHED C
0298 LATIN LETTER BILABIAL CLICK
0299 LATIN LETTER SMALL CAPITAL B
029A LATIN SMALL LETTER CLOSED OPEN E
029B LATIN LETTER SMALL CAPITAL G WITH HOOK
029C LATIN LETTER SMALL CAPITAL H
029D LATIN SMALL LETTER J WITH CROSSED-TAIL
029E LATIN SMALL LETTER TURNED K
029F LATIN LETTER SMALL CAPITAL L
02A0 LATIN SMALL LETTER Q WITH HOOK
02A1 LATIN LETTER GLOTTAL STOP WITH STROKE
02A2 LATIN LETTER REVERSED GLOTTAL STOP WITH STROKE
02A3 LATIN SMALL LETTER DZ DIGRAPH
02A4 LATIN SMALL LETTER DEZH DIGRAPH
02A5 LATIN SMALL LETTER DZ DIGRAPH WITH CURL
02A6 LATIN SMALL LETTER TS DIGRAPH
02A7 LATIN SMALL LETTER TESH DIGRAPH
02A8 LATIN SMALL LETTER TC DIGRAPH WITH CURL
02B0 MODIFIER LETTER SMALL H
02B1 MODIFIER LETTER SMALL H WITH HOOK
02B2 MODIFIER LETTER SMALL J
02B3 MODIFIER LETTER SMALL R
02B4 MODIFIER LETTER SMALL TURNED R
02B5 MODIFIER LETTER SMALL TURNED R WITH HOOK
02B6 MODIFIER LETTER SMALL CAPITAL INVERTED R
02B7 MODIFIER LETTER SMALL W
02B8 MODIFIER LETTER SMALL Y
02B9 MODIFIER LETTER PRIME
02BA MODIFIER LETTER DOUBLE PRIME
02BB MODIFIER LETTER TURNED COMMA
02BC MODIFIER LETTER APOSTROPHE
02BD MODIFIER LETTER REVERSED COMMA
02BE MODIFIER LETTER RIGHT HALF RING
02BF MODIFIER LETTER LEFT HALF RING
02C0 MODIFIER LETTER GLOTTAL STOP
02C1 MODIFIER LETTER REVERSED GLOTTAL STOP
02C2 MODIFIER LETTER LEFT ARROWHEAD
02C3 MODIFIER LETTER RIGHT ARROWHEAD
02C4 MODIFIER LETTER UP ARROWHEAD
02C5 MODIFIER LETTER DOWN ARROWHEAD
02C6 MODIFIER LETTER CIRCUMFLEX ACCENT
02C7 CARON
02C8 MODIFIER LETTER VERTICAL LINE
02C9 MODIFIER LETTER MACRON
02CA MODIFIER LETTER ACUTE ACCENT
02CB MODIFIER LETTER GRAVE ACCENT
02CC MODIFIER LETTER LOW VERTICAL LINE
02CD MODIFIER LETTER LOW MACRON
02CE MODIFIER LETTER LOW GRAVE ACCENT
02CF MODIFIER LETTER LOW ACUTE ACCENT
02D0 MODIFIER LETTER TRIANGULAR COLON
02D1 MODIFIER LETTER HALF TRIANGULAR COLON
02D2 MODIFIER LETTER CENTRED RIGHT HALF RING
02D3 MODIFIER LETTER CENTRED LEFT HALF RING
02D4 MODIFIER LETTER UP TACK
02D5 MODIFIER LETTER DOWN TACK
02D6 MODIFIER LETTER PLUS SIGN
02D7 MODIFIER LETTER MINUS SIGN
02D8 BREVE
02D9 DOT ABOVE
02DA RING ABOVE
02DB OGONEK
02DC SMALL TILDE
02DD DOUBLE ACUTE ACCENT
02DE MODIFIER LETTER RHOTIC HOOK
02E0 MODIFIER LETTER SMALL GAMMA
02E1 MODIFIER LETTER SMALL L
02E2 MODIFIER LETTER SMALL S
02E3 MODIFIER LETTER SMALL X
02E4 MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
02E5 MODIFIER LETTER EXTRA-HIGH TONE BAR
02E6 MODIFIER LETTER HIGH TONE BAR
02E7 MODIFIER LETTER MID TONE BAR
02E8 MODIFIER LETTER LOW TONE BAR
02E9 MODIFIER LETTER EXTRA-LOW TONE BAR
0300 COMBINING GRAVE ACCENT
0301 COMBINING ACUTE ACCENT
0302 COMBINING CIRCUMFLEX ACCENT
0303 COMBINING TILDE
0304 COMBINING MACRON
0305 COMBINING OVERLINE
0306 COMBINING BREVE
0307 COMBINING DOT ABOVE
0308 COMBINING DIAERESIS
0309 COMBINING HOOK ABOVE
030A COMBINING RING ABOVE
030B COMBINING DOUBLE ACUTE ACCENT
030C COMBINING CARON
030D COMBINING VERTICAL LINE ABOVE
030E COMBINING DOUBLE VERTICAL LINE ABOVE
030F COMBINING DOUBLE GRAVE ACCENT
0310 COMBINING CANDRABINDU
0311 COMBINING INVERTED BREVE
0312 COMBINING TURNED COMMA ABOVE
0313 COMBINING COMMA ABOVE
0314 COMBINING REVERSED COMMA ABOVE
0315 COMBINING COMMA ABOVE RIGHT
0316 COMBINING GRAVE ACCENT BELOW
0317 COMBINING ACUTE ACCENT BELOW
0318 COMBINING LEFT TACK BELOW
0319 COMBINING RIGHT TACK BELOW
031A COMBINING LEFT ANGLE ABOVE
031B COMBINING HORN
031C COMBINING LEFT HALF RING BELOW
031D COMBINING UP TACK BELOW
031E COMBINING DOWN TACK BELOW
031F COMBINING PLUS SIGN BELOW
0320 COMBINING MINUS SIGN BELOW
0321 COMBINING PALATALIZED HOOK BELOW
0322 COMBINING RETROFLEX HOOK BELOW
0323 COMBINING DOT BELOW
0324 COMBINING DIAERESIS BELOW
0325 COMBINING RING BELOW
0326 COMBINING COMMA BELOW
0327 COMBINING CEDILLA
0328 COMBINING OGONEK
0329 COMBINING VERTICAL LINE BELOW
032A COMBINING BRIDGE BELOW
032B COMBINING INVERTED DOUBLE ARCH BELOW
032C COMBINING CARON BELOW
032D COMBINING CIRCUMFLEX ACCENT BELOW
032E COMBINING BREVE BELOW
032F COMBINING INVERTED BREVE BELOW
0330 COMBINING TILDE BELOW
0331 COMBINING MACRON BELOW
0332 COMBINING LOW LINE
0333 COMBINING DOUBLE LOW LINE
0334 COMBINING TILDE OVERLAY
0335 COMBINING SHORT STROKE OVERLAY
0336 COMBINING LONG STROKE OVERLAY
0337 COMBINING SHORT SOLIDUS OVERLAY
0338 COMBINING LONG SOLIDUS OVERLAY
0339 COMBINING RIGHT HALF RING BELOW
033A COMBINING INVERTED BRIDGE BELOW
033B COMBINING SQUARE BELOW
033C COMBINING SEAGULL BELOW
033D COMBINING X ABOVE
033E COMBINING VERTICAL TILDE
033F COMBINING DOUBLE OVERLINE
0340 COMBINING GRAVE TONE MARK
0341 COMBINING ACUTE TONE MARK
0342 COMBINING GREEK PERISPOMENI
0343 COMBINING GREEK KORONIS
0344 COMBINING GREEK DIALYTIKA TONOS
0345 COMBINING GREEK YPOGEGRAMMENI
0360 COMBINING DOUBLE TILDE
0361 COMBINING DOUBLE INVERTED BREVE
0374 GREEK NUMERAL SIGN
0375 GREEK LOWER NUMERAL SIGN
037A GREEK YPOGEGRAMMENI
037E GREEK QUESTION MARK
0384 GREEK TONOS
0385 GREEK DIALYTIKA TONOS
0386 GREEK CAPITAL LETTER ALPHA WITH TONOS
0387 GREEK ANO TELEIA
0388 GREEK CAPITAL LETTER EPSILON WITH TONOS
0389 GREEK CAPITAL LETTER ETA WITH TONOS
038A GREEK CAPITAL LETTER IOTA WITH TONOS
038C GREEK CAPITAL LETTER OMICRON WITH TONOS
038E GREEK CAPITAL LETTER UPSILON WITH TONOS
038F GREEK CAPITAL LETTER OMEGA WITH TONOS
0390 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
0391 GREEK CAPITAL LETTER ALPHA
0392 GREEK CAPITAL LETTER BETA
0393 GREEK CAPITAL LETTER GAMMA
0394 GREEK CAPITAL LETTER DELTA
0395 GREEK CAPITAL LETTER EPSILON
0396 GREEK CAPITAL LETTER ZETA
0397 GREEK CAPITAL LETTER ETA
0398 GREEK CAPITAL LETTER THETA
0399 GREEK CAPITAL LETTER IOTA
039A GREEK CAPITAL LETTER KAPPA
039B GREEK CAPITAL LETTER LAMDA
039C GREEK CAPITAL LETTER MU
039D GREEK CAPITAL LETTER NU
039E GREEK CAPITAL LETTER XI
039F GREEK CAPITAL LETTER OMICRON
03A0 GREEK CAPITAL LETTER PI
03A1 GREEK CAPITAL LETTER RHO
03A3 GREEK CAPITAL LETTER SIGMA
03A4 GREEK CAPITAL LETTER TAU
03A5 GREEK CAPITAL LETTER UPSILON
03A6 GREEK CAPITAL LETTER PHI
03A7 GREEK CAPITAL LETTER CHI
03A8 GREEK CAPITAL LETTER PSI
03A9 GREEK CAPITAL LETTER OMEGA
03AA GREEK CAPITAL LETTER IOTA WITH DIALYTIKA
03AB GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
03AC GREEK SMALL LETTER ALPHA WITH TONOS
03AD GREEK SMALL LETTER EPSILON WITH TONOS
03AE GREEK SMALL LETTER ETA WITH TONOS
03AF GREEK SMALL LETTER IOTA WITH TONOS
03B0 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS
03B1 GREEK SMALL LETTER ALPHA
03B2 GREEK SMALL LETTER BETA
03B3 GREEK SMALL LETTER GAMMA
03B4 GREEK SMALL LETTER DELTA
03B5 GREEK SMALL LETTER EPSILON
03B6 GREEK SMALL LETTER ZETA
03B7 GREEK SMALL LETTER ETA
03B8 GREEK SMALL LETTER THETA
03B9 GREEK SMALL LETTER IOTA
03BA GREEK SMALL LETTER KAPPA
03BB GREEK SMALL LETTER LAMDA
03BC GREEK SMALL LETTER MU
03BD GREEK SMALL LETTER NU
03BE GREEK SMALL LETTER XI
03BF GREEK SMALL LETTER OMICRON
03C0 GREEK SMALL LETTER PI
03C1 GREEK SMALL LETTER RHO
03C2 GREEK SMALL LETTER FINAL SIGMA
03C3 GREEK SMALL LETTER SIGMA
03C4 GREEK SMALL LETTER TAU
03C5 GREEK SMALL LETTER UPSILON
03C6 GREEK SMALL LETTER PHI
03C7 GREEK SMALL LETTER CHI
03C8 GREEK SMALL LETTER PSI
03C9 GREEK SMALL LETTER OMEGA
03CA GREEK SMALL LETTER IOTA WITH DIALYTIKA
03CB GREEK SMALL LETTER UPSILON WITH DIALYTIKA
03CC GREEK SMALL LETTER OMICRON WITH TONOS
03CD GREEK SMALL LETTER UPSILON WITH TONOS
03CE GREEK SMALL LETTER OMEGA WITH TONOS
03D0 GREEK BETA SYMBOL
03D1 GREEK THETA SYMBOL
03D2 GREEK UPSILON WITH HOOK SYMBOL
03D3 GREEK UPSILON WITH ACUTE AND HOOK SYMBOL
03D4 GREEK UPSILON WITH DIAERESIS AND HOOK SYMBOL
03D5 GREEK PHI SYMBOL
03D6 GREEK PI SYMBOL
03DA GREEK LETTER STIGMA
03DC GREEK LETTER DIGAMMA
03DE GREEK LETTER KOPPA
03E0 GREEK LETTER SAMPI
03E2 COPTIC CAPITAL LETTER SHEI
03E3 COPTIC SMALL LETTER SHEI
03E4 COPTIC CAPITAL LETTER FEI
03E5 COPTIC SMALL LETTER FEI
03E6 COPTIC CAPITAL LETTER KHEI
03E7 COPTIC SMALL LETTER KHEI
03E8 COPTIC CAPITAL LETTER HORI
03E9 COPTIC SMALL LETTER HORI
03EA COPTIC CAPITAL LETTER GANGIA
03EB COPTIC SMALL LETTER GANGIA
03EC COPTIC CAPITAL LETTER SHIMA
03ED COPTIC SMALL LETTER SHIMA
03EE COPTIC CAPITAL LETTER DEI
03EF COPTIC SMALL LETTER DEI
03F0 GREEK KAPPA SYMBOL
03F1 GREEK RHO SYMBOL
03F2 GREEK LUNATE SIGMA SYMBOL
03F3 GREEK LETTER YOT
0401 CYRILLIC CAPITAL LETTER IO
0402 CYRILLIC CAPITAL LETTER DJE
0403 CYRILLIC CAPITAL LETTER GJE
0404 CYRILLIC CAPITAL LETTER UKRAINIAN IE
0405 CYRILLIC CAPITAL LETTER DZE
0406 CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
0407 CYRILLIC CAPITAL LETTER YI
0408 CYRILLIC CAPITAL LETTER JE
0409 CYRILLIC CAPITAL LETTER LJE
040A CYRILLIC CAPITAL LETTER NJE
040B CYRILLIC CAPITAL LETTER TSHE
040C CYRILLIC CAPITAL LETTER KJE
040E CYRILLIC CAPITAL LETTER SHORT U
040F CYRILLIC CAPITAL LETTER DZHE
0410 CYRILLIC CAPITAL LETTER A
0411 CYRILLIC CAPITAL LETTER BE
0412 CYRILLIC CAPITAL LETTER VE
0413 CYRILLIC CAPITAL LETTER GHE
0414 CYRILLIC CAPITAL LETTER DE
0415 CYRILLIC CAPITAL LETTER IE
0416 CYRILLIC CAPITAL LETTER ZHE
0417 CYRILLIC CAPITAL LETTER ZE
0418 CYRILLIC CAPITAL LETTER I
0419 CYRILLIC CAPITAL LETTER SHORT I
041A CYRILLIC CAPITAL LETTER KA
041B CYRILLIC CAPITAL LETTER EL
041C CYRILLIC CAPITAL LETTER EM
041D CYRILLIC CAPITAL LETTER EN
041E CYRILLIC CAPITAL LETTER O
041F CYRILLIC CAPITAL LETTER PE
0420 CYRILLIC CAPITAL LETTER ER
0421 CYRILLIC CAPITAL LETTER ES
0422 CYRILLIC CAPITAL LETTER TE
0423 CYRILLIC CAPITAL LETTER U
0424 CYRILLIC CAPITAL LETTER EF
0425 CYRILLIC CAPITAL LETTER HA
0426 CYRILLIC CAPITAL LETTER TSE
0427 CYRILLIC CAPITAL LETTER CHE
0428 CYRILLIC CAPITAL LETTER SHA
0429 CYRILLIC CAPITAL LETTER SHCHA
042A CYRILLIC CAPITAL LETTER HARD SIGN
042B CYRILLIC CAPITAL LETTER YERU
042C CYRILLIC CAPITAL LETTER SOFT SIGN
042D CYRILLIC CAPITAL LETTER E
042E CYRILLIC CAPITAL LETTER YU
042F CYRILLIC CAPITAL LETTER YA
0430 CYRILLIC SMALL LETTER A
0431 CYRILLIC SMALL LETTER BE
0432 CYRILLIC SMALL LETTER VE
0433 CYRILLIC SMALL LETTER GHE
0434 CYRILLIC SMALL LETTER DE
0435 CYRILLIC SMALL LETTER IE
0436 CYRILLIC SMALL LETTER ZHE
0437 CYRILLIC SMALL LETTER ZE
0438 CYRILLIC SMALL LETTER I
0439 CYRILLIC SMALL LETTER SHORT I
043A CYRILLIC SMALL LETTER KA
043B CYRILLIC SMALL LETTER EL
043C CYRILLIC SMALL LETTER EM
043D CYRILLIC SMALL LETTER EN
043E CYRILLIC SMALL LETTER O
043F CYRILLIC SMALL LETTER PE
0440 CYRILLIC SMALL LETTER ER
0441 CYRILLIC SMALL LETTER ES
0442 CYRILLIC SMALL LETTER TE
0443 CYRILLIC SMALL LETTER U
0444 CYRILLIC SMALL LETTER EF
0445 CYRILLIC SMALL LETTER HA
0446 CYRILLIC SMALL LETTER TSE
0447 CYRILLIC SMALL LETTER CHE
0448 CYRILLIC SMALL LETTER SHA
0449 CYRILLIC SMALL LETTER SHCHA
044A CYRILLIC SMALL LETTER HARD SIGN
044B CYRILLIC SMALL LETTER YERU
044C CYRILLIC SMALL LETTER SOFT SIGN
044D CYRILLIC SMALL LETTER E
044E CYRILLIC SMALL LETTER YU
044F CYRILLIC SMALL LETTER YA
0451 CYRILLIC SMALL LETTER IO
0452 CYRILLIC SMALL LETTER DJE
0453 CYRILLIC SMALL LETTER GJE
0454 CYRILLIC SMALL LETTER UKRAINIAN IE
0455 CYRILLIC SMALL LETTER DZE
0456 CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
0457 CYRILLIC SMALL LETTER YI
0458 CYRILLIC SMALL LETTER JE
0459 CYRILLIC SMALL LETTER LJE
045A CYRILLIC SMALL LETTER NJE
045B CYRILLIC SMALL LETTER TSHE
045C CYRILLIC SMALL LETTER KJE
045E CYRILLIC SMALL LETTER SHORT U
045F CYRILLIC SMALL LETTER DZHE
0460 CYRILLIC CAPITAL LETTER OMEGA
0461 CYRILLIC SMALL LETTER OMEGA
0462 CYRILLIC CAPITAL LETTER YAT
0463 CYRILLIC SMALL LETTER YAT
0464 CYRILLIC CAPITAL LETTER IOTIFIED E
0465 CYRILLIC SMALL LETTER IOTIFIED E
0466 CYRILLIC CAPITAL LETTER LITTLE YUS
0467 CYRILLIC SMALL LETTER LITTLE YUS
0468 CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
0469 CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS
046A CYRILLIC CAPITAL LETTER BIG YUS
046B CYRILLIC SMALL LETTER BIG YUS
046C CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
046D CYRILLIC SMALL LETTER IOTIFIED BIG YUS
046E CYRILLIC CAPITAL LETTER KSI
046F CYRILLIC SMALL LETTER KSI
0470 CYRILLIC CAPITAL LETTER PSI
0471 CYRILLIC SMALL LETTER PSI
0472 CYRILLIC CAPITAL LETTER FITA
0473 CYRILLIC SMALL LETTER FITA
0474 CYRILLIC CAPITAL LETTER IZHITSA
0475 CYRILLIC SMALL LETTER IZHITSA
0476 CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
0477 CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
0478 CYRILLIC CAPITAL LETTER UK
0479 CYRILLIC SMALL LETTER UK
047A CYRILLIC CAPITAL LETTER ROUND OMEGA
047B CYRILLIC SMALL LETTER ROUND OMEGA
047C CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
047D CYRILLIC SMALL LETTER OMEGA WITH TITLO
047E CYRILLIC CAPITAL LETTER OT
047F CYRILLIC SMALL LETTER OT
0480 CYRILLIC CAPITAL LETTER KOPPA
0481 CYRILLIC SMALL LETTER KOPPA
0482 CYRILLIC THOUSANDS SIGN
0483 COMBINING CYRILLIC TITLO
0484 COMBINING CYRILLIC PALATALIZATION
0485 COMBINING CYRILLIC DASIA PNEUMATA
0486 COMBINING CYRILLIC PSILI PNEUMATA
0490 CYRILLIC CAPITAL LETTER GHE WITH UPTURN
0491 CYRILLIC SMALL LETTER GHE WITH UPTURN
0492 CYRILLIC CAPITAL LETTER GHE WITH STROKE
0493 CYRILLIC SMALL LETTER GHE WITH STROKE
0494 CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
0495 CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK
0496 CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
0497 CYRILLIC SMALL LETTER ZHE WITH DESCENDER
0498 CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
0499 CYRILLIC SMALL LETTER ZE WITH DESCENDER
049A CYRILLIC CAPITAL LETTER KA WITH DESCENDER
049B CYRILLIC SMALL LETTER KA WITH DESCENDER
049C CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
049D CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE
049E CYRILLIC CAPITAL LETTER KA WITH STROKE
049F CYRILLIC SMALL LETTER KA WITH STROKE
04A0 CYRILLIC CAPITAL LETTER BASHKIR KA
04A1 CYRILLIC SMALL LETTER BASHKIR KA
04A2 CYRILLIC CAPITAL LETTER EN WITH DESCENDER
04A3 CYRILLIC SMALL LETTER EN WITH DESCENDER
04A4 CYRILLIC CAPITAL LIGATURE EN GHE
04A5 CYRILLIC SMALL LIGATURE EN GHE
04A6 CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
04A7 CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK
04A8 CYRILLIC CAPITAL LETTER ABKHASIAN HA
04A9 CYRILLIC SMALL LETTER ABKHASIAN HA
04AA CYRILLIC CAPITAL LETTER ES WITH DESCENDER
04AB CYRILLIC SMALL LETTER ES WITH DESCENDER
04AC CYRILLIC CAPITAL LETTER TE WITH DESCENDER
04AD CYRILLIC SMALL LETTER TE WITH DESCENDER
04AE CYRILLIC CAPITAL LETTER STRAIGHT U
04AF CYRILLIC SMALL LETTER STRAIGHT U
04B0 CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
04B1 CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
04B2 CYRILLIC CAPITAL LETTER HA WITH DESCENDER
04B3 CYRILLIC SMALL LETTER HA WITH DESCENDER
04B4 CYRILLIC CAPITAL LIGATURE TE TSE
04B5 CYRILLIC SMALL LIGATURE TE TSE
04B6 CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
04B7 CYRILLIC SMALL LETTER CHE WITH DESCENDER
04B8 CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
04B9 CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE
04BA CYRILLIC CAPITAL LETTER SHHA
04BB CYRILLIC SMALL LETTER SHHA
04BC CYRILLIC CAPITAL LETTER ABKHASIAN CHE
04BD CYRILLIC SMALL LETTER ABKHASIAN CHE
04BE CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
04BF CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER
04C0 CYRILLIC LETTER PALOCHKA
04C1 CYRILLIC CAPITAL LETTER ZHE WITH BREVE
04C2 CYRILLIC SMALL LETTER ZHE WITH BREVE
04C3 CYRILLIC CAPITAL LETTER KA WITH HOOK
04C4 CYRILLIC SMALL LETTER KA WITH HOOK
04C7 CYRILLIC CAPITAL LETTER EN WITH HOOK
04C8 CYRILLIC SMALL LETTER EN WITH HOOK
04CB CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
04CC CYRILLIC SMALL LETTER KHAKASSIAN CHE
04D0 CYRILLIC CAPITAL LETTER A WITH BREVE
04D1 CYRILLIC SMALL LETTER A WITH BREVE
04D2 CYRILLIC CAPITAL LETTER A WITH DIAERESIS
04D3 CYRILLIC SMALL LETTER A WITH DIAERESIS
04D4 CYRILLIC CAPITAL LIGATURE A IE
04D5 CYRILLIC SMALL LIGATURE A IE
04D6 CYRILLIC CAPITAL LETTER IE WITH BREVE
04D7 CYRILLIC SMALL LETTER IE WITH BREVE
04D8 CYRILLIC CAPITAL LETTER SCHWA
04D9 CYRILLIC SMALL LETTER SCHWA
04DA CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
04DB CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS
04DC CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
04DD CYRILLIC SMALL LETTER ZHE WITH DIAERESIS
04DE CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
04DF CYRILLIC SMALL LETTER ZE WITH DIAERESIS
04E0 CYRILLIC CAPITAL LETTER ABKHASIAN DZE
04E1 CYRILLIC SMALL LETTER ABKHASIAN DZE
04E2 CYRILLIC CAPITAL LETTER I WITH MACRON
04E3 CYRILLIC SMALL LETTER I WITH MACRON
04E4 CYRILLIC CAPITAL LETTER I WITH DIAERESIS
04E5 CYRILLIC SMALL LETTER I WITH DIAERESIS
04E6 CYRILLIC CAPITAL LETTER O WITH DIAERESIS
04E7 CYRILLIC SMALL LETTER O WITH DIAERESIS
04E8 CYRILLIC CAPITAL LETTER BARRED O
04E9 CYRILLIC SMALL LETTER BARRED O
04EA CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
04EB CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS
04EE CYRILLIC CAPITAL LETTER U WITH MACRON
04EF CYRILLIC SMALL LETTER U WITH MACRON
04F0 CYRILLIC CAPITAL LETTER U WITH DIAERESIS
04F1 CYRILLIC SMALL LETTER U WITH DIAERESIS
04F2 CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
04F3 CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE
04F4 CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
04F5 CYRILLIC SMALL LETTER CHE WITH DIAERESIS
04F8 CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
04F9 CYRILLIC SMALL LETTER YERU WITH DIAERESIS
0531 ARMENIAN CAPITAL LETTER AYB
0532 ARMENIAN CAPITAL LETTER BEN
0533 ARMENIAN CAPITAL LETTER GIM
0534 ARMENIAN CAPITAL LETTER DA
0535 ARMENIAN CAPITAL LETTER ECH
0536 ARMENIAN CAPITAL LETTER ZA
0537 ARMENIAN CAPITAL LETTER EH
0538 ARMENIAN CAPITAL LETTER ET
0539 ARMENIAN CAPITAL LETTER TO
053A ARMENIAN CAPITAL LETTER ZHE
053B ARMENIAN CAPITAL LETTER INI
053C ARMENIAN CAPITAL LETTER LIWN
053D ARMENIAN CAPITAL LETTER XEH
053E ARMENIAN CAPITAL LETTER CA
053F ARMENIAN CAPITAL LETTER KEN
0540 ARMENIAN CAPITAL LETTER HO
0541 ARMENIAN CAPITAL LETTER JA
0542 ARMENIAN CAPITAL LETTER GHAD
0543 ARMENIAN CAPITAL LETTER CHEH
0544 ARMENIAN CAPITAL LETTER MEN
0545 ARMENIAN CAPITAL LETTER YI
0546 ARMENIAN CAPITAL LETTER NOW
0547 ARMENIAN CAPITAL LETTER SHA
0548 ARMENIAN CAPITAL LETTER VO
0549 ARMENIAN CAPITAL LETTER CHA
054A ARMENIAN CAPITAL LETTER PEH
054B ARMENIAN CAPITAL LETTER JHEH
054C ARMENIAN CAPITAL LETTER RA
054D ARMENIAN CAPITAL LETTER SEH
054E ARMENIAN CAPITAL LETTER VEW
054F ARMENIAN CAPITAL LETTER TIWN
0550 ARMENIAN CAPITAL LETTER REH
0551 ARMENIAN CAPITAL LETTER CO
0552 ARMENIAN CAPITAL LETTER YIWN
0553 ARMENIAN CAPITAL LETTER PIWR
0554 ARMENIAN CAPITAL LETTER KEH
0555 ARMENIAN CAPITAL LETTER OH
0556 ARMENIAN CAPITAL LETTER FEH
0559 ARMENIAN MODIFIER LETTER LEFT HALF RING
055A ARMENIAN APOSTROPHE
055B ARMENIAN EMPHASIS MARK
055C ARMENIAN EXCLAMATION MARK
055D ARMENIAN COMMA
055E ARMENIAN QUESTION MARK
055F ARMENIAN ABBREVIATION MARK
0561 ARMENIAN SMALL LETTER AYB
0562 ARMENIAN SMALL LETTER BEN
0563 ARMENIAN SMALL LETTER GIM
0564 ARMENIAN SMALL LETTER DA
0565 ARMENIAN SMALL LETTER ECH
0566 ARMENIAN SMALL LETTER ZA
0567 ARMENIAN SMALL LETTER EH
0568 ARMENIAN SMALL LETTER ET
0569 ARMENIAN SMALL LETTER TO
056A ARMENIAN SMALL LETTER ZHE
056B ARMENIAN SMALL LETTER INI
056C ARMENIAN SMALL LETTER LIWN
056D ARMENIAN SMALL LETTER XEH
056E ARMENIAN SMALL LETTER CA
056F ARMENIAN SMALL LETTER KEN
0570 ARMENIAN SMALL LETTER HO
0571 ARMENIAN SMALL LETTER JA
0572 ARMENIAN SMALL LETTER GHAD
0573 ARMENIAN SMALL LETTER CHEH
0574 ARMENIAN SMALL LETTER MEN
0575 ARMENIAN SMALL LETTER YI
0576 ARMENIAN SMALL LETTER NOW
0577 ARMENIAN SMALL LETTER SHA
0578 ARMENIAN SMALL LETTER VO
0579 ARMENIAN SMALL LETTER CHA
057A ARMENIAN SMALL LETTER PEH
057B ARMENIAN SMALL LETTER JHEH
057C ARMENIAN SMALL LETTER RA
057D ARMENIAN SMALL LETTER SEH
057E ARMENIAN SMALL LETTER VEW
057F ARMENIAN SMALL LETTER TIWN
0580 ARMENIAN SMALL LETTER REH
0581 ARMENIAN SMALL LETTER CO
0582 ARMENIAN SMALL LETTER YIWN
0583 ARMENIAN SMALL LETTER PIWR
0584 ARMENIAN SMALL LETTER KEH
0585 ARMENIAN SMALL LETTER OH
0586 ARMENIAN SMALL LETTER FEH
0587 ARMENIAN SMALL LIGATURE ECH YIWN
0589 ARMENIAN FULL STOP
0591 HEBREW ACCENT ETNAHTA
0592 HEBREW ACCENT SEGOL
0593 HEBREW ACCENT SHALSHELET
0594 HEBREW ACCENT ZAQEF QATAN
0595 HEBREW ACCENT ZAQEF GADOL
0596 HEBREW ACCENT TIPEHA
0597 HEBREW ACCENT REVIA
0598 HEBREW ACCENT ZARQA
0599 HEBREW ACCENT PASHTA
059A HEBREW ACCENT YETIV
059B HEBREW ACCENT TEVIR
059C HEBREW ACCENT GERESH
059D HEBREW ACCENT GERESH MUQDAM
059E HEBREW ACCENT GERSHAYIM
059F HEBREW ACCENT QARNEY PARA
05A0 HEBREW ACCENT TELISHA GEDOLA
05A1 HEBREW ACCENT PAZER
05A3 HEBREW ACCENT MUNAH
05A4 HEBREW ACCENT MAHAPAKH
05A5 HEBREW ACCENT MERKHA
05A6 HEBREW ACCENT MERKHA KEFULA
05A7 HEBREW ACCENT DARGA
05A8 HEBREW ACCENT QADMA
05A9 HEBREW ACCENT TELISHA QETANA
05AA HEBREW ACCENT YERAH BEN YOMO
05AB HEBREW ACCENT OLE
05AC HEBREW ACCENT ILUY
05AD HEBREW ACCENT DEHI
05AE HEBREW ACCENT ZINOR
05AF HEBREW MARK MASORA CIRCLE
05B0 HEBREW POINT SHEVA
05B1 HEBREW POINT HATAF SEGOL
05B2 HEBREW POINT HATAF PATAH
05B3 HEBREW POINT HATAF QAMATS
05B4 HEBREW POINT HIRIQ
05B5 HEBREW POINT TSERE
05B6 HEBREW POINT SEGOL
05B7 HEBREW POINT PATAH
05B8 HEBREW POINT QAMATS
05B9 HEBREW POINT HOLAM
05BB HEBREW POINT QUBUTS
05BC HEBREW POINT DAGESH OR MAPIQ
05BD HEBREW POINT METEG
05BE HEBREW PUNCTUATION MAQAF
05BF HEBREW POINT RAFE
05C0 HEBREW PUNCTUATION PASEQ
05C1 HEBREW POINT SHIN DOT
05C2 HEBREW POINT SIN DOT
05C3 HEBREW PUNCTUATION SOF PASUQ
05C4 HEBREW MARK UPPER DOT
05D0 HEBREW LETTER ALEF
05D1 HEBREW LETTER BET
05D2 HEBREW LETTER GIMEL
05D3 HEBREW LETTER DALET
05D4 HEBREW LETTER HE
05D5 HEBREW LETTER VAV
05D6 HEBREW LETTER ZAYIN
05D7 HEBREW LETTER HET
05D8 HEBREW LETTER TET
05D9 HEBREW LETTER YOD
05DA HEBREW LETTER FINAL KAF
05DB HEBREW LETTER KAF
05DC HEBREW LETTER LAMED
05DD HEBREW LETTER FINAL MEM
05DE HEBREW LETTER MEM
05DF HEBREW LETTER FINAL NUN
05E0 HEBREW LETTER NUN
05E1 HEBREW LETTER SAMEKH
05E2 HEBREW LETTER AYIN
05E3 HEBREW LETTER FINAL PE
05E4 HEBREW LETTER PE
05E5 HEBREW LETTER FINAL TSADI
05E6 HEBREW LETTER TSADI
05E7 HEBREW LETTER QOF
05E8 HEBREW LETTER RESH
05E9 HEBREW LETTER SHIN
05EA HEBREW LETTER TAV
05F0 HEBREW LIGATURE YIDDISH DOUBLE VAV
05F1 HEBREW LIGATURE YIDDISH VAV YOD
05F2 HEBREW LIGATURE YIDDISH DOUBLE YOD
05F3 HEBREW PUNCTUATION GERESH
05F4 HEBREW PUNCTUATION GERSHAYIM
060C ARABIC COMMA
061B ARABIC SEMICOLON
061F ARABIC QUESTION MARK
0621 ARABIC LETTER HAMZA
0622 ARABIC LETTER ALEF WITH MADDA ABOVE
0623 ARABIC LETTER ALEF WITH HAMZA ABOVE
0624 ARABIC LETTER WAW WITH HAMZA ABOVE
0625 ARABIC LETTER ALEF WITH HAMZA BELOW
0626 ARABIC LETTER YEH WITH HAMZA ABOVE
0627 ARABIC LETTER ALEF
0628 ARABIC LETTER BEH
0629 ARABIC LETTER TEH MARBUTA
062A ARABIC LETTER TEH
062B ARABIC LETTER THEH
062C ARABIC LETTER JEEM
062D ARABIC LETTER HAH
062E ARABIC LETTER KHAH
062F ARABIC LETTER DAL
0630 ARABIC LETTER THAL
0631 ARABIC LETTER REH
0632 ARABIC LETTER ZAIN
0633 ARABIC LETTER SEEN
0634 ARABIC LETTER SHEEN
0635 ARABIC LETTER SAD
0636 ARABIC LETTER DAD
0637 ARABIC LETTER TAH
0638 ARABIC LETTER ZAH
0639 ARABIC LETTER AIN
063A ARABIC LETTER GHAIN
0640 ARABIC TATWEEL
0641 ARABIC LETTER FEH
0642 ARABIC LETTER QAF
0643 ARABIC LETTER KAF
0644 ARABIC LETTER LAM
0645 ARABIC LETTER MEEM
0646 ARABIC LETTER NOON
0647 ARABIC LETTER HEH
0648 ARABIC LETTER WAW
0649 ARABIC LETTER ALEF MAKSURA
064A ARABIC LETTER YEH
064B ARABIC FATHATAN
064C ARABIC DAMMATAN
064D ARABIC KASRATAN
064E ARABIC FATHA
064F ARABIC DAMMA
0650 ARABIC KASRA
0651 ARABIC SHADDA
0652 ARABIC SUKUN
0660 ARABIC-INDIC DIGIT ZERO
0661 ARABIC-INDIC DIGIT ONE
0662 ARABIC-INDIC DIGIT TWO
0663 ARABIC-INDIC DIGIT THREE
0664 ARABIC-INDIC DIGIT FOUR
0665 ARABIC-INDIC DIGIT FIVE
0666 ARABIC-INDIC DIGIT SIX
0667 ARABIC-INDIC DIGIT SEVEN
0668 ARABIC-INDIC DIGIT EIGHT
0669 ARABIC-INDIC DIGIT NINE
066A ARABIC PERCENT SIGN
066B ARABIC DECIMAL SEPARATOR
066C ARABIC THOUSANDS SEPARATOR
066D ARABIC FIVE POINTED STAR
0670 ARABIC LETTER SUPERSCRIPT ALEF
0671 ARABIC LETTER ALEF WASLA
0672 ARABIC LETTER ALEF WITH WAVY HAMZA ABOVE
0673 ARABIC LETTER ALEF WITH WAVY HAMZA BELOW
0674 ARABIC LETTER HIGH HAMZA
0675 ARABIC LETTER HIGH HAMZA ALEF
0676 ARABIC LETTER HIGH HAMZA WAW
0677 ARABIC LETTER U WITH HAMZA ABOVE
0678 ARABIC LETTER HIGH HAMZA YEH
0679 ARABIC LETTER TTEH
067A ARABIC LETTER TTEHEH
067B ARABIC LETTER BEEH
067C ARABIC LETTER TEH WITH RING
067D ARABIC LETTER TEH WITH THREE DOTS ABOVE DOWNWARDS
067E ARABIC LETTER PEH
067F ARABIC LETTER TEHEH
0680 ARABIC LETTER BEHEH
0681 ARABIC LETTER HAH WITH HAMZA ABOVE
0682 ARABIC LETTER HAH WITH TWO DOTS VERTICAL ABOVE
0683 ARABIC LETTER NYEH
0684 ARABIC LETTER DYEH
0685 ARABIC LETTER HAH WITH THREE DOTS ABOVE
0686 ARABIC LETTER TCHEH
0687 ARABIC LETTER TCHEHEH
0688 ARABIC LETTER DDAL
0689 ARABIC LETTER DAL WITH RING
068A ARABIC LETTER DAL WITH DOT BELOW
068B ARABIC LETTER DAL WITH DOT BELOW AND SMALL TAH
068C ARABIC LETTER DAHAL
068D ARABIC LETTER DDAHAL
068E ARABIC LETTER DUL
068F ARABIC LETTER DAL WITH THREE DOTS ABOVE DOWNWARDS
0690 ARABIC LETTER DAL WITH FOUR DOTS ABOVE
0691 ARABIC LETTER RREH
0692 ARABIC LETTER REH WITH SMALL V
0693 ARABIC LETTER REH WITH RING
0694 ARABIC LETTER REH WITH DOT BELOW
0695 ARABIC LETTER REH WITH SMALL V BELOW
0696 ARABIC LETTER REH WITH DOT BELOW AND DOT ABOVE
0697 ARABIC LETTER REH WITH TWO DOTS ABOVE
0698 ARABIC LETTER JEH
0699 ARABIC LETTER REH WITH FOUR DOTS ABOVE
069A ARABIC LETTER SEEN WITH DOT BELOW AND DOT ABOVE
069B ARABIC LETTER SEEN WITH THREE DOTS BELOW
069C ARABIC LETTER SEEN WITH THREE DOTS BELOW AND THREE DOTS ABOVE
069D ARABIC LETTER SAD WITH TWO DOTS BELOW
069E ARABIC LETTER SAD WITH THREE DOTS ABOVE
069F ARABIC LETTER TAH WITH THREE DOTS ABOVE
06A0 ARABIC LETTER AIN WITH THREE DOTS ABOVE
06A1 ARABIC LETTER DOTLESS FEH
06A2 ARABIC LETTER FEH WITH DOT MOVED BELOW
06A3 ARABIC LETTER FEH WITH DOT BELOW
06A4 ARABIC LETTER VEH
06A5 ARABIC LETTER FEH WITH THREE DOTS BELOW
06A6 ARABIC LETTER PEHEH
06A7 ARABIC LETTER QAF WITH DOT ABOVE
06A8 ARABIC LETTER QAF WITH THREE DOTS ABOVE
06A9 ARABIC LETTER KEHEH
06AA ARABIC LETTER SWASH KAF
06AB ARABIC LETTER KAF WITH RING
06AC ARABIC LETTER KAF WITH DOT ABOVE
06AD ARABIC LETTER NG
06AE ARABIC LETTER KAF WITH THREE DOTS BELOW
06AF ARABIC LETTER GAF
06B0 ARABIC LETTER GAF WITH RING
06B1 ARABIC LETTER NGOEH
06B2 ARABIC LETTER GAF WITH TWO DOTS BELOW
06B3 ARABIC LETTER GUEH
06B4 ARABIC LETTER GAF WITH THREE DOTS ABOVE
06B5 ARABIC LETTER LAM WITH SMALL V
06B6 ARABIC LETTER LAM WITH DOT ABOVE
06B7 ARABIC LETTER LAM WITH THREE DOTS ABOVE
06BA ARABIC LETTER NOON GHUNNA
06BB ARABIC LETTER RNOON
06BC ARABIC LETTER NOON WITH RING
06BD ARABIC LETTER NOON WITH THREE DOTS ABOVE
06BE ARABIC LETTER HEH DOACHASHMEE
06C0 ARABIC LETTER HEH WITH YEH ABOVE
06C1 ARABIC LETTER HEH GOAL
06C2 ARABIC LETTER HEH GOAL WITH HAMZA ABOVE
06C3 ARABIC LETTER TEH MARBUTA GOAL
06C4 ARABIC LETTER WAW WITH RING
06C5 ARABIC LETTER KIRGHIZ OE
06C6 ARABIC LETTER OE
06C7 ARABIC LETTER U
06C8 ARABIC LETTER YU
06C9 ARABIC LETTER KIRGHIZ YU
06CA ARABIC LETTER WAW WITH TWO DOTS ABOVE
06CB ARABIC LETTER VE
06CC ARABIC LETTER FARSI YEH
06CD ARABIC LETTER YEH WITH TAIL
06CE ARABIC LETTER YEH WITH SMALL V
06D0 ARABIC LETTER E
06D1 ARABIC LETTER YEH WITH THREE DOTS BELOW
06D2 ARABIC LETTER YEH BARREE
06D3 ARABIC LETTER YEH BARREE WITH HAMZA ABOVE
06D4 ARABIC FULL STOP
06D5 ARABIC LETTER AE
06D6 ARABIC SMALL HIGH LIGATURE SAD WITH LAM WITH ALEF MAKSURA
06D7 ARABIC SMALL HIGH LIGATURE QAF WITH LAM WITH ALEF MAKSURA
06D8 ARABIC SMALL HIGH MEEM INITIAL FORM
06D9 ARABIC SMALL HIGH LAM ALEF
06DA ARABIC SMALL HIGH JEEM
06DB ARABIC SMALL HIGH THREE DOTS
06DC ARABIC SMALL HIGH SEEN
06DD ARABIC END OF AYAH
06DE ARABIC START OF RUB EL HIZB
06DF ARABIC SMALL HIGH ROUNDED ZERO
06E0 ARABIC SMALL HIGH UPRIGHT RECTANGULAR ZERO
06E1 ARABIC SMALL HIGH DOTLESS HEAD OF KHAH
06E2 ARABIC SMALL HIGH MEEM ISOLATED FORM
06E3 ARABIC SMALL LOW SEEN
06E4 ARABIC SMALL HIGH MADDA
06E5 ARABIC SMALL WAW
06E6 ARABIC SMALL YEH
06E7 ARABIC SMALL HIGH YEH
06E8 ARABIC SMALL HIGH NOON
06E9 ARABIC PLACE OF SAJDAH
06EA ARABIC EMPTY CENTRE LOW STOP
06EB ARABIC EMPTY CENTRE HIGH STOP
06EC ARABIC ROUNDED HIGH STOP WITH FILLED CENTRE
06ED ARABIC SMALL LOW MEEM
06F0 EXTENDED ARABIC-INDIC DIGIT ZERO
06F1 EXTENDED ARABIC-INDIC DIGIT ONE
06F2 EXTENDED ARABIC-INDIC DIGIT TWO
06F3 EXTENDED ARABIC-INDIC DIGIT THREE
06F4 EXTENDED ARABIC-INDIC DIGIT FOUR
06F5 EXTENDED ARABIC-INDIC DIGIT FIVE
06F6 EXTENDED ARABIC-INDIC DIGIT SIX
06F7 EXTENDED ARABIC-INDIC DIGIT SEVEN
06F8 EXTENDED ARABIC-INDIC DIGIT EIGHT
06F9 EXTENDED ARABIC-INDIC DIGIT NINE
0901 DEVANAGARI SIGN CANDRABINDU
0902 DEVANAGARI SIGN ANUSVARA
0903 DEVANAGARI SIGN VISARGA
0905 DEVANAGARI LETTER A
0906 DEVANAGARI LETTER AA
0907 DEVANAGARI LETTER I
0908 DEVANAGARI LETTER II
0909 DEVANAGARI LETTER U
090A DEVANAGARI LETTER UU
090B DEVANAGARI LETTER VOCALIC R
090C DEVANAGARI LETTER VOCALIC L
090D DEVANAGARI LETTER CANDRA E
090E DEVANAGARI LETTER SHORT E
090F DEVANAGARI LETTER E
0910 DEVANAGARI LETTER AI
0911 DEVANAGARI LETTER CANDRA O
0912 DEVANAGARI LETTER SHORT O
0913 DEVANAGARI LETTER O
0914 DEVANAGARI LETTER AU
0915 DEVANAGARI LETTER KA
0916 DEVANAGARI LETTER KHA
0917 DEVANAGARI LETTER GA
0918 DEVANAGARI LETTER GHA
0919 DEVANAGARI LETTER NGA
091A DEVANAGARI LETTER CA
091B DEVANAGARI LETTER CHA
091C DEVANAGARI LETTER JA
091D DEVANAGARI LETTER JHA
091E DEVANAGARI LETTER NYA
091F DEVANAGARI LETTER TTA
0920 DEVANAGARI LETTER TTHA
0921 DEVANAGARI LETTER DDA
0922 DEVANAGARI LETTER DDHA
0923 DEVANAGARI LETTER NNA
0924 DEVANAGARI LETTER TA
0925 DEVANAGARI LETTER THA
0926 DEVANAGARI LETTER DA
0927 DEVANAGARI LETTER DHA
0928 DEVANAGARI LETTER NA
0929 DEVANAGARI LETTER NNNA
092A DEVANAGARI LETTER PA
092B DEVANAGARI LETTER PHA
092C DEVANAGARI LETTER BA
092D DEVANAGARI LETTER BHA
092E DEVANAGARI LETTER MA
092F DEVANAGARI LETTER YA
0930 DEVANAGARI LETTER RA
0931 DEVANAGARI LETTER RRA
0932 DEVANAGARI LETTER LA
0933 DEVANAGARI LETTER LLA
0934 DEVANAGARI LETTER LLLA
0935 DEVANAGARI LETTER VA
0936 DEVANAGARI LETTER SHA
0937 DEVANAGARI LETTER SSA
0938 DEVANAGARI LETTER SA
0939 DEVANAGARI LETTER HA
093C DEVANAGARI SIGN NUKTA
093D DEVANAGARI SIGN AVAGRAHA
093E DEVANAGARI VOWEL SIGN AA
093F DEVANAGARI VOWEL SIGN I
0940 DEVANAGARI VOWEL SIGN II
0941 DEVANAGARI VOWEL SIGN U
0942 DEVANAGARI VOWEL SIGN UU
0943 DEVANAGARI VOWEL SIGN VOCALIC R
0944 DEVANAGARI VOWEL SIGN VOCALIC RR
0945 DEVANAGARI VOWEL SIGN CANDRA E
0946 DEVANAGARI VOWEL SIGN SHORT E
0947 DEVANAGARI VOWEL SIGN E
0948 DEVANAGARI VOWEL SIGN AI
0949 DEVANAGARI VOWEL SIGN CANDRA O
094A DEVANAGARI VOWEL SIGN SHORT O
094B DEVANAGARI VOWEL SIGN O
094C DEVANAGARI VOWEL SIGN AU
094D DEVANAGARI SIGN VIRAMA
0950 DEVANAGARI OM
0951 DEVANAGARI STRESS SIGN UDATTA
0952 DEVANAGARI STRESS SIGN ANUDATTA
0953 DEVANAGARI GRAVE ACCENT
0954 DEVANAGARI ACUTE ACCENT
0958 DEVANAGARI LETTER QA
0959 DEVANAGARI LETTER KHHA
095A DEVANAGARI LETTER GHHA
095B DEVANAGARI LETTER ZA
095C DEVANAGARI LETTER DDDHA
095D DEVANAGARI LETTER RHA
095E DEVANAGARI LETTER FA
095F DEVANAGARI LETTER YYA
0960 DEVANAGARI LETTER VOCALIC RR
0961 DEVANAGARI LETTER VOCALIC LL
0962 DEVANAGARI VOWEL SIGN VOCALIC L
0963 DEVANAGARI VOWEL SIGN VOCALIC LL
0964 DEVANAGARI DANDA
0965 DEVANAGARI DOUBLE DANDA
0966 DEVANAGARI DIGIT ZERO
0967 DEVANAGARI DIGIT ONE
0968 DEVANAGARI DIGIT TWO
0969 DEVANAGARI DIGIT THREE
096A DEVANAGARI DIGIT FOUR
096B DEVANAGARI DIGIT FIVE
096C DEVANAGARI DIGIT SIX
096D DEVANAGARI DIGIT SEVEN
096E DEVANAGARI DIGIT EIGHT
096F DEVANAGARI DIGIT NINE
0970 DEVANAGARI ABBREVIATION SIGN
0981 BENGALI SIGN CANDRABINDU
0982 BENGALI SIGN ANUSVARA
0983 BENGALI SIGN VISARGA
0985 BENGALI LETTER A
0986 BENGALI LETTER AA
0987 BENGALI LETTER I
0988 BENGALI LETTER II
0989 BENGALI LETTER U
098A BENGALI LETTER UU
098B BENGALI LETTER VOCALIC R
098C BENGALI LETTER VOCALIC L
098F BENGALI LETTER E
0990 BENGALI LETTER AI
0993 BENGALI LETTER O
0994 BENGALI LETTER AU
0995 BENGALI LETTER KA
0996 BENGALI LETTER KHA
0997 BENGALI LETTER GA
0998 BENGALI LETTER GHA
0999 BENGALI LETTER NGA
099A BENGALI LETTER CA
099B BENGALI LETTER CHA
099C BENGALI LETTER JA
099D BENGALI LETTER JHA
099E BENGALI LETTER NYA
099F BENGALI LETTER TTA
09A0 BENGALI LETTER TTHA
09A1 BENGALI LETTER DDA
09A2 BENGALI LETTER DDHA
09A3 BENGALI LETTER NNA
09A4 BENGALI LETTER TA
09A5 BENGALI LETTER THA
09A6 BENGALI LETTER DA
09A7 BENGALI LETTER DHA
09A8 BENGALI LETTER NA
09AA BENGALI LETTER PA
09AB BENGALI LETTER PHA
09AC BENGALI LETTER BA
09AD BENGALI LETTER BHA
09AE BENGALI LETTER MA
09AF BENGALI LETTER YA
09B0 BENGALI LETTER RA
09B2 BENGALI LETTER LA
09B6 BENGALI LETTER SHA
09B7 BENGALI LETTER SSA
09B8 BENGALI LETTER SA
09B9 BENGALI LETTER HA
09BC BENGALI SIGN NUKTA
09BE BENGALI VOWEL SIGN AA
09BF BENGALI VOWEL SIGN I
09C0 BENGALI VOWEL SIGN II
09C1 BENGALI VOWEL SIGN U
09C2 BENGALI VOWEL SIGN UU
09C3 BENGALI VOWEL SIGN VOCALIC R
09C4 BENGALI VOWEL SIGN VOCALIC RR
09C7 BENGALI VOWEL SIGN E
09C8 BENGALI VOWEL SIGN AI
09CB BENGALI VOWEL SIGN O
09CC BENGALI VOWEL SIGN AU
09CD BENGALI SIGN VIRAMA
09D7 BENGALI AU LENGTH MARK
09DC BENGALI LETTER RRA
09DD BENGALI LETTER RHA
09DF BENGALI LETTER YYA
09E0 BENGALI LETTER VOCALIC RR
09E1 BENGALI LETTER VOCALIC LL
09E2 BENGALI VOWEL SIGN VOCALIC L
09E3 BENGALI VOWEL SIGN VOCALIC LL
09E6 BENGALI DIGIT ZERO
09E7 BENGALI DIGIT ONE
09E8 BENGALI DIGIT TWO
09E9 BENGALI DIGIT THREE
09EA BENGALI DIGIT FOUR
09EB BENGALI DIGIT FIVE
09EC BENGALI DIGIT SIX
09ED BENGALI DIGIT SEVEN
09EE BENGALI DIGIT EIGHT
09EF BENGALI DIGIT NINE
09F0 BENGALI LETTER RA WITH MIDDLE DIAGONAL
09F1 BENGALI LETTER RA WITH LOWER DIAGONAL
09F2 BENGALI RUPEE MARK
09F3 BENGALI RUPEE SIGN
09F4 BENGALI CURRENCY NUMERATOR ONE
09F5 BENGALI CURRENCY NUMERATOR TWO
09F6 BENGALI CURRENCY NUMERATOR THREE
09F7 BENGALI CURRENCY NUMERATOR FOUR
09F8 BENGALI CURRENCY NUMERATOR ONE LESS THAN THE DENOMINATOR
09F9 BENGALI CURRENCY DENOMINATOR SIXTEEN
09FA BENGALI ISSHAR
0A02 GURMUKHI SIGN BINDI
0A05 GURMUKHI LETTER A
0A06 GURMUKHI LETTER AA
0A07 GURMUKHI LETTER I
0A08 GURMUKHI LETTER II
0A09 GURMUKHI LETTER U
0A0A GURMUKHI LETTER UU
0A0F GURMUKHI LETTER EE
0A10 GURMUKHI LETTER AI
0A13 GURMUKHI LETTER OO
0A14 GURMUKHI LETTER AU
0A15 GURMUKHI LETTER KA
0A16 GURMUKHI LETTER KHA
0A17 GURMUKHI LETTER GA
0A18 GURMUKHI LETTER GHA
0A19 GURMUKHI LETTER NGA
0A1A GURMUKHI LETTER CA
0A1B GURMUKHI LETTER CHA
0A1C GURMUKHI LETTER JA
0A1D GURMUKHI LETTER JHA
0A1E GURMUKHI LETTER NYA
0A1F GURMUKHI LETTER TTA
0A20 GURMUKHI LETTER TTHA
0A21 GURMUKHI LETTER DDA
0A22 GURMUKHI LETTER DDHA
0A23 GURMUKHI LETTER NNA
0A24 GURMUKHI LETTER TA
0A25 GURMUKHI LETTER THA
0A26 GURMUKHI LETTER DA
0A27 GURMUKHI LETTER DHA
0A28 GURMUKHI LETTER NA
0A2A GURMUKHI LETTER PA
0A2B GURMUKHI LETTER PHA
0A2C GURMUKHI LETTER BA
0A2D GURMUKHI LETTER BHA
0A2E GURMUKHI LETTER MA
0A2F GURMUKHI LETTER YA
0A30 GURMUKHI LETTER RA
0A32 GURMUKHI LETTER LA
0A33 GURMUKHI LETTER LLA
0A35 GURMUKHI LETTER VA
0A36 GURMUKHI LETTER SHA
0A38 GURMUKHI LETTER SA
0A39 GURMUKHI LETTER HA
0A3C GURMUKHI SIGN NUKTA
0A3E GURMUKHI VOWEL SIGN AA
0A3F GURMUKHI VOWEL SIGN I
0A40 GURMUKHI VOWEL SIGN II
0A41 GURMUKHI VOWEL SIGN U
0A42 GURMUKHI VOWEL SIGN UU
0A47 GURMUKHI VOWEL SIGN EE
0A48 GURMUKHI VOWEL SIGN AI
0A4B GURMUKHI VOWEL SIGN OO
0A4C GURMUKHI VOWEL SIGN AU
0A4D GURMUKHI SIGN VIRAMA
0A59 GURMUKHI LETTER KHHA
0A5A GURMUKHI LETTER GHHA
0A5B GURMUKHI LETTER ZA
0A5C GURMUKHI LETTER RRA
0A5E GURMUKHI LETTER FA
0A66 GURMUKHI DIGIT ZERO
0A67 GURMUKHI DIGIT ONE
0A68 GURMUKHI DIGIT TWO
0A69 GURMUKHI DIGIT THREE
0A6A GURMUKHI DIGIT FOUR
0A6B GURMUKHI DIGIT FIVE
0A6C GURMUKHI DIGIT SIX
0A6D GURMUKHI DIGIT SEVEN
0A6E GURMUKHI DIGIT EIGHT
0A6F GURMUKHI DIGIT NINE
0A70 GURMUKHI TIPPI
0A71 GURMUKHI ADDAK
0A72 GURMUKHI IRI
0A73 GURMUKHI URA
0A74 GURMUKHI EK ONKAR
0A81 GUJARATI SIGN CANDRABINDU
0A82 GUJARATI SIGN ANUSVARA
0A83 GUJARATI SIGN VISARGA
0A85 GUJARATI LETTER A
0A86 GUJARATI LETTER AA
0A87 GUJARATI LETTER I
0A88 GUJARATI LETTER II
0A89 GUJARATI LETTER U
0A8A GUJARATI LETTER UU
0A8B GUJARATI LETTER VOCALIC R
0A8D GUJARATI VOWEL CANDRA E
0A8F GUJARATI LETTER E
0A90 GUJARATI LETTER AI
0A91 GUJARATI VOWEL CANDRA O
0A93 GUJARATI LETTER O
0A94 GUJARATI LETTER AU
0A95 GUJARATI LETTER KA
0A96 GUJARATI LETTER KHA
0A97 GUJARATI LETTER GA
0A98 GUJARATI LETTER GHA
0A99 GUJARATI LETTER NGA
0A9A GUJARATI LETTER CA
0A9B GUJARATI LETTER CHA
0A9C GUJARATI LETTER JA
0A9D GUJARATI LETTER JHA
0A9E GUJARATI LETTER NYA
0A9F GUJARATI LETTER TTA
0AA0 GUJARATI LETTER TTHA
0AA1 GUJARATI LETTER DDA
0AA2 GUJARATI LETTER DDHA
0AA3 GUJARATI LETTER NNA
0AA4 GUJARATI LETTER TA
0AA5 GUJARATI LETTER THA
0AA6 GUJARATI LETTER DA
0AA7 GUJARATI LETTER DHA
0AA8 GUJARATI LETTER NA
0AAA GUJARATI LETTER PA
0AAB GUJARATI LETTER PHA
0AAC GUJARATI LETTER BA
0AAD GUJARATI LETTER BHA
0AAE GUJARATI LETTER MA
0AAF GUJARATI LETTER YA
0AB0 GUJARATI LETTER RA
0AB2 GUJARATI LETTER LA
0AB3 GUJARATI LETTER LLA
0AB5 GUJARATI LETTER VA
0AB6 GUJARATI LETTER SHA
0AB7 GUJARATI LETTER SSA
0AB8 GUJARATI LETTER SA
0AB9 GUJARATI LETTER HA
0ABC GUJARATI SIGN NUKTA
0ABD GUJARATI SIGN AVAGRAHA
0ABE GUJARATI VOWEL SIGN AA
0ABF GUJARATI VOWEL SIGN I
0AC0 GUJARATI VOWEL SIGN II
0AC1 GUJARATI VOWEL SIGN U
0AC2 GUJARATI VOWEL SIGN UU
0AC3 GUJARATI VOWEL SIGN VOCALIC R
0AC4 GUJARATI VOWEL SIGN VOCALIC RR
0AC5 GUJARATI VOWEL SIGN CANDRA E
0AC7 GUJARATI VOWEL SIGN E
0AC8 GUJARATI VOWEL SIGN AI
0AC9 GUJARATI VOWEL SIGN CANDRA O
0ACB GUJARATI VOWEL SIGN O
0ACC GUJARATI VOWEL SIGN AU
0ACD GUJARATI SIGN VIRAMA
0AD0 GUJARATI OM
0AE0 GUJARATI LETTER VOCALIC RR
0AE6 GUJARATI DIGIT ZERO
0AE7 GUJARATI DIGIT ONE
0AE8 GUJARATI DIGIT TWO
0AE9 GUJARATI DIGIT THREE
0AEA GUJARATI DIGIT FOUR
0AEB GUJARATI DIGIT FIVE
0AEC GUJARATI DIGIT SIX
0AED GUJARATI DIGIT SEVEN
0AEE GUJARATI DIGIT EIGHT
0AEF GUJARATI DIGIT NINE
0B01 ORIYA SIGN CANDRABINDU
0B02 ORIYA SIGN ANUSVARA
0B03 ORIYA SIGN VISARGA
0B05 ORIYA LETTER A
0B06 ORIYA LETTER AA
0B07 ORIYA LETTER I
0B08 ORIYA LETTER II
0B09 ORIYA LETTER U
0B0A ORIYA LETTER UU
0B0B ORIYA LETTER VOCALIC R
0B0C ORIYA LETTER VOCALIC L
0B0F ORIYA LETTER E
0B10 ORIYA LETTER AI
0B13 ORIYA LETTER O
0B14 ORIYA LETTER AU
0B15 ORIYA LETTER KA
0B16 ORIYA LETTER KHA
0B17 ORIYA LETTER GA
0B18 ORIYA LETTER GHA
0B19 ORIYA LETTER NGA
0B1A ORIYA LETTER CA
0B1B ORIYA LETTER CHA
0B1C ORIYA LETTER JA
0B1D ORIYA LETTER JHA
0B1E ORIYA LETTER NYA
0B1F ORIYA LETTER TTA
0B20 ORIYA LETTER TTHA
0B21 ORIYA LETTER DDA
0B22 ORIYA LETTER DDHA
0B23 ORIYA LETTER NNA
0B24 ORIYA LETTER TA
0B25 ORIYA LETTER THA
0B26 ORIYA LETTER DA
0B27 ORIYA LETTER DHA
0B28 ORIYA LETTER NA
0B2A ORIYA LETTER PA
0B2B ORIYA LETTER PHA
0B2C ORIYA LETTER BA
0B2D ORIYA LETTER BHA
0B2E ORIYA LETTER MA
0B2F ORIYA LETTER YA
0B30 ORIYA LETTER RA
0B32 ORIYA LETTER LA
0B33 ORIYA LETTER LLA
0B36 ORIYA LETTER SHA
0B37 ORIYA LETTER SSA
0B38 ORIYA LETTER SA
0B39 ORIYA LETTER HA
0B3C ORIYA SIGN NUKTA
0B3D ORIYA SIGN AVAGRAHA
0B3E ORIYA VOWEL SIGN AA
0B3F ORIYA VOWEL SIGN I
0B40 ORIYA VOWEL SIGN II
0B41 ORIYA VOWEL SIGN U
0B42 ORIYA VOWEL SIGN UU
0B43 ORIYA VOWEL SIGN VOCALIC R
0B47 ORIYA VOWEL SIGN E
0B48 ORIYA VOWEL SIGN AI
0B4B ORIYA VOWEL SIGN O
0B4C ORIYA VOWEL SIGN AU
0B4D ORIYA SIGN VIRAMA
0B56 ORIYA AI LENGTH MARK
0B57 ORIYA AU LENGTH MARK
0B5C ORIYA LETTER RRA
0B5D ORIYA LETTER RHA
0B5F ORIYA LETTER YYA
0B60 ORIYA LETTER VOCALIC RR
0B61 ORIYA LETTER VOCALIC LL
0B66 ORIYA DIGIT ZERO
0B67 ORIYA DIGIT ONE
0B68 ORIYA DIGIT TWO
0B69 ORIYA DIGIT THREE
0B6A ORIYA DIGIT FOUR
0B6B ORIYA DIGIT FIVE
0B6C ORIYA DIGIT SIX
0B6D ORIYA DIGIT SEVEN
0B6E ORIYA DIGIT EIGHT
0B6F ORIYA DIGIT NINE
0B70 ORIYA ISSHAR
0B82 TAMIL SIGN ANUSVARA
0B83 TAMIL SIGN VISARGA
0B85 TAMIL LETTER A
0B86 TAMIL LETTER AA
0B87 TAMIL LETTER I
0B88 TAMIL LETTER II
0B89 TAMIL LETTER U
0B8A TAMIL LETTER UU
0B8E TAMIL LETTER E
0B8F TAMIL LETTER EE
0B90 TAMIL LETTER AI
0B92 TAMIL LETTER O
0B93 TAMIL LETTER OO
0B94 TAMIL LETTER AU
0B95 TAMIL LETTER KA
0B99 TAMIL LETTER NGA
0B9A TAMIL LETTER CA
0B9C TAMIL LETTER JA
0B9E TAMIL LETTER NYA
0B9F TAMIL LETTER TTA
0BA3 TAMIL LETTER NNA
0BA4 TAMIL LETTER TA
0BA8 TAMIL LETTER NA
0BA9 TAMIL LETTER NNNA
0BAA TAMIL LETTER PA
0BAE TAMIL LETTER MA
0BAF TAMIL LETTER YA
0BB0 TAMIL LETTER RA
0BB1 TAMIL LETTER RRA
0BB2 TAMIL LETTER LA
0BB3 TAMIL LETTER LLA
0BB4 TAMIL LETTER LLLA
0BB5 TAMIL LETTER VA
0BB7 TAMIL LETTER SSA
0BB8 TAMIL LETTER SA
0BB9 TAMIL LETTER HA
0BBE TAMIL VOWEL SIGN AA
0BBF TAMIL VOWEL SIGN I
0BC0 TAMIL VOWEL SIGN II
0BC1 TAMIL VOWEL SIGN U
0BC2 TAMIL VOWEL SIGN UU
0BC6 TAMIL VOWEL SIGN E
0BC7 TAMIL VOWEL SIGN EE
0BC8 TAMIL VOWEL SIGN AI
0BCA TAMIL VOWEL SIGN O
0BCB TAMIL VOWEL SIGN OO
0BCC TAMIL VOWEL SIGN AU
0BCD TAMIL SIGN VIRAMA
0BD7 TAMIL AU LENGTH MARK
0BE7 TAMIL DIGIT ONE
0BE8 TAMIL DIGIT TWO
0BE9 TAMIL DIGIT THREE
0BEA TAMIL DIGIT FOUR
0BEB TAMIL DIGIT FIVE
0BEC TAMIL DIGIT SIX
0BED TAMIL DIGIT SEVEN
0BEE TAMIL DIGIT EIGHT
0BEF TAMIL DIGIT NINE
0BF0 TAMIL NUMBER TEN
0BF1 TAMIL NUMBER ONE HUNDRED
0BF2 TAMIL NUMBER ONE THOUSAND
0C01 TELUGU SIGN CANDRABINDU
0C02 TELUGU SIGN ANUSVARA
0C03 TELUGU SIGN VISARGA
0C05 TELUGU LETTER A
0C06 TELUGU LETTER AA
0C07 TELUGU LETTER I
0C08 TELUGU LETTER II
0C09 TELUGU LETTER U
0C0A TELUGU LETTER UU
0C0B TELUGU LETTER VOCALIC R
0C0C TELUGU LETTER VOCALIC L
0C0E TELUGU LETTER E
0C0F TELUGU LETTER EE
0C10 TELUGU LETTER AI
0C12 TELUGU LETTER O
0C13 TELUGU LETTER OO
0C14 TELUGU LETTER AU
0C15 TELUGU LETTER KA
0C16 TELUGU LETTER KHA
0C17 TELUGU LETTER GA
0C18 TELUGU LETTER GHA
0C19 TELUGU LETTER NGA
0C1A TELUGU LETTER CA
0C1B TELUGU LETTER CHA
0C1C TELUGU LETTER JA
0C1D TELUGU LETTER JHA
0C1E TELUGU LETTER NYA
0C1F TELUGU LETTER TTA
0C20 TELUGU LETTER TTHA
0C21 TELUGU LETTER DDA
0C22 TELUGU LETTER DDHA
0C23 TELUGU LETTER NNA
0C24 TELUGU LETTER TA
0C25 TELUGU LETTER THA
0C26 TELUGU LETTER DA
0C27 TELUGU LETTER DHA
0C28 TELUGU LETTER NA
0C2A TELUGU LETTER PA
0C2B TELUGU LETTER PHA
0C2C TELUGU LETTER BA
0C2D TELUGU LETTER BHA
0C2E TELUGU LETTER MA
0C2F TELUGU LETTER YA
0C30 TELUGU LETTER RA
0C31 TELUGU LETTER RRA
0C32 TELUGU LETTER LA
0C33 TELUGU LETTER LLA
0C35 TELUGU LETTER VA
0C36 TELUGU LETTER SHA
0C37 TELUGU LETTER SSA
0C38 TELUGU LETTER SA
0C39 TELUGU LETTER HA
0C3E TELUGU VOWEL SIGN AA
0C3F TELUGU VOWEL SIGN I
0C40 TELUGU VOWEL SIGN II
0C41 TELUGU VOWEL SIGN U
0C42 TELUGU VOWEL SIGN UU
0C43 TELUGU VOWEL SIGN VOCALIC R
0C44 TELUGU VOWEL SIGN VOCALIC RR
0C46 TELUGU VOWEL SIGN E
0C47 TELUGU VOWEL SIGN EE
0C48 TELUGU VOWEL SIGN AI
0C4A TELUGU VOWEL SIGN O
0C4B TELUGU VOWEL SIGN OO
0C4C TELUGU VOWEL SIGN AU
0C4D TELUGU SIGN VIRAMA
0C55 TELUGU LENGTH MARK
0C56 TELUGU AI LENGTH MARK
0C60 TELUGU LETTER VOCALIC RR
0C61 TELUGU LETTER VOCALIC LL
0C66 TELUGU DIGIT ZERO
0C67 TELUGU DIGIT ONE
0C68 TELUGU DIGIT TWO
0C69 TELUGU DIGIT THREE
0C6A TELUGU DIGIT FOUR
0C6B TELUGU DIGIT FIVE
0C6C TELUGU DIGIT SIX
0C6D TELUGU DIGIT SEVEN
0C6E TELUGU DIGIT EIGHT
0C6F TELUGU DIGIT NINE
0C82 KANNADA SIGN ANUSVARA
0C83 KANNADA SIGN VISARGA
0C85 KANNADA LETTER A
0C86 KANNADA LETTER AA
0C87 KANNADA LETTER I
0C88 KANNADA LETTER II
0C89 KANNADA LETTER U
0C8A KANNADA LETTER UU
0C8B KANNADA LETTER VOCALIC R
0C8C KANNADA LETTER VOCALIC L
0C8E KANNADA LETTER E
0C8F KANNADA LETTER EE
0C90 KANNADA LETTER AI
0C92 KANNADA LETTER O
0C93 KANNADA LETTER OO
0C94 KANNADA LETTER AU
0C95 KANNADA LETTER KA
0C96 KANNADA LETTER KHA
0C97 KANNADA LETTER GA
0C98 KANNADA LETTER GHA
0C99 KANNADA LETTER NGA
0C9A KANNADA LETTER CA
0C9B KANNADA LETTER CHA
0C9C KANNADA LETTER JA
0C9D KANNADA LETTER JHA
0C9E KANNADA LETTER NYA
0C9F KANNADA LETTER TTA
0CA0 KANNADA LETTER TTHA
0CA1 KANNADA LETTER DDA
0CA2 KANNADA LETTER DDHA
0CA3 KANNADA LETTER NNA
0CA4 KANNADA LETTER TA
0CA5 KANNADA LETTER THA
0CA6 KANNADA LETTER DA
0CA7 KANNADA LETTER DHA
0CA8 KANNADA LETTER NA
0CAA KANNADA LETTER PA
0CAB KANNADA LETTER PHA
0CAC KANNADA LETTER BA
0CAD KANNADA LETTER BHA
0CAE KANNADA LETTER MA
0CAF KANNADA LETTER YA
0CB0 KANNADA LETTER RA
0CB1 KANNADA LETTER RRA
0CB2 KANNADA LETTER LA
0CB3 KANNADA LETTER LLA
0CB5 KANNADA LETTER VA
0CB6 KANNADA LETTER SHA
0CB7 KANNADA LETTER SSA
0CB8 KANNADA LETTER SA
0CB9 KANNADA LETTER HA
0CBE KANNADA VOWEL SIGN AA
0CBF KANNADA VOWEL SIGN I
0CC0 KANNADA VOWEL SIGN II
0CC1 KANNADA VOWEL SIGN U
0CC2 KANNADA VOWEL SIGN UU
0CC3 KANNADA VOWEL SIGN VOCALIC R
0CC4 KANNADA VOWEL SIGN VOCALIC RR
0CC6 KANNADA VOWEL SIGN E
0CC7 KANNADA VOWEL SIGN EE
0CC8 KANNADA VOWEL SIGN AI
0CCA KANNADA VOWEL SIGN O
0CCB KANNADA VOWEL SIGN OO
0CCC KANNADA VOWEL SIGN AU
0CCD KANNADA SIGN VIRAMA
0CD5 KANNADA LENGTH MARK
0CD6 KANNADA AI LENGTH MARK
0CDE KANNADA LETTER FA
0CE0 KANNADA LETTER VOCALIC RR
0CE1 KANNADA LETTER VOCALIC LL
0CE6 KANNADA DIGIT ZERO
0CE7 KANNADA DIGIT ONE
0CE8 KANNADA DIGIT TWO
0CE9 KANNADA DIGIT THREE
0CEA KANNADA DIGIT FOUR
0CEB KANNADA DIGIT FIVE
0CEC KANNADA DIGIT SIX
0CED KANNADA DIGIT SEVEN
0CEE KANNADA DIGIT EIGHT
0CEF KANNADA DIGIT NINE
0D02 MALAYALAM SIGN ANUSVARA
0D03 MALAYALAM SIGN VISARGA
0D05 MALAYALAM LETTER A
0D06 MALAYALAM LETTER AA
0D07 MALAYALAM LETTER I
0D08 MALAYALAM LETTER II
0D09 MALAYALAM LETTER U
0D0A MALAYALAM LETTER UU
0D0B MALAYALAM LETTER VOCALIC R
0D0C MALAYALAM LETTER VOCALIC L
0D0E MALAYALAM LETTER E
0D0F MALAYALAM LETTER EE
0D10 MALAYALAM LETTER AI
0D12 MALAYALAM LETTER O
0D13 MALAYALAM LETTER OO
0D14 MALAYALAM LETTER AU
0D15 MALAYALAM LETTER KA
0D16 MALAYALAM LETTER KHA
0D17 MALAYALAM LETTER GA
0D18 MALAYALAM LETTER GHA
0D19 MALAYALAM LETTER NGA
0D1A MALAYALAM LETTER CA
0D1B MALAYALAM LETTER CHA
0D1C MALAYALAM LETTER JA
0D1D MALAYALAM LETTER JHA
0D1E MALAYALAM LETTER NYA
0D1F MALAYALAM LETTER TTA
0D20 MALAYALAM LETTER TTHA
0D21 MALAYALAM LETTER DDA
0D22 MALAYALAM LETTER DDHA
0D23 MALAYALAM LETTER NNA
0D24 MALAYALAM LETTER TA
0D25 MALAYALAM LETTER THA
0D26 MALAYALAM LETTER DA
0D27 MALAYALAM LETTER DHA
0D28 MALAYALAM LETTER NA
0D2A MALAYALAM LETTER PA
0D2B MALAYALAM LETTER PHA
0D2C MALAYALAM LETTER BA
0D2D MALAYALAM LETTER BHA
0D2E MALAYALAM LETTER MA
0D2F MALAYALAM LETTER YA
0D30 MALAYALAM LETTER RA
0D31 MALAYALAM LETTER RRA
0D32 MALAYALAM LETTER LA
0D33 MALAYALAM LETTER LLA
0D34 MALAYALAM LETTER LLLA
0D35 MALAYALAM LETTER VA
0D36 MALAYALAM LETTER SHA
0D37 MALAYALAM LETTER SSA
0D38 MALAYALAM LETTER SA
0D39 MALAYALAM LETTER HA
0D3E MALAYALAM VOWEL SIGN AA
0D3F MALAYALAM VOWEL SIGN I
0D40 MALAYALAM VOWEL SIGN II
0D41 MALAYALAM VOWEL SIGN U
0D42 MALAYALAM VOWEL SIGN UU
0D43 MALAYALAM VOWEL SIGN VOCALIC R
0D46 MALAYALAM VOWEL SIGN E
0D47 MALAYALAM VOWEL SIGN EE
0D48 MALAYALAM VOWEL SIGN AI
0D4A MALAYALAM VOWEL SIGN O
0D4B MALAYALAM VOWEL SIGN OO
0D4C MALAYALAM VOWEL SIGN AU
0D4D MALAYALAM SIGN VIRAMA
0D57 MALAYALAM AU LENGTH MARK
0D60 MALAYALAM LETTER VOCALIC RR
0D61 MALAYALAM LETTER VOCALIC LL
0D66 MALAYALAM DIGIT ZERO
0D67 MALAYALAM DIGIT ONE
0D68 MALAYALAM DIGIT TWO
0D69 MALAYALAM DIGIT THREE
0D6A MALAYALAM DIGIT FOUR
0D6B MALAYALAM DIGIT FIVE
0D6C MALAYALAM DIGIT SIX
0D6D MALAYALAM DIGIT SEVEN
0D6E MALAYALAM DIGIT EIGHT
0D6F MALAYALAM DIGIT NINE
0E01 THAI CHARACTER KO KAI
0E02 THAI CHARACTER KHO KHAI
0E03 THAI CHARACTER KHO KHUAT
0E04 THAI CHARACTER KHO KHWAI
0E05 THAI CHARACTER KHO KHON
0E06 THAI CHARACTER KHO RAKHANG
0E07 THAI CHARACTER NGO NGU
0E08 THAI CHARACTER CHO CHAN
0E09 THAI CHARACTER CHO CHING
0E0A THAI CHARACTER CHO CHANG
0E0B THAI CHARACTER SO SO
0E0C THAI CHARACTER CHO CHOE
0E0D THAI CHARACTER YO YING
0E0E THAI CHARACTER DO CHADA
0E0F THAI CHARACTER TO PATAK
0E10 THAI CHARACTER THO THAN
0E11 THAI CHARACTER THO NANGMONTHO
0E12 THAI CHARACTER THO PHUTHAO
0E13 THAI CHARACTER NO NEN
0E14 THAI CHARACTER DO DEK
0E15 THAI CHARACTER TO TAO
0E16 THAI CHARACTER THO THUNG
0E17 THAI CHARACTER THO THAHAN
0E18 THAI CHARACTER THO THONG
0E19 THAI CHARACTER NO NU
0E1A THAI CHARACTER BO BAIMAI
0E1B THAI CHARACTER PO PLA
0E1C THAI CHARACTER PHO PHUNG
0E1D THAI CHARACTER FO FA
0E1E THAI CHARACTER PHO PHAN
0E1F THAI CHARACTER FO FAN
0E20 THAI CHARACTER PHO SAMPHAO
0E21 THAI CHARACTER MO MA
0E22 THAI CHARACTER YO YAK
0E23 THAI CHARACTER RO RUA
0E24 THAI CHARACTER RU
0E25 THAI CHARACTER LO LING
0E26 THAI CHARACTER LU
0E27 THAI CHARACTER WO WAEN
0E28 THAI CHARACTER SO SALA
0E29 THAI CHARACTER SO RUSI
0E2A THAI CHARACTER SO SUA
0E2B THAI CHARACTER HO HIP
0E2C THAI CHARACTER LO CHULA
0E2D THAI CHARACTER O ANG
0E2E THAI CHARACTER HO NOKHUK
0E2F THAI CHARACTER PAIYANNOI
0E30 THAI CHARACTER SARA A
0E31 THAI CHARACTER MAI HAN-AKAT
0E32 THAI CHARACTER SARA AA
0E33 THAI CHARACTER SARA AM
0E34 THAI CHARACTER SARA I
0E35 THAI CHARACTER SARA II
0E36 THAI CHARACTER SARA UE
0E37 THAI CHARACTER SARA UEE
0E38 THAI CHARACTER SARA U
0E39 THAI CHARACTER SARA UU
0E3A THAI CHARACTER PHINTHU
0E3F THAI CURRENCY SYMBOL BAHT
0E40 THAI CHARACTER SARA E
0E41 THAI CHARACTER SARA AE
0E42 THAI CHARACTER SARA O
0E43 THAI CHARACTER SARA AI MAIMUAN
0E44 THAI CHARACTER SARA AI MAIMALAI
0E45 THAI CHARACTER LAKKHANGYAO
0E46 THAI CHARACTER MAIYAMOK
0E47 THAI CHARACTER MAITAIKHU
0E48 THAI CHARACTER MAI EK
0E49 THAI CHARACTER MAI THO
0E4A THAI CHARACTER MAI TRI
0E4B THAI CHARACTER MAI CHATTAWA
0E4C THAI CHARACTER THANTHAKHAT
0E4D THAI CHARACTER NIKHAHIT
0E4E THAI CHARACTER YAMAKKAN
0E4F THAI CHARACTER FONGMAN
0E50 THAI DIGIT ZERO
0E51 THAI DIGIT ONE
0E52 THAI DIGIT TWO
0E53 THAI DIGIT THREE
0E54 THAI DIGIT FOUR
0E55 THAI DIGIT FIVE
0E56 THAI DIGIT SIX
0E57 THAI DIGIT SEVEN
0E58 THAI DIGIT EIGHT
0E59 THAI DIGIT NINE
0E5A THAI CHARACTER ANGKHANKHU
0E5B THAI CHARACTER KHOMUT
0E81 LAO LETTER KO
0E82 LAO LETTER KHO SUNG
0E84 LAO LETTER KHO TAM
0E87 LAO LETTER NGO
0E88 LAO LETTER CO
0E8A LAO LETTER SO TAM
0E8D LAO LETTER NYO
0E94 LAO LETTER DO
0E95 LAO LETTER TO
0E96 LAO LETTER THO SUNG
0E97 LAO LETTER THO TAM
0E99 LAO LETTER NO
0E9A LAO LETTER BO
0E9B LAO LETTER PO
0E9C LAO LETTER PHO SUNG
0E9D LAO LETTER FO TAM
0E9E LAO LETTER PHO TAM
0E9F LAO LETTER FO SUNG
0EA1 LAO LETTER MO
0EA2 LAO LETTER YO
0EA3 LAO LETTER LO LING
0EA5 LAO LETTER LO LOOT
0EA7 LAO LETTER WO
0EAA LAO LETTER SO SUNG
0EAB LAO LETTER HO SUNG
0EAD LAO LETTER O
0EAE LAO LETTER HO TAM
0EAF LAO ELLIPSIS
0EB0 LAO VOWEL SIGN A
0EB1 LAO VOWEL SIGN MAI KAN
0EB2 LAO VOWEL SIGN AA
0EB3 LAO VOWEL SIGN AM
0EB4 LAO VOWEL SIGN I
0EB5 LAO VOWEL SIGN II
0EB6 LAO VOWEL SIGN Y
0EB7 LAO VOWEL SIGN YY
0EB8 LAO VOWEL SIGN U
0EB9 LAO VOWEL SIGN UU
0EBB LAO VOWEL SIGN MAI KON
0EBC LAO SEMIVOWEL SIGN LO
0EBD LAO SEMIVOWEL SIGN NYO
0EC0 LAO VOWEL SIGN E
0EC1 LAO VOWEL SIGN EI
0EC2 LAO VOWEL SIGN O
0EC3 LAO VOWEL SIGN AY
0EC4 LAO VOWEL SIGN AI
0EC6 LAO KO LA
0EC8 LAO TONE MAI EK
0EC9 LAO TONE MAI THO
0ECA LAO TONE MAI TI
0ECB LAO TONE MAI CATAWA
0ECC LAO CANCELLATION MARK
0ECD LAO NIGGAHITA
0ED0 LAO DIGIT ZERO
0ED1 LAO DIGIT ONE
0ED2 LAO DIGIT TWO
0ED3 LAO DIGIT THREE
0ED4 LAO DIGIT FOUR
0ED5 LAO DIGIT FIVE
0ED6 LAO DIGIT SIX
0ED7 LAO DIGIT SEVEN
0ED8 LAO DIGIT EIGHT
0ED9 LAO DIGIT NINE
0EDC LAO HO NO
0EDD LAO HO MO
0F00 TIBETAN SYLLABLE OM
0F01 TIBETAN MARK GTER YIG MGO TRUNCATED A
0F02 TIBETAN MARK GTER YIG MGO -UM RNAM BCAD MA
0F03 TIBETAN MARK GTER YIG MGO -UM GTER TSHEG MA
0F04 TIBETAN MARK INITIAL YIG MGO MDUN MA
0F05 TIBETAN MARK CLOSING YIG MGO SGAB MA
0F06 TIBETAN MARK CARET YIG MGO PHUR SHAD MA
0F07 TIBETAN MARK YIG MGO TSHEG SHAD MA
0F08 TIBETAN MARK SBRUL SHAD
0F09 TIBETAN MARK BSKUR YIG MGO
0F0A TIBETAN MARK BKA- SHOG YIG MGO
0F0B TIBETAN MARK INTERSYLLABIC TSHEG
0F0C TIBETAN MARK DELIMITER TSHEG BSTAR
0F0D TIBETAN MARK SHAD
0F0E TIBETAN MARK NYIS SHAD
0F0F TIBETAN MARK TSHEG SHAD
0F10 TIBETAN MARK NYIS TSHEG SHAD
0F11 TIBETAN MARK RIN CHEN SPUNGS SHAD
0F12 TIBETAN MARK RGYA GRAM SHAD
0F13 TIBETAN MARK CARET -DZUD RTAGS ME LONG CAN
0F14 TIBETAN MARK GTER TSHEG
0F15 TIBETAN LOGOTYPE SIGN CHAD RTAGS
0F16 TIBETAN LOGOTYPE SIGN LHAG RTAGS
0F17 TIBETAN ASTROLOGICAL SIGN SGRA GCAN -CHAR RTAGS
0F18 TIBETAN ASTROLOGICAL SIGN -KHYUD PA
0F19 TIBETAN ASTROLOGICAL SIGN SDONG TSHUGS
0F1A TIBETAN SIGN RDEL DKAR GCIG
0F1B TIBETAN SIGN RDEL DKAR GNYIS
0F1C TIBETAN SIGN RDEL DKAR GSUM
0F1D TIBETAN SIGN RDEL NAG GCIG
0F1E TIBETAN SIGN RDEL NAG GNYIS
0F1F TIBETAN SIGN RDEL DKAR RDEL NAG
0F20 TIBETAN DIGIT ZERO
0F21 TIBETAN DIGIT ONE
0F22 TIBETAN DIGIT TWO
0F23 TIBETAN DIGIT THREE
0F24 TIBETAN DIGIT FOUR
0F25 TIBETAN DIGIT FIVE
0F26 TIBETAN DIGIT SIX
0F27 TIBETAN DIGIT SEVEN
0F28 TIBETAN DIGIT EIGHT
0F29 TIBETAN DIGIT NINE
0F2A TIBETAN DIGIT HALF ONE
0F2B TIBETAN DIGIT HALF TWO
0F2C TIBETAN DIGIT HALF THREE
0F2D TIBETAN DIGIT HALF FOUR
0F2E TIBETAN DIGIT HALF FIVE
0F2F TIBETAN DIGIT HALF SIX
0F30 TIBETAN DIGIT HALF SEVEN
0F31 TIBETAN DIGIT HALF EIGHT
0F32 TIBETAN DIGIT HALF NINE
0F33 TIBETAN DIGIT HALF ZERO
0F34 TIBETAN MARK BSDUS RTAGS
0F35 TIBETAN MARK NGAS BZUNG NYI ZLA
0F36 TIBETAN MARK CARET -DZUD RTAGS BZHI MIG CAN
0F37 TIBETAN MARK NGAS BZUNG SGOR RTAGS
0F38 TIBETAN MARK CHE MGO
0F39 TIBETAN MARK TSA -PHRU
0F3A TIBETAN MARK GUG RTAGS GYON
0F3B TIBETAN MARK GUG RTAGS GYAS
0F3C TIBETAN MARK ANG KHANG GYON
0F3D TIBETAN MARK ANG KHANG GYAS
0F3E TIBETAN SIGN YAR TSHES
0F3F TIBETAN SIGN MAR TSHES
0F40 TIBETAN LETTER KA
0F41 TIBETAN LETTER KHA
0F42 TIBETAN LETTER GA
0F43 TIBETAN LETTER GHA
0F44 TIBETAN LETTER NGA
0F45 TIBETAN LETTER CA
0F46 TIBETAN LETTER CHA
0F47 TIBETAN LETTER JA
0F49 TIBETAN LETTER NYA
0F4A TIBETAN LETTER TTA
0F4B TIBETAN LETTER TTHA
0F4C TIBETAN LETTER DDA
0F4D TIBETAN LETTER DDHA
0F4E TIBETAN LETTER NNA
0F4F TIBETAN LETTER TA
0F50 TIBETAN LETTER THA
0F51 TIBETAN LETTER DA
0F52 TIBETAN LETTER DHA
0F53 TIBETAN LETTER NA
0F54 TIBETAN LETTER PA
0F55 TIBETAN LETTER PHA
0F56 TIBETAN LETTER BA
0F57 TIBETAN LETTER BHA
0F58 TIBETAN LETTER MA
0F59 TIBETAN LETTER TSA
0F5A TIBETAN LETTER TSHA
0F5B TIBETAN LETTER DZA
0F5C TIBETAN LETTER DZHA
0F5D TIBETAN LETTER WA
0F5E TIBETAN LETTER ZHA
0F5F TIBETAN LETTER ZA
0F60 TIBETAN LETTER -A
0F61 TIBETAN LETTER YA
0F62 TIBETAN LETTER RA
0F63 TIBETAN LETTER LA
0F64 TIBETAN LETTER SHA
0F65 TIBETAN LETTER SSA
0F66 TIBETAN LETTER SA
0F67 TIBETAN LETTER HA
0F68 TIBETAN LETTER A
0F69 TIBETAN LETTER KSSA
0F71 TIBETAN VOWEL SIGN AA
0F72 TIBETAN VOWEL SIGN I
0F73 TIBETAN VOWEL SIGN II
0F74 TIBETAN VOWEL SIGN U
0F75 TIBETAN VOWEL SIGN UU
0F76 TIBETAN VOWEL SIGN VOCALIC R
0F77 TIBETAN VOWEL SIGN VOCALIC RR
0F78 TIBETAN VOWEL SIGN VOCALIC L
0F79 TIBETAN VOWEL SIGN VOCALIC LL
0F7A TIBETAN VOWEL SIGN E
0F7B TIBETAN VOWEL SIGN EE
0F7C TIBETAN VOWEL SIGN O
0F7D TIBETAN VOWEL SIGN OO
0F7E TIBETAN SIGN RJES SU NGA RO
0F7F TIBETAN SIGN RNAM BCAD
0F80 TIBETAN VOWEL SIGN REVERSED I
0F81 TIBETAN VOWEL SIGN REVERSED II
0F82 TIBETAN SIGN NYI ZLA NAA DA
0F83 TIBETAN SIGN SNA LDAN
0F84 TIBETAN MARK HALANTA
0F85 TIBETAN MARK PALUTA
0F86 TIBETAN SIGN LCI RTAGS
0F87 TIBETAN SIGN YANG RTAGS
0F88 TIBETAN SIGN LCE TSA CAN
0F89 TIBETAN SIGN MCHU CAN
0F8A TIBETAN SIGN GRU CAN RGYINGS
0F8B TIBETAN SIGN GRU MED RGYINGS
0F90 TIBETAN SUBJOINED LETTER KA
0F91 TIBETAN SUBJOINED LETTER KHA
0F92 TIBETAN SUBJOINED LETTER GA
0F93 TIBETAN SUBJOINED LETTER GHA
0F94 TIBETAN SUBJOINED LETTER NGA
0F95 TIBETAN SUBJOINED LETTER CA
0F97 TIBETAN SUBJOINED LETTER JA
0F99 TIBETAN SUBJOINED LETTER NYA
0F9A TIBETAN SUBJOINED LETTER TTA
0F9B TIBETAN SUBJOINED LETTER TTHA
0F9C TIBETAN SUBJOINED LETTER DDA
0F9D TIBETAN SUBJOINED LETTER DDHA
0F9E TIBETAN SUBJOINED LETTER NNA
0F9F TIBETAN SUBJOINED LETTER TA
0FA0 TIBETAN SUBJOINED LETTER THA
0FA1 TIBETAN SUBJOINED LETTER DA
0FA2 TIBETAN SUBJOINED LETTER DHA
0FA3 TIBETAN SUBJOINED LETTER NA
0FA4 TIBETAN SUBJOINED LETTER PA
0FA5 TIBETAN SUBJOINED LETTER PHA
0FA6 TIBETAN SUBJOINED LETTER BA
0FA7 TIBETAN SUBJOINED LETTER BHA
0FA8 TIBETAN SUBJOINED LETTER MA
0FA9 TIBETAN SUBJOINED LETTER TSA
0FAA TIBETAN SUBJOINED LETTER TSHA
0FAB TIBETAN SUBJOINED LETTER DZA
0FAC TIBETAN SUBJOINED LETTER DZHA
0FAD TIBETAN SUBJOINED LETTER WA
0FB1 TIBETAN SUBJOINED LETTER YA
0FB2 TIBETAN SUBJOINED LETTER RA
0FB3 TIBETAN SUBJOINED LETTER LA
0FB4 TIBETAN SUBJOINED LETTER SHA
0FB5 TIBETAN SUBJOINED LETTER SSA
0FB6 TIBETAN SUBJOINED LETTER SA
0FB7 TIBETAN SUBJOINED LETTER HA
0FB9 TIBETAN SUBJOINED LETTER KSSA
10A0 GEORGIAN CAPITAL LETTER AN
10A1 GEORGIAN CAPITAL LETTER BAN
10A2 GEORGIAN CAPITAL LETTER GAN
10A3 GEORGIAN CAPITAL LETTER DON
10A4 GEORGIAN CAPITAL LETTER EN
10A5 GEORGIAN CAPITAL LETTER VIN
10A6 GEORGIAN CAPITAL LETTER ZEN
10A7 GEORGIAN CAPITAL LETTER TAN
10A8 GEORGIAN CAPITAL LETTER IN
10A9 GEORGIAN CAPITAL LETTER KAN
10AA GEORGIAN CAPITAL LETTER LAS
10AB GEORGIAN CAPITAL LETTER MAN
10AC GEORGIAN CAPITAL LETTER NAR
10AD GEORGIAN CAPITAL LETTER ON
10AE GEORGIAN CAPITAL LETTER PAR
10AF GEORGIAN CAPITAL LETTER ZHAR
10B0 GEORGIAN CAPITAL LETTER RAE
10B1 GEORGIAN CAPITAL LETTER SAN
10B2 GEORGIAN CAPITAL LETTER TAR
10B3 GEORGIAN CAPITAL LETTER UN
10B4 GEORGIAN CAPITAL LETTER PHAR
10B5 GEORGIAN CAPITAL LETTER KHAR
10B6 GEORGIAN CAPITAL LETTER GHAN
10B7 GEORGIAN CAPITAL LETTER QAR
10B8 GEORGIAN CAPITAL LETTER SHIN
10B9 GEORGIAN CAPITAL LETTER CHIN
10BA GEORGIAN CAPITAL LETTER CAN
10BB GEORGIAN CAPITAL LETTER JIL
10BC GEORGIAN CAPITAL LETTER CIL
10BD GEORGIAN CAPITAL LETTER CHAR
10BE GEORGIAN CAPITAL LETTER XAN
10BF GEORGIAN CAPITAL LETTER JHAN
10C0 GEORGIAN CAPITAL LETTER HAE
10C1 GEORGIAN CAPITAL LETTER HE
10C2 GEORGIAN CAPITAL LETTER HIE
10C3 GEORGIAN CAPITAL LETTER WE
10C4 GEORGIAN CAPITAL LETTER HAR
10C5 GEORGIAN CAPITAL LETTER HOE
10D0 GEORGIAN LETTER AN
10D1 GEORGIAN LETTER BAN
10D2 GEORGIAN LETTER GAN
10D3 GEORGIAN LETTER DON
10D4 GEORGIAN LETTER EN
10D5 GEORGIAN LETTER VIN
10D6 GEORGIAN LETTER ZEN
10D7 GEORGIAN LETTER TAN
10D8 GEORGIAN LETTER IN
10D9 GEORGIAN LETTER KAN
10DA GEORGIAN LETTER LAS
10DB GEORGIAN LETTER MAN
10DC GEORGIAN LETTER NAR
10DD GEORGIAN LETTER ON
10DE GEORGIAN LETTER PAR
10DF GEORGIAN LETTER ZHAR
10E0 GEORGIAN LETTER RAE
10E1 GEORGIAN LETTER SAN
10E2 GEORGIAN LETTER TAR
10E3 GEORGIAN LETTER UN
10E4 GEORGIAN LETTER PHAR
10E5 GEORGIAN LETTER KHAR
10E6 GEORGIAN LETTER GHAN
10E7 GEORGIAN LETTER QAR
10E8 GEORGIAN LETTER SHIN
10E9 GEORGIAN LETTER CHIN
10EA GEORGIAN LETTER CAN
10EB GEORGIAN LETTER JIL
10EC GEORGIAN LETTER CIL
10ED GEORGIAN LETTER CHAR
10EE GEORGIAN LETTER XAN
10EF GEORGIAN LETTER JHAN
10F0 GEORGIAN LETTER HAE
10F1 GEORGIAN LETTER HE
10F2 GEORGIAN LETTER HIE
10F3 GEORGIAN LETTER WE
10F4 GEORGIAN LETTER HAR
10F5 GEORGIAN LETTER HOE
10F6 GEORGIAN LETTER FI
10FB GEORGIAN PARAGRAPH SEPARATOR
1100 HANGUL CHOSEONG KIYEOK
1101 HANGUL CHOSEONG SSANGKIYEOK
1102 HANGUL CHOSEONG NIEUN
1103 HANGUL CHOSEONG TIKEUT
1104 HANGUL CHOSEONG SSANGTIKEUT
1105 HANGUL CHOSEONG RIEUL
1106 HANGUL CHOSEONG MIEUM
1107 HANGUL CHOSEONG PIEUP
1108 HANGUL CHOSEONG SSANGPIEUP
1109 HANGUL CHOSEONG SIOS
110A HANGUL CHOSEONG SSANGSIOS
110B HANGUL CHOSEONG IEUNG
110C HANGUL CHOSEONG CIEUC
110D HANGUL CHOSEONG SSANGCIEUC
110E HANGUL CHOSEONG CHIEUCH
110F HANGUL CHOSEONG KHIEUKH
1110 HANGUL CHOSEONG THIEUTH
1111 HANGUL CHOSEONG PHIEUPH
1112 HANGUL CHOSEONG HIEUH
1113 HANGUL CHOSEONG NIEUN-KIYEOK
1114 HANGUL CHOSEONG SSANGNIEUN
1115 HANGUL CHOSEONG NIEUN-TIKEUT
1116 HANGUL CHOSEONG NIEUN-PIEUP
1117 HANGUL CHOSEONG TIKEUT-KIYEOK
1118 HANGUL CHOSEONG RIEUL-NIEUN
1119 HANGUL CHOSEONG SSANGRIEUL
111A HANGUL CHOSEONG RIEUL-HIEUH
111B HANGUL CHOSEONG KAPYEOUNRIEUL
111C HANGUL CHOSEONG MIEUM-PIEUP
111D HANGUL CHOSEONG KAPYEOUNMIEUM
111E HANGUL CHOSEONG PIEUP-KIYEOK
111F HANGUL CHOSEONG PIEUP-NIEUN
1120 HANGUL CHOSEONG PIEUP-TIKEUT
1121 HANGUL CHOSEONG PIEUP-SIOS
1122 HANGUL CHOSEONG PIEUP-SIOS-KIYEOK
1123 HANGUL CHOSEONG PIEUP-SIOS-TIKEUT
1124 HANGUL CHOSEONG PIEUP-SIOS-PIEUP
1125 HANGUL CHOSEONG PIEUP-SSANGSIOS
1126 HANGUL CHOSEONG PIEUP-SIOS-CIEUC
1127 HANGUL CHOSEONG PIEUP-CIEUC
1128 HANGUL CHOSEONG PIEUP-CHIEUCH
1129 HANGUL CHOSEONG PIEUP-THIEUTH
112A HANGUL CHOSEONG PIEUP-PHIEUPH
112B HANGUL CHOSEONG KAPYEOUNPIEUP
112C HANGUL CHOSEONG KAPYEOUNSSANGPIEUP
112D HANGUL CHOSEONG SIOS-KIYEOK
112E HANGUL CHOSEONG SIOS-NIEUN
112F HANGUL CHOSEONG SIOS-TIKEUT
1130 HANGUL CHOSEONG SIOS-RIEUL
1131 HANGUL CHOSEONG SIOS-MIEUM
1132 HANGUL CHOSEONG SIOS-PIEUP
1133 HANGUL CHOSEONG SIOS-PIEUP-KIYEOK
1134 HANGUL CHOSEONG SIOS-SSANGSIOS
1135 HANGUL CHOSEONG SIOS-IEUNG
1136 HANGUL CHOSEONG SIOS-CIEUC
1137 HANGUL CHOSEONG SIOS-CHIEUCH
1138 HANGUL CHOSEONG SIOS-KHIEUKH
1139 HANGUL CHOSEONG SIOS-THIEUTH
113A HANGUL CHOSEONG SIOS-PHIEUPH
113B HANGUL CHOSEONG SIOS-HIEUH
113C HANGUL CHOSEONG CHITUEUMSIOS
113D HANGUL CHOSEONG CHITUEUMSSANGSIOS
113E HANGUL CHOSEONG CEONGCHIEUMSIOS
113F HANGUL CHOSEONG CEONGCHIEUMSSANGSIOS
1140 HANGUL CHOSEONG PANSIOS
1141 HANGUL CHOSEONG IEUNG-KIYEOK
1142 HANGUL CHOSEONG IEUNG-TIKEUT
1143 HANGUL CHOSEONG IEUNG-MIEUM
1144 HANGUL CHOSEONG IEUNG-PIEUP
1145 HANGUL CHOSEONG IEUNG-SIOS
1146 HANGUL CHOSEONG IEUNG-PANSIOS
1147 HANGUL CHOSEONG SSANGIEUNG
1148 HANGUL CHOSEONG IEUNG-CIEUC
1149 HANGUL CHOSEONG IEUNG-CHIEUCH
114A HANGUL CHOSEONG IEUNG-THIEUTH
114B HANGUL CHOSEONG IEUNG-PHIEUPH
114C HANGUL CHOSEONG YESIEUNG
114D HANGUL CHOSEONG CIEUC-IEUNG
114E HANGUL CHOSEONG CHITUEUMCIEUC
114F HANGUL CHOSEONG CHITUEUMSSANGCIEUC
1150 HANGUL CHOSEONG CEONGCHIEUMCIEUC
1151 HANGUL CHOSEONG CEONGCHIEUMSSANGCIEUC
1152 HANGUL CHOSEONG CHIEUCH-KHIEUKH
1153 HANGUL CHOSEONG CHIEUCH-HIEUH
1154 HANGUL CHOSEONG CHITUEUMCHIEUCH
1155 HANGUL CHOSEONG CEONGCHIEUMCHIEUCH
1156 HANGUL CHOSEONG PHIEUPH-PIEUP
1157 HANGUL CHOSEONG KAPYEOUNPHIEUPH
1158 HANGUL CHOSEONG SSANGHIEUH
1159 HANGUL CHOSEONG YEORINHIEUH
115F HANGUL CHOSEONG FILLER
1160 HANGUL JUNGSEONG FILLER
1161 HANGUL JUNGSEONG A
1162 HANGUL JUNGSEONG AE
1163 HANGUL JUNGSEONG YA
1164 HANGUL JUNGSEONG YAE
1165 HANGUL JUNGSEONG EO
1166 HANGUL JUNGSEONG E
1167 HANGUL JUNGSEONG YEO
1168 HANGUL JUNGSEONG YE
1169 HANGUL JUNGSEONG O
116A HANGUL JUNGSEONG WA
116B HANGUL JUNGSEONG WAE
116C HANGUL JUNGSEONG OE
116D HANGUL JUNGSEONG YO
116E HANGUL JUNGSEONG U
116F HANGUL JUNGSEONG WEO
1170 HANGUL JUNGSEONG WE
1171 HANGUL JUNGSEONG WI
1172 HANGUL JUNGSEONG YU
1173 HANGUL JUNGSEONG EU
1174 HANGUL JUNGSEONG YI
1175 HANGUL JUNGSEONG I
1176 HANGUL JUNGSEONG A-O
1177 HANGUL JUNGSEONG A-U
1178 HANGUL JUNGSEONG YA-O
1179 HANGUL JUNGSEONG YA-YO
117A HANGUL JUNGSEONG EO-O
117B HANGUL JUNGSEONG EO-U
117C HANGUL JUNGSEONG EO-EU
117D HANGUL JUNGSEONG YEO-O
117E HANGUL JUNGSEONG YEO-U
117F HANGUL JUNGSEONG O-EO
1180 HANGUL JUNGSEONG O-E
1181 HANGUL JUNGSEONG O-YE
1182 HANGUL JUNGSEONG O-O
1183 HANGUL JUNGSEONG O-U
1184 HANGUL JUNGSEONG YO-YA
1185 HANGUL JUNGSEONG YO-YAE
1186 HANGUL JUNGSEONG YO-YEO
1187 HANGUL JUNGSEONG YO-O
1188 HANGUL JUNGSEONG YO-I
1189 HANGUL JUNGSEONG U-A
118A HANGUL JUNGSEONG U-AE
118B HANGUL JUNGSEONG U-EO-EU
118C HANGUL JUNGSEONG U-YE
118D HANGUL JUNGSEONG U-U
118E HANGUL JUNGSEONG YU-A
118F HANGUL JUNGSEONG YU-EO
1190 HANGUL JUNGSEONG YU-E
1191 HANGUL JUNGSEONG YU-YEO
1192 HANGUL JUNGSEONG YU-YE
1193 HANGUL JUNGSEONG YU-U
1194 HANGUL JUNGSEONG YU-I
1195 HANGUL JUNGSEONG EU-U
1196 HANGUL JUNGSEONG EU-EU
1197 HANGUL JUNGSEONG YI-U
1198 HANGUL JUNGSEONG I-A
1199 HANGUL JUNGSEONG I-YA
119A HANGUL JUNGSEONG I-O
119B HANGUL JUNGSEONG I-U
119C HANGUL JUNGSEONG I-EU
119D HANGUL JUNGSEONG I-ARAEA
119E HANGUL JUNGSEONG ARAEA
119F HANGUL JUNGSEONG ARAEA-EO
11A0 HANGUL JUNGSEONG ARAEA-U
11A1 HANGUL JUNGSEONG ARAEA-I
11A2 HANGUL JUNGSEONG SSANGARAEA
11A8 HANGUL JONGSEONG KIYEOK
11A9 HANGUL JONGSEONG SSANGKIYEOK
11AA HANGUL JONGSEONG KIYEOK-SIOS
11AB HANGUL JONGSEONG NIEUN
11AC HANGUL JONGSEONG NIEUN-CIEUC
11AD HANGUL JONGSEONG NIEUN-HIEUH
11AE HANGUL JONGSEONG TIKEUT
11AF HANGUL JONGSEONG RIEUL
11B0 HANGUL JONGSEONG RIEUL-KIYEOK
11B1 HANGUL JONGSEONG RIEUL-MIEUM
11B2 HANGUL JONGSEONG RIEUL-PIEUP
11B3 HANGUL JONGSEONG RIEUL-SIOS
11B4 HANGUL JONGSEONG RIEUL-THIEUTH
11B5 HANGUL JONGSEONG RIEUL-PHIEUPH
11B6 HANGUL JONGSEONG RIEUL-HIEUH
11B7 HANGUL JONGSEONG MIEUM
11B8 HANGUL JONGSEONG PIEUP
11B9 HANGUL JONGSEONG PIEUP-SIOS
11BA HANGUL JONGSEONG SIOS
11BB HANGUL JONGSEONG SSANGSIOS
11BC HANGUL JONGSEONG IEUNG
11BD HANGUL JONGSEONG CIEUC
11BE HANGUL JONGSEONG CHIEUCH
11BF HANGUL JONGSEONG KHIEUKH
11C0 HANGUL JONGSEONG THIEUTH
11C1 HANGUL JONGSEONG PHIEUPH
11C2 HANGUL JONGSEONG HIEUH
11C3 HANGUL JONGSEONG KIYEOK-RIEUL
11C4 HANGUL JONGSEONG KIYEOK-SIOS-KIYEOK
11C5 HANGUL JONGSEONG NIEUN-KIYEOK
11C6 HANGUL JONGSEONG NIEUN-TIKEUT
11C7 HANGUL JONGSEONG NIEUN-SIOS
11C8 HANGUL JONGSEONG NIEUN-PANSIOS
11C9 HANGUL JONGSEONG NIEUN-THIEUTH
11CA HANGUL JONGSEONG TIKEUT-KIYEOK
11CB HANGUL JONGSEONG TIKEUT-RIEUL
11CC HANGUL JONGSEONG RIEUL-KIYEOK-SIOS
11CD HANGUL JONGSEONG RIEUL-NIEUN
11CE HANGUL JONGSEONG RIEUL-TIKEUT
11CF HANGUL JONGSEONG RIEUL-TIKEUT-HIEUH
11D0 HANGUL JONGSEONG SSANGRIEUL
11D1 HANGUL JONGSEONG RIEUL-MIEUM-KIYEOK
11D2 HANGUL JONGSEONG RIEUL-MIEUM-SIOS
11D3 HANGUL JONGSEONG RIEUL-PIEUP-SIOS
11D4 HANGUL JONGSEONG RIEUL-PIEUP-HIEUH
11D5 HANGUL JONGSEONG RIEUL-KAPYEOUNPIEUP
11D6 HANGUL JONGSEONG RIEUL-SSANGSIOS
11D7 HANGUL JONGSEONG RIEUL-PANSIOS
11D8 HANGUL JONGSEONG RIEUL-KHIEUKH
11D9 HANGUL JONGSEONG RIEUL-YEORINHIEUH
11DA HANGUL JONGSEONG MIEUM-KIYEOK
11DB HANGUL JONGSEONG MIEUM-RIEUL
11DC HANGUL JONGSEONG MIEUM-PIEUP
11DD HANGUL JONGSEONG MIEUM-SIOS
11DE HANGUL JONGSEONG MIEUM-SSANGSIOS
11DF HANGUL JONGSEONG MIEUM-PANSIOS
11E0 HANGUL JONGSEONG MIEUM-CHIEUCH
11E1 HANGUL JONGSEONG MIEUM-HIEUH
11E2 HANGUL JONGSEONG KAPYEOUNMIEUM
11E3 HANGUL JONGSEONG PIEUP-RIEUL
11E4 HANGUL JONGSEONG PIEUP-PHIEUPH
11E5 HANGUL JONGSEONG PIEUP-HIEUH
11E6 HANGUL JONGSEONG KAPYEOUNPIEUP
11E7 HANGUL JONGSEONG SIOS-KIYEOK
11E8 HANGUL JONGSEONG SIOS-TIKEUT
11E9 HANGUL JONGSEONG SIOS-RIEUL
11EA HANGUL JONGSEONG SIOS-PIEUP
11EB HANGUL JONGSEONG PANSIOS
11EC HANGUL JONGSEONG IEUNG-KIYEOK
11ED HANGUL JONGSEONG IEUNG-SSANGKIYEOK
11EE HANGUL JONGSEONG SSANGIEUNG
11EF HANGUL JONGSEONG IEUNG-KHIEUKH
11F0 HANGUL JONGSEONG YESIEUNG
11F1 HANGUL JONGSEONG YESIEUNG-SIOS
11F2 HANGUL JONGSEONG YESIEUNG-PANSIOS
11F3 HANGUL JONGSEONG PHIEUPH-PIEUP
11F4 HANGUL JONGSEONG KAPYEOUNPHIEUPH
11F5 HANGUL JONGSEONG HIEUH-NIEUN
11F6 HANGUL JONGSEONG HIEUH-RIEUL
11F7 HANGUL JONGSEONG HIEUH-MIEUM
11F8 HANGUL JONGSEONG HIEUH-PIEUP
11F9 HANGUL JONGSEONG YEORINHIEUH
1E00 LATIN CAPITAL LETTER A WITH RING BELOW
1E01 LATIN SMALL LETTER A WITH RING BELOW
1E02 LATIN CAPITAL LETTER B WITH DOT ABOVE
1E03 LATIN SMALL LETTER B WITH DOT ABOVE
1E04 LATIN CAPITAL LETTER B WITH DOT BELOW
1E05 LATIN SMALL LETTER B WITH DOT BELOW
1E06 LATIN CAPITAL LETTER B WITH LINE BELOW
1E07 LATIN SMALL LETTER B WITH LINE BELOW
1E08 LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
1E09 LATIN SMALL LETTER C WITH CEDILLA AND ACUTE
1E0A LATIN CAPITAL LETTER D WITH DOT ABOVE
1E0B LATIN SMALL LETTER D WITH DOT ABOVE
1E0C LATIN CAPITAL LETTER D WITH DOT BELOW
1E0D LATIN SMALL LETTER D WITH DOT BELOW
1E0E LATIN CAPITAL LETTER D WITH LINE BELOW
1E0F LATIN SMALL LETTER D WITH LINE BELOW
1E10 LATIN CAPITAL LETTER D WITH CEDILLA
1E11 LATIN SMALL LETTER D WITH CEDILLA
1E12 LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
1E13 LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW
1E14 LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
1E15 LATIN SMALL LETTER E WITH MACRON AND GRAVE
1E16 LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
1E17 LATIN SMALL LETTER E WITH MACRON AND ACUTE
1E18 LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
1E19 LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW
1E1A LATIN CAPITAL LETTER E WITH TILDE BELOW
1E1B LATIN SMALL LETTER E WITH TILDE BELOW
1E1C LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
1E1D LATIN SMALL LETTER E WITH CEDILLA AND BREVE
1E1E LATIN CAPITAL LETTER F WITH DOT ABOVE
1E1F LATIN SMALL LETTER F WITH DOT ABOVE
1E20 LATIN CAPITAL LETTER G WITH MACRON
1E21 LATIN SMALL LETTER G WITH MACRON
1E22 LATIN CAPITAL LETTER H WITH DOT ABOVE
1E23 LATIN SMALL LETTER H WITH DOT ABOVE
1E24 LATIN CAPITAL LETTER H WITH DOT BELOW
1E25 LATIN SMALL LETTER H WITH DOT BELOW
1E26 LATIN CAPITAL LETTER H WITH DIAERESIS
1E27 LATIN SMALL LETTER H WITH DIAERESIS
1E28 LATIN CAPITAL LETTER H WITH CEDILLA
1E29 LATIN SMALL LETTER H WITH CEDILLA
1E2A LATIN CAPITAL LETTER H WITH BREVE BELOW
1E2B LATIN SMALL LETTER H WITH BREVE BELOW
1E2C LATIN CAPITAL LETTER I WITH TILDE BELOW
1E2D LATIN SMALL LETTER I WITH TILDE BELOW
1E2E LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
1E2F LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE
1E30 LATIN CAPITAL LETTER K WITH ACUTE
1E31 LATIN SMALL LETTER K WITH ACUTE
1E32 LATIN CAPITAL LETTER K WITH DOT BELOW
1E33 LATIN SMALL LETTER K WITH DOT BELOW
1E34 LATIN CAPITAL LETTER K WITH LINE BELOW
1E35 LATIN SMALL LETTER K WITH LINE BELOW
1E36 LATIN CAPITAL LETTER L WITH DOT BELOW
1E37 LATIN SMALL LETTER L WITH DOT BELOW
1E38 LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
1E39 LATIN SMALL LETTER L WITH DOT BELOW AND MACRON
1E3A LATIN CAPITAL LETTER L WITH LINE BELOW
1E3B LATIN SMALL LETTER L WITH LINE BELOW
1E3C LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
1E3D LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW
1E3E LATIN CAPITAL LETTER M WITH ACUTE
1E3F LATIN SMALL LETTER M WITH ACUTE
1E40 LATIN CAPITAL LETTER M WITH DOT ABOVE
1E41 LATIN SMALL LETTER M WITH DOT ABOVE
1E42 LATIN CAPITAL LETTER M WITH DOT BELOW
1E43 LATIN SMALL LETTER M WITH DOT BELOW
1E44 LATIN CAPITAL LETTER N WITH DOT ABOVE
1E45 LATIN SMALL LETTER N WITH DOT ABOVE
1E46 LATIN CAPITAL LETTER N WITH DOT BELOW
1E47 LATIN SMALL LETTER N WITH DOT BELOW
1E48 LATIN CAPITAL LETTER N WITH LINE BELOW
1E49 LATIN SMALL LETTER N WITH LINE BELOW
1E4A LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
1E4B LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW
1E4C LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
1E4D LATIN SMALL LETTER O WITH TILDE AND ACUTE
1E4E LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
1E4F LATIN SMALL LETTER O WITH TILDE AND DIAERESIS
1E50 LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
1E51 LATIN SMALL LETTER O WITH MACRON AND GRAVE
1E52 LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
1E53 LATIN SMALL LETTER O WITH MACRON AND ACUTE
1E54 LATIN CAPITAL LETTER P WITH ACUTE
1E55 LATIN SMALL LETTER P WITH ACUTE
1E56 LATIN CAPITAL LETTER P WITH DOT ABOVE
1E57 LATIN SMALL LETTER P WITH DOT ABOVE
1E58 LATIN CAPITAL LETTER R WITH DOT ABOVE
1E59 LATIN SMALL LETTER R WITH DOT ABOVE
1E5A LATIN CAPITAL LETTER R WITH DOT BELOW
1E5B LATIN SMALL LETTER R WITH DOT BELOW
1E5C LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
1E5D LATIN SMALL LETTER R WITH DOT BELOW AND MACRON
1E5E LATIN CAPITAL LETTER R WITH LINE BELOW
1E5F LATIN SMALL LETTER R WITH LINE BELOW
1E60 LATIN CAPITAL LETTER S WITH DOT ABOVE
1E61 LATIN SMALL LETTER S WITH DOT ABOVE
1E62 LATIN CAPITAL LETTER S WITH DOT BELOW
1E63 LATIN SMALL LETTER S WITH DOT BELOW
1E64 LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
1E65 LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE
1E66 LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
1E67 LATIN SMALL LETTER S WITH CARON AND DOT ABOVE
1E68 LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
1E69 LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE
1E6A LATIN CAPITAL LETTER T WITH DOT ABOVE
1E6B LATIN SMALL LETTER T WITH DOT ABOVE
1E6C LATIN CAPITAL LETTER T WITH DOT BELOW
1E6D LATIN SMALL LETTER T WITH DOT BELOW
1E6E LATIN CAPITAL LETTER T WITH LINE BELOW
1E6F LATIN SMALL LETTER T WITH LINE BELOW
1E70 LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
1E71 LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW
1E72 LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
1E73 LATIN SMALL LETTER U WITH DIAERESIS BELOW
1E74 LATIN CAPITAL LETTER U WITH TILDE BELOW
1E75 LATIN SMALL LETTER U WITH TILDE BELOW
1E76 LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
1E77 LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW
1E78 LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
1E79 LATIN SMALL LETTER U WITH TILDE AND ACUTE
1E7A LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
1E7B LATIN SMALL LETTER U WITH MACRON AND DIAERESIS
1E7C LATIN CAPITAL LETTER V WITH TILDE
1E7D LATIN SMALL LETTER V WITH TILDE
1E7E LATIN CAPITAL LETTER V WITH DOT BELOW
1E7F LATIN SMALL LETTER V WITH DOT BELOW
1E80 LATIN CAPITAL LETTER W WITH GRAVE
1E81 LATIN SMALL LETTER W WITH GRAVE
1E82 LATIN CAPITAL LETTER W WITH ACUTE
1E83 LATIN SMALL LETTER W WITH ACUTE
1E84 LATIN CAPITAL LETTER W WITH DIAERESIS
1E85 LATIN SMALL LETTER W WITH DIAERESIS
1E86 LATIN CAPITAL LETTER W WITH DOT ABOVE
1E87 LATIN SMALL LETTER W WITH DOT ABOVE
1E88 LATIN CAPITAL LETTER W WITH DOT BELOW
1E89 LATIN SMALL LETTER W WITH DOT BELOW
1E8A LATIN CAPITAL LETTER X WITH DOT ABOVE
1E8B LATIN SMALL LETTER X WITH DOT ABOVE
1E8C LATIN CAPITAL LETTER X WITH DIAERESIS
1E8D LATIN SMALL LETTER X WITH DIAERESIS
1E8E LATIN CAPITAL LETTER Y WITH DOT ABOVE
1E8F LATIN SMALL LETTER Y WITH DOT ABOVE
1E90 LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
1E91 LATIN SMALL LETTER Z WITH CIRCUMFLEX
1E92 LATIN CAPITAL LETTER Z WITH DOT BELOW
1E93 LATIN SMALL LETTER Z WITH DOT BELOW
1E94 LATIN CAPITAL LETTER Z WITH LINE BELOW
1E95 LATIN SMALL LETTER Z WITH LINE BELOW
1E96 LATIN SMALL LETTER H WITH LINE BELOW
1E97 LATIN SMALL LETTER T WITH DIAERESIS
1E98 LATIN SMALL LETTER W WITH RING ABOVE
1E99 LATIN SMALL LETTER Y WITH RING ABOVE
1E9A LATIN SMALL LETTER A WITH RIGHT HALF RING
1E9B LATIN SMALL LETTER LONG S WITH DOT ABOVE
1EA0 LATIN CAPITAL LETTER A WITH DOT BELOW
1EA1 LATIN SMALL LETTER A WITH DOT BELOW
1EA2 LATIN CAPITAL LETTER A WITH HOOK ABOVE
1EA3 LATIN SMALL LETTER A WITH HOOK ABOVE
1EA4 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
1EA5 LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
1EA6 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
1EA7 LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
1EA8 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
1EA9 LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
1EAA LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
1EAB LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
1EAC LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
1EAD LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
1EAE LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
1EAF LATIN SMALL LETTER A WITH BREVE AND ACUTE
1EB0 LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
1EB1 LATIN SMALL LETTER A WITH BREVE AND GRAVE
1EB2 LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
1EB3 LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
1EB4 LATIN CAPITAL LETTER A WITH BREVE AND TILDE
1EB5 LATIN SMALL LETTER A WITH BREVE AND TILDE
1EB6 LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
1EB7 LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
1EB8 LATIN CAPITAL LETTER E WITH DOT BELOW
1EB9 LATIN SMALL LETTER E WITH DOT BELOW
1EBA LATIN CAPITAL LETTER E WITH HOOK ABOVE
1EBB LATIN SMALL LETTER E WITH HOOK ABOVE
1EBC LATIN CAPITAL LETTER E WITH TILDE
1EBD LATIN SMALL LETTER E WITH TILDE
1EBE LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
1EBF LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
1EC0 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
1EC1 LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
1EC2 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
1EC3 LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
1EC4 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
1EC5 LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
1EC6 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
1EC7 LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
1EC8 LATIN CAPITAL LETTER I WITH HOOK ABOVE
1EC9 LATIN SMALL LETTER I WITH HOOK ABOVE
1ECA LATIN CAPITAL LETTER I WITH DOT BELOW
1ECB LATIN SMALL LETTER I WITH DOT BELOW
1ECC LATIN CAPITAL LETTER O WITH DOT BELOW
1ECD LATIN SMALL LETTER O WITH DOT BELOW
1ECE LATIN CAPITAL LETTER O WITH HOOK ABOVE
1ECF LATIN SMALL LETTER O WITH HOOK ABOVE
1ED0 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
1ED1 LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
1ED2 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
1ED3 LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
1ED4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
1ED5 LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
1ED6 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
1ED7 LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
1ED8 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
1ED9 LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
1EDA LATIN CAPITAL LETTER O WITH HORN AND ACUTE
1EDB LATIN SMALL LETTER O WITH HORN AND ACUTE
1EDC LATIN CAPITAL LETTER O WITH HORN AND GRAVE
1EDD LATIN SMALL LETTER O WITH HORN AND GRAVE
1EDE LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
1EDF LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
1EE0 LATIN CAPITAL LETTER O WITH HORN AND TILDE
1EE1 LATIN SMALL LETTER O WITH HORN AND TILDE
1EE2 LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
1EE3 LATIN SMALL LETTER O WITH HORN AND DOT BELOW
1EE4 LATIN CAPITAL LETTER U WITH DOT BELOW
1EE5 LATIN SMALL LETTER U WITH DOT BELOW
1EE6 LATIN CAPITAL LETTER U WITH HOOK ABOVE
1EE7 LATIN SMALL LETTER U WITH HOOK ABOVE
1EE8 LATIN CAPITAL LETTER U WITH HORN AND ACUTE
1EE9 LATIN SMALL LETTER U WITH HORN AND ACUTE
1EEA LATIN CAPITAL LETTER U WITH HORN AND GRAVE
1EEB LATIN SMALL LETTER U WITH HORN AND GRAVE
1EEC LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
1EED LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
1EEE LATIN CAPITAL LETTER U WITH HORN AND TILDE
1EEF LATIN SMALL LETTER U WITH HORN AND TILDE
1EF0 LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
1EF1 LATIN SMALL LETTER U WITH HORN AND DOT BELOW
1EF2 LATIN CAPITAL LETTER Y WITH GRAVE
1EF3 LATIN SMALL LETTER Y WITH GRAVE
1EF4 LATIN CAPITAL LETTER Y WITH DOT BELOW
1EF5 LATIN SMALL LETTER Y WITH DOT BELOW
1EF6 LATIN CAPITAL LETTER Y WITH HOOK ABOVE
1EF7 LATIN SMALL LETTER Y WITH HOOK ABOVE
1EF8 LATIN CAPITAL LETTER Y WITH TILDE
1EF9 LATIN SMALL LETTER Y WITH TILDE
1F00 GREEK SMALL LETTER ALPHA WITH PSILI
1F01 GREEK SMALL LETTER ALPHA WITH DASIA
1F02 GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA
1F03 GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA
1F04 GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA
1F05 GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA
1F06 GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI
1F07 GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI
1F08 GREEK CAPITAL LETTER ALPHA WITH PSILI
1F09 GREEK CAPITAL LETTER ALPHA WITH DASIA
1F0A GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA
1F0B GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA
1F0C GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA
1F0D GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA
1F0E GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI
1F0F GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
1F10 GREEK SMALL LETTER EPSILON WITH PSILI
1F11 GREEK SMALL LETTER EPSILON WITH DASIA
1F12 GREEK SMALL LETTER EPSILON WITH PSILI AND VARIA
1F13 GREEK SMALL LETTER EPSILON WITH DASIA AND VARIA
1F14 GREEK SMALL LETTER EPSILON WITH PSILI AND OXIA
1F15 GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
1F18 GREEK CAPITAL LETTER EPSILON WITH PSILI
1F19 GREEK CAPITAL LETTER EPSILON WITH DASIA
1F1A GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA
1F1B GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA
1F1C GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA
1F1D GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
1F20 GREEK SMALL LETTER ETA WITH PSILI
1F21 GREEK SMALL LETTER ETA WITH DASIA
1F22 GREEK SMALL LETTER ETA WITH PSILI AND VARIA
1F23 GREEK SMALL LETTER ETA WITH DASIA AND VARIA
1F24 GREEK SMALL LETTER ETA WITH PSILI AND OXIA
1F25 GREEK SMALL LETTER ETA WITH DASIA AND OXIA
1F26 GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI
1F27 GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI
1F28 GREEK CAPITAL LETTER ETA WITH PSILI
1F29 GREEK CAPITAL LETTER ETA WITH DASIA
1F2A GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA
1F2B GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA
1F2C GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA
1F2D GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA
1F2E GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI
1F2F GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
1F30 GREEK SMALL LETTER IOTA WITH PSILI
1F31 GREEK SMALL LETTER IOTA WITH DASIA
1F32 GREEK SMALL LETTER IOTA WITH PSILI AND VARIA
1F33 GREEK SMALL LETTER IOTA WITH DASIA AND VARIA
1F34 GREEK SMALL LETTER IOTA WITH PSILI AND OXIA
1F35 GREEK SMALL LETTER IOTA WITH DASIA AND OXIA
1F36 GREEK SMALL LETTER IOTA WITH PSILI AND PERISPOMENI
1F37 GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI
1F38 GREEK CAPITAL LETTER IOTA WITH PSILI
1F39 GREEK CAPITAL LETTER IOTA WITH DASIA
1F3A GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA
1F3B GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA
1F3C GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA
1F3D GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA
1F3E GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI
1F3F GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
1F40 GREEK SMALL LETTER OMICRON WITH PSILI
1F41 GREEK SMALL LETTER OMICRON WITH DASIA
1F42 GREEK SMALL LETTER OMICRON WITH PSILI AND VARIA
1F43 GREEK SMALL LETTER OMICRON WITH DASIA AND VARIA
1F44 GREEK SMALL LETTER OMICRON WITH PSILI AND OXIA
1F45 GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
1F48 GREEK CAPITAL LETTER OMICRON WITH PSILI
1F49 GREEK CAPITAL LETTER OMICRON WITH DASIA
1F4A GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA
1F4B GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA
1F4C GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA
1F4D GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
1F50 GREEK SMALL LETTER UPSILON WITH PSILI
1F51 GREEK SMALL LETTER UPSILON WITH DASIA
1F52 GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA
1F53 GREEK SMALL LETTER UPSILON WITH DASIA AND VARIA
1F54 GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA
1F55 GREEK SMALL LETTER UPSILON WITH DASIA AND OXIA
1F56 GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI
1F57 GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
1F59 GREEK CAPITAL LETTER UPSILON WITH DASIA
1F5B GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
1F5D GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
1F5F GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
1F60 GREEK SMALL LETTER OMEGA WITH PSILI
1F61 GREEK SMALL LETTER OMEGA WITH DASIA
1F62 GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA
1F63 GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA
1F64 GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA
1F65 GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA
1F66 GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI
1F67 GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI
1F68 GREEK CAPITAL LETTER OMEGA WITH PSILI
1F69 GREEK CAPITAL LETTER OMEGA WITH DASIA
1F6A GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA
1F6B GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA
1F6C GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA
1F6D GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA
1F6E GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI
1F6F GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
1F70 GREEK SMALL LETTER ALPHA WITH VARIA
1F71 GREEK SMALL LETTER ALPHA WITH OXIA
1F72 GREEK SMALL LETTER EPSILON WITH VARIA
1F73 GREEK SMALL LETTER EPSILON WITH OXIA
1F74 GREEK SMALL LETTER ETA WITH VARIA
1F75 GREEK SMALL LETTER ETA WITH OXIA
1F76 GREEK SMALL LETTER IOTA WITH VARIA
1F77 GREEK SMALL LETTER IOTA WITH OXIA
1F78 GREEK SMALL LETTER OMICRON WITH VARIA
1F79 GREEK SMALL LETTER OMICRON WITH OXIA
1F7A GREEK SMALL LETTER UPSILON WITH VARIA
1F7B GREEK SMALL LETTER UPSILON WITH OXIA
1F7C GREEK SMALL LETTER OMEGA WITH VARIA
1F7D GREEK SMALL LETTER OMEGA WITH OXIA
1F80 GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI
1F81 GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI
1F82 GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI
1F83 GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI
1F84 GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI
1F85 GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI
1F86 GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
1F87 GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
1F88 GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI
1F89 GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI
1F8A GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI
1F8B GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI
1F8C GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI
1F8D GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI
1F8E GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
1F8F GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
1F90 GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI
1F91 GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI
1F92 GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI
1F93 GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI
1F94 GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI
1F95 GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI
1F96 GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
1F97 GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
1F98 GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI
1F99 GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI
1F9A GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI
1F9B GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI
1F9C GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI
1F9D GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI
1F9E GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
1F9F GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
1FA0 GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI
1FA1 GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI
1FA2 GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI
1FA3 GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI
1FA4 GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI
1FA5 GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI
1FA6 GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
1FA7 GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
1FA8 GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI
1FA9 GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI
1FAA GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI
1FAB GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI
1FAC GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI
1FAD GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI
1FAE GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
1FAF GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
1FB0 GREEK SMALL LETTER ALPHA WITH VRACHY
1FB1 GREEK SMALL LETTER ALPHA WITH MACRON
1FB2 GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI
1FB3 GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI
1FB4 GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
1FB6 GREEK SMALL LETTER ALPHA WITH PERISPOMENI
1FB7 GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
1FB8 GREEK CAPITAL LETTER ALPHA WITH VRACHY
1FB9 GREEK CAPITAL LETTER ALPHA WITH MACRON
1FBA GREEK CAPITAL LETTER ALPHA WITH VARIA
1FBB GREEK CAPITAL LETTER ALPHA WITH OXIA
1FBC GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
1FBD GREEK KORONIS
1FBE GREEK PROSGEGRAMMENI
1FBF GREEK PSILI
1FC0 GREEK PERISPOMENI
1FC1 GREEK DIALYTIKA AND PERISPOMENI
1FC2 GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI
1FC3 GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI
1FC4 GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
1FC6 GREEK SMALL LETTER ETA WITH PERISPOMENI
1FC7 GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
1FC8 GREEK CAPITAL LETTER EPSILON WITH VARIA
1FC9 GREEK CAPITAL LETTER EPSILON WITH OXIA
1FCA GREEK CAPITAL LETTER ETA WITH VARIA
1FCB GREEK CAPITAL LETTER ETA WITH OXIA
1FCC GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
1FCD GREEK PSILI AND VARIA
1FCE GREEK PSILI AND OXIA
1FCF GREEK PSILI AND PERISPOMENI
1FD0 GREEK SMALL LETTER IOTA WITH VRACHY
1FD1 GREEK SMALL LETTER IOTA WITH MACRON
1FD2 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA
1FD3 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
1FD6 GREEK SMALL LETTER IOTA WITH PERISPOMENI
1FD7 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
1FD8 GREEK CAPITAL LETTER IOTA WITH VRACHY
1FD9 GREEK CAPITAL LETTER IOTA WITH MACRON
1FDA GREEK CAPITAL LETTER IOTA WITH VARIA
1FDB GREEK CAPITAL LETTER IOTA WITH OXIA
1FDD GREEK DASIA AND VARIA
1FDE GREEK DASIA AND OXIA
1FDF GREEK DASIA AND PERISPOMENI
1FE0 GREEK SMALL LETTER UPSILON WITH VRACHY
1FE1 GREEK SMALL LETTER UPSILON WITH MACRON
1FE2 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA
1FE3 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA
1FE4 GREEK SMALL LETTER RHO WITH PSILI
1FE5 GREEK SMALL LETTER RHO WITH DASIA
1FE6 GREEK SMALL LETTER UPSILON WITH PERISPOMENI
1FE7 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
1FE8 GREEK CAPITAL LETTER UPSILON WITH VRACHY
1FE9 GREEK CAPITAL LETTER UPSILON WITH MACRON
1FEA GREEK CAPITAL LETTER UPSILON WITH VARIA
1FEB GREEK CAPITAL LETTER UPSILON WITH OXIA
1FEC GREEK CAPITAL LETTER RHO WITH DASIA
1FED GREEK DIALYTIKA AND VARIA
1FEE GREEK DIALYTIKA AND OXIA
1FEF GREEK VARIA
1FF2 GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI
1FF3 GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI
1FF4 GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
1FF6 GREEK SMALL LETTER OMEGA WITH PERISPOMENI
1FF7 GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
1FF8 GREEK CAPITAL LETTER OMICRON WITH VARIA
1FF9 GREEK CAPITAL LETTER OMICRON WITH OXIA
1FFA GREEK CAPITAL LETTER OMEGA WITH VARIA
1FFB GREEK CAPITAL LETTER OMEGA WITH OXIA
1FFC GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
1FFD GREEK OXIA
1FFE GREEK DASIA
2000 EN QUAD
2001 EM QUAD
2002 EN SPACE
2003 EM SPACE
2004 THREE-PER-EM SPACE
2005 FOUR-PER-EM SPACE
2006 SIX-PER-EM SPACE
2007 FIGURE SPACE
2008 PUNCTUATION SPACE
2009 THIN SPACE
200A HAIR SPACE
200B ZERO WIDTH SPACE
200C ZERO WIDTH NON-JOINER
200D ZERO WIDTH JOINER
200E LEFT-TO-RIGHT MARK
200F RIGHT-TO-LEFT MARK
2010 HYPHEN
2011 NON-BREAKING HYPHEN
2012 FIGURE DASH
2013 EN DASH
2014 EM DASH
2015 HORIZONTAL BAR
2016 DOUBLE VERTICAL LINE
2017 DOUBLE LOW LINE
2018 LEFT SINGLE QUOTATION MARK
2019 RIGHT SINGLE QUOTATION MARK
201A SINGLE LOW-9 QUOTATION MARK
201B SINGLE HIGH-REVERSED-9 QUOTATION MARK
201C LEFT DOUBLE QUOTATION MARK
201D RIGHT DOUBLE QUOTATION MARK
201E DOUBLE LOW-9 QUOTATION MARK
201F DOUBLE HIGH-REVERSED-9 QUOTATION MARK
2020 DAGGER
2021 DOUBLE DAGGER
2022 BULLET
2023 TRIANGULAR BULLET
2024 ONE DOT LEADER
2025 TWO DOT LEADER
2026 HORIZONTAL ELLIPSIS
2027 HYPHENATION POINT
2028 LINE SEPARATOR
2029 PARAGRAPH SEPARATOR
202A LEFT-TO-RIGHT EMBEDDING
202B RIGHT-TO-LEFT EMBEDDING
202C POP DIRECTIONAL FORMATTING
202D LEFT-TO-RIGHT OVERRIDE
202E RIGHT-TO-LEFT OVERRIDE
2030 PER MILLE SIGN
2031 PER TEN THOUSAND SIGN
2032 PRIME
2033 DOUBLE PRIME
2034 TRIPLE PRIME
2035 REVERSED PRIME
2036 REVERSED DOUBLE PRIME
2037 REVERSED TRIPLE PRIME
2038 CARET
2039 SINGLE LEFT-POINTING ANGLE QUOTATION MARK
203A SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
203B REFERENCE MARK
203C DOUBLE EXCLAMATION MARK
203D INTERROBANG
203E OVERLINE
203F UNDERTIE
2040 CHARACTER TIE
2041 CARET INSERTION POINT
2042 ASTERISM
2043 HYPHEN BULLET
2044 FRACTION SLASH
2045 LEFT SQUARE BRACKET WITH QUILL
2046 RIGHT SQUARE BRACKET WITH QUILL
206A INHIBIT SYMMETRIC SWAPPING
206B ACTIVATE SYMMETRIC SWAPPING
206C INHIBIT ARABIC FORM SHAPING
206D ACTIVATE ARABIC FORM SHAPING
206E NATIONAL DIGIT SHAPES
206F NOMINAL DIGIT SHAPES
2070 SUPERSCRIPT ZERO
2074 SUPERSCRIPT FOUR
2075 SUPERSCRIPT FIVE
2076 SUPERSCRIPT SIX
2077 SUPERSCRIPT SEVEN
2078 SUPERSCRIPT EIGHT
2079 SUPERSCRIPT NINE
207A SUPERSCRIPT PLUS SIGN
207B SUPERSCRIPT MINUS
207C SUPERSCRIPT EQUALS SIGN
207D SUPERSCRIPT LEFT PARENTHESIS
207E SUPERSCRIPT RIGHT PARENTHESIS
207F SUPERSCRIPT LATIN SMALL LETTER N
2080 SUBSCRIPT ZERO
2081 SUBSCRIPT ONE
2082 SUBSCRIPT TWO
2083 SUBSCRIPT THREE
2084 SUBSCRIPT FOUR
2085 SUBSCRIPT FIVE
2086 SUBSCRIPT SIX
2087 SUBSCRIPT SEVEN
2088 SUBSCRIPT EIGHT
2089 SUBSCRIPT NINE
208A SUBSCRIPT PLUS SIGN
208B SUBSCRIPT MINUS
208C SUBSCRIPT EQUALS SIGN
208D SUBSCRIPT LEFT PARENTHESIS
208E SUBSCRIPT RIGHT PARENTHESIS
20A0 EURO-CURRENCY SIGN
20A1 COLON SIGN
20A2 CRUZEIRO SIGN
20A3 FRENCH FRANC SIGN
20A4 LIRA SIGN
20A5 MILL SIGN
20A6 NAIRA SIGN
20A7 PESETA SIGN
20A8 RUPEE SIGN
20A9 WON SIGN
20AA NEW SHEQEL SIGN
20AB DONG SIGN
20D0 COMBINING LEFT HARPOON ABOVE
20D1 COMBINING RIGHT HARPOON ABOVE
20D2 COMBINING LONG VERTICAL LINE OVERLAY
20D3 COMBINING SHORT VERTICAL LINE OVERLAY
20D4 COMBINING ANTICLOCKWISE ARROW ABOVE
20D5 COMBINING CLOCKWISE ARROW ABOVE
20D6 COMBINING LEFT ARROW ABOVE
20D7 COMBINING RIGHT ARROW ABOVE
20D8 COMBINING RING OVERLAY
20D9 COMBINING CLOCKWISE RING OVERLAY
20DA COMBINING ANTICLOCKWISE RING OVERLAY
20DB COMBINING THREE DOTS ABOVE
20DC COMBINING FOUR DOTS ABOVE
20DD COMBINING ENCLOSING CIRCLE
20DE COMBINING ENCLOSING SQUARE
20DF COMBINING ENCLOSING DIAMOND
20E0 COMBINING ENCLOSING CIRCLE BACKSLASH
20E1 COMBINING LEFT RIGHT ARROW ABOVE
2100 ACCOUNT OF
2101 ADDRESSED TO THE SUBJECT
2102 DOUBLE-STRUCK CAPITAL C
2103 DEGREE CELSIUS
2104 CENTRE LINE SYMBOL
2105 CARE OF
2106 CADA UNA
2107 EULER CONSTANT
2108 SCRUPLE
2109 DEGREE FAHRENHEIT
210A SCRIPT SMALL G
210B SCRIPT CAPITAL H
210C BLACK-LETTER CAPITAL H
210D DOUBLE-STRUCK CAPITAL H
210E PLANCK CONSTANT
210F PLANCK CONSTANT OVER TWO PI
2110 SCRIPT CAPITAL I
2111 BLACK-LETTER CAPITAL I
2112 SCRIPT CAPITAL L
2113 SCRIPT SMALL L
2114 L B BAR SYMBOL
2115 DOUBLE-STRUCK CAPITAL N
2116 NUMERO SIGN
2117 SOUND RECORDING COPYRIGHT
2118 SCRIPT CAPITAL P
2119 DOUBLE-STRUCK CAPITAL P
211A DOUBLE-STRUCK CAPITAL Q
211B SCRIPT CAPITAL R
211C BLACK-LETTER CAPITAL R
211D DOUBLE-STRUCK CAPITAL R
211E PRESCRIPTION TAKE
211F RESPONSE
2120 SERVICE MARK
2121 TELEPHONE SIGN
2122 TRADE MARK SIGN
2123 VERSICLE
2124 DOUBLE-STRUCK CAPITAL Z
2125 OUNCE SIGN
2126 OHM SIGN
2127 INVERTED OHM SIGN
2128 BLACK-LETTER CAPITAL Z
2129 TURNED GREEK SMALL LETTER IOTA
212A KELVIN SIGN
212B ANGSTROM SIGN
212C SCRIPT CAPITAL B
212D BLACK-LETTER CAPITAL C
212E ESTIMATED SYMBOL
212F SCRIPT SMALL E
2130 SCRIPT CAPITAL E
2131 SCRIPT CAPITAL F
2132 TURNED CAPITAL F
2133 SCRIPT CAPITAL M
2134 SCRIPT SMALL O
2135 ALEF SYMBOL
2136 BET SYMBOL
2137 GIMEL SYMBOL
2138 DALET SYMBOL
2153 VULGAR FRACTION ONE THIRD
2154 VULGAR FRACTION TWO THIRDS
2155 VULGAR FRACTION ONE FIFTH
2156 VULGAR FRACTION TWO FIFTHS
2157 VULGAR FRACTION THREE FIFTHS
2158 VULGAR FRACTION FOUR FIFTHS
2159 VULGAR FRACTION ONE SIXTH
215A VULGAR FRACTION FIVE SIXTHS
215B VULGAR FRACTION ONE EIGHTH
215C VULGAR FRACTION THREE EIGHTHS
215D VULGAR FRACTION FIVE EIGHTHS
215E VULGAR FRACTION SEVEN EIGHTHS
215F FRACTION NUMERATOR ONE
2160 ROMAN NUMERAL ONE
2161 ROMAN NUMERAL TWO
2162 ROMAN NUMERAL THREE
2163 ROMAN NUMERAL FOUR
2164 ROMAN NUMERAL FIVE
2165 ROMAN NUMERAL SIX
2166 ROMAN NUMERAL SEVEN
2167 ROMAN NUMERAL EIGHT
2168 ROMAN NUMERAL NINE
2169 ROMAN NUMERAL TEN
216A ROMAN NUMERAL ELEVEN
216B ROMAN NUMERAL TWELVE
216C ROMAN NUMERAL FIFTY
216D ROMAN NUMERAL ONE HUNDRED
216E ROMAN NUMERAL FIVE HUNDRED
216F ROMAN NUMERAL ONE THOUSAND
2170 SMALL ROMAN NUMERAL ONE
2171 SMALL ROMAN NUMERAL TWO
2172 SMALL ROMAN NUMERAL THREE
2173 SMALL ROMAN NUMERAL FOUR
2174 SMALL ROMAN NUMERAL FIVE
2175 SMALL ROMAN NUMERAL SIX
2176 SMALL ROMAN NUMERAL SEVEN
2177 SMALL ROMAN NUMERAL EIGHT
2178 SMALL ROMAN NUMERAL NINE
2179 SMALL ROMAN NUMERAL TEN
217A SMALL ROMAN NUMERAL ELEVEN
217B SMALL ROMAN NUMERAL TWELVE
217C SMALL ROMAN NUMERAL FIFTY
217D SMALL ROMAN NUMERAL ONE HUNDRED
217E SMALL ROMAN NUMERAL FIVE HUNDRED
217F SMALL ROMAN NUMERAL ONE THOUSAND
2180 ROMAN NUMERAL ONE THOUSAND C D
2181 ROMAN NUMERAL FIVE THOUSAND
2182 ROMAN NUMERAL TEN THOUSAND
2190 LEFTWARDS ARROW
2191 UPWARDS ARROW
2192 RIGHTWARDS ARROW
2193 DOWNWARDS ARROW
2194 LEFT RIGHT ARROW
2195 UP DOWN ARROW
2196 NORTH WEST ARROW
2197 NORTH EAST ARROW
2198 SOUTH EAST ARROW
2199 SOUTH WEST ARROW
219A LEFTWARDS ARROW WITH STROKE
219B RIGHTWARDS ARROW WITH STROKE
219C LEFTWARDS WAVE ARROW
219D RIGHTWARDS WAVE ARROW
219E LEFTWARDS TWO HEADED ARROW
219F UPWARDS TWO HEADED ARROW
21A0 RIGHTWARDS TWO HEADED ARROW
21A1 DOWNWARDS TWO HEADED ARROW
21A2 LEFTWARDS ARROW WITH TAIL
21A3 RIGHTWARDS ARROW WITH TAIL
21A4 LEFTWARDS ARROW FROM BAR
21A5 UPWARDS ARROW FROM BAR
21A6 RIGHTWARDS ARROW FROM BAR
21A7 DOWNWARDS ARROW FROM BAR
21A8 UP DOWN ARROW WITH BASE
21A9 LEFTWARDS ARROW WITH HOOK
21AA RIGHTWARDS ARROW WITH HOOK
21AB LEFTWARDS ARROW WITH LOOP
21AC RIGHTWARDS ARROW WITH LOOP
21AD LEFT RIGHT WAVE ARROW
21AE LEFT RIGHT ARROW WITH STROKE
21AF DOWNWARDS ZIGZAG ARROW
21B0 UPWARDS ARROW WITH TIP LEFTWARDS
21B1 UPWARDS ARROW WITH TIP RIGHTWARDS
21B2 DOWNWARDS ARROW WITH TIP LEFTWARDS
21B3 DOWNWARDS ARROW WITH TIP RIGHTWARDS
21B4 RIGHTWARDS ARROW WITH CORNER DOWNWARDS
21B5 DOWNWARDS ARROW WITH CORNER LEFTWARDS
21B6 ANTICLOCKWISE TOP SEMICIRCLE ARROW
21B7 CLOCKWISE TOP SEMICIRCLE ARROW
21B8 NORTH WEST ARROW TO LONG BAR
21B9 LEFTWARDS ARROW TO BAR OVER RIGHTWARDS ARROW TO BAR
21BA ANTICLOCKWISE OPEN CIRCLE ARROW
21BB CLOCKWISE OPEN CIRCLE ARROW
21BC LEFTWARDS HARPOON WITH BARB UPWARDS
21BD LEFTWARDS HARPOON WITH BARB DOWNWARDS
21BE UPWARDS HARPOON WITH BARB RIGHTWARDS
21BF UPWARDS HARPOON WITH BARB LEFTWARDS
21C0 RIGHTWARDS HARPOON WITH BARB UPWARDS
21C1 RIGHTWARDS HARPOON WITH BARB DOWNWARDS
21C2 DOWNWARDS HARPOON WITH BARB RIGHTWARDS
21C3 DOWNWARDS HARPOON WITH BARB LEFTWARDS
21C4 RIGHTWARDS ARROW OVER LEFTWARDS ARROW
21C5 UPWARDS ARROW LEFTWARDS OF DOWNWARDS ARROW
21C6 LEFTWARDS ARROW OVER RIGHTWARDS ARROW
21C7 LEFTWARDS PAIRED ARROWS
21C8 UPWARDS PAIRED ARROWS
21C9 RIGHTWARDS PAIRED ARROWS
21CA DOWNWARDS PAIRED ARROWS
21CB LEFTWARDS HARPOON OVER RIGHTWARDS HARPOON
21CC RIGHTWARDS HARPOON OVER LEFTWARDS HARPOON
21CD LEFTWARDS DOUBLE ARROW WITH STROKE
21CE LEFT RIGHT DOUBLE ARROW WITH STROKE
21CF RIGHTWARDS DOUBLE ARROW WITH STROKE
21D0 LEFTWARDS DOUBLE ARROW
21D1 UPWARDS DOUBLE ARROW
21D2 RIGHTWARDS DOUBLE ARROW
21D3 DOWNWARDS DOUBLE ARROW
21D4 LEFT RIGHT DOUBLE ARROW
21D5 UP DOWN DOUBLE ARROW
21D6 NORTH WEST DOUBLE ARROW
21D7 NORTH EAST DOUBLE ARROW
21D8 SOUTH EAST DOUBLE ARROW
21D9 SOUTH WEST DOUBLE ARROW
21DA LEFTWARDS TRIPLE ARROW
21DB RIGHTWARDS TRIPLE ARROW
21DC LEFTWARDS SQUIGGLE ARROW
21DD RIGHTWARDS SQUIGGLE ARROW
21DE UPWARDS ARROW WITH DOUBLE STROKE
21DF DOWNWARDS ARROW WITH DOUBLE STROKE
21E0 LEFTWARDS DASHED ARROW
21E1 UPWARDS DASHED ARROW
21E2 RIGHTWARDS DASHED ARROW
21E3 DOWNWARDS DASHED ARROW
21E4 LEFTWARDS ARROW TO BAR
21E5 RIGHTWARDS ARROW TO BAR
21E6 LEFTWARDS WHITE ARROW
21E7 UPWARDS WHITE ARROW
21E8 RIGHTWARDS WHITE ARROW
21E9 DOWNWARDS WHITE ARROW
21EA UPWARDS WHITE ARROW FROM BAR
2200 FOR ALL
2201 COMPLEMENT
2202 PARTIAL DIFFERENTIAL
2203 THERE EXISTS
2204 THERE DOES NOT EXIST
2205 EMPTY SET
2206 INCREMENT
2207 NABLA
2208 ELEMENT OF
2209 NOT AN ELEMENT OF
220A SMALL ELEMENT OF
220B CONTAINS AS MEMBER
220C DOES NOT CONTAIN AS MEMBER
220D SMALL CONTAINS AS MEMBER
220E END OF PROOF
220F N-ARY PRODUCT
2210 N-ARY COPRODUCT
2211 N-ARY SUMMATION
2212 MINUS SIGN
2213 MINUS-OR-PLUS SIGN
2214 DOT PLUS
2215 DIVISION SLASH
2216 SET MINUS
2217 ASTERISK OPERATOR
2218 RING OPERATOR
2219 BULLET OPERATOR
221A SQUARE ROOT
221B CUBE ROOT
221C FOURTH ROOT
221D PROPORTIONAL TO
221E INFINITY
221F RIGHT ANGLE
2220 ANGLE
2221 MEASURED ANGLE
2222 SPHERICAL ANGLE
2223 DIVIDES
2224 DOES NOT DIVIDE
2225 PARALLEL TO
2226 NOT PARALLEL TO
2227 LOGICAL AND
2228 LOGICAL OR
2229 INTERSECTION
222A UNION
222B INTEGRAL
222C DOUBLE INTEGRAL
222D TRIPLE INTEGRAL
222E CONTOUR INTEGRAL
222F SURFACE INTEGRAL
2230 VOLUME INTEGRAL
2231 CLOCKWISE INTEGRAL
2232 CLOCKWISE CONTOUR INTEGRAL
2233 ANTICLOCKWISE CONTOUR INTEGRAL
2234 THEREFORE
2235 BECAUSE
2236 RATIO
2237 PROPORTION
2238 DOT MINUS
2239 EXCESS
223A GEOMETRIC PROPORTION
223B HOMOTHETIC
223C TILDE OPERATOR
223D REVERSED TILDE
223E INVERTED LAZY S
223F SINE WAVE
2240 WREATH PRODUCT
2241 NOT TILDE
2242 MINUS TILDE
2243 ASYMPTOTICALLY EQUAL TO
2244 NOT ASYMPTOTICALLY EQUAL TO
2245 APPROXIMATELY EQUAL TO
2246 APPROXIMATELY BUT NOT ACTUALLY EQUAL TO
2247 NEITHER APPROXIMATELY NOR ACTUALLY EQUAL TO
2248 ALMOST EQUAL TO
2249 NOT ALMOST EQUAL TO
224A ALMOST EQUAL OR EQUAL TO
224B TRIPLE TILDE
224C ALL EQUAL TO
224D EQUIVALENT TO
224E GEOMETRICALLY EQUIVALENT TO
224F DIFFERENCE BETWEEN
2250 APPROACHES THE LIMIT
2251 GEOMETRICALLY EQUAL TO
2252 APPROXIMATELY EQUAL TO OR THE IMAGE OF
2253 IMAGE OF OR APPROXIMATELY EQUAL TO
2254 COLON EQUALS
2255 EQUALS COLON
2256 RING IN EQUAL TO
2257 RING EQUAL TO
2258 CORRESPONDS TO
2259 ESTIMATES
225A EQUIANGULAR TO
225B STAR EQUALS
225C DELTA EQUAL TO
225D EQUAL TO BY DEFINITION
225E MEASURED BY
225F QUESTIONED EQUAL TO
2260 NOT EQUAL TO
2261 IDENTICAL TO
2262 NOT IDENTICAL TO
2263 STRICTLY EQUIVALENT TO
2264 LESS-THAN OR EQUAL TO
2265 GREATER-THAN OR EQUAL TO
2266 LESS-THAN OVER EQUAL TO
2267 GREATER-THAN OVER EQUAL TO
2268 LESS-THAN BUT NOT EQUAL TO
2269 GREATER-THAN BUT NOT EQUAL TO
226A MUCH LESS-THAN
226B MUCH GREATER-THAN
226C BETWEEN
226D NOT EQUIVALENT TO
226E NOT LESS-THAN
226F NOT GREATER-THAN
2270 NEITHER LESS-THAN NOR EQUAL TO
2271 NEITHER GREATER-THAN NOR EQUAL TO
2272 LESS-THAN OR EQUIVALENT TO
2273 GREATER-THAN OR EQUIVALENT TO
2274 NEITHER LESS-THAN NOR EQUIVALENT TO
2275 NEITHER GREATER-THAN NOR EQUIVALENT TO
2276 LESS-THAN OR GREATER-THAN
2277 GREATER-THAN OR LESS-THAN
2278 NEITHER LESS-THAN NOR GREATER-THAN
2279 NEITHER GREATER-THAN NOR LESS-THAN
227A PRECEDES
227B SUCCEEDS
227C PRECEDES OR EQUAL TO
227D SUCCEEDS OR EQUAL TO
227E PRECEDES OR EQUIVALENT TO
227F SUCCEEDS OR EQUIVALENT TO
2280 DOES NOT PRECEDE
2281 DOES NOT SUCCEED
2282 SUBSET OF
2283 SUPERSET OF
2284 NOT A SUBSET OF
2285 NOT A SUPERSET OF
2286 SUBSET OF OR EQUAL TO
2287 SUPERSET OF OR EQUAL TO
2288 NEITHER A SUBSET OF NOR EQUAL TO
2289 NEITHER A SUPERSET OF NOR EQUAL TO
228A SUBSET OF WITH NOT EQUAL TO
228B SUPERSET OF WITH NOT EQUAL TO
228C MULTISET
228D MULTISET MULTIPLICATION
228E MULTISET UNION
228F SQUARE IMAGE OF
2290 SQUARE ORIGINAL OF
2291 SQUARE IMAGE OF OR EQUAL TO
2292 SQUARE ORIGINAL OF OR EQUAL TO
2293 SQUARE CAP
2294 SQUARE CUP
2295 CIRCLED PLUS
2296 CIRCLED MINUS
2297 CIRCLED TIMES
2298 CIRCLED DIVISION SLASH
2299 CIRCLED DOT OPERATOR
229A CIRCLED RING OPERATOR
229B CIRCLED ASTERISK OPERATOR
229C CIRCLED EQUALS
229D CIRCLED DASH
229E SQUARED PLUS
229F SQUARED MINUS
22A0 SQUARED TIMES
22A1 SQUARED DOT OPERATOR
22A2 RIGHT TACK
22A3 LEFT TACK
22A4 DOWN TACK
22A5 UP TACK
22A6 ASSERTION
22A7 MODELS
22A8 TRUE
22A9 FORCES
22AA TRIPLE VERTICAL BAR RIGHT TURNSTILE
22AB DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE
22AC DOES NOT PROVE
22AD NOT TRUE
22AE DOES NOT FORCE
22AF NEGATED DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE
22B0 PRECEDES UNDER RELATION
22B1 SUCCEEDS UNDER RELATION
22B2 NORMAL SUBGROUP OF
22B3 CONTAINS AS NORMAL SUBGROUP
22B4 NORMAL SUBGROUP OF OR EQUAL TO
22B5 CONTAINS AS NORMAL SUBGROUP OR EQUAL TO
22B6 ORIGINAL OF
22B7 IMAGE OF
22B8 MULTIMAP
22B9 HERMITIAN CONJUGATE MATRIX
22BA INTERCALATE
22BB XOR
22BC NAND
22BD NOR
22BE RIGHT ANGLE WITH ARC
22BF RIGHT TRIANGLE
22C0 N-ARY LOGICAL AND
22C1 N-ARY LOGICAL OR
22C2 N-ARY INTERSECTION
22C3 N-ARY UNION
22C4 DIAMOND OPERATOR
22C5 DOT OPERATOR
22C6 STAR OPERATOR
22C7 DIVISION TIMES
22C8 BOWTIE
22C9 LEFT NORMAL FACTOR SEMIDIRECT PRODUCT
22CA RIGHT NORMAL FACTOR SEMIDIRECT PRODUCT
22CB LEFT SEMIDIRECT PRODUCT
22CC RIGHT SEMIDIRECT PRODUCT
22CD REVERSED TILDE EQUALS
22CE CURLY LOGICAL OR
22CF CURLY LOGICAL AND
22D0 DOUBLE SUBSET
22D1 DOUBLE SUPERSET
22D2 DOUBLE INTERSECTION
22D3 DOUBLE UNION
22D4 PITCHFORK
22D5 EQUAL AND PARALLEL TO
22D6 LESS-THAN WITH DOT
22D7 GREATER-THAN WITH DOT
22D8 VERY MUCH LESS-THAN
22D9 VERY MUCH GREATER-THAN
22DA LESS-THAN EQUAL TO OR GREATER-THAN
22DB GREATER-THAN EQUAL TO OR LESS-THAN
22DC EQUAL TO OR LESS-THAN
22DD EQUAL TO OR GREATER-THAN
22DE EQUAL TO OR PRECEDES
22DF EQUAL TO OR SUCCEEDS
22E0 DOES NOT PRECEDE OR EQUAL
22E1 DOES NOT SUCCEED OR EQUAL
22E2 NOT SQUARE IMAGE OF OR EQUAL TO
22E3 NOT SQUARE ORIGINAL OF OR EQUAL TO
22E4 SQUARE IMAGE OF OR NOT EQUAL TO
22E5 SQUARE ORIGINAL OF OR NOT EQUAL TO
22E6 LESS-THAN BUT NOT EQUIVALENT TO
22E7 GREATER-THAN BUT NOT EQUIVALENT TO
22E8 PRECEDES BUT NOT EQUIVALENT TO
22E9 SUCCEEDS BUT NOT EQUIVALENT TO
22EA NOT NORMAL SUBGROUP OF
22EB DOES NOT CONTAIN AS NORMAL SUBGROUP
22EC NOT NORMAL SUBGROUP OF OR EQUAL TO
22ED DOES NOT CONTAIN AS NORMAL SUBGROUP OR EQUAL
22EE VERTICAL ELLIPSIS
22EF MIDLINE HORIZONTAL ELLIPSIS
22F0 UP RIGHT DIAGONAL ELLIPSIS
22F1 DOWN RIGHT DIAGONAL ELLIPSIS
2300 DIAMETER SIGN
2302 HOUSE
2303 UP ARROWHEAD
2304 DOWN ARROWHEAD
2305 PROJECTIVE
2306 PERSPECTIVE
2307 WAVY LINE
2308 LEFT CEILING
2309 RIGHT CEILING
230A LEFT FLOOR
230B RIGHT FLOOR
230C BOTTOM RIGHT CROP
230D BOTTOM LEFT CROP
230E TOP RIGHT CROP
230F TOP LEFT CROP
2310 REVERSED NOT SIGN
2311 SQUARE LOZENGE
2312 ARC
2313 SEGMENT
2314 SECTOR
2315 TELEPHONE RECORDER
2316 POSITION INDICATOR
2317 VIEWDATA SQUARE
2318 PLACE OF INTEREST SIGN
2319 TURNED NOT SIGN
231A WATCH
231B HOURGLASS
231C TOP LEFT CORNER
231D TOP RIGHT CORNER
231E BOTTOM LEFT CORNER
231F BOTTOM RIGHT CORNER
2320 TOP HALF INTEGRAL
2321 BOTTOM HALF INTEGRAL
2322 FROWN
2323 SMILE
2324 UP ARROWHEAD BETWEEN TWO HORIZONTAL BARS
2325 OPTION KEY
2326 ERASE TO THE RIGHT
2327 X IN A RECTANGLE BOX
2328 KEYBOARD
2329 LEFT-POINTING ANGLE BRACKET
232A RIGHT-POINTING ANGLE BRACKET
232B ERASE TO THE LEFT
232C BENZENE RING
232D CYLINDRICITY
232E ALL AROUND-PROFILE
232F SYMMETRY
2330 TOTAL RUNOUT
2331 DIMENSION ORIGIN
2332 CONICAL TAPER
2333 SLOPE
2334 COUNTERBORE
2335 COUNTERSINK
2336 APL FUNCTIONAL SYMBOL I-BEAM
2337 APL FUNCTIONAL SYMBOL SQUISH QUAD
2338 APL FUNCTIONAL SYMBOL QUAD EQUAL
2339 APL FUNCTIONAL SYMBOL QUAD DIVIDE
233A APL FUNCTIONAL SYMBOL QUAD DIAMOND
233B APL FUNCTIONAL SYMBOL QUAD JOT
233C APL FUNCTIONAL SYMBOL QUAD CIRCLE
233D APL FUNCTIONAL SYMBOL CIRCLE STILE
233E APL FUNCTIONAL SYMBOL CIRCLE JOT
233F APL FUNCTIONAL SYMBOL SLASH BAR
2340 APL FUNCTIONAL SYMBOL BACKSLASH BAR
2341 APL FUNCTIONAL SYMBOL QUAD SLASH
2342 APL FUNCTIONAL SYMBOL QUAD BACKSLASH
2343 APL FUNCTIONAL SYMBOL QUAD LESS-THAN
2344 APL FUNCTIONAL SYMBOL QUAD GREATER-THAN
2345 APL FUNCTIONAL SYMBOL LEFTWARDS VANE
2346 APL FUNCTIONAL SYMBOL RIGHTWARDS VANE
2347 APL FUNCTIONAL SYMBOL QUAD LEFTWARDS ARROW
2348 APL FUNCTIONAL SYMBOL QUAD RIGHTWARDS ARROW
2349 APL FUNCTIONAL SYMBOL CIRCLE BACKSLASH
234A APL FUNCTIONAL SYMBOL DOWN TACK UNDERBAR
234B APL FUNCTIONAL SYMBOL DELTA STILE
234C APL FUNCTIONAL SYMBOL QUAD DOWN CARET
234D APL FUNCTIONAL SYMBOL QUAD DELTA
234E APL FUNCTIONAL SYMBOL DOWN TACK JOT
234F APL FUNCTIONAL SYMBOL UPWARDS VANE
2350 APL FUNCTIONAL SYMBOL QUAD UPWARDS ARROW
2351 APL FUNCTIONAL SYMBOL UP TACK OVERBAR
2352 APL FUNCTIONAL SYMBOL DEL STILE
2353 APL FUNCTIONAL SYMBOL QUAD UP CARET
2354 APL FUNCTIONAL SYMBOL QUAD DEL
2355 APL FUNCTIONAL SYMBOL UP TACK JOT
2356 APL FUNCTIONAL SYMBOL DOWNWARDS VANE
2357 APL FUNCTIONAL SYMBOL QUAD DOWNWARDS ARROW
2358 APL FUNCTIONAL SYMBOL QUOTE UNDERBAR
2359 APL FUNCTIONAL SYMBOL DELTA UNDERBAR
235A APL FUNCTIONAL SYMBOL DIAMOND UNDERBAR
235B APL FUNCTIONAL SYMBOL JOT UNDERBAR
235C APL FUNCTIONAL SYMBOL CIRCLE UNDERBAR
235D APL FUNCTIONAL SYMBOL UP SHOE JOT
235E APL FUNCTIONAL SYMBOL QUOTE QUAD
235F APL FUNCTIONAL SYMBOL CIRCLE STAR
2360 APL FUNCTIONAL SYMBOL QUAD COLON
2361 APL FUNCTIONAL SYMBOL UP TACK DIAERESIS
2362 APL FUNCTIONAL SYMBOL DEL DIAERESIS
2363 APL FUNCTIONAL SYMBOL STAR DIAERESIS
2364 APL FUNCTIONAL SYMBOL JOT DIAERESIS
2365 APL FUNCTIONAL SYMBOL CIRCLE DIAERESIS
2366 APL FUNCTIONAL SYMBOL DOWN SHOE STILE
2367 APL FUNCTIONAL SYMBOL LEFT SHOE STILE
2368 APL FUNCTIONAL SYMBOL TILDE DIAERESIS
2369 APL FUNCTIONAL SYMBOL GREATER-THAN DIAERESIS
236A APL FUNCTIONAL SYMBOL COMMA BAR
236B APL FUNCTIONAL SYMBOL DEL TILDE
236C APL FUNCTIONAL SYMBOL ZILDE
236D APL FUNCTIONAL SYMBOL STILE TILDE
236E APL FUNCTIONAL SYMBOL SEMICOLON UNDERBAR
236F APL FUNCTIONAL SYMBOL QUAD NOT EQUAL
2370 APL FUNCTIONAL SYMBOL QUAD QUESTION
2371 APL FUNCTIONAL SYMBOL DOWN CARET TILDE
2372 APL FUNCTIONAL SYMBOL UP CARET TILDE
2373 APL FUNCTIONAL SYMBOL IOTA
2374 APL FUNCTIONAL SYMBOL RHO
2375 APL FUNCTIONAL SYMBOL OMEGA
2376 APL FUNCTIONAL SYMBOL ALPHA UNDERBAR
2377 APL FUNCTIONAL SYMBOL EPSILON UNDERBAR
2378 APL FUNCTIONAL SYMBOL IOTA UNDERBAR
2379 APL FUNCTIONAL SYMBOL OMEGA UNDERBAR
237A APL FUNCTIONAL SYMBOL ALPHA
2400 SYMBOL FOR NULL
2401 SYMBOL FOR START OF HEADING
2402 SYMBOL FOR START OF TEXT
2403 SYMBOL FOR END OF TEXT
2404 SYMBOL FOR END OF TRANSMISSION
2405 SYMBOL FOR ENQUIRY
2406 SYMBOL FOR ACKNOWLEDGE
2407 SYMBOL FOR BELL
2408 SYMBOL FOR BACKSPACE
2409 SYMBOL FOR HORIZONTAL TABULATION
240A SYMBOL FOR LINE FEED
240B SYMBOL FOR VERTICAL TABULATION
240C SYMBOL FOR FORM FEED
240D SYMBOL FOR CARRIAGE RETURN
240E SYMBOL FOR SHIFT OUT
240F SYMBOL FOR SHIFT IN
2410 SYMBOL FOR DATA LINK ESCAPE
2411 SYMBOL FOR DEVICE CONTROL ONE
2412 SYMBOL FOR DEVICE CONTROL TWO
2413 SYMBOL FOR DEVICE CONTROL THREE
2414 SYMBOL FOR DEVICE CONTROL FOUR
2415 SYMBOL FOR NEGATIVE ACKNOWLEDGE
2416 SYMBOL FOR SYNCHRONOUS IDLE
2417 SYMBOL FOR END OF TRANSMISSION BLOCK
2418 SYMBOL FOR CANCEL
2419 SYMBOL FOR END OF MEDIUM
241A SYMBOL FOR SUBSTITUTE
241B SYMBOL FOR ESCAPE
241C SYMBOL FOR FILE SEPARATOR
241D SYMBOL FOR GROUP SEPARATOR
241E SYMBOL FOR RECORD SEPARATOR
241F SYMBOL FOR UNIT SEPARATOR
2420 SYMBOL FOR SPACE
2421 SYMBOL FOR DELETE
2422 BLANK SYMBOL
2423 OPEN BOX
2424 SYMBOL FOR NEWLINE
2440 OCR HOOK
2441 OCR CHAIR
2442 OCR FORK
2443 OCR INVERTED FORK
2444 OCR BELT BUCKLE
2445 OCR BOW TIE
2446 OCR BRANCH BANK IDENTIFICATION
2447 OCR AMOUNT OF CHECK
2448 OCR DASH
2449 OCR CUSTOMER ACCOUNT NUMBER
244A OCR DOUBLE BACKSLASH
2460 CIRCLED DIGIT ONE
2461 CIRCLED DIGIT TWO
2462 CIRCLED DIGIT THREE
2463 CIRCLED DIGIT FOUR
2464 CIRCLED DIGIT FIVE
2465 CIRCLED DIGIT SIX
2466 CIRCLED DIGIT SEVEN
2467 CIRCLED DIGIT EIGHT
2468 CIRCLED DIGIT NINE
2469 CIRCLED NUMBER TEN
246A CIRCLED NUMBER ELEVEN
246B CIRCLED NUMBER TWELVE
246C CIRCLED NUMBER THIRTEEN
246D CIRCLED NUMBER FOURTEEN
246E CIRCLED NUMBER FIFTEEN
246F CIRCLED NUMBER SIXTEEN
2470 CIRCLED NUMBER SEVENTEEN
2471 CIRCLED NUMBER EIGHTEEN
2472 CIRCLED NUMBER NINETEEN
2473 CIRCLED NUMBER TWENTY
2474 PARENTHESIZED DIGIT ONE
2475 PARENTHESIZED DIGIT TWO
2476 PARENTHESIZED DIGIT THREE
2477 PARENTHESIZED DIGIT FOUR
2478 PARENTHESIZED DIGIT FIVE
2479 PARENTHESIZED DIGIT SIX
247A PARENTHESIZED DIGIT SEVEN
247B PARENTHESIZED DIGIT EIGHT
247C PARENTHESIZED DIGIT NINE
247D PARENTHESIZED NUMBER TEN
247E PARENTHESIZED NUMBER ELEVEN
247F PARENTHESIZED NUMBER TWELVE
2480 PARENTHESIZED NUMBER THIRTEEN
2481 PARENTHESIZED NUMBER FOURTEEN
2482 PARENTHESIZED NUMBER FIFTEEN
2483 PARENTHESIZED NUMBER SIXTEEN
2484 PARENTHESIZED NUMBER SEVENTEEN
2485 PARENTHESIZED NUMBER EIGHTEEN
2486 PARENTHESIZED NUMBER NINETEEN
2487 PARENTHESIZED NUMBER TWENTY
2488 DIGIT ONE FULL STOP
2489 DIGIT TWO FULL STOP
248A DIGIT THREE FULL STOP
248B DIGIT FOUR FULL STOP
248C DIGIT FIVE FULL STOP
248D DIGIT SIX FULL STOP
248E DIGIT SEVEN FULL STOP
248F DIGIT EIGHT FULL STOP
2490 DIGIT NINE FULL STOP
2491 NUMBER TEN FULL STOP
2492 NUMBER ELEVEN FULL STOP
2493 NUMBER TWELVE FULL STOP
2494 NUMBER THIRTEEN FULL STOP
2495 NUMBER FOURTEEN FULL STOP
2496 NUMBER FIFTEEN FULL STOP
2497 NUMBER SIXTEEN FULL STOP
2498 NUMBER SEVENTEEN FULL STOP
2499 NUMBER EIGHTEEN FULL STOP
249A NUMBER NINETEEN FULL STOP
249B NUMBER TWENTY FULL STOP
249C PARENTHESIZED LATIN SMALL LETTER A
249D PARENTHESIZED LATIN SMALL LETTER B
249E PARENTHESIZED LATIN SMALL LETTER C
249F PARENTHESIZED LATIN SMALL LETTER D
24A0 PARENTHESIZED LATIN SMALL LETTER E
24A1 PARENTHESIZED LATIN SMALL LETTER F
24A2 PARENTHESIZED LATIN SMALL LETTER G
24A3 PARENTHESIZED LATIN SMALL LETTER H
24A4 PARENTHESIZED LATIN SMALL LETTER I
24A5 PARENTHESIZED LATIN SMALL LETTER J
24A6 PARENTHESIZED LATIN SMALL LETTER K
24A7 PARENTHESIZED LATIN SMALL LETTER L
24A8 PARENTHESIZED LATIN SMALL LETTER M
24A9 PARENTHESIZED LATIN SMALL LETTER N
24AA PARENTHESIZED LATIN SMALL LETTER O
24AB PARENTHESIZED LATIN SMALL LETTER P
24AC PARENTHESIZED LATIN SMALL LETTER Q
24AD PARENTHESIZED LATIN SMALL LETTER R
24AE PARENTHESIZED LATIN SMALL LETTER S
24AF PARENTHESIZED LATIN SMALL LETTER T
24B0 PARENTHESIZED LATIN SMALL LETTER U
24B1 PARENTHESIZED LATIN SMALL LETTER V
24B2 PARENTHESIZED LATIN SMALL LETTER W
24B3 PARENTHESIZED LATIN SMALL LETTER X
24B4 PARENTHESIZED LATIN SMALL LETTER Y
24B5 PARENTHESIZED LATIN SMALL LETTER Z
24B6 CIRCLED LATIN CAPITAL LETTER A
24B7 CIRCLED LATIN CAPITAL LETTER B
24B8 CIRCLED LATIN CAPITAL LETTER C
24B9 CIRCLED LATIN CAPITAL LETTER D
24BA CIRCLED LATIN CAPITAL LETTER E
24BB CIRCLED LATIN CAPITAL LETTER F
24BC CIRCLED LATIN CAPITAL LETTER G
24BD CIRCLED LATIN CAPITAL LETTER H
24BE CIRCLED LATIN CAPITAL LETTER I
24BF CIRCLED LATIN CAPITAL LETTER J
24C0 CIRCLED LATIN CAPITAL LETTER K
24C1 CIRCLED LATIN CAPITAL LETTER L
24C2 CIRCLED LATIN CAPITAL LETTER M
24C3 CIRCLED LATIN CAPITAL LETTER N
24C4 CIRCLED LATIN CAPITAL LETTER O
24C5 CIRCLED LATIN CAPITAL LETTER P
24C6 CIRCLED LATIN CAPITAL LETTER Q
24C7 CIRCLED LATIN CAPITAL LETTER R
24C8 CIRCLED LATIN CAPITAL LETTER S
24C9 CIRCLED LATIN CAPITAL LETTER T
24CA CIRCLED LATIN CAPITAL LETTER U
24CB CIRCLED LATIN CAPITAL LETTER V
24CC CIRCLED LATIN CAPITAL LETTER W
24CD CIRCLED LATIN CAPITAL LETTER X
24CE CIRCLED LATIN CAPITAL LETTER Y
24CF CIRCLED LATIN CAPITAL LETTER Z
24D0 CIRCLED LATIN SMALL LETTER A
24D1 CIRCLED LATIN SMALL LETTER B
24D2 CIRCLED LATIN SMALL LETTER C
24D3 CIRCLED LATIN SMALL LETTER D
24D4 CIRCLED LATIN SMALL LETTER E
24D5 CIRCLED LATIN SMALL LETTER F
24D6 CIRCLED LATIN SMALL LETTER G
24D7 CIRCLED LATIN SMALL LETTER H
24D8 CIRCLED LATIN SMALL LETTER I
24D9 CIRCLED LATIN SMALL LETTER J
24DA CIRCLED LATIN SMALL LETTER K
24DB CIRCLED LATIN SMALL LETTER L
24DC CIRCLED LATIN SMALL LETTER M
24DD CIRCLED LATIN SMALL LETTER N
24DE CIRCLED LATIN SMALL LETTER O
24DF CIRCLED LATIN SMALL LETTER P
24E0 CIRCLED LATIN SMALL LETTER Q
24E1 CIRCLED LATIN SMALL LETTER R
24E2 CIRCLED LATIN SMALL LETTER S
24E3 CIRCLED LATIN SMALL LETTER T
24E4 CIRCLED LATIN SMALL LETTER U
24E5 CIRCLED LATIN SMALL LETTER V
24E6 CIRCLED LATIN SMALL LETTER W
24E7 CIRCLED LATIN SMALL LETTER X
24E8 CIRCLED LATIN SMALL LETTER Y
24E9 CIRCLED LATIN SMALL LETTER Z
24EA CIRCLED DIGIT ZERO
2500 BOX DRAWINGS LIGHT HORIZONTAL
2501 BOX DRAWINGS HEAVY HORIZONTAL
2502 BOX DRAWINGS LIGHT VERTICAL
2503 BOX DRAWINGS HEAVY VERTICAL
2504 BOX DRAWINGS LIGHT TRIPLE DASH HORIZONTAL
2505 BOX DRAWINGS HEAVY TRIPLE DASH HORIZONTAL
2506 BOX DRAWINGS LIGHT TRIPLE DASH VERTICAL
2507 BOX DRAWINGS HEAVY TRIPLE DASH VERTICAL
2508 BOX DRAWINGS LIGHT QUADRUPLE DASH HORIZONTAL
2509 BOX DRAWINGS HEAVY QUADRUPLE DASH HORIZONTAL
250A BOX DRAWINGS LIGHT QUADRUPLE DASH VERTICAL
250B BOX DRAWINGS HEAVY QUADRUPLE DASH VERTICAL
250C BOX DRAWINGS LIGHT DOWN AND RIGHT
250D BOX DRAWINGS DOWN LIGHT AND RIGHT HEAVY
250E BOX DRAWINGS DOWN HEAVY AND RIGHT LIGHT
250F BOX DRAWINGS HEAVY DOWN AND RIGHT
2510 BOX DRAWINGS LIGHT DOWN AND LEFT
2511 BOX DRAWINGS DOWN LIGHT AND LEFT HEAVY
2512 BOX DRAWINGS DOWN HEAVY AND LEFT LIGHT
2513 BOX DRAWINGS HEAVY DOWN AND LEFT
2514 BOX DRAWINGS LIGHT UP AND RIGHT
2515 BOX DRAWINGS UP LIGHT AND RIGHT HEAVY
2516 BOX DRAWINGS UP HEAVY AND RIGHT LIGHT
2517 BOX DRAWINGS HEAVY UP AND RIGHT
2518 BOX DRAWINGS LIGHT UP AND LEFT
2519 BOX DRAWINGS UP LIGHT AND LEFT HEAVY
251A BOX DRAWINGS UP HEAVY AND LEFT LIGHT
251B BOX DRAWINGS HEAVY UP AND LEFT
251C BOX DRAWINGS LIGHT VERTICAL AND RIGHT
251D BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY
251E BOX DRAWINGS UP HEAVY AND RIGHT DOWN LIGHT
251F BOX DRAWINGS DOWN HEAVY AND RIGHT UP LIGHT
2520 BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT
2521 BOX DRAWINGS DOWN LIGHT AND RIGHT UP HEAVY
2522 BOX DRAWINGS UP LIGHT AND RIGHT DOWN HEAVY
2523 BOX DRAWINGS HEAVY VERTICAL AND RIGHT
2524 BOX DRAWINGS LIGHT VERTICAL AND LEFT
2525 BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY
2526 BOX DRAWINGS UP HEAVY AND LEFT DOWN LIGHT
2527 BOX DRAWINGS DOWN HEAVY AND LEFT UP LIGHT
2528 BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT
2529 BOX DRAWINGS DOWN LIGHT AND LEFT UP HEAVY
252A BOX DRAWINGS UP LIGHT AND LEFT DOWN HEAVY
252B BOX DRAWINGS HEAVY VERTICAL AND LEFT
252C BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
252D BOX DRAWINGS LEFT HEAVY AND RIGHT DOWN LIGHT
252E BOX DRAWINGS RIGHT HEAVY AND LEFT DOWN LIGHT
252F BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY
2530 BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT
2531 BOX DRAWINGS RIGHT LIGHT AND LEFT DOWN HEAVY
2532 BOX DRAWINGS LEFT LIGHT AND RIGHT DOWN HEAVY
2533 BOX DRAWINGS HEAVY DOWN AND HORIZONTAL
2534 BOX DRAWINGS LIGHT UP AND HORIZONTAL
2535 BOX DRAWINGS LEFT HEAVY AND RIGHT UP LIGHT
2536 BOX DRAWINGS RIGHT HEAVY AND LEFT UP LIGHT
2537 BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY
2538 BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT
2539 BOX DRAWINGS RIGHT LIGHT AND LEFT UP HEAVY
253A BOX DRAWINGS LEFT LIGHT AND RIGHT UP HEAVY
253B BOX DRAWINGS HEAVY UP AND HORIZONTAL
253C BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
253D BOX DRAWINGS LEFT HEAVY AND RIGHT VERTICAL LIGHT
253E BOX DRAWINGS RIGHT HEAVY AND LEFT VERTICAL LIGHT
253F BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY
2540 BOX DRAWINGS UP HEAVY AND DOWN HORIZONTAL LIGHT
2541 BOX DRAWINGS DOWN HEAVY AND UP HORIZONTAL LIGHT
2542 BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT
2543 BOX DRAWINGS LEFT UP HEAVY AND RIGHT DOWN LIGHT
2544 BOX DRAWINGS RIGHT UP HEAVY AND LEFT DOWN LIGHT
2545 BOX DRAWINGS LEFT DOWN HEAVY AND RIGHT UP LIGHT
2546 BOX DRAWINGS RIGHT DOWN HEAVY AND LEFT UP LIGHT
2547 BOX DRAWINGS DOWN LIGHT AND UP HORIZONTAL HEAVY
2548 BOX DRAWINGS UP LIGHT AND DOWN HORIZONTAL HEAVY
2549 BOX DRAWINGS RIGHT LIGHT AND LEFT VERTICAL HEAVY
254A BOX DRAWINGS LEFT LIGHT AND RIGHT VERTICAL HEAVY
254B BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL
254C BOX DRAWINGS LIGHT DOUBLE DASH HORIZONTAL
254D BOX DRAWINGS HEAVY DOUBLE DASH HORIZONTAL
254E BOX DRAWINGS LIGHT DOUBLE DASH VERTICAL
254F BOX DRAWINGS HEAVY DOUBLE DASH VERTICAL
2550 BOX DRAWINGS DOUBLE HORIZONTAL
2551 BOX DRAWINGS DOUBLE VERTICAL
2552 BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE
2553 BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE
2554 BOX DRAWINGS DOUBLE DOWN AND RIGHT
2555 BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE
2556 BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE
2557 BOX DRAWINGS DOUBLE DOWN AND LEFT
2558 BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE
2559 BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE
255A BOX DRAWINGS DOUBLE UP AND RIGHT
255B BOX DRAWINGS UP SINGLE AND LEFT DOUBLE
255C BOX DRAWINGS UP DOUBLE AND LEFT SINGLE
255D BOX DRAWINGS DOUBLE UP AND LEFT
255E BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE
255F BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE
2560 BOX DRAWINGS DOUBLE VERTICAL AND RIGHT
2561 BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE
2562 BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE
2563 BOX DRAWINGS DOUBLE VERTICAL AND LEFT
2564 BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE
2565 BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE
2566 BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL
2567 BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE
2568 BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE
2569 BOX DRAWINGS DOUBLE UP AND HORIZONTAL
256A BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE
256B BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE
256C BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL
256D BOX DRAWINGS LIGHT ARC DOWN AND RIGHT
256E BOX DRAWINGS LIGHT ARC DOWN AND LEFT
256F BOX DRAWINGS LIGHT ARC UP AND LEFT
2570 BOX DRAWINGS LIGHT ARC UP AND RIGHT
2571 BOX DRAWINGS LIGHT DIAGONAL UPPER RIGHT TO LOWER LEFT
2572 BOX DRAWINGS LIGHT DIAGONAL UPPER LEFT TO LOWER RIGHT
2573 BOX DRAWINGS LIGHT DIAGONAL CROSS
2574 BOX DRAWINGS LIGHT LEFT
2575 BOX DRAWINGS LIGHT UP
2576 BOX DRAWINGS LIGHT RIGHT
2577 BOX DRAWINGS LIGHT DOWN
2578 BOX DRAWINGS HEAVY LEFT
2579 BOX DRAWINGS HEAVY UP
257A BOX DRAWINGS HEAVY RIGHT
257B BOX DRAWINGS HEAVY DOWN
257C BOX DRAWINGS LIGHT LEFT AND HEAVY RIGHT
257D BOX DRAWINGS LIGHT UP AND HEAVY DOWN
257E BOX DRAWINGS HEAVY LEFT AND LIGHT RIGHT
257F BOX DRAWINGS HEAVY UP AND LIGHT DOWN
2580 UPPER HALF BLOCK
2581 LOWER ONE EIGHTH BLOCK
2582 LOWER ONE QUARTER BLOCK
2583 LOWER THREE EIGHTHS BLOCK
2584 LOWER HALF BLOCK
2585 LOWER FIVE EIGHTHS BLOCK
2586 LOWER THREE QUARTERS BLOCK
2587 LOWER SEVEN EIGHTHS BLOCK
2588 FULL BLOCK
2589 LEFT SEVEN EIGHTHS BLOCK
258A LEFT THREE QUARTERS BLOCK
258B LEFT FIVE EIGHTHS BLOCK
258C LEFT HALF BLOCK
258D LEFT THREE EIGHTHS BLOCK
258E LEFT ONE QUARTER BLOCK
258F LEFT ONE EIGHTH BLOCK
2590 RIGHT HALF BLOCK
2591 LIGHT SHADE
2592 MEDIUM SHADE
2593 DARK SHADE
2594 UPPER ONE EIGHTH BLOCK
2595 RIGHT ONE EIGHTH BLOCK
25A0 BLACK SQUARE
25A1 WHITE SQUARE
25A2 WHITE SQUARE WITH ROUNDED CORNERS
25A3 WHITE SQUARE CONTAINING BLACK SMALL SQUARE
25A4 SQUARE WITH HORIZONTAL FILL
25A5 SQUARE WITH VERTICAL FILL
25A6 SQUARE WITH ORTHOGONAL CROSSHATCH FILL
25A7 SQUARE WITH UPPER LEFT TO LOWER RIGHT FILL
25A8 SQUARE WITH UPPER RIGHT TO LOWER LEFT FILL
25A9 SQUARE WITH DIAGONAL CROSSHATCH FILL
25AA BLACK SMALL SQUARE
25AB WHITE SMALL SQUARE
25AC BLACK RECTANGLE
25AD WHITE RECTANGLE
25AE BLACK VERTICAL RECTANGLE
25AF WHITE VERTICAL RECTANGLE
25B0 BLACK PARALLELOGRAM
25B1 WHITE PARALLELOGRAM
25B2 BLACK UP-POINTING TRIANGLE
25B3 WHITE UP-POINTING TRIANGLE
25B4 BLACK UP-POINTING SMALL TRIANGLE
25B5 WHITE UP-POINTING SMALL TRIANGLE
25B6 BLACK RIGHT-POINTING TRIANGLE
25B7 WHITE RIGHT-POINTING TRIANGLE
25B8 BLACK RIGHT-POINTING SMALL TRIANGLE
25B9 WHITE RIGHT-POINTING SMALL TRIANGLE
25BA BLACK RIGHT-POINTING POINTER
25BB WHITE RIGHT-POINTING POINTER
25BC BLACK DOWN-POINTING TRIANGLE
25BD WHITE DOWN-POINTING TRIANGLE
25BE BLACK DOWN-POINTING SMALL TRIANGLE
25BF WHITE DOWN-POINTING SMALL TRIANGLE
25C0 BLACK LEFT-POINTING TRIANGLE
25C1 WHITE LEFT-POINTING TRIANGLE
25C2 BLACK LEFT-POINTING SMALL TRIANGLE
25C3 WHITE LEFT-POINTING SMALL TRIANGLE
25C4 BLACK LEFT-POINTING POINTER
25C5 WHITE LEFT-POINTING POINTER
25C6 BLACK DIAMOND
25C7 WHITE DIAMOND
25C8 WHITE DIAMOND CONTAINING BLACK SMALL DIAMOND
25C9 FISHEYE
25CA LOZENGE
25CB WHITE CIRCLE
25CC DOTTED CIRCLE
25CD CIRCLE WITH VERTICAL FILL
25CE BULLSEYE
25CF BLACK CIRCLE
25D0 CIRCLE WITH LEFT HALF BLACK
25D1 CIRCLE WITH RIGHT HALF BLACK
25D2 CIRCLE WITH LOWER HALF BLACK
25D3 CIRCLE WITH UPPER HALF BLACK
25D4 CIRCLE WITH UPPER RIGHT QUADRANT BLACK
25D5 CIRCLE WITH ALL BUT UPPER LEFT QUADRANT BLACK
25D6 LEFT HALF BLACK CIRCLE
25D7 RIGHT HALF BLACK CIRCLE
25D8 INVERSE BULLET
25D9 INVERSE WHITE CIRCLE
25DA UPPER HALF INVERSE WHITE CIRCLE
25DB LOWER HALF INVERSE WHITE CIRCLE
25DC UPPER LEFT QUADRANT CIRCULAR ARC
25DD UPPER RIGHT QUADRANT CIRCULAR ARC
25DE LOWER RIGHT QUADRANT CIRCULAR ARC
25DF LOWER LEFT QUADRANT CIRCULAR ARC
25E0 UPPER HALF CIRCLE
25E1 LOWER HALF CIRCLE
25E2 BLACK LOWER RIGHT TRIANGLE
25E3 BLACK LOWER LEFT TRIANGLE
25E4 BLACK UPPER LEFT TRIANGLE
25E5 BLACK UPPER RIGHT TRIANGLE
25E6 WHITE BULLET
25E7 SQUARE WITH LEFT HALF BLACK
25E8 SQUARE WITH RIGHT HALF BLACK
25E9 SQUARE WITH UPPER LEFT DIAGONAL HALF BLACK
25EA SQUARE WITH LOWER RIGHT DIAGONAL HALF BLACK
25EB WHITE SQUARE WITH VERTICAL BISECTING LINE
25EC WHITE UP-POINTING TRIANGLE WITH DOT
25ED UP-POINTING TRIANGLE WITH LEFT HALF BLACK
25EE UP-POINTING TRIANGLE WITH RIGHT HALF BLACK
25EF LARGE CIRCLE
2600 BLACK SUN WITH RAYS
2601 CLOUD
2602 UMBRELLA
2603 SNOWMAN
2604 COMET
2605 BLACK STAR
2606 WHITE STAR
2607 LIGHTNING
2608 THUNDERSTORM
2609 SUN
260A ASCENDING NODE
260B DESCENDING NODE
260C CONJUNCTION
260D OPPOSITION
260E BLACK TELEPHONE
260F WHITE TELEPHONE
2610 BALLOT BOX
2611 BALLOT BOX WITH CHECK
2612 BALLOT BOX WITH X
2613 SALTIRE
261A BLACK LEFT POINTING INDEX
261B BLACK RIGHT POINTING INDEX
261C WHITE LEFT POINTING INDEX
261D WHITE UP POINTING INDEX
261E WHITE RIGHT POINTING INDEX
261F WHITE DOWN POINTING INDEX
2620 SKULL AND CROSSBONES
2621 CAUTION SIGN
2622 RADIOACTIVE SIGN
2623 BIOHAZARD SIGN
2624 CADUCEUS
2625 ANKH
2626 ORTHODOX CROSS
2627 CHI RHO
2628 CROSS OF LORRAINE
2629 CROSS OF JERUSALEM
262A STAR AND CRESCENT
262B FARSI SYMBOL
262C ADI SHAKTI
262D HAMMER AND SICKLE
262E PEACE SYMBOL
262F YIN YANG
2630 TRIGRAM FOR HEAVEN
2631 TRIGRAM FOR LAKE
2632 TRIGRAM FOR FIRE
2633 TRIGRAM FOR THUNDER
2634 TRIGRAM FOR WIND
2635 TRIGRAM FOR WATER
2636 TRIGRAM FOR MOUNTAIN
2637 TRIGRAM FOR EARTH
2638 WHEEL OF DHARMA
2639 WHITE FROWNING FACE
263A WHITE SMILING FACE
263B BLACK SMILING FACE
263C WHITE SUN WITH RAYS
263D FIRST QUARTER MOON
263E LAST QUARTER MOON
263F MERCURY
2640 FEMALE SIGN
2641 EARTH
2642 MALE SIGN
2643 JUPITER
2644 SATURN
2645 URANUS
2646 NEPTUNE
2647 PLUTO
2648 ARIES
2649 TAURUS
264A GEMINI
264B CANCER
264C LEO
264D VIRGO
264E LIBRA
264F SCORPIUS
2650 SAGITTARIUS
2651 CAPRICORN
2652 AQUARIUS
2653 PISCES
2654 WHITE CHESS KING
2655 WHITE CHESS QUEEN
2656 WHITE CHESS ROOK
2657 WHITE CHESS BISHOP
2658 WHITE CHESS KNIGHT
2659 WHITE CHESS PAWN
265A BLACK CHESS KING
265B BLACK CHESS QUEEN
265C BLACK CHESS ROOK
265D BLACK CHESS BISHOP
265E BLACK CHESS KNIGHT
265F BLACK CHESS PAWN
2660 BLACK SPADE SUIT
2661 WHITE HEART SUIT
2662 WHITE DIAMOND SUIT
2663 BLACK CLUB SUIT
2664 WHITE SPADE SUIT
2665 BLACK HEART SUIT
2666 BLACK DIAMOND SUIT
2667 WHITE CLUB SUIT
2668 HOT SPRINGS
2669 QUARTER NOTE
266A EIGHTH NOTE
266B BEAMED EIGHTH NOTES
266C BEAMED SIXTEENTH NOTES
266D MUSIC FLAT SIGN
266E MUSIC NATURAL SIGN
266F MUSIC SHARP SIGN
2701 UPPER BLADE SCISSORS
2702 BLACK SCISSORS
2703 LOWER BLADE SCISSORS
2704 WHITE SCISSORS
2706 TELEPHONE LOCATION SIGN
2707 TAPE DRIVE
2708 AIRPLANE
2709 ENVELOPE
270C VICTORY HAND
270D WRITING HAND
270E LOWER RIGHT PENCIL
270F PENCIL
2710 UPPER RIGHT PENCIL
2711 WHITE NIB
2712 BLACK NIB
2713 CHECK MARK
2714 HEAVY CHECK MARK
2715 MULTIPLICATION X
2716 HEAVY MULTIPLICATION X
2717 BALLOT X
2718 HEAVY BALLOT X
2719 OUTLINED GREEK CROSS
271A HEAVY GREEK CROSS
271B OPEN CENTRE CROSS
271C HEAVY OPEN CENTRE CROSS
271D LATIN CROSS
271E SHADOWED WHITE LATIN CROSS
271F OUTLINED LATIN CROSS
2720 MALTESE CROSS
2721 STAR OF DAVID
2722 FOUR TEARDROP-SPOKED ASTERISK
2723 FOUR BALLOON-SPOKED ASTERISK
2724 HEAVY FOUR BALLOON-SPOKED ASTERISK
2725 FOUR CLUB-SPOKED ASTERISK
2726 BLACK FOUR POINTED STAR
2727 WHITE FOUR POINTED STAR
2729 STRESS OUTLINED WHITE STAR
272A CIRCLED WHITE STAR
272B OPEN CENTRE BLACK STAR
272C BLACK CENTRE WHITE STAR
272D OUTLINED BLACK STAR
272E HEAVY OUTLINED BLACK STAR
272F PINWHEEL STAR
2730 SHADOWED WHITE STAR
2731 HEAVY ASTERISK
2732 OPEN CENTRE ASTERISK
2733 EIGHT SPOKED ASTERISK
2734 EIGHT POINTED BLACK STAR
2735 EIGHT POINTED PINWHEEL STAR
2736 SIX POINTED BLACK STAR
2737 EIGHT POINTED RECTILINEAR BLACK STAR
2738 HEAVY EIGHT POINTED RECTILINEAR BLACK STAR
2739 TWELVE POINTED BLACK STAR
273A SIXTEEN POINTED ASTERISK
273B TEARDROP-SPOKED ASTERISK
273C OPEN CENTRE TEARDROP-SPOKED ASTERISK
273D HEAVY TEARDROP-SPOKED ASTERISK
273E SIX PETALLED BLACK AND WHITE FLORETTE
273F BLACK FLORETTE
2740 WHITE FLORETTE
2741 EIGHT PETALLED OUTLINED BLACK FLORETTE
2742 CIRCLED OPEN CENTRE EIGHT POINTED STAR
2743 HEAVY TEARDROP-SPOKED PINWHEEL ASTERISK
2744 SNOWFLAKE
2745 TIGHT TRIFOLIATE SNOWFLAKE
2746 HEAVY CHEVRON SNOWFLAKE
2747 SPARKLE
2748 HEAVY SPARKLE
2749 BALLOON-SPOKED ASTERISK
274A EIGHT TEARDROP-SPOKED PROPELLER ASTERISK
274B HEAVY EIGHT TEARDROP-SPOKED PROPELLER ASTERISK
274D SHADOWED WHITE CIRCLE
274F LOWER RIGHT DROP-SHADOWED WHITE SQUARE
2750 UPPER RIGHT DROP-SHADOWED WHITE SQUARE
2751 LOWER RIGHT SHADOWED WHITE SQUARE
2752 UPPER RIGHT SHADOWED WHITE SQUARE
2756 BLACK DIAMOND MINUS WHITE X
2758 LIGHT VERTICAL BAR
2759 MEDIUM VERTICAL BAR
275A HEAVY VERTICAL BAR
275B HEAVY SINGLE TURNED COMMA QUOTATION MARK ORNAMENT
275C HEAVY SINGLE COMMA QUOTATION MARK ORNAMENT
275D HEAVY DOUBLE TURNED COMMA QUOTATION MARK ORNAMENT
275E HEAVY DOUBLE COMMA QUOTATION MARK ORNAMENT
2761 CURVED STEM PARAGRAPH SIGN ORNAMENT
2762 HEAVY EXCLAMATION MARK ORNAMENT
2763 HEAVY HEART EXCLAMATION MARK ORNAMENT
2764 HEAVY BLACK HEART
2765 ROTATED HEAVY BLACK HEART BULLET
2766 FLORAL HEART
2767 ROTATED FLORAL HEART BULLET
2776 DINGBAT NEGATIVE CIRCLED DIGIT ONE
2777 DINGBAT NEGATIVE CIRCLED DIGIT TWO
2778 DINGBAT NEGATIVE CIRCLED DIGIT THREE
2779 DINGBAT NEGATIVE CIRCLED DIGIT FOUR
277A DINGBAT NEGATIVE CIRCLED DIGIT FIVE
277B DINGBAT NEGATIVE CIRCLED DIGIT SIX
277C DINGBAT NEGATIVE CIRCLED DIGIT SEVEN
277D DINGBAT NEGATIVE CIRCLED DIGIT EIGHT
277E DINGBAT NEGATIVE CIRCLED DIGIT NINE
277F DINGBAT NEGATIVE CIRCLED NUMBER TEN
2780 DINGBAT CIRCLED SANS-SERIF DIGIT ONE
2781 DINGBAT CIRCLED SANS-SERIF DIGIT TWO
2782 DINGBAT CIRCLED SANS-SERIF DIGIT THREE
2783 DINGBAT CIRCLED SANS-SERIF DIGIT FOUR
2784 DINGBAT CIRCLED SANS-SERIF DIGIT FIVE
2785 DINGBAT CIRCLED SANS-SERIF DIGIT SIX
2786 DINGBAT CIRCLED SANS-SERIF DIGIT SEVEN
2787 DINGBAT CIRCLED SANS-SERIF DIGIT EIGHT
2788 DINGBAT CIRCLED SANS-SERIF DIGIT NINE
2789 DINGBAT CIRCLED SANS-SERIF NUMBER TEN
278A DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT ONE
278B DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT TWO
278C DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT THREE
278D DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT FOUR
278E DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT FIVE
278F DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT SIX
2790 DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT SEVEN
2791 DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT EIGHT
2792 DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT NINE
2793 DINGBAT NEGATIVE CIRCLED SANS-SERIF NUMBER TEN
2794 HEAVY WIDE-HEADED RIGHTWARDS ARROW
2798 HEAVY SOUTH EAST ARROW
2799 HEAVY RIGHTWARDS ARROW
279A HEAVY NORTH EAST ARROW
279B DRAFTING POINT RIGHTWARDS ARROW
279C HEAVY ROUND-TIPPED RIGHTWARDS ARROW
279D TRIANGLE-HEADED RIGHTWARDS ARROW
279E HEAVY TRIANGLE-HEADED RIGHTWARDS ARROW
279F DASHED TRIANGLE-HEADED RIGHTWARDS ARROW
27A0 HEAVY DASHED TRIANGLE-HEADED RIGHTWARDS ARROW
27A1 BLACK RIGHTWARDS ARROW
27A2 THREE-D TOP-LIGHTED RIGHTWARDS ARROWHEAD
27A3 THREE-D BOTTOM-LIGHTED RIGHTWARDS ARROWHEAD
27A4 BLACK RIGHTWARDS ARROWHEAD
27A5 HEAVY BLACK CURVED DOWNWARDS AND RIGHTWARDS ARROW
27A6 HEAVY BLACK CURVED UPWARDS AND RIGHTWARDS ARROW
27A7 SQUAT BLACK RIGHTWARDS ARROW
27A8 HEAVY CONCAVE-POINTED BLACK RIGHTWARDS ARROW
27A9 RIGHT-SHADED WHITE RIGHTWARDS ARROW
27AA LEFT-SHADED WHITE RIGHTWARDS ARROW
27AB BACK-TILTED SHADOWED WHITE RIGHTWARDS ARROW
27AC FRONT-TILTED SHADOWED WHITE RIGHTWARDS ARROW
27AD HEAVY LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW
27AE HEAVY UPPER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW
27AF NOTCHED LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW
27B1 NOTCHED UPPER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW
27B2 CIRCLED HEAVY WHITE RIGHTWARDS ARROW
27B3 WHITE-FEATHERED RIGHTWARDS ARROW
27B4 BLACK-FEATHERED SOUTH EAST ARROW
27B5 BLACK-FEATHERED RIGHTWARDS ARROW
27B6 BLACK-FEATHERED NORTH EAST ARROW
27B7 HEAVY BLACK-FEATHERED SOUTH EAST ARROW
27B8 HEAVY BLACK-FEATHERED RIGHTWARDS ARROW
27B9 HEAVY BLACK-FEATHERED NORTH EAST ARROW
27BA TEARDROP-BARBED RIGHTWARDS ARROW
27BB HEAVY TEARDROP-SHANKED RIGHTWARDS ARROW
27BC WEDGE-TAILED RIGHTWARDS ARROW
27BD HEAVY WEDGE-TAILED RIGHTWARDS ARROW
27BE OPEN-OUTLINED RIGHTWARDS ARROW
3000 IDEOGRAPHIC SPACE
3001 IDEOGRAPHIC COMMA
3002 IDEOGRAPHIC FULL STOP
3003 DITTO MARK
3004 JAPANESE INDUSTRIAL STANDARD SYMBOL
3005 IDEOGRAPHIC ITERATION MARK
3006 IDEOGRAPHIC CLOSING MARK
3007 IDEOGRAPHIC NUMBER ZERO
3008 LEFT ANGLE BRACKET
3009 RIGHT ANGLE BRACKET
300A LEFT DOUBLE ANGLE BRACKET
300B RIGHT DOUBLE ANGLE BRACKET
300C LEFT CORNER BRACKET
300D RIGHT CORNER BRACKET
300E LEFT WHITE CORNER BRACKET
300F RIGHT WHITE CORNER BRACKET
3010 LEFT BLACK LENTICULAR BRACKET
3011 RIGHT BLACK LENTICULAR BRACKET
3012 POSTAL MARK
3013 GETA MARK
3014 LEFT TORTOISE SHELL BRACKET
3015 RIGHT TORTOISE SHELL BRACKET
3016 LEFT WHITE LENTICULAR BRACKET
3017 RIGHT WHITE LENTICULAR BRACKET
3018 LEFT WHITE TORTOISE SHELL BRACKET
3019 RIGHT WHITE TORTOISE SHELL BRACKET
301A LEFT WHITE SQUARE BRACKET
301B RIGHT WHITE SQUARE BRACKET
301C WAVE DASH
301D REVERSED DOUBLE PRIME QUOTATION MARK
301E DOUBLE PRIME QUOTATION MARK
301F LOW DOUBLE PRIME QUOTATION MARK
3020 POSTAL MARK FACE
3021 HANGZHOU NUMERAL ONE
3022 HANGZHOU NUMERAL TWO
3023 HANGZHOU NUMERAL THREE
3024 HANGZHOU NUMERAL FOUR
3025 HANGZHOU NUMERAL FIVE
3026 HANGZHOU NUMERAL SIX
3027 HANGZHOU NUMERAL SEVEN
3028 HANGZHOU NUMERAL EIGHT
3029 HANGZHOU NUMERAL NINE
302A IDEOGRAPHIC LEVEL TONE MARK
302B IDEOGRAPHIC RISING TONE MARK
302C IDEOGRAPHIC DEPARTING TONE MARK
302D IDEOGRAPHIC ENTERING TONE MARK
302E HANGUL SINGLE DOT TONE MARK
302F HANGUL DOUBLE DOT TONE MARK
3030 WAVY DASH
3031 VERTICAL KANA REPEAT MARK
3032 VERTICAL KANA REPEAT WITH VOICED SOUND MARK
3033 VERTICAL KANA REPEAT MARK UPPER HALF
3034 VERTICAL KANA REPEAT WITH VOICED SOUND MARK UPPER HALF
3035 VERTICAL KANA REPEAT MARK LOWER HALF
3036 CIRCLED POSTAL MARK
3037 IDEOGRAPHIC TELEGRAPH LINE FEED SEPARATOR SYMBOL
303F IDEOGRAPHIC HALF FILL SPACE
3041 HIRAGANA LETTER SMALL A
3042 HIRAGANA LETTER A
3043 HIRAGANA LETTER SMALL I
3044 HIRAGANA LETTER I
3045 HIRAGANA LETTER SMALL U
3046 HIRAGANA LETTER U
3047 HIRAGANA LETTER SMALL E
3048 HIRAGANA LETTER E
3049 HIRAGANA LETTER SMALL O
304A HIRAGANA LETTER O
304B HIRAGANA LETTER KA
304C HIRAGANA LETTER GA
304D HIRAGANA LETTER KI
304E HIRAGANA LETTER GI
304F HIRAGANA LETTER KU
3050 HIRAGANA LETTER GU
3051 HIRAGANA LETTER KE
3052 HIRAGANA LETTER GE
3053 HIRAGANA LETTER KO
3054 HIRAGANA LETTER GO
3055 HIRAGANA LETTER SA
3056 HIRAGANA LETTER ZA
3057 HIRAGANA LETTER SI
3058 HIRAGANA LETTER ZI
3059 HIRAGANA LETTER SU
305A HIRAGANA LETTER ZU
305B HIRAGANA LETTER SE
305C HIRAGANA LETTER ZE
305D HIRAGANA LETTER SO
305E HIRAGANA LETTER ZO
305F HIRAGANA LETTER TA
3060 HIRAGANA LETTER DA
3061 HIRAGANA LETTER TI
3062 HIRAGANA LETTER DI
3063 HIRAGANA LETTER SMALL TU
3064 HIRAGANA LETTER TU
3065 HIRAGANA LETTER DU
3066 HIRAGANA LETTER TE
3067 HIRAGANA LETTER DE
3068 HIRAGANA LETTER TO
3069 HIRAGANA LETTER DO
306A HIRAGANA LETTER NA
306B HIRAGANA LETTER NI
306C HIRAGANA LETTER NU
306D HIRAGANA LETTER NE
306E HIRAGANA LETTER NO
306F HIRAGANA LETTER HA
3070 HIRAGANA LETTER BA
3071 HIRAGANA LETTER PA
3072 HIRAGANA LETTER HI
3073 HIRAGANA LETTER BI
3074 HIRAGANA LETTER PI
3075 HIRAGANA LETTER HU
3076 HIRAGANA LETTER BU
3077 HIRAGANA LETTER PU
3078 HIRAGANA LETTER HE
3079 HIRAGANA LETTER BE
307A HIRAGANA LETTER PE
307B HIRAGANA LETTER HO
307C HIRAGANA LETTER BO
307D HIRAGANA LETTER PO
307E HIRAGANA LETTER MA
307F HIRAGANA LETTER MI
3080 HIRAGANA LETTER MU
3081 HIRAGANA LETTER ME
3082 HIRAGANA LETTER MO
3083 HIRAGANA LETTER SMALL YA
3084 HIRAGANA LETTER YA
3085 HIRAGANA LETTER SMALL YU
3086 HIRAGANA LETTER YU
3087 HIRAGANA LETTER SMALL YO
3088 HIRAGANA LETTER YO
3089 HIRAGANA LETTER RA
308A HIRAGANA LETTER RI
308B HIRAGANA LETTER RU
308C HIRAGANA LETTER RE
308D HIRAGANA LETTER RO
308E HIRAGANA LETTER SMALL WA
308F HIRAGANA LETTER WA
3090 HIRAGANA LETTER WI
3091 HIRAGANA LETTER WE
3092 HIRAGANA LETTER WO
3093 HIRAGANA LETTER N
3094 HIRAGANA LETTER VU
3099 COMBINING KATAKANA-HIRAGANA VOICED SOUND MARK
309A COMBINING KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
309B KATAKANA-HIRAGANA VOICED SOUND MARK
309C KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
309D HIRAGANA ITERATION MARK
309E HIRAGANA VOICED ITERATION MARK
30A1 KATAKANA LETTER SMALL A
30A2 KATAKANA LETTER A
30A3 KATAKANA LETTER SMALL I
30A4 KATAKANA LETTER I
30A5 KATAKANA LETTER SMALL U
30A6 KATAKANA LETTER U
30A7 KATAKANA LETTER SMALL E
30A8 KATAKANA LETTER E
30A9 KATAKANA LETTER SMALL O
30AA KATAKANA LETTER O
30AB KATAKANA LETTER KA
30AC KATAKANA LETTER GA
30AD KATAKANA LETTER KI
30AE KATAKANA LETTER GI
30AF KATAKANA LETTER KU
30B0 KATAKANA LETTER GU
30B1 KATAKANA LETTER KE
30B2 KATAKANA LETTER GE
30B3 KATAKANA LETTER KO
30B4 KATAKANA LETTER GO
30B5 KATAKANA LETTER SA
30B6 KATAKANA LETTER ZA
30B7 KATAKANA LETTER SI
30B8 KATAKANA LETTER ZI
30B9 KATAKANA LETTER SU
30BA KATAKANA LETTER ZU
30BB KATAKANA LETTER SE
30BC KATAKANA LETTER ZE
30BD KATAKANA LETTER SO
30BE KATAKANA LETTER ZO
30BF KATAKANA LETTER TA
30C0 KATAKANA LETTER DA
30C1 KATAKANA LETTER TI
30C2 KATAKANA LETTER DI
30C3 KATAKANA LETTER SMALL TU
30C4 KATAKANA LETTER TU
30C5 KATAKANA LETTER DU
30C6 KATAKANA LETTER TE
30C7 KATAKANA LETTER DE
30C8 KATAKANA LETTER TO
30C9 KATAKANA LETTER DO
30CA KATAKANA LETTER NA
30CB KATAKANA LETTER NI
30CC KATAKANA LETTER NU
30CD KATAKANA LETTER NE
30CE KATAKANA LETTER NO
30CF KATAKANA LETTER HA
30D0 KATAKANA LETTER BA
30D1 KATAKANA LETTER PA
30D2 KATAKANA LETTER HI
30D3 KATAKANA LETTER BI
30D4 KATAKANA LETTER PI
30D5 KATAKANA LETTER HU
30D6 KATAKANA LETTER BU
30D7 KATAKANA LETTER PU
30D8 KATAKANA LETTER HE
30D9 KATAKANA LETTER BE
30DA KATAKANA LETTER PE
30DB KATAKANA LETTER HO
30DC KATAKANA LETTER BO
30DD KATAKANA LETTER PO
30DE KATAKANA LETTER MA
30DF KATAKANA LETTER MI
30E0 KATAKANA LETTER MU
30E1 KATAKANA LETTER ME
30E2 KATAKANA LETTER MO
30E3 KATAKANA LETTER SMALL YA
30E4 KATAKANA LETTER YA
30E5 KATAKANA LETTER SMALL YU
30E6 KATAKANA LETTER YU
30E7 KATAKANA LETTER SMALL YO
30E8 KATAKANA LETTER YO
30E9 KATAKANA LETTER RA
30EA KATAKANA LETTER RI
30EB KATAKANA LETTER RU
30EC KATAKANA LETTER RE
30ED KATAKANA LETTER RO
30EE KATAKANA LETTER SMALL WA
30EF KATAKANA LETTER WA
30F0 KATAKANA LETTER WI
30F1 KATAKANA LETTER WE
30F2 KATAKANA LETTER WO
30F3 KATAKANA LETTER N
30F4 KATAKANA LETTER VU
30F5 KATAKANA LETTER SMALL KA
30F6 KATAKANA LETTER SMALL KE
30F7 KATAKANA LETTER VA
30F8 KATAKANA LETTER VI
30F9 KATAKANA LETTER VE
30FA KATAKANA LETTER VO
30FB KATAKANA MIDDLE DOT
30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
30FD KATAKANA ITERATION MARK
30FE KATAKANA VOICED ITERATION MARK
3105 BOPOMOFO LETTER B
3106 BOPOMOFO LETTER P
3107 BOPOMOFO LETTER M
3108 BOPOMOFO LETTER F
3109 BOPOMOFO LETTER D
310A BOPOMOFO LETTER T
310B BOPOMOFO LETTER N
310C BOPOMOFO LETTER L
310D BOPOMOFO LETTER G
310E BOPOMOFO LETTER K
310F BOPOMOFO LETTER H
3110 BOPOMOFO LETTER J
3111 BOPOMOFO LETTER Q
3112 BOPOMOFO LETTER X
3113 BOPOMOFO LETTER ZH
3114 BOPOMOFO LETTER CH
3115 BOPOMOFO LETTER SH
3116 BOPOMOFO LETTER R
3117 BOPOMOFO LETTER Z
3118 BOPOMOFO LETTER C
3119 BOPOMOFO LETTER S
311A BOPOMOFO LETTER A
311B BOPOMOFO LETTER O
311C BOPOMOFO LETTER E
311D BOPOMOFO LETTER EH
311E BOPOMOFO LETTER AI
311F BOPOMOFO LETTER EI
3120 BOPOMOFO LETTER AU
3121 BOPOMOFO LETTER OU
3122 BOPOMOFO LETTER AN
3123 BOPOMOFO LETTER EN
3124 BOPOMOFO LETTER ANG
3125 BOPOMOFO LETTER ENG
3126 BOPOMOFO LETTER ER
3127 BOPOMOFO LETTER I
3128 BOPOMOFO LETTER U
3129 BOPOMOFO LETTER IU
312A BOPOMOFO LETTER V
312B BOPOMOFO LETTER NG
312C BOPOMOFO LETTER GN
3131 HANGUL LETTER KIYEOK
3132 HANGUL LETTER SSANGKIYEOK
3133 HANGUL LETTER KIYEOK-SIOS
3134 HANGUL LETTER NIEUN
3135 HANGUL LETTER NIEUN-CIEUC
3136 HANGUL LETTER NIEUN-HIEUH
3137 HANGUL LETTER TIKEUT
3138 HANGUL LETTER SSANGTIKEUT
3139 HANGUL LETTER RIEUL
313A HANGUL LETTER RIEUL-KIYEOK
313B HANGUL LETTER RIEUL-MIEUM
313C HANGUL LETTER RIEUL-PIEUP
313D HANGUL LETTER RIEUL-SIOS
313E HANGUL LETTER RIEUL-THIEUTH
313F HANGUL LETTER RIEUL-PHIEUPH
3140 HANGUL LETTER RIEUL-HIEUH
3141 HANGUL LETTER MIEUM
3142 HANGUL LETTER PIEUP
3143 HANGUL LETTER SSANGPIEUP
3144 HANGUL LETTER PIEUP-SIOS
3145 HANGUL LETTER SIOS
3146 HANGUL LETTER SSANGSIOS
3147 HANGUL LETTER IEUNG
3148 HANGUL LETTER CIEUC
3149 HANGUL LETTER SSANGCIEUC
314A HANGUL LETTER CHIEUCH
314B HANGUL LETTER KHIEUKH
314C HANGUL LETTER THIEUTH
314D HANGUL LETTER PHIEUPH
314E HANGUL LETTER HIEUH
314F HANGUL LETTER A
3150 HANGUL LETTER AE
3151 HANGUL LETTER YA
3152 HANGUL LETTER YAE
3153 HANGUL LETTER EO
3154 HANGUL LETTER E
3155 HANGUL LETTER YEO
3156 HANGUL LETTER YE
3157 HANGUL LETTER O
3158 HANGUL LETTER WA
3159 HANGUL LETTER WAE
315A HANGUL LETTER OE
315B HANGUL LETTER YO
315C HANGUL LETTER U
315D HANGUL LETTER WEO
315E HANGUL LETTER WE
315F HANGUL LETTER WI
3160 HANGUL LETTER YU
3161 HANGUL LETTER EU
3162 HANGUL LETTER YI
3163 HANGUL LETTER I
3164 HANGUL FILLER
3165 HANGUL LETTER SSANGNIEUN
3166 HANGUL LETTER NIEUN-TIKEUT
3167 HANGUL LETTER NIEUN-SIOS
3168 HANGUL LETTER NIEUN-PANSIOS
3169 HANGUL LETTER RIEUL-KIYEOK-SIOS
316A HANGUL LETTER RIEUL-TIKEUT
316B HANGUL LETTER RIEUL-PIEUP-SIOS
316C HANGUL LETTER RIEUL-PANSIOS
316D HANGUL LETTER RIEUL-YEORINHIEUH
316E HANGUL LETTER MIEUM-PIEUP
316F HANGUL LETTER MIEUM-SIOS
3170 HANGUL LETTER MIEUM-PANSIOS
3171 HANGUL LETTER KAPYEOUNMIEUM
3172 HANGUL LETTER PIEUP-KIYEOK
3173 HANGUL LETTER PIEUP-TIKEUT
3174 HANGUL LETTER PIEUP-SIOS-KIYEOK
3175 HANGUL LETTER PIEUP-SIOS-TIKEUT
3176 HANGUL LETTER PIEUP-CIEUC
3177 HANGUL LETTER PIEUP-THIEUTH
3178 HANGUL LETTER KAPYEOUNPIEUP
3179 HANGUL LETTER KAPYEOUNSSANGPIEUP
317A HANGUL LETTER SIOS-KIYEOK
317B HANGUL LETTER SIOS-NIEUN
317C HANGUL LETTER SIOS-TIKEUT
317D HANGUL LETTER SIOS-PIEUP
317E HANGUL LETTER SIOS-CIEUC
317F HANGUL LETTER PANSIOS
3180 HANGUL LETTER SSANGIEUNG
3181 HANGUL LETTER YESIEUNG
3182 HANGUL LETTER YESIEUNG-SIOS
3183 HANGUL LETTER YESIEUNG-PANSIOS
3184 HANGUL LETTER KAPYEOUNPHIEUPH
3185 HANGUL LETTER SSANGHIEUH
3186 HANGUL LETTER YEORINHIEUH
3187 HANGUL LETTER YO-YA
3188 HANGUL LETTER YO-YAE
3189 HANGUL LETTER YO-I
318A HANGUL LETTER YU-YEO
318B HANGUL LETTER YU-YE
318C HANGUL LETTER YU-I
318D HANGUL LETTER ARAEA
318E HANGUL LETTER ARAEAE
3190 IDEOGRAPHIC ANNOTATION LINKING MARK
3191 IDEOGRAPHIC ANNOTATION REVERSE MARK
3192 IDEOGRAPHIC ANNOTATION ONE MARK
3193 IDEOGRAPHIC ANNOTATION TWO MARK
3194 IDEOGRAPHIC ANNOTATION THREE MARK
3195 IDEOGRAPHIC ANNOTATION FOUR MARK
3196 IDEOGRAPHIC ANNOTATION TOP MARK
3197 IDEOGRAPHIC ANNOTATION MIDDLE MARK
3198 IDEOGRAPHIC ANNOTATION BOTTOM MARK
3199 IDEOGRAPHIC ANNOTATION FIRST MARK
319A IDEOGRAPHIC ANNOTATION SECOND MARK
319B IDEOGRAPHIC ANNOTATION THIRD MARK
319C IDEOGRAPHIC ANNOTATION FOURTH MARK
319D IDEOGRAPHIC ANNOTATION HEAVEN MARK
319E IDEOGRAPHIC ANNOTATION EARTH MARK
319F IDEOGRAPHIC ANNOTATION MAN MARK
3200 PARENTHESIZED HANGUL KIYEOK
3201 PARENTHESIZED HANGUL NIEUN
3202 PARENTHESIZED HANGUL TIKEUT
3203 PARENTHESIZED HANGUL RIEUL
3204 PARENTHESIZED HANGUL MIEUM
3205 PARENTHESIZED HANGUL PIEUP
3206 PARENTHESIZED HANGUL SIOS
3207 PARENTHESIZED HANGUL IEUNG
3208 PARENTHESIZED HANGUL CIEUC
3209 PARENTHESIZED HANGUL CHIEUCH
320A PARENTHESIZED HANGUL KHIEUKH
320B PARENTHESIZED HANGUL THIEUTH
320C PARENTHESIZED HANGUL PHIEUPH
320D PARENTHESIZED HANGUL HIEUH
320E PARENTHESIZED HANGUL KIYEOK A
320F PARENTHESIZED HANGUL NIEUN A
3210 PARENTHESIZED HANGUL TIKEUT A
3211 PARENTHESIZED HANGUL RIEUL A
3212 PARENTHESIZED HANGUL MIEUM A
3213 PARENTHESIZED HANGUL PIEUP A
3214 PARENTHESIZED HANGUL SIOS A
3215 PARENTHESIZED HANGUL IEUNG A
3216 PARENTHESIZED HANGUL CIEUC A
3217 PARENTHESIZED HANGUL CHIEUCH A
3218 PARENTHESIZED HANGUL KHIEUKH A
3219 PARENTHESIZED HANGUL THIEUTH A
321A PARENTHESIZED HANGUL PHIEUPH A
321B PARENTHESIZED HANGUL HIEUH A
321C PARENTHESIZED HANGUL CIEUC U
3220 PARENTHESIZED IDEOGRAPH ONE
3221 PARENTHESIZED IDEOGRAPH TWO
3222 PARENTHESIZED IDEOGRAPH THREE
3223 PARENTHESIZED IDEOGRAPH FOUR
3224 PARENTHESIZED IDEOGRAPH FIVE
3225 PARENTHESIZED IDEOGRAPH SIX
3226 PARENTHESIZED IDEOGRAPH SEVEN
3227 PARENTHESIZED IDEOGRAPH EIGHT
3228 PARENTHESIZED IDEOGRAPH NINE
3229 PARENTHESIZED IDEOGRAPH TEN
322A PARENTHESIZED IDEOGRAPH MOON
322B PARENTHESIZED IDEOGRAPH FIRE
322C PARENTHESIZED IDEOGRAPH WATER
322D PARENTHESIZED IDEOGRAPH WOOD
322E PARENTHESIZED IDEOGRAPH METAL
322F PARENTHESIZED IDEOGRAPH EARTH
3230 PARENTHESIZED IDEOGRAPH SUN
3231 PARENTHESIZED IDEOGRAPH STOCK
3232 PARENTHESIZED IDEOGRAPH HAVE
3233 PARENTHESIZED IDEOGRAPH SOCIETY
3234 PARENTHESIZED IDEOGRAPH NAME
3235 PARENTHESIZED IDEOGRAPH SPECIAL
3236 PARENTHESIZED IDEOGRAPH FINANCIAL
3237 PARENTHESIZED IDEOGRAPH CONGRATULATION
3238 PARENTHESIZED IDEOGRAPH LABOR
3239 PARENTHESIZED IDEOGRAPH REPRESENT
323A PARENTHESIZED IDEOGRAPH CALL
323B PARENTHESIZED IDEOGRAPH STUDY
323C PARENTHESIZED IDEOGRAPH SUPERVISE
323D PARENTHESIZED IDEOGRAPH ENTERPRISE
323E PARENTHESIZED IDEOGRAPH RESOURCE
323F PARENTHESIZED IDEOGRAPH ALLIANCE
3240 PARENTHESIZED IDEOGRAPH FESTIVAL
3241 PARENTHESIZED IDEOGRAPH REST
3242 PARENTHESIZED IDEOGRAPH SELF
3243 PARENTHESIZED IDEOGRAPH REACH
3260 CIRCLED HANGUL KIYEOK
3261 CIRCLED HANGUL NIEUN
3262 CIRCLED HANGUL TIKEUT
3263 CIRCLED HANGUL RIEUL
3264 CIRCLED HANGUL MIEUM
3265 CIRCLED HANGUL PIEUP
3266 CIRCLED HANGUL SIOS
3267 CIRCLED HANGUL IEUNG
3268 CIRCLED HANGUL CIEUC
3269 CIRCLED HANGUL CHIEUCH
326A CIRCLED HANGUL KHIEUKH
326B CIRCLED HANGUL THIEUTH
326C CIRCLED HANGUL PHIEUPH
326D CIRCLED HANGUL HIEUH
326E CIRCLED HANGUL KIYEOK A
326F CIRCLED HANGUL NIEUN A
3270 CIRCLED HANGUL TIKEUT A
3271 CIRCLED HANGUL RIEUL A
3272 CIRCLED HANGUL MIEUM A
3273 CIRCLED HANGUL PIEUP A
3274 CIRCLED HANGUL SIOS A
3275 CIRCLED HANGUL IEUNG A
3276 CIRCLED HANGUL CIEUC A
3277 CIRCLED HANGUL CHIEUCH A
3278 CIRCLED HANGUL KHIEUKH A
3279 CIRCLED HANGUL THIEUTH A
327A CIRCLED HANGUL PHIEUPH A
327B CIRCLED HANGUL HIEUH A
327F KOREAN STANDARD SYMBOL
3280 CIRCLED IDEOGRAPH ONE
3281 CIRCLED IDEOGRAPH TWO
3282 CIRCLED IDEOGRAPH THREE
3283 CIRCLED IDEOGRAPH FOUR
3284 CIRCLED IDEOGRAPH FIVE
3285 CIRCLED IDEOGRAPH SIX
3286 CIRCLED IDEOGRAPH SEVEN
3287 CIRCLED IDEOGRAPH EIGHT
3288 CIRCLED IDEOGRAPH NINE
3289 CIRCLED IDEOGRAPH TEN
328A CIRCLED IDEOGRAPH MOON
328B CIRCLED IDEOGRAPH FIRE
328C CIRCLED IDEOGRAPH WATER
328D CIRCLED IDEOGRAPH WOOD
328E CIRCLED IDEOGRAPH METAL
328F CIRCLED IDEOGRAPH EARTH
3290 CIRCLED IDEOGRAPH SUN
3291 CIRCLED IDEOGRAPH STOCK
3292 CIRCLED IDEOGRAPH HAVE
3293 CIRCLED IDEOGRAPH SOCIETY
3294 CIRCLED IDEOGRAPH NAME
3295 CIRCLED IDEOGRAPH SPECIAL
3296 CIRCLED IDEOGRAPH FINANCIAL
3297 CIRCLED IDEOGRAPH CONGRATULATION
3298 CIRCLED IDEOGRAPH LABOR
3299 CIRCLED IDEOGRAPH SECRET
329A CIRCLED IDEOGRAPH MALE
329B CIRCLED IDEOGRAPH FEMALE
329C CIRCLED IDEOGRAPH SUITABLE
329D CIRCLED IDEOGRAPH EXCELLENT
329E CIRCLED IDEOGRAPH PRINT
329F CIRCLED IDEOGRAPH ATTENTION
32A0 CIRCLED IDEOGRAPH ITEM
32A1 CIRCLED IDEOGRAPH REST
32A2 CIRCLED IDEOGRAPH COPY
32A3 CIRCLED IDEOGRAPH CORRECT
32A4 CIRCLED IDEOGRAPH HIGH
32A5 CIRCLED IDEOGRAPH CENTRE
32A6 CIRCLED IDEOGRAPH LOW
32A7 CIRCLED IDEOGRAPH LEFT
32A8 CIRCLED IDEOGRAPH RIGHT
32A9 CIRCLED IDEOGRAPH MEDICINE
32AA CIRCLED IDEOGRAPH RELIGION
32AB CIRCLED IDEOGRAPH STUDY
32AC CIRCLED IDEOGRAPH SUPERVISE
32AD CIRCLED IDEOGRAPH ENTERPRISE
32AE CIRCLED IDEOGRAPH RESOURCE
32AF CIRCLED IDEOGRAPH ALLIANCE
32B0 CIRCLED IDEOGRAPH NIGHT
32C0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JANUARY
32C1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR FEBRUARY
32C2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR MARCH
32C3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR APRIL
32C4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR MAY
32C5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JUNE
32C6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JULY
32C7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR AUGUST
32C8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR SEPTEMBER
32C9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR OCTOBER
32CA IDEOGRAPHIC TELEGRAPH SYMBOL FOR NOVEMBER
32CB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DECEMBER
32D0 CIRCLED KATAKANA A
32D1 CIRCLED KATAKANA I
32D2 CIRCLED KATAKANA U
32D3 CIRCLED KATAKANA E
32D4 CIRCLED KATAKANA O
32D5 CIRCLED KATAKANA KA
32D6 CIRCLED KATAKANA KI
32D7 CIRCLED KATAKANA KU
32D8 CIRCLED KATAKANA KE
32D9 CIRCLED KATAKANA KO
32DA CIRCLED KATAKANA SA
32DB CIRCLED KATAKANA SI
32DC CIRCLED KATAKANA SU
32DD CIRCLED KATAKANA SE
32DE CIRCLED KATAKANA SO
32DF CIRCLED KATAKANA TA
32E0 CIRCLED KATAKANA TI
32E1 CIRCLED KATAKANA TU
32E2 CIRCLED KATAKANA TE
32E3 CIRCLED KATAKANA TO
32E4 CIRCLED KATAKANA NA
32E5 CIRCLED KATAKANA NI
32E6 CIRCLED KATAKANA NU
32E7 CIRCLED KATAKANA NE
32E8 CIRCLED KATAKANA NO
32E9 CIRCLED KATAKANA HA
32EA CIRCLED KATAKANA HI
32EB CIRCLED KATAKANA HU
32EC CIRCLED KATAKANA HE
32ED CIRCLED KATAKANA HO
32EE CIRCLED KATAKANA MA
32EF CIRCLED KATAKANA MI
32F0 CIRCLED KATAKANA MU
32F1 CIRCLED KATAKANA ME
32F2 CIRCLED KATAKANA MO
32F3 CIRCLED KATAKANA YA
32F4 CIRCLED KATAKANA YU
32F5 CIRCLED KATAKANA YO
32F6 CIRCLED KATAKANA RA
32F7 CIRCLED KATAKANA RI
32F8 CIRCLED KATAKANA RU
32F9 CIRCLED KATAKANA RE
32FA CIRCLED KATAKANA RO
32FB CIRCLED KATAKANA WA
32FC CIRCLED KATAKANA WI
32FD CIRCLED KATAKANA WE
32FE CIRCLED KATAKANA WO
3300 SQUARE APAATO
3301 SQUARE ARUHUA
3302 SQUARE ANPEA
3303 SQUARE AARU
3304 SQUARE ININGU
3305 SQUARE INTI
3306 SQUARE UON
3307 SQUARE ESUKUUDO
3308 SQUARE EEKAA
3309 SQUARE ONSU
330A SQUARE OOMU
330B SQUARE KAIRI
330C SQUARE KARATTO
330D SQUARE KARORII
330E SQUARE GARON
330F SQUARE GANMA
3310 SQUARE GIGA
3311 SQUARE GINII
3312 SQUARE KYURII
3313 SQUARE GIRUDAA
3314 SQUARE KIRO
3315 SQUARE KIROGURAMU
3316 SQUARE KIROMEETORU
3317 SQUARE KIROWATTO
3318 SQUARE GURAMU
3319 SQUARE GURAMUTON
331A SQUARE KURUZEIRO
331B SQUARE KUROONE
331C SQUARE KEESU
331D SQUARE KORUNA
331E SQUARE KOOPO
331F SQUARE SAIKURU
3320 SQUARE SANTIIMU
3321 SQUARE SIRINGU
3322 SQUARE SENTI
3323 SQUARE SENTO
3324 SQUARE DAASU
3325 SQUARE DESI
3326 SQUARE DORU
3327 SQUARE TON
3328 SQUARE NANO
3329 SQUARE NOTTO
332A SQUARE HAITU
332B SQUARE PAASENTO
332C SQUARE PAATU
332D SQUARE BAARERU
332E SQUARE PIASUTORU
332F SQUARE PIKURU
3330 SQUARE PIKO
3331 SQUARE BIRU
3332 SQUARE HUARADDO
3333 SQUARE HUIITO
3334 SQUARE BUSSYERU
3335 SQUARE HURAN
3336 SQUARE HEKUTAARU
3337 SQUARE PESO
3338 SQUARE PENIHI
3339 SQUARE HERUTU
333A SQUARE PENSU
333B SQUARE PEEZI
333C SQUARE BEETA
333D SQUARE POINTO
333E SQUARE BORUTO
333F SQUARE HON
3340 SQUARE PONDO
3341 SQUARE HOORU
3342 SQUARE HOON
3343 SQUARE MAIKURO
3344 SQUARE MAIRU
3345 SQUARE MAHHA
3346 SQUARE MARUKU
3347 SQUARE MANSYON
3348 SQUARE MIKURON
3349 SQUARE MIRI
334A SQUARE MIRIBAARU
334B SQUARE MEGA
334C SQUARE MEGATON
334D SQUARE MEETORU
334E SQUARE YAADO
334F SQUARE YAARU
3350 SQUARE YUAN
3351 SQUARE RITTORU
3352 SQUARE RIRA
3353 SQUARE RUPII
3354 SQUARE RUUBURU
3355 SQUARE REMU
3356 SQUARE RENTOGEN
3357 SQUARE WATTO
3358 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ZERO
3359 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ONE
335A IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWO
335B IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THREE
335C IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOUR
335D IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIVE
335E IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIX
335F IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVEN
3360 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHT
3361 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINE
3362 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TEN
3363 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ELEVEN
3364 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWELVE
3365 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THIRTEEN
3366 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOURTEEN
3367 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIFTEEN
3368 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIXTEEN
3369 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVENTEEN
336A IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHTEEN
336B IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINETEEN
336C IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY
336D IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-ONE
336E IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-TWO
336F IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-THREE
3370 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-FOUR
3371 SQUARE HPA
3372 SQUARE DA
3373 SQUARE AU
3374 SQUARE BAR
3375 SQUARE OV
3376 SQUARE PC
337B SQUARE ERA NAME HEISEI
337C SQUARE ERA NAME SYOUWA
337D SQUARE ERA NAME TAISYOU
337E SQUARE ERA NAME MEIZI
337F SQUARE CORPORATION
3380 SQUARE PA AMPS
3381 SQUARE NA
3382 SQUARE MU A
3383 SQUARE MA
3384 SQUARE KA
3385 SQUARE KB
3386 SQUARE MB
3387 SQUARE GB
3388 SQUARE CAL
3389 SQUARE KCAL
338A SQUARE PF
338B SQUARE NF
338C SQUARE MU F
338D SQUARE MU G
338E SQUARE MG
338F SQUARE KG
3390 SQUARE HZ
3391 SQUARE KHZ
3392 SQUARE MHZ
3393 SQUARE GHZ
3394 SQUARE THZ
3395 SQUARE MU L
3396 SQUARE ML
3397 SQUARE DL
3398 SQUARE KL
3399 SQUARE FM
339A SQUARE NM
339B SQUARE MU M
339C SQUARE MM
339D SQUARE CM
339E SQUARE KM
339F SQUARE MM SQUARED
33A0 SQUARE CM SQUARED
33A1 SQUARE M SQUARED
33A2 SQUARE KM SQUARED
33A3 SQUARE MM CUBED
33A4 SQUARE CM CUBED
33A5 SQUARE M CUBED
33A6 SQUARE KM CUBED
33A7 SQUARE M OVER S
33A8 SQUARE M OVER S SQUARED
33A9 SQUARE PA
33AA SQUARE KPA
33AB SQUARE MPA
33AC SQUARE GPA
33AD SQUARE RAD
33AE SQUARE RAD OVER S
33AF SQUARE RAD OVER S SQUARED
33B0 SQUARE PS
33B1 SQUARE NS
33B2 SQUARE MU S
33B3 SQUARE MS
33B4 SQUARE PV
33B5 SQUARE NV
33B6 SQUARE MU V
33B7 SQUARE MV
33B8 SQUARE KV
33B9 SQUARE MV MEGA
33BA SQUARE PW
33BB SQUARE NW
33BC SQUARE MU W
33BD SQUARE MW
33BE SQUARE KW
33BF SQUARE MW MEGA
33C0 SQUARE K OHM
33C1 SQUARE M OHM
33C2 SQUARE AM
33C3 SQUARE BQ
33C4 SQUARE CC
33C5 SQUARE CD
33C6 SQUARE C OVER KG
33C7 SQUARE CO
33C8 SQUARE DB
33C9 SQUARE GY
33CA SQUARE HA
33CB SQUARE HP
33CC SQUARE IN
33CD SQUARE KK
33CE SQUARE KM CAPITAL
33CF SQUARE KT
33D0 SQUARE LM
33D1 SQUARE LN
33D2 SQUARE LOG
33D3 SQUARE LX
33D4 SQUARE MB SMALL
33D5 SQUARE MIL
33D6 SQUARE MOL
33D7 SQUARE PH
33D8 SQUARE PM
33D9 SQUARE PPM
33DA SQUARE PR
33DB SQUARE SR
33DC SQUARE SV
33DD SQUARE WB
33E0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ONE
33E1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWO
33E2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THREE
33E3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOUR
33E4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIVE
33E5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIX
33E6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVEN
33E7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHT
33E8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINE
33E9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TEN
33EA IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ELEVEN
33EB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWELVE
33EC IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTEEN
33ED IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOURTEEN
33EE IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIFTEEN
33EF IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIXTEEN
33F0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVENTEEN
33F1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHTEEN
33F2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINETEEN
33F3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY
33F4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-ONE
33F5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-TWO
33F6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-THREE
33F7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FOUR
33F8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FIVE
33F9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SIX
33FA IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SEVEN
33FB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-EIGHT
33FC IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-NINE
33FD IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY
33FE IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY-ONE
4E00 <CJK Ideograph, First>
9FA5 <CJK Ideograph, Last>
AC00 <Hangul Syllable, First>
D7A3 <Hangul Syllable, Last>
D800 <Unassigned High Surrogate, First>
DB7F <Unassigned High Surrogate, Last>
DB80 <Private Use High Surrogate, First>
DBFF <Private Use High Surrogate, Last>
DC00 <Low Surrogate, First>
DFFF <Low Surrogate, Last>
E000 <Private Use, First>
F8FF <Private Use, Last>
F900 <CJK Compatibility Ideograph, First>
FA2D <CJK Compatibility Ideograph, Last>
FB00 LATIN SMALL LIGATURE FF
FB01 LATIN SMALL LIGATURE FI
FB02 LATIN SMALL LIGATURE FL
FB03 LATIN SMALL LIGATURE FFI
FB04 LATIN SMALL LIGATURE FFL
FB05 LATIN SMALL LIGATURE LONG S T
FB06 LATIN SMALL LIGATURE ST
FB13 ARMENIAN SMALL LIGATURE MEN NOW
FB14 ARMENIAN SMALL LIGATURE MEN ECH
FB15 ARMENIAN SMALL LIGATURE MEN INI
FB16 ARMENIAN SMALL LIGATURE VEW NOW
FB17 ARMENIAN SMALL LIGATURE MEN XEH
FB1E HEBREW POINT JUDEO-SPANISH VARIKA
FB1F HEBREW LIGATURE YIDDISH YOD YOD PATAH
FB20 HEBREW LETTER ALTERNATIVE AYIN
FB21 HEBREW LETTER WIDE ALEF
FB22 HEBREW LETTER WIDE DALET
FB23 HEBREW LETTER WIDE HE
FB24 HEBREW LETTER WIDE KAF
FB25 HEBREW LETTER WIDE LAMED
FB26 HEBREW LETTER WIDE FINAL MEM
FB27 HEBREW LETTER WIDE RESH
FB28 HEBREW LETTER WIDE TAV
FB29 HEBREW LETTER ALTERNATIVE PLUS SIGN
FB2A HEBREW LETTER SHIN WITH SHIN DOT
FB2B HEBREW LETTER SHIN WITH SIN DOT
FB2C HEBREW LETTER SHIN WITH DAGESH AND SHIN DOT
FB2D HEBREW LETTER SHIN WITH DAGESH AND SIN DOT
FB2E HEBREW LETTER ALEF WITH PATAH
FB2F HEBREW LETTER ALEF WITH QAMATS
FB30 HEBREW LETTER ALEF WITH MAPIQ
FB31 HEBREW LETTER BET WITH DAGESH
FB32 HEBREW LETTER GIMEL WITH DAGESH
FB33 HEBREW LETTER DALET WITH DAGESH
FB34 HEBREW LETTER HE WITH MAPIQ
FB35 HEBREW LETTER VAV WITH DAGESH
FB36 HEBREW LETTER ZAYIN WITH DAGESH
FB38 HEBREW LETTER TET WITH DAGESH
FB39 HEBREW LETTER YOD WITH DAGESH
FB3A HEBREW LETTER FINAL KAF WITH DAGESH
FB3B HEBREW LETTER KAF WITH DAGESH
FB3C HEBREW LETTER LAMED WITH DAGESH
FB3E HEBREW LETTER MEM WITH DAGESH
FB40 HEBREW LETTER NUN WITH DAGESH
FB41 HEBREW LETTER SAMEKH WITH DAGESH
FB43 HEBREW LETTER FINAL PE WITH DAGESH
FB44 HEBREW LETTER PE WITH DAGESH
FB46 HEBREW LETTER TSADI WITH DAGESH
FB47 HEBREW LETTER QOF WITH DAGESH
FB48 HEBREW LETTER RESH WITH DAGESH
FB49 HEBREW LETTER SHIN WITH DAGESH
FB4A HEBREW LETTER TAV WITH DAGESH
FB4B HEBREW LETTER VAV WITH HOLAM
FB4C HEBREW LETTER BET WITH RAFE
FB4D HEBREW LETTER KAF WITH RAFE
FB4E HEBREW LETTER PE WITH RAFE
FB4F HEBREW LIGATURE ALEF LAMED
FB50 ARABIC LETTER ALEF WASLA ISOLATED FORM
FB51 ARABIC LETTER ALEF WASLA FINAL FORM
FB52 ARABIC LETTER BEEH ISOLATED FORM
FB53 ARABIC LETTER BEEH FINAL FORM
FB54 ARABIC LETTER BEEH INITIAL FORM
FB55 ARABIC LETTER BEEH MEDIAL FORM
FB56 ARABIC LETTER PEH ISOLATED FORM
FB57 ARABIC LETTER PEH FINAL FORM
FB58 ARABIC LETTER PEH INITIAL FORM
FB59 ARABIC LETTER PEH MEDIAL FORM
FB5A ARABIC LETTER BEHEH ISOLATED FORM
FB5B ARABIC LETTER BEHEH FINAL FORM
FB5C ARABIC LETTER BEHEH INITIAL FORM
FB5D ARABIC LETTER BEHEH MEDIAL FORM
FB5E ARABIC LETTER TTEHEH ISOLATED FORM
FB5F ARABIC LETTER TTEHEH FINAL FORM
FB60 ARABIC LETTER TTEHEH INITIAL FORM
FB61 ARABIC LETTER TTEHEH MEDIAL FORM
FB62 ARABIC LETTER TEHEH ISOLATED FORM
FB63 ARABIC LETTER TEHEH FINAL FORM
FB64 ARABIC LETTER TEHEH INITIAL FORM
FB65 ARABIC LETTER TEHEH MEDIAL FORM
FB66 ARABIC LETTER TTEH ISOLATED FORM
FB67 ARABIC LETTER TTEH FINAL FORM
FB68 ARABIC LETTER TTEH INITIAL FORM
FB69 ARABIC LETTER TTEH MEDIAL FORM
FB6A ARABIC LETTER VEH ISOLATED FORM
FB6B ARABIC LETTER VEH FINAL FORM
FB6C ARABIC LETTER VEH INITIAL FORM
FB6D ARABIC LETTER VEH MEDIAL FORM
FB6E ARABIC LETTER PEHEH ISOLATED FORM
FB6F ARABIC LETTER PEHEH FINAL FORM
FB70 ARABIC LETTER PEHEH INITIAL FORM
FB71 ARABIC LETTER PEHEH MEDIAL FORM
FB72 ARABIC LETTER DYEH ISOLATED FORM
FB73 ARABIC LETTER DYEH FINAL FORM
FB74 ARABIC LETTER DYEH INITIAL FORM
FB75 ARABIC LETTER DYEH MEDIAL FORM
FB76 ARABIC LETTER NYEH ISOLATED FORM
FB77 ARABIC LETTER NYEH FINAL FORM
FB78 ARABIC LETTER NYEH INITIAL FORM
FB79 ARABIC LETTER NYEH MEDIAL FORM
FB7A ARABIC LETTER TCHEH ISOLATED FORM
FB7B ARABIC LETTER TCHEH FINAL FORM
FB7C ARABIC LETTER TCHEH INITIAL FORM
FB7D ARABIC LETTER TCHEH MEDIAL FORM
FB7E ARABIC LETTER TCHEHEH ISOLATED FORM
FB7F ARABIC LETTER TCHEHEH FINAL FORM
FB80 ARABIC LETTER TCHEHEH INITIAL FORM
FB81 ARABIC LETTER TCHEHEH MEDIAL FORM
FB82 ARABIC LETTER DDAHAL ISOLATED FORM
FB83 ARABIC LETTER DDAHAL FINAL FORM
FB84 ARABIC LETTER DAHAL ISOLATED FORM
FB85 ARABIC LETTER DAHAL FINAL FORM
FB86 ARABIC LETTER DUL ISOLATED FORM
FB87 ARABIC LETTER DUL FINAL FORM
FB88 ARABIC LETTER DDAL ISOLATED FORM
FB89 ARABIC LETTER DDAL FINAL FORM
FB8A ARABIC LETTER JEH ISOLATED FORM
FB8B ARABIC LETTER JEH FINAL FORM
FB8C ARABIC LETTER RREH ISOLATED FORM
FB8D ARABIC LETTER RREH FINAL FORM
FB8E ARABIC LETTER KEHEH ISOLATED FORM
FB8F ARABIC LETTER KEHEH FINAL FORM
FB90 ARABIC LETTER KEHEH INITIAL FORM
FB91 ARABIC LETTER KEHEH MEDIAL FORM
FB92 ARABIC LETTER GAF ISOLATED FORM
FB93 ARABIC LETTER GAF FINAL FORM
FB94 ARABIC LETTER GAF INITIAL FORM
FB95 ARABIC LETTER GAF MEDIAL FORM
FB96 ARABIC LETTER GUEH ISOLATED FORM
FB97 ARABIC LETTER GUEH FINAL FORM
FB98 ARABIC LETTER GUEH INITIAL FORM
FB99 ARABIC LETTER GUEH MEDIAL FORM
FB9A ARABIC LETTER NGOEH ISOLATED FORM
FB9B ARABIC LETTER NGOEH FINAL FORM
FB9C ARABIC LETTER NGOEH INITIAL FORM
FB9D ARABIC LETTER NGOEH MEDIAL FORM
FB9E ARABIC LETTER NOON GHUNNA ISOLATED FORM
FB9F ARABIC LETTER NOON GHUNNA FINAL FORM
FBA0 ARABIC LETTER RNOON ISOLATED FORM
FBA1 ARABIC LETTER RNOON FINAL FORM
FBA2 ARABIC LETTER RNOON INITIAL FORM
FBA3 ARABIC LETTER RNOON MEDIAL FORM
FBA4 ARABIC LETTER HEH WITH YEH ABOVE ISOLATED FORM
FBA5 ARABIC LETTER HEH WITH YEH ABOVE FINAL FORM
FBA6 ARABIC LETTER HEH GOAL ISOLATED FORM
FBA7 ARABIC LETTER HEH GOAL FINAL FORM
FBA8 ARABIC LETTER HEH GOAL INITIAL FORM
FBA9 ARABIC LETTER HEH GOAL MEDIAL FORM
FBAA ARABIC LETTER HEH DOACHASHMEE ISOLATED FORM
FBAB ARABIC LETTER HEH DOACHASHMEE FINAL FORM
FBAC ARABIC LETTER HEH DOACHASHMEE INITIAL FORM
FBAD ARABIC LETTER HEH DOACHASHMEE MEDIAL FORM
FBAE ARABIC LETTER YEH BARREE ISOLATED FORM
FBAF ARABIC LETTER YEH BARREE FINAL FORM
FBB0 ARABIC LETTER YEH BARREE WITH HAMZA ABOVE ISOLATED FORM
FBB1 ARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORM
FBD3 ARABIC LETTER NG ISOLATED FORM
FBD4 ARABIC LETTER NG FINAL FORM
FBD5 ARABIC LETTER NG INITIAL FORM
FBD6 ARABIC LETTER NG MEDIAL FORM
FBD7 ARABIC LETTER U ISOLATED FORM
FBD8 ARABIC LETTER U FINAL FORM
FBD9 ARABIC LETTER OE ISOLATED FORM
FBDA ARABIC LETTER OE FINAL FORM
FBDB ARABIC LETTER YU ISOLATED FORM
FBDC ARABIC LETTER YU FINAL FORM
FBDD ARABIC LETTER U WITH HAMZA ABOVE ISOLATED FORM
FBDE ARABIC LETTER VE ISOLATED FORM
FBDF ARABIC LETTER VE FINAL FORM
FBE0 ARABIC LETTER KIRGHIZ OE ISOLATED FORM
FBE1 ARABIC LETTER KIRGHIZ OE FINAL FORM
FBE2 ARABIC LETTER KIRGHIZ YU ISOLATED FORM
FBE3 ARABIC LETTER KIRGHIZ YU FINAL FORM
FBE4 ARABIC LETTER E ISOLATED FORM
FBE5 ARABIC LETTER E FINAL FORM
FBE6 ARABIC LETTER E INITIAL FORM
FBE7 ARABIC LETTER E MEDIAL FORM
FBE8 ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA INITIAL FORM
FBE9 ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA MEDIAL FORM
FBEA ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF ISOLATED FORM
FBEB ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF FINAL FORM
FBEC ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE ISOLATED FORM
FBED ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE FINAL FORM
FBEE ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW ISOLATED FORM
FBEF ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW FINAL FORM
FBF0 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U ISOLATED FORM
FBF1 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U FINAL FORM
FBF2 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE ISOLATED FORM
FBF3 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE FINAL FORM
FBF4 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU ISOLATED FORM
FBF5 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU FINAL FORM
FBF6 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E ISOLATED FORM
FBF7 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E FINAL FORM
FBF8 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E INITIAL FORM
FBF9 ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORM
FBFA ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FINAL FORM
FBFB ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA INITIAL FORM
FBFC ARABIC LETTER FARSI YEH ISOLATED FORM
FBFD ARABIC LETTER FARSI YEH FINAL FORM
FBFE ARABIC LETTER FARSI YEH INITIAL FORM
FBFF ARABIC LETTER FARSI YEH MEDIAL FORM
FC00 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM ISOLATED FORM
FC01 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH ISOLATED FORM
FC02 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM ISOLATED FORM
FC03 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORM
FC04 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YEH ISOLATED FORM
FC05 ARABIC LIGATURE BEH WITH JEEM ISOLATED FORM
FC06 ARABIC LIGATURE BEH WITH HAH ISOLATED FORM
FC07 ARABIC LIGATURE BEH WITH KHAH ISOLATED FORM
FC08 ARABIC LIGATURE BEH WITH MEEM ISOLATED FORM
FC09 ARABIC LIGATURE BEH WITH ALEF MAKSURA ISOLATED FORM
FC0A ARABIC LIGATURE BEH WITH YEH ISOLATED FORM
FC0B ARABIC LIGATURE TEH WITH JEEM ISOLATED FORM
FC0C ARABIC LIGATURE TEH WITH HAH ISOLATED FORM
FC0D ARABIC LIGATURE TEH WITH KHAH ISOLATED FORM
FC0E ARABIC LIGATURE TEH WITH MEEM ISOLATED FORM
FC0F ARABIC LIGATURE TEH WITH ALEF MAKSURA ISOLATED FORM
FC10 ARABIC LIGATURE TEH WITH YEH ISOLATED FORM
FC11 ARABIC LIGATURE THEH WITH JEEM ISOLATED FORM
FC12 ARABIC LIGATURE THEH WITH MEEM ISOLATED FORM
FC13 ARABIC LIGATURE THEH WITH ALEF MAKSURA ISOLATED FORM
FC14 ARABIC LIGATURE THEH WITH YEH ISOLATED FORM
FC15 ARABIC LIGATURE JEEM WITH HAH ISOLATED FORM
FC16 ARABIC LIGATURE JEEM WITH MEEM ISOLATED FORM
FC17 ARABIC LIGATURE HAH WITH JEEM ISOLATED FORM
FC18 ARABIC LIGATURE HAH WITH MEEM ISOLATED FORM
FC19 ARABIC LIGATURE KHAH WITH JEEM ISOLATED FORM
FC1A ARABIC LIGATURE KHAH WITH HAH ISOLATED FORM
FC1B ARABIC LIGATURE KHAH WITH MEEM ISOLATED FORM
FC1C ARABIC LIGATURE SEEN WITH JEEM ISOLATED FORM
FC1D ARABIC LIGATURE SEEN WITH HAH ISOLATED FORM
FC1E ARABIC LIGATURE SEEN WITH KHAH ISOLATED FORM
FC1F ARABIC LIGATURE SEEN WITH MEEM ISOLATED FORM
FC20 ARABIC LIGATURE SAD WITH HAH ISOLATED FORM
FC21 ARABIC LIGATURE SAD WITH MEEM ISOLATED FORM
FC22 ARABIC LIGATURE DAD WITH JEEM ISOLATED FORM
FC23 ARABIC LIGATURE DAD WITH HAH ISOLATED FORM
FC24 ARABIC LIGATURE DAD WITH KHAH ISOLATED FORM
FC25 ARABIC LIGATURE DAD WITH MEEM ISOLATED FORM
FC26 ARABIC LIGATURE TAH WITH HAH ISOLATED FORM
FC27 ARABIC LIGATURE TAH WITH MEEM ISOLATED FORM
FC28 ARABIC LIGATURE ZAH WITH MEEM ISOLATED FORM
FC29 ARABIC LIGATURE AIN WITH JEEM ISOLATED FORM
FC2A ARABIC LIGATURE AIN WITH MEEM ISOLATED FORM
FC2B ARABIC LIGATURE GHAIN WITH JEEM ISOLATED FORM
FC2C ARABIC LIGATURE GHAIN WITH MEEM ISOLATED FORM
FC2D ARABIC LIGATURE FEH WITH JEEM ISOLATED FORM
FC2E ARABIC LIGATURE FEH WITH HAH ISOLATED FORM
FC2F ARABIC LIGATURE FEH WITH KHAH ISOLATED FORM
FC30 ARABIC LIGATURE FEH WITH MEEM ISOLATED FORM
FC31 ARABIC LIGATURE FEH WITH ALEF MAKSURA ISOLATED FORM
FC32 ARABIC LIGATURE FEH WITH YEH ISOLATED FORM
FC33 ARABIC LIGATURE QAF WITH HAH ISOLATED FORM
FC34 ARABIC LIGATURE QAF WITH MEEM ISOLATED FORM
FC35 ARABIC LIGATURE QAF WITH ALEF MAKSURA ISOLATED FORM
FC36 ARABIC LIGATURE QAF WITH YEH ISOLATED FORM
FC37 ARABIC LIGATURE KAF WITH ALEF ISOLATED FORM
FC38 ARABIC LIGATURE KAF WITH JEEM ISOLATED FORM
FC39 ARABIC LIGATURE KAF WITH HAH ISOLATED FORM
FC3A ARABIC LIGATURE KAF WITH KHAH ISOLATED FORM
FC3B ARABIC LIGATURE KAF WITH LAM ISOLATED FORM
FC3C ARABIC LIGATURE KAF WITH MEEM ISOLATED FORM
FC3D ARABIC LIGATURE KAF WITH ALEF MAKSURA ISOLATED FORM
FC3E ARABIC LIGATURE KAF WITH YEH ISOLATED FORM
FC3F ARABIC LIGATURE LAM WITH JEEM ISOLATED FORM
FC40 ARABIC LIGATURE LAM WITH HAH ISOLATED FORM
FC41 ARABIC LIGATURE LAM WITH KHAH ISOLATED FORM
FC42 ARABIC LIGATURE LAM WITH MEEM ISOLATED FORM
FC43 ARABIC LIGATURE LAM WITH ALEF MAKSURA ISOLATED FORM
FC44 ARABIC LIGATURE LAM WITH YEH ISOLATED FORM
FC45 ARABIC LIGATURE MEEM WITH JEEM ISOLATED FORM
FC46 ARABIC LIGATURE MEEM WITH HAH ISOLATED FORM
FC47 ARABIC LIGATURE MEEM WITH KHAH ISOLATED FORM
FC48 ARABIC LIGATURE MEEM WITH MEEM ISOLATED FORM
FC49 ARABIC LIGATURE MEEM WITH ALEF MAKSURA ISOLATED FORM
FC4A ARABIC LIGATURE MEEM WITH YEH ISOLATED FORM
FC4B ARABIC LIGATURE NOON WITH JEEM ISOLATED FORM
FC4C ARABIC LIGATURE NOON WITH HAH ISOLATED FORM
FC4D ARABIC LIGATURE NOON WITH KHAH ISOLATED FORM
FC4E ARABIC LIGATURE NOON WITH MEEM ISOLATED FORM
FC4F ARABIC LIGATURE NOON WITH ALEF MAKSURA ISOLATED FORM
FC50 ARABIC LIGATURE NOON WITH YEH ISOLATED FORM
FC51 ARABIC LIGATURE HEH WITH JEEM ISOLATED FORM
FC52 ARABIC LIGATURE HEH WITH MEEM ISOLATED FORM
FC53 ARABIC LIGATURE HEH WITH ALEF MAKSURA ISOLATED FORM
FC54 ARABIC LIGATURE HEH WITH YEH ISOLATED FORM
FC55 ARABIC LIGATURE YEH WITH JEEM ISOLATED FORM
FC56 ARABIC LIGATURE YEH WITH HAH ISOLATED FORM
FC57 ARABIC LIGATURE YEH WITH KHAH ISOLATED FORM
FC58 ARABIC LIGATURE YEH WITH MEEM ISOLATED FORM
FC59 ARABIC LIGATURE YEH WITH ALEF MAKSURA ISOLATED FORM
FC5A ARABIC LIGATURE YEH WITH YEH ISOLATED FORM
FC5B ARABIC LIGATURE THAL WITH SUPERSCRIPT ALEF ISOLATED FORM
FC5C ARABIC LIGATURE REH WITH SUPERSCRIPT ALEF ISOLATED FORM
FC5D ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF ISOLATED FORM
FC5E ARABIC LIGATURE SHADDA WITH DAMMATAN ISOLATED FORM
FC5F ARABIC LIGATURE SHADDA WITH KASRATAN ISOLATED FORM
FC60 ARABIC LIGATURE SHADDA WITH FATHA ISOLATED FORM
FC61 ARABIC LIGATURE SHADDA WITH DAMMA ISOLATED FORM
FC62 ARABIC LIGATURE SHADDA WITH KASRA ISOLATED FORM
FC63 ARABIC LIGATURE SHADDA WITH SUPERSCRIPT ALEF ISOLATED FORM
FC64 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH REH FINAL FORM
FC65 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ZAIN FINAL FORM
FC66 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM FINAL FORM
FC67 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH NOON FINAL FORM
FC68 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FINAL FORM
FC69 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YEH FINAL FORM
FC6A ARABIC LIGATURE BEH WITH REH FINAL FORM
FC6B ARABIC LIGATURE BEH WITH ZAIN FINAL FORM
FC6C ARABIC LIGATURE BEH WITH MEEM FINAL FORM
FC6D ARABIC LIGATURE BEH WITH NOON FINAL FORM
FC6E ARABIC LIGATURE BEH WITH ALEF MAKSURA FINAL FORM
FC6F ARABIC LIGATURE BEH WITH YEH FINAL FORM
FC70 ARABIC LIGATURE TEH WITH REH FINAL FORM
FC71 ARABIC LIGATURE TEH WITH ZAIN FINAL FORM
FC72 ARABIC LIGATURE TEH WITH MEEM FINAL FORM
FC73 ARABIC LIGATURE TEH WITH NOON FINAL FORM
FC74 ARABIC LIGATURE TEH WITH ALEF MAKSURA FINAL FORM
FC75 ARABIC LIGATURE TEH WITH YEH FINAL FORM
FC76 ARABIC LIGATURE THEH WITH REH FINAL FORM
FC77 ARABIC LIGATURE THEH WITH ZAIN FINAL FORM
FC78 ARABIC LIGATURE THEH WITH MEEM FINAL FORM
FC79 ARABIC LIGATURE THEH WITH NOON FINAL FORM
FC7A ARABIC LIGATURE THEH WITH ALEF MAKSURA FINAL FORM
FC7B ARABIC LIGATURE THEH WITH YEH FINAL FORM
FC7C ARABIC LIGATURE FEH WITH ALEF MAKSURA FINAL FORM
FC7D ARABIC LIGATURE FEH WITH YEH FINAL FORM
FC7E ARABIC LIGATURE QAF WITH ALEF MAKSURA FINAL FORM
FC7F ARABIC LIGATURE QAF WITH YEH FINAL FORM
FC80 ARABIC LIGATURE KAF WITH ALEF FINAL FORM
FC81 ARABIC LIGATURE KAF WITH LAM FINAL FORM
FC82 ARABIC LIGATURE KAF WITH MEEM FINAL FORM
FC83 ARABIC LIGATURE KAF WITH ALEF MAKSURA FINAL FORM
FC84 ARABIC LIGATURE KAF WITH YEH FINAL FORM
FC85 ARABIC LIGATURE LAM WITH MEEM FINAL FORM
FC86 ARABIC LIGATURE LAM WITH ALEF MAKSURA FINAL FORM
FC87 ARABIC LIGATURE LAM WITH YEH FINAL FORM
FC88 ARABIC LIGATURE MEEM WITH ALEF FINAL FORM
FC89 ARABIC LIGATURE MEEM WITH MEEM FINAL FORM
FC8A ARABIC LIGATURE NOON WITH REH FINAL FORM
FC8B ARABIC LIGATURE NOON WITH ZAIN FINAL FORM
FC8C ARABIC LIGATURE NOON WITH MEEM FINAL FORM
FC8D ARABIC LIGATURE NOON WITH NOON FINAL FORM
FC8E ARABIC LIGATURE NOON WITH ALEF MAKSURA FINAL FORM
FC8F ARABIC LIGATURE NOON WITH YEH FINAL FORM
FC90 ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF FINAL FORM
FC91 ARABIC LIGATURE YEH WITH REH FINAL FORM
FC92 ARABIC LIGATURE YEH WITH ZAIN FINAL FORM
FC93 ARABIC LIGATURE YEH WITH MEEM FINAL FORM
FC94 ARABIC LIGATURE YEH WITH NOON FINAL FORM
FC95 ARABIC LIGATURE YEH WITH ALEF MAKSURA FINAL FORM
FC96 ARABIC LIGATURE YEH WITH YEH FINAL FORM
FC97 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM INITIAL FORM
FC98 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH INITIAL FORM
FC99 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH KHAH INITIAL FORM
FC9A ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM INITIAL FORM
FC9B ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH INITIAL FORM
FC9C ARABIC LIGATURE BEH WITH JEEM INITIAL FORM
FC9D ARABIC LIGATURE BEH WITH HAH INITIAL FORM
FC9E ARABIC LIGATURE BEH WITH KHAH INITIAL FORM
FC9F ARABIC LIGATURE BEH WITH MEEM INITIAL FORM
FCA0 ARABIC LIGATURE BEH WITH HEH INITIAL FORM
FCA1 ARABIC LIGATURE TEH WITH JEEM INITIAL FORM
FCA2 ARABIC LIGATURE TEH WITH HAH INITIAL FORM
FCA3 ARABIC LIGATURE TEH WITH KHAH INITIAL FORM
FCA4 ARABIC LIGATURE TEH WITH MEEM INITIAL FORM
FCA5 ARABIC LIGATURE TEH WITH HEH INITIAL FORM
FCA6 ARABIC LIGATURE THEH WITH MEEM INITIAL FORM
FCA7 ARABIC LIGATURE JEEM WITH HAH INITIAL FORM
FCA8 ARABIC LIGATURE JEEM WITH MEEM INITIAL FORM
FCA9 ARABIC LIGATURE HAH WITH JEEM INITIAL FORM
FCAA ARABIC LIGATURE HAH WITH MEEM INITIAL FORM
FCAB ARABIC LIGATURE KHAH WITH JEEM INITIAL FORM
FCAC ARABIC LIGATURE KHAH WITH MEEM INITIAL FORM
FCAD ARABIC LIGATURE SEEN WITH JEEM INITIAL FORM
FCAE ARABIC LIGATURE SEEN WITH HAH INITIAL FORM
FCAF ARABIC LIGATURE SEEN WITH KHAH INITIAL FORM
FCB0 ARABIC LIGATURE SEEN WITH MEEM INITIAL FORM
FCB1 ARABIC LIGATURE SAD WITH HAH INITIAL FORM
FCB2 ARABIC LIGATURE SAD WITH KHAH INITIAL FORM
FCB3 ARABIC LIGATURE SAD WITH MEEM INITIAL FORM
FCB4 ARABIC LIGATURE DAD WITH JEEM INITIAL FORM
FCB5 ARABIC LIGATURE DAD WITH HAH INITIAL FORM
FCB6 ARABIC LIGATURE DAD WITH KHAH INITIAL FORM
FCB7 ARABIC LIGATURE DAD WITH MEEM INITIAL FORM
FCB8 ARABIC LIGATURE TAH WITH HAH INITIAL FORM
FCB9 ARABIC LIGATURE ZAH WITH MEEM INITIAL FORM
FCBA ARABIC LIGATURE AIN WITH JEEM INITIAL FORM
FCBB ARABIC LIGATURE AIN WITH MEEM INITIAL FORM
FCBC ARABIC LIGATURE GHAIN WITH JEEM INITIAL FORM
FCBD ARABIC LIGATURE GHAIN WITH MEEM INITIAL FORM
FCBE ARABIC LIGATURE FEH WITH JEEM INITIAL FORM
FCBF ARABIC LIGATURE FEH WITH HAH INITIAL FORM
FCC0 ARABIC LIGATURE FEH WITH KHAH INITIAL FORM
FCC1 ARABIC LIGATURE FEH WITH MEEM INITIAL FORM
FCC2 ARABIC LIGATURE QAF WITH HAH INITIAL FORM
FCC3 ARABIC LIGATURE QAF WITH MEEM INITIAL FORM
FCC4 ARABIC LIGATURE KAF WITH JEEM INITIAL FORM
FCC5 ARABIC LIGATURE KAF WITH HAH INITIAL FORM
FCC6 ARABIC LIGATURE KAF WITH KHAH INITIAL FORM
FCC7 ARABIC LIGATURE KAF WITH LAM INITIAL FORM
FCC8 ARABIC LIGATURE KAF WITH MEEM INITIAL FORM
FCC9 ARABIC LIGATURE LAM WITH JEEM INITIAL FORM
FCCA ARABIC LIGATURE LAM WITH HAH INITIAL FORM
FCCB ARABIC LIGATURE LAM WITH KHAH INITIAL FORM
FCCC ARABIC LIGATURE LAM WITH MEEM INITIAL FORM
FCCD ARABIC LIGATURE LAM WITH HEH INITIAL FORM
FCCE ARABIC LIGATURE MEEM WITH JEEM INITIAL FORM
FCCF ARABIC LIGATURE MEEM WITH HAH INITIAL FORM
FCD0 ARABIC LIGATURE MEEM WITH KHAH INITIAL FORM
FCD1 ARABIC LIGATURE MEEM WITH MEEM INITIAL FORM
FCD2 ARABIC LIGATURE NOON WITH JEEM INITIAL FORM
FCD3 ARABIC LIGATURE NOON WITH HAH INITIAL FORM
FCD4 ARABIC LIGATURE NOON WITH KHAH INITIAL FORM
FCD5 ARABIC LIGATURE NOON WITH MEEM INITIAL FORM
FCD6 ARABIC LIGATURE NOON WITH HEH INITIAL FORM
FCD7 ARABIC LIGATURE HEH WITH JEEM INITIAL FORM
FCD8 ARABIC LIGATURE HEH WITH MEEM INITIAL FORM
FCD9 ARABIC LIGATURE HEH WITH SUPERSCRIPT ALEF INITIAL FORM
FCDA ARABIC LIGATURE YEH WITH JEEM INITIAL FORM
FCDB ARABIC LIGATURE YEH WITH HAH INITIAL FORM
FCDC ARABIC LIGATURE YEH WITH KHAH INITIAL FORM
FCDD ARABIC LIGATURE YEH WITH MEEM INITIAL FORM
FCDE ARABIC LIGATURE YEH WITH HEH INITIAL FORM
FCDF ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM MEDIAL FORM
FCE0 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH MEDIAL FORM
FCE1 ARABIC LIGATURE BEH WITH MEEM MEDIAL FORM
FCE2 ARABIC LIGATURE BEH WITH HEH MEDIAL FORM
FCE3 ARABIC LIGATURE TEH WITH MEEM MEDIAL FORM
FCE4 ARABIC LIGATURE TEH WITH HEH MEDIAL FORM
FCE5 ARABIC LIGATURE THEH WITH MEEM MEDIAL FORM
FCE6 ARABIC LIGATURE THEH WITH HEH MEDIAL FORM
FCE7 ARABIC LIGATURE SEEN WITH MEEM MEDIAL FORM
FCE8 ARABIC LIGATURE SEEN WITH HEH MEDIAL FORM
FCE9 ARABIC LIGATURE SHEEN WITH MEEM MEDIAL FORM
FCEA ARABIC LIGATURE SHEEN WITH HEH MEDIAL FORM
FCEB ARABIC LIGATURE KAF WITH LAM MEDIAL FORM
FCEC ARABIC LIGATURE KAF WITH MEEM MEDIAL FORM
FCED ARABIC LIGATURE LAM WITH MEEM MEDIAL FORM
FCEE ARABIC LIGATURE NOON WITH MEEM MEDIAL FORM
FCEF ARABIC LIGATURE NOON WITH HEH MEDIAL FORM
FCF0 ARABIC LIGATURE YEH WITH MEEM MEDIAL FORM
FCF1 ARABIC LIGATURE YEH WITH HEH MEDIAL FORM
FCF2 ARABIC LIGATURE SHADDA WITH FATHA MEDIAL FORM
FCF3 ARABIC LIGATURE SHADDA WITH DAMMA MEDIAL FORM
FCF4 ARABIC LIGATURE SHADDA WITH KASRA MEDIAL FORM
FCF5 ARABIC LIGATURE TAH WITH ALEF MAKSURA ISOLATED FORM
FCF6 ARABIC LIGATURE TAH WITH YEH ISOLATED FORM
FCF7 ARABIC LIGATURE AIN WITH ALEF MAKSURA ISOLATED FORM
FCF8 ARABIC LIGATURE AIN WITH YEH ISOLATED FORM
FCF9 ARABIC LIGATURE GHAIN WITH ALEF MAKSURA ISOLATED FORM
FCFA ARABIC LIGATURE GHAIN WITH YEH ISOLATED FORM
FCFB ARABIC LIGATURE SEEN WITH ALEF MAKSURA ISOLATED FORM
FCFC ARABIC LIGATURE SEEN WITH YEH ISOLATED FORM
FCFD ARABIC LIGATURE SHEEN WITH ALEF MAKSURA ISOLATED FORM
FCFE ARABIC LIGATURE SHEEN WITH YEH ISOLATED FORM
FCFF ARABIC LIGATURE HAH WITH ALEF MAKSURA ISOLATED FORM
FD00 ARABIC LIGATURE HAH WITH YEH ISOLATED FORM
FD01 ARABIC LIGATURE JEEM WITH ALEF MAKSURA ISOLATED FORM
FD02 ARABIC LIGATURE JEEM WITH YEH ISOLATED FORM
FD03 ARABIC LIGATURE KHAH WITH ALEF MAKSURA ISOLATED FORM
FD04 ARABIC LIGATURE KHAH WITH YEH ISOLATED FORM
FD05 ARABIC LIGATURE SAD WITH ALEF MAKSURA ISOLATED FORM
FD06 ARABIC LIGATURE SAD WITH YEH ISOLATED FORM
FD07 ARABIC LIGATURE DAD WITH ALEF MAKSURA ISOLATED FORM
FD08 ARABIC LIGATURE DAD WITH YEH ISOLATED FORM
FD09 ARABIC LIGATURE SHEEN WITH JEEM ISOLATED FORM
FD0A ARABIC LIGATURE SHEEN WITH HAH ISOLATED FORM
FD0B ARABIC LIGATURE SHEEN WITH KHAH ISOLATED FORM
FD0C ARABIC LIGATURE SHEEN WITH MEEM ISOLATED FORM
FD0D ARABIC LIGATURE SHEEN WITH REH ISOLATED FORM
FD0E ARABIC LIGATURE SEEN WITH REH ISOLATED FORM
FD0F ARABIC LIGATURE SAD WITH REH ISOLATED FORM
FD10 ARABIC LIGATURE DAD WITH REH ISOLATED FORM
FD11 ARABIC LIGATURE TAH WITH ALEF MAKSURA FINAL FORM
FD12 ARABIC LIGATURE TAH WITH YEH FINAL FORM
FD13 ARABIC LIGATURE AIN WITH ALEF MAKSURA FINAL FORM
FD14 ARABIC LIGATURE AIN WITH YEH FINAL FORM
FD15 ARABIC LIGATURE GHAIN WITH ALEF MAKSURA FINAL FORM
FD16 ARABIC LIGATURE GHAIN WITH YEH FINAL FORM
FD17 ARABIC LIGATURE SEEN WITH ALEF MAKSURA FINAL FORM
FD18 ARABIC LIGATURE SEEN WITH YEH FINAL FORM
FD19 ARABIC LIGATURE SHEEN WITH ALEF MAKSURA FINAL FORM
FD1A ARABIC LIGATURE SHEEN WITH YEH FINAL FORM
FD1B ARABIC LIGATURE HAH WITH ALEF MAKSURA FINAL FORM
FD1C ARABIC LIGATURE HAH WITH YEH FINAL FORM
FD1D ARABIC LIGATURE JEEM WITH ALEF MAKSURA FINAL FORM
FD1E ARABIC LIGATURE JEEM WITH YEH FINAL FORM
FD1F ARABIC LIGATURE KHAH WITH ALEF MAKSURA FINAL FORM
FD20 ARABIC LIGATURE KHAH WITH YEH FINAL FORM
FD21 ARABIC LIGATURE SAD WITH ALEF MAKSURA FINAL FORM
FD22 ARABIC LIGATURE SAD WITH YEH FINAL FORM
FD23 ARABIC LIGATURE DAD WITH ALEF MAKSURA FINAL FORM
FD24 ARABIC LIGATURE DAD WITH YEH FINAL FORM
FD25 ARABIC LIGATURE SHEEN WITH JEEM FINAL FORM
FD26 ARABIC LIGATURE SHEEN WITH HAH FINAL FORM
FD27 ARABIC LIGATURE SHEEN WITH KHAH FINAL FORM
FD28 ARABIC LIGATURE SHEEN WITH MEEM FINAL FORM
FD29 ARABIC LIGATURE SHEEN WITH REH FINAL FORM
FD2A ARABIC LIGATURE SEEN WITH REH FINAL FORM
FD2B ARABIC LIGATURE SAD WITH REH FINAL FORM
FD2C ARABIC LIGATURE DAD WITH REH FINAL FORM
FD2D ARABIC LIGATURE SHEEN WITH JEEM INITIAL FORM
FD2E ARABIC LIGATURE SHEEN WITH HAH INITIAL FORM
FD2F ARABIC LIGATURE SHEEN WITH KHAH INITIAL FORM
FD30 ARABIC LIGATURE SHEEN WITH MEEM INITIAL FORM
FD31 ARABIC LIGATURE SEEN WITH HEH INITIAL FORM
FD32 ARABIC LIGATURE SHEEN WITH HEH INITIAL FORM
FD33 ARABIC LIGATURE TAH WITH MEEM INITIAL FORM
FD34 ARABIC LIGATURE SEEN WITH JEEM MEDIAL FORM
FD35 ARABIC LIGATURE SEEN WITH HAH MEDIAL FORM
FD36 ARABIC LIGATURE SEEN WITH KHAH MEDIAL FORM
FD37 ARABIC LIGATURE SHEEN WITH JEEM MEDIAL FORM
FD38 ARABIC LIGATURE SHEEN WITH HAH MEDIAL FORM
FD39 ARABIC LIGATURE SHEEN WITH KHAH MEDIAL FORM
FD3A ARABIC LIGATURE TAH WITH MEEM MEDIAL FORM
FD3B ARABIC LIGATURE ZAH WITH MEEM MEDIAL FORM
FD3C ARABIC LIGATURE ALEF WITH FATHATAN FINAL FORM
FD3D ARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORM
FD3E ORNATE LEFT PARENTHESIS
FD3F ORNATE RIGHT PARENTHESIS
FD50 ARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORM
FD51 ARABIC LIGATURE TEH WITH HAH WITH JEEM FINAL FORM
FD52 ARABIC LIGATURE TEH WITH HAH WITH JEEM INITIAL FORM
FD53 ARABIC LIGATURE TEH WITH HAH WITH MEEM INITIAL FORM
FD54 ARABIC LIGATURE TEH WITH KHAH WITH MEEM INITIAL FORM
FD55 ARABIC LIGATURE TEH WITH MEEM WITH JEEM INITIAL FORM
FD56 ARABIC LIGATURE TEH WITH MEEM WITH HAH INITIAL FORM
FD57 ARABIC LIGATURE TEH WITH MEEM WITH KHAH INITIAL FORM
FD58 ARABIC LIGATURE JEEM WITH MEEM WITH HAH FINAL FORM
FD59 ARABIC LIGATURE JEEM WITH MEEM WITH HAH INITIAL FORM
FD5A ARABIC LIGATURE HAH WITH MEEM WITH YEH FINAL FORM
FD5B ARABIC LIGATURE HAH WITH MEEM WITH ALEF MAKSURA FINAL FORM
FD5C ARABIC LIGATURE SEEN WITH HAH WITH JEEM INITIAL FORM
FD5D ARABIC LIGATURE SEEN WITH JEEM WITH HAH INITIAL FORM
FD5E ARABIC LIGATURE SEEN WITH JEEM WITH ALEF MAKSURA FINAL FORM
FD5F ARABIC LIGATURE SEEN WITH MEEM WITH HAH FINAL FORM
FD60 ARABIC LIGATURE SEEN WITH MEEM WITH HAH INITIAL FORM
FD61 ARABIC LIGATURE SEEN WITH MEEM WITH JEEM INITIAL FORM
FD62 ARABIC LIGATURE SEEN WITH MEEM WITH MEEM FINAL FORM
FD63 ARABIC LIGATURE SEEN WITH MEEM WITH MEEM INITIAL FORM
FD64 ARABIC LIGATURE SAD WITH HAH WITH HAH FINAL FORM
FD65 ARABIC LIGATURE SAD WITH HAH WITH HAH INITIAL FORM
FD66 ARABIC LIGATURE SAD WITH MEEM WITH MEEM FINAL FORM
FD67 ARABIC LIGATURE SHEEN WITH HAH WITH MEEM FINAL FORM
FD68 ARABIC LIGATURE SHEEN WITH HAH WITH MEEM INITIAL FORM
FD69 ARABIC LIGATURE SHEEN WITH JEEM WITH YEH FINAL FORM
FD6A ARABIC LIGATURE SHEEN WITH MEEM WITH KHAH FINAL FORM
FD6B ARABIC LIGATURE SHEEN WITH MEEM WITH KHAH INITIAL FORM
FD6C ARABIC LIGATURE SHEEN WITH MEEM WITH MEEM FINAL FORM
FD6D ARABIC LIGATURE SHEEN WITH MEEM WITH MEEM INITIAL FORM
FD6E ARABIC LIGATURE DAD WITH HAH WITH ALEF MAKSURA FINAL FORM
FD6F ARABIC LIGATURE DAD WITH KHAH WITH MEEM FINAL FORM
FD70 ARABIC LIGATURE DAD WITH KHAH WITH MEEM INITIAL FORM
FD71 ARABIC LIGATURE TAH WITH MEEM WITH HAH FINAL FORM
FD72 ARABIC LIGATURE TAH WITH MEEM WITH HAH INITIAL FORM
FD73 ARABIC LIGATURE TAH WITH MEEM WITH MEEM INITIAL FORM
FD74 ARABIC LIGATURE TAH WITH MEEM WITH YEH FINAL FORM
FD75 ARABIC LIGATURE AIN WITH JEEM WITH MEEM FINAL FORM
FD76 ARABIC LIGATURE AIN WITH MEEM WITH MEEM FINAL FORM
FD77 ARABIC LIGATURE AIN WITH MEEM WITH MEEM INITIAL FORM
FD78 ARABIC LIGATURE AIN WITH MEEM WITH ALEF MAKSURA FINAL FORM
FD79 ARABIC LIGATURE GHAIN WITH MEEM WITH MEEM FINAL FORM
FD7A ARABIC LIGATURE GHAIN WITH MEEM WITH YEH FINAL FORM
FD7B ARABIC LIGATURE GHAIN WITH MEEM WITH ALEF MAKSURA FINAL FORM
FD7C ARABIC LIGATURE FEH WITH KHAH WITH MEEM FINAL FORM
FD7D ARABIC LIGATURE FEH WITH KHAH WITH MEEM INITIAL FORM
FD7E ARABIC LIGATURE QAF WITH MEEM WITH HAH FINAL FORM
FD7F ARABIC LIGATURE QAF WITH MEEM WITH MEEM FINAL FORM
FD80 ARABIC LIGATURE LAM WITH HAH WITH MEEM FINAL FORM
FD81 ARABIC LIGATURE LAM WITH HAH WITH YEH FINAL FORM
FD82 ARABIC LIGATURE LAM WITH HAH WITH ALEF MAKSURA FINAL FORM
FD83 ARABIC LIGATURE LAM WITH JEEM WITH JEEM INITIAL FORM
FD84 ARABIC LIGATURE LAM WITH JEEM WITH JEEM FINAL FORM
FD85 ARABIC LIGATURE LAM WITH KHAH WITH MEEM FINAL FORM
FD86 ARABIC LIGATURE LAM WITH KHAH WITH MEEM INITIAL FORM
FD87 ARABIC LIGATURE LAM WITH MEEM WITH HAH FINAL FORM
FD88 ARABIC LIGATURE LAM WITH MEEM WITH HAH INITIAL FORM
FD89 ARABIC LIGATURE MEEM WITH HAH WITH JEEM INITIAL FORM
FD8A ARABIC LIGATURE MEEM WITH HAH WITH MEEM INITIAL FORM
FD8B ARABIC LIGATURE MEEM WITH HAH WITH YEH FINAL FORM
FD8C ARABIC LIGATURE MEEM WITH JEEM WITH HAH INITIAL FORM
FD8D ARABIC LIGATURE MEEM WITH JEEM WITH MEEM INITIAL FORM
FD8E ARABIC LIGATURE MEEM WITH KHAH WITH JEEM INITIAL FORM
FD8F ARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORM
FD92 ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORM
FD93 ARABIC LIGATURE HEH WITH MEEM WITH JEEM INITIAL FORM
FD94 ARABIC LIGATURE HEH WITH MEEM WITH MEEM INITIAL FORM
FD95 ARABIC LIGATURE NOON WITH HAH WITH MEEM INITIAL FORM
FD96 ARABIC LIGATURE NOON WITH HAH WITH ALEF MAKSURA FINAL FORM
FD97 ARABIC LIGATURE NOON WITH JEEM WITH MEEM FINAL FORM
FD98 ARABIC LIGATURE NOON WITH JEEM WITH MEEM INITIAL FORM
FD99 ARABIC LIGATURE NOON WITH JEEM WITH ALEF MAKSURA FINAL FORM
FD9A ARABIC LIGATURE NOON WITH MEEM WITH YEH FINAL FORM
FD9B ARABIC LIGATURE NOON WITH MEEM WITH ALEF MAKSURA FINAL FORM
FD9C ARABIC LIGATURE YEH WITH MEEM WITH MEEM FINAL FORM
FD9D ARABIC LIGATURE YEH WITH MEEM WITH MEEM INITIAL FORM
FD9E ARABIC LIGATURE BEH WITH KHAH WITH YEH FINAL FORM
FD9F ARABIC LIGATURE TEH WITH JEEM WITH YEH FINAL FORM
FDA0 ARABIC LIGATURE TEH WITH JEEM WITH ALEF MAKSURA FINAL FORM
FDA1 ARABIC LIGATURE TEH WITH KHAH WITH YEH FINAL FORM
FDA2 ARABIC LIGATURE TEH WITH KHAH WITH ALEF MAKSURA FINAL FORM
FDA3 ARABIC LIGATURE TEH WITH MEEM WITH YEH FINAL FORM
FDA4 ARABIC LIGATURE TEH WITH MEEM WITH ALEF MAKSURA FINAL FORM
FDA5 ARABIC LIGATURE JEEM WITH MEEM WITH YEH FINAL FORM
FDA6 ARABIC LIGATURE JEEM WITH HAH WITH ALEF MAKSURA FINAL FORM
FDA7 ARABIC LIGATURE JEEM WITH MEEM WITH ALEF MAKSURA FINAL FORM
FDA8 ARABIC LIGATURE SEEN WITH KHAH WITH ALEF MAKSURA FINAL FORM
FDA9 ARABIC LIGATURE SAD WITH HAH WITH YEH FINAL FORM
FDAA ARABIC LIGATURE SHEEN WITH HAH WITH YEH FINAL FORM
FDAB ARABIC LIGATURE DAD WITH HAH WITH YEH FINAL FORM
FDAC ARABIC LIGATURE LAM WITH JEEM WITH YEH FINAL FORM
FDAD ARABIC LIGATURE LAM WITH MEEM WITH YEH FINAL FORM
FDAE ARABIC LIGATURE YEH WITH HAH WITH YEH FINAL FORM
FDAF ARABIC LIGATURE YEH WITH JEEM WITH YEH FINAL FORM
FDB0 ARABIC LIGATURE YEH WITH MEEM WITH YEH FINAL FORM
FDB1 ARABIC LIGATURE MEEM WITH MEEM WITH YEH FINAL FORM
FDB2 ARABIC LIGATURE QAF WITH MEEM WITH YEH FINAL FORM
FDB3 ARABIC LIGATURE NOON WITH HAH WITH YEH FINAL FORM
FDB4 ARABIC LIGATURE QAF WITH MEEM WITH HAH INITIAL FORM
FDB5 ARABIC LIGATURE LAM WITH HAH WITH MEEM INITIAL FORM
FDB6 ARABIC LIGATURE AIN WITH MEEM WITH YEH FINAL FORM
FDB7 ARABIC LIGATURE KAF WITH MEEM WITH YEH FINAL FORM
FDB8 ARABIC LIGATURE NOON WITH JEEM WITH HAH INITIAL FORM
FDB9 ARABIC LIGATURE MEEM WITH KHAH WITH YEH FINAL FORM
FDBA ARABIC LIGATURE LAM WITH JEEM WITH MEEM INITIAL FORM
FDBB ARABIC LIGATURE KAF WITH MEEM WITH MEEM FINAL FORM
FDBC ARABIC LIGATURE LAM WITH JEEM WITH MEEM FINAL FORM
FDBD ARABIC LIGATURE NOON WITH JEEM WITH HAH FINAL FORM
FDBE ARABIC LIGATURE JEEM WITH HAH WITH YEH FINAL FORM
FDBF ARABIC LIGATURE HAH WITH JEEM WITH YEH FINAL FORM
FDC0 ARABIC LIGATURE MEEM WITH JEEM WITH YEH FINAL FORM
FDC1 ARABIC LIGATURE FEH WITH MEEM WITH YEH FINAL FORM
FDC2 ARABIC LIGATURE BEH WITH HAH WITH YEH FINAL FORM
FDC3 ARABIC LIGATURE KAF WITH MEEM WITH MEEM INITIAL FORM
FDC4 ARABIC LIGATURE AIN WITH JEEM WITH MEEM INITIAL FORM
FDC5 ARABIC LIGATURE SAD WITH MEEM WITH MEEM INITIAL FORM
FDC6 ARABIC LIGATURE SEEN WITH KHAH WITH YEH FINAL FORM
FDC7 ARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORM
FDF0 ARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORM
FDF1 ARABIC LIGATURE QALA USED AS KORANIC STOP SIGN ISOLATED FORM
FDF2 ARABIC LIGATURE ALLAH ISOLATED FORM
FDF3 ARABIC LIGATURE AKBAR ISOLATED FORM
FDF4 ARABIC LIGATURE MOHAMMAD ISOLATED FORM
FDF5 ARABIC LIGATURE SALAM ISOLATED FORM
FDF6 ARABIC LIGATURE RASOUL ISOLATED FORM
FDF7 ARABIC LIGATURE ALAYHE ISOLATED FORM
FDF8 ARABIC LIGATURE WASALLAM ISOLATED FORM
FDF9 ARABIC LIGATURE SALLA ISOLATED FORM
FDFA ARABIC LIGATURE SALLALLAHOU ALAYHE WASALLAM
FDFB ARABIC LIGATURE JALLAJALALOUHOU
FE20 COMBINING LIGATURE LEFT HALF
FE21 COMBINING LIGATURE RIGHT HALF
FE22 COMBINING DOUBLE TILDE LEFT HALF
FE23 COMBINING DOUBLE TILDE RIGHT HALF
FE30 PRESENTATION FORM FOR VERTICAL TWO DOT LEADER
FE31 PRESENTATION FORM FOR VERTICAL EM DASH
FE32 PRESENTATION FORM FOR VERTICAL EN DASH
FE33 PRESENTATION FORM FOR VERTICAL LOW LINE
FE34 PRESENTATION FORM FOR VERTICAL WAVY LOW LINE
FE35 PRESENTATION FORM FOR VERTICAL LEFT PARENTHESIS
FE36 PRESENTATION FORM FOR VERTICAL RIGHT PARENTHESIS
FE37 PRESENTATION FORM FOR VERTICAL LEFT CURLY BRACKET
FE38 PRESENTATION FORM FOR VERTICAL RIGHT CURLY BRACKET
FE39 PRESENTATION FORM FOR VERTICAL LEFT TORTOISE SHELL BRACKET
FE3A PRESENTATION FORM FOR VERTICAL RIGHT TORTOISE SHELL BRACKET
FE3B PRESENTATION FORM FOR VERTICAL LEFT BLACK LENTICULAR BRACKET
FE3C PRESENTATION FORM FOR VERTICAL RIGHT BLACK LENTICULAR BRACKET
FE3D PRESENTATION FORM FOR VERTICAL LEFT DOUBLE ANGLE BRACKET
FE3E PRESENTATION FORM FOR VERTICAL RIGHT DOUBLE ANGLE BRACKET
FE3F PRESENTATION FORM FOR VERTICAL LEFT ANGLE BRACKET
FE40 PRESENTATION FORM FOR VERTICAL RIGHT ANGLE BRACKET
FE41 PRESENTATION FORM FOR VERTICAL LEFT CORNER BRACKET
FE42 PRESENTATION FORM FOR VERTICAL RIGHT CORNER BRACKET
FE43 PRESENTATION FORM FOR VERTICAL LEFT WHITE CORNER BRACKET
FE44 PRESENTATION FORM FOR VERTICAL RIGHT WHITE CORNER BRACKET
FE49 DASHED OVERLINE
FE4A CENTRELINE OVERLINE
FE4B WAVY OVERLINE
FE4C DOUBLE WAVY OVERLINE
FE4D DASHED LOW LINE
FE4E CENTRELINE LOW LINE
FE4F WAVY LOW LINE
FE50 SMALL COMMA
FE51 SMALL IDEOGRAPHIC COMMA
FE52 SMALL FULL STOP
FE54 SMALL SEMICOLON
FE55 SMALL COLON
FE56 SMALL QUESTION MARK
FE57 SMALL EXCLAMATION MARK
FE58 SMALL EM DASH
FE59 SMALL LEFT PARENTHESIS
FE5A SMALL RIGHT PARENTHESIS
FE5B SMALL LEFT CURLY BRACKET
FE5C SMALL RIGHT CURLY BRACKET
FE5D SMALL LEFT TORTOISE SHELL BRACKET
FE5E SMALL RIGHT TORTOISE SHELL BRACKET
FE5F SMALL NUMBER SIGN
FE60 SMALL AMPERSAND
FE61 SMALL ASTERISK
FE62 SMALL PLUS SIGN
FE63 SMALL HYPHEN-MINUS
FE64 SMALL LESS-THAN SIGN
FE65 SMALL GREATER-THAN SIGN
FE66 SMALL EQUALS SIGN
FE68 SMALL REVERSE SOLIDUS
FE69 SMALL DOLLAR SIGN
FE6A SMALL PERCENT SIGN
FE6B SMALL COMMERCIAL AT
FE70 ARABIC FATHATAN ISOLATED FORM
FE71 ARABIC TATWEEL WITH FATHATAN ABOVE
FE72 ARABIC DAMMATAN ISOLATED FORM
FE74 ARABIC KASRATAN ISOLATED FORM
FE76 ARABIC FATHA ISOLATED FORM
FE77 ARABIC FATHA MEDIAL FORM
FE78 ARABIC DAMMA ISOLATED FORM
FE79 ARABIC DAMMA MEDIAL FORM
FE7A ARABIC KASRA ISOLATED FORM
FE7B ARABIC KASRA MEDIAL FORM
FE7C ARABIC SHADDA ISOLATED FORM
FE7D ARABIC SHADDA MEDIAL FORM
FE7E ARABIC SUKUN ISOLATED FORM
FE7F ARABIC SUKUN MEDIAL FORM
FE80 ARABIC LETTER HAMZA ISOLATED FORM
FE81 ARABIC LETTER ALEF WITH MADDA ABOVE ISOLATED FORM
FE82 ARABIC LETTER ALEF WITH MADDA ABOVE FINAL FORM
FE83 ARABIC LETTER ALEF WITH HAMZA ABOVE ISOLATED FORM
FE84 ARABIC LETTER ALEF WITH HAMZA ABOVE FINAL FORM
FE85 ARABIC LETTER WAW WITH HAMZA ABOVE ISOLATED FORM
FE86 ARABIC LETTER WAW WITH HAMZA ABOVE FINAL FORM
FE87 ARABIC LETTER ALEF WITH HAMZA BELOW ISOLATED FORM
FE88 ARABIC LETTER ALEF WITH HAMZA BELOW FINAL FORM
FE89 ARABIC LETTER YEH WITH HAMZA ABOVE ISOLATED FORM
FE8A ARABIC LETTER YEH WITH HAMZA ABOVE FINAL FORM
FE8B ARABIC LETTER YEH WITH HAMZA ABOVE INITIAL FORM
FE8C ARABIC LETTER YEH WITH HAMZA ABOVE MEDIAL FORM
FE8D ARABIC LETTER ALEF ISOLATED FORM
FE8E ARABIC LETTER ALEF FINAL FORM
FE8F ARABIC LETTER BEH ISOLATED FORM
FE90 ARABIC LETTER BEH FINAL FORM
FE91 ARABIC LETTER BEH INITIAL FORM
FE92 ARABIC LETTER BEH MEDIAL FORM
FE93 ARABIC LETTER TEH MARBUTA ISOLATED FORM
FE94 ARABIC LETTER TEH MARBUTA FINAL FORM
FE95 ARABIC LETTER TEH ISOLATED FORM
FE96 ARABIC LETTER TEH FINAL FORM
FE97 ARABIC LETTER TEH INITIAL FORM
FE98 ARABIC LETTER TEH MEDIAL FORM
FE99 ARABIC LETTER THEH ISOLATED FORM
FE9A ARABIC LETTER THEH FINAL FORM
FE9B ARABIC LETTER THEH INITIAL FORM
FE9C ARABIC LETTER THEH MEDIAL FORM
FE9D ARABIC LETTER JEEM ISOLATED FORM
FE9E ARABIC LETTER JEEM FINAL FORM
FE9F ARABIC LETTER JEEM INITIAL FORM
FEA0 ARABIC LETTER JEEM MEDIAL FORM
FEA1 ARABIC LETTER HAH ISOLATED FORM
FEA2 ARABIC LETTER HAH FINAL FORM
FEA3 ARABIC LETTER HAH INITIAL FORM
FEA4 ARABIC LETTER HAH MEDIAL FORM
FEA5 ARABIC LETTER KHAH ISOLATED FORM
FEA6 ARABIC LETTER KHAH FINAL FORM
FEA7 ARABIC LETTER KHAH INITIAL FORM
FEA8 ARABIC LETTER KHAH MEDIAL FORM
FEA9 ARABIC LETTER DAL ISOLATED FORM
FEAA ARABIC LETTER DAL FINAL FORM
FEAB ARABIC LETTER THAL ISOLATED FORM
FEAC ARABIC LETTER THAL FINAL FORM
FEAD ARABIC LETTER REH ISOLATED FORM
FEAE ARABIC LETTER REH FINAL FORM
FEAF ARABIC LETTER ZAIN ISOLATED FORM
FEB0 ARABIC LETTER ZAIN FINAL FORM
FEB1 ARABIC LETTER SEEN ISOLATED FORM
FEB2 ARABIC LETTER SEEN FINAL FORM
FEB3 ARABIC LETTER SEEN INITIAL FORM
FEB4 ARABIC LETTER SEEN MEDIAL FORM
FEB5 ARABIC LETTER SHEEN ISOLATED FORM
FEB6 ARABIC LETTER SHEEN FINAL FORM
FEB7 ARABIC LETTER SHEEN INITIAL FORM
FEB8 ARABIC LETTER SHEEN MEDIAL FORM
FEB9 ARABIC LETTER SAD ISOLATED FORM
FEBA ARABIC LETTER SAD FINAL FORM
FEBB ARABIC LETTER SAD INITIAL FORM
FEBC ARABIC LETTER SAD MEDIAL FORM
FEBD ARABIC LETTER DAD ISOLATED FORM
FEBE ARABIC LETTER DAD FINAL FORM
FEBF ARABIC LETTER DAD INITIAL FORM
FEC0 ARABIC LETTER DAD MEDIAL FORM
FEC1 ARABIC LETTER TAH ISOLATED FORM
FEC2 ARABIC LETTER TAH FINAL FORM
FEC3 ARABIC LETTER TAH INITIAL FORM
FEC4 ARABIC LETTER TAH MEDIAL FORM
FEC5 ARABIC LETTER ZAH ISOLATED FORM
FEC6 ARABIC LETTER ZAH FINAL FORM
FEC7 ARABIC LETTER ZAH INITIAL FORM
FEC8 ARABIC LETTER ZAH MEDIAL FORM
FEC9 ARABIC LETTER AIN ISOLATED FORM
FECA ARABIC LETTER AIN FINAL FORM
FECB ARABIC LETTER AIN INITIAL FORM
FECC ARABIC LETTER AIN MEDIAL FORM
FECD ARABIC LETTER GHAIN ISOLATED FORM
FECE ARABIC LETTER GHAIN FINAL FORM
FECF ARABIC LETTER GHAIN INITIAL FORM
FED0 ARABIC LETTER GHAIN MEDIAL FORM
FED1 ARABIC LETTER FEH ISOLATED FORM
FED2 ARABIC LETTER FEH FINAL FORM
FED3 ARABIC LETTER FEH INITIAL FORM
FED4 ARABIC LETTER FEH MEDIAL FORM
FED5 ARABIC LETTER QAF ISOLATED FORM
FED6 ARABIC LETTER QAF FINAL FORM
FED7 ARABIC LETTER QAF INITIAL FORM
FED8 ARABIC LETTER QAF MEDIAL FORM
FED9 ARABIC LETTER KAF ISOLATED FORM
FEDA ARABIC LETTER KAF FINAL FORM
FEDB ARABIC LETTER KAF INITIAL FORM
FEDC ARABIC LETTER KAF MEDIAL FORM
FEDD ARABIC LETTER LAM ISOLATED FORM
FEDE ARABIC LETTER LAM FINAL FORM
FEDF ARABIC LETTER LAM INITIAL FORM
FEE0 ARABIC LETTER LAM MEDIAL FORM
FEE1 ARABIC LETTER MEEM ISOLATED FORM
FEE2 ARABIC LETTER MEEM FINAL FORM
FEE3 ARABIC LETTER MEEM INITIAL FORM
FEE4 ARABIC LETTER MEEM MEDIAL FORM
FEE5 ARABIC LETTER NOON ISOLATED FORM
FEE6 ARABIC LETTER NOON FINAL FORM
FEE7 ARABIC LETTER NOON INITIAL FORM
FEE8 ARABIC LETTER NOON MEDIAL FORM
FEE9 ARABIC LETTER HEH ISOLATED FORM
FEEA ARABIC LETTER HEH FINAL FORM
FEEB ARABIC LETTER HEH INITIAL FORM
FEEC ARABIC LETTER HEH MEDIAL FORM
FEED ARABIC LETTER WAW ISOLATED FORM
FEEE ARABIC LETTER WAW FINAL FORM
FEEF ARABIC LETTER ALEF MAKSURA ISOLATED FORM
FEF0 ARABIC LETTER ALEF MAKSURA FINAL FORM
FEF1 ARABIC LETTER YEH ISOLATED FORM
FEF2 ARABIC LETTER YEH FINAL FORM
FEF3 ARABIC LETTER YEH INITIAL FORM
FEF4 ARABIC LETTER YEH MEDIAL FORM
FEF5 ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE ISOLATED FORM
FEF6 ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE FINAL FORM
FEF7 ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE ISOLATED FORM
FEF8 ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE FINAL FORM
FEF9 ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW ISOLATED FORM
FEFA ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW FINAL FORM
FEFB ARABIC LIGATURE LAM WITH ALEF ISOLATED FORM
FEFC ARABIC LIGATURE LAM WITH ALEF FINAL FORM
FEFF ZERO WIDTH NO-BREAK SPACE
FF01 FULLWIDTH EXCLAMATION MARK
FF02 FULLWIDTH QUOTATION MARK
FF03 FULLWIDTH NUMBER SIGN
FF04 FULLWIDTH DOLLAR SIGN
FF05 FULLWIDTH PERCENT SIGN
FF06 FULLWIDTH AMPERSAND
FF07 FULLWIDTH APOSTROPHE
FF08 FULLWIDTH LEFT PARENTHESIS
FF09 FULLWIDTH RIGHT PARENTHESIS
FF0A FULLWIDTH ASTERISK
FF0B FULLWIDTH PLUS SIGN
FF0C FULLWIDTH COMMA
FF0D FULLWIDTH HYPHEN-MINUS
FF0E FULLWIDTH FULL STOP
FF0F FULLWIDTH SOLIDUS
FF10 FULLWIDTH DIGIT ZERO
FF11 FULLWIDTH DIGIT ONE
FF12 FULLWIDTH DIGIT TWO
FF13 FULLWIDTH DIGIT THREE
FF14 FULLWIDTH DIGIT FOUR
FF15 FULLWIDTH DIGIT FIVE
FF16 FULLWIDTH DIGIT SIX
FF17 FULLWIDTH DIGIT SEVEN
FF18 FULLWIDTH DIGIT EIGHT
FF19 FULLWIDTH DIGIT NINE
FF1A FULLWIDTH COLON
FF1B FULLWIDTH SEMICOLON
FF1C FULLWIDTH LESS-THAN SIGN
FF1D FULLWIDTH EQUALS SIGN
FF1E FULLWIDTH GREATER-THAN SIGN
FF1F FULLWIDTH QUESTION MARK
FF20 FULLWIDTH COMMERCIAL AT
FF21 FULLWIDTH LATIN CAPITAL LETTER A
FF22 FULLWIDTH LATIN CAPITAL LETTER B
FF23 FULLWIDTH LATIN CAPITAL LETTER C
FF24 FULLWIDTH LATIN CAPITAL LETTER D
FF25 FULLWIDTH LATIN CAPITAL LETTER E
FF26 FULLWIDTH LATIN CAPITAL LETTER F
FF27 FULLWIDTH LATIN CAPITAL LETTER G
FF28 FULLWIDTH LATIN CAPITAL LETTER H
FF29 FULLWIDTH LATIN CAPITAL LETTER I
FF2A FULLWIDTH LATIN CAPITAL LETTER J
FF2B FULLWIDTH LATIN CAPITAL LETTER K
FF2C FULLWIDTH LATIN CAPITAL LETTER L
FF2D FULLWIDTH LATIN CAPITAL LETTER M
FF2E FULLWIDTH LATIN CAPITAL LETTER N
FF2F FULLWIDTH LATIN CAPITAL LETTER O
FF30 FULLWIDTH LATIN CAPITAL LETTER P
FF31 FULLWIDTH LATIN CAPITAL LETTER Q
FF32 FULLWIDTH LATIN CAPITAL LETTER R
FF33 FULLWIDTH LATIN CAPITAL LETTER S
FF34 FULLWIDTH LATIN CAPITAL LETTER T
FF35 FULLWIDTH LATIN CAPITAL LETTER U
FF36 FULLWIDTH LATIN CAPITAL LETTER V
FF37 FULLWIDTH LATIN CAPITAL LETTER W
FF38 FULLWIDTH LATIN CAPITAL LETTER X
FF39 FULLWIDTH LATIN CAPITAL LETTER Y
FF3A FULLWIDTH LATIN CAPITAL LETTER Z
FF3B FULLWIDTH LEFT SQUARE BRACKET
FF3C FULLWIDTH REVERSE SOLIDUS
FF3D FULLWIDTH RIGHT SQUARE BRACKET
FF3E FULLWIDTH CIRCUMFLEX ACCENT
FF3F FULLWIDTH LOW LINE
FF40 FULLWIDTH GRAVE ACCENT
FF41 FULLWIDTH LATIN SMALL LETTER A
FF42 FULLWIDTH LATIN SMALL LETTER B
FF43 FULLWIDTH LATIN SMALL LETTER C
FF44 FULLWIDTH LATIN SMALL LETTER D
FF45 FULLWIDTH LATIN SMALL LETTER E
FF46 FULLWIDTH LATIN SMALL LETTER F
FF47 FULLWIDTH LATIN SMALL LETTER G
FF48 FULLWIDTH LATIN SMALL LETTER H
FF49 FULLWIDTH LATIN SMALL LETTER I
FF4A FULLWIDTH LATIN SMALL LETTER J
FF4B FULLWIDTH LATIN SMALL LETTER K
FF4C FULLWIDTH LATIN SMALL LETTER L
FF4D FULLWIDTH LATIN SMALL LETTER M
FF4E FULLWIDTH LATIN SMALL LETTER N
FF4F FULLWIDTH LATIN SMALL LETTER O
FF50 FULLWIDTH LATIN SMALL LETTER P
FF51 FULLWIDTH LATIN SMALL LETTER Q
FF52 FULLWIDTH LATIN SMALL LETTER R
FF53 FULLWIDTH LATIN SMALL LETTER S
FF54 FULLWIDTH LATIN SMALL LETTER T
FF55 FULLWIDTH LATIN SMALL LETTER U
FF56 FULLWIDTH LATIN SMALL LETTER V
FF57 FULLWIDTH LATIN SMALL LETTER W
FF58 FULLWIDTH LATIN SMALL LETTER X
FF59 FULLWIDTH LATIN SMALL LETTER Y
FF5A FULLWIDTH LATIN SMALL LETTER Z
FF5B FULLWIDTH LEFT CURLY BRACKET
FF5C FULLWIDTH VERTICAL LINE
FF5D FULLWIDTH RIGHT CURLY BRACKET
FF5E FULLWIDTH TILDE
FF61 HALFWIDTH IDEOGRAPHIC FULL STOP
FF62 HALFWIDTH LEFT CORNER BRACKET
FF63 HALFWIDTH RIGHT CORNER BRACKET
FF64 HALFWIDTH IDEOGRAPHIC COMMA
FF65 HALFWIDTH KATAKANA MIDDLE DOT
FF66 HALFWIDTH KATAKANA LETTER WO
FF67 HALFWIDTH KATAKANA LETTER SMALL A
FF68 HALFWIDTH KATAKANA LETTER SMALL I
FF69 HALFWIDTH KATAKANA LETTER SMALL U
FF6A HALFWIDTH KATAKANA LETTER SMALL E
FF6B HALFWIDTH KATAKANA LETTER SMALL O
FF6C HALFWIDTH KATAKANA LETTER SMALL YA
FF6D HALFWIDTH KATAKANA LETTER SMALL YU
FF6E HALFWIDTH KATAKANA LETTER SMALL YO
FF6F HALFWIDTH KATAKANA LETTER SMALL TU
FF70 HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
FF71 HALFWIDTH KATAKANA LETTER A
FF72 HALFWIDTH KATAKANA LETTER I
FF73 HALFWIDTH KATAKANA LETTER U
FF74 HALFWIDTH KATAKANA LETTER E
FF75 HALFWIDTH KATAKANA LETTER O
FF76 HALFWIDTH KATAKANA LETTER KA
FF77 HALFWIDTH KATAKANA LETTER KI
FF78 HALFWIDTH KATAKANA LETTER KU
FF79 HALFWIDTH KATAKANA LETTER KE
FF7A HALFWIDTH KATAKANA LETTER KO
FF7B HALFWIDTH KATAKANA LETTER SA
FF7C HALFWIDTH KATAKANA LETTER SI
FF7D HALFWIDTH KATAKANA LETTER SU
FF7E HALFWIDTH KATAKANA LETTER SE
FF7F HALFWIDTH KATAKANA LETTER SO
FF80 HALFWIDTH KATAKANA LETTER TA
FF81 HALFWIDTH KATAKANA LETTER TI
FF82 HALFWIDTH KATAKANA LETTER TU
FF83 HALFWIDTH KATAKANA LETTER TE
FF84 HALFWIDTH KATAKANA LETTER TO
FF85 HALFWIDTH KATAKANA LETTER NA
FF86 HALFWIDTH KATAKANA LETTER NI
FF87 HALFWIDTH KATAKANA LETTER NU
FF88 HALFWIDTH KATAKANA LETTER NE
FF89 HALFWIDTH KATAKANA LETTER NO
FF8A HALFWIDTH KATAKANA LETTER HA
FF8B HALFWIDTH KATAKANA LETTER HI
FF8C HALFWIDTH KATAKANA LETTER HU
FF8D HALFWIDTH KATAKANA LETTER HE
FF8E HALFWIDTH KATAKANA LETTER HO
FF8F HALFWIDTH KATAKANA LETTER MA
FF90 HALFWIDTH KATAKANA LETTER MI
FF91 HALFWIDTH KATAKANA LETTER MU
FF92 HALFWIDTH KATAKANA LETTER ME
FF93 HALFWIDTH KATAKANA LETTER MO
FF94 HALFWIDTH KATAKANA LETTER YA
FF95 HALFWIDTH KATAKANA LETTER YU
FF96 HALFWIDTH KATAKANA LETTER YO
FF97 HALFWIDTH KATAKANA LETTER RA
FF98 HALFWIDTH KATAKANA LETTER RI
FF99 HALFWIDTH KATAKANA LETTER RU
FF9A HALFWIDTH KATAKANA LETTER RE
FF9B HALFWIDTH KATAKANA LETTER RO
FF9C HALFWIDTH KATAKANA LETTER WA
FF9D HALFWIDTH KATAKANA LETTER N
FF9E HALFWIDTH KATAKANA VOICED SOUND MARK
FF9F HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK
FFA0 HALFWIDTH HANGUL FILLER
FFA1 HALFWIDTH HANGUL LETTER KIYEOK
FFA2 HALFWIDTH HANGUL LETTER SSANGKIYEOK
FFA3 HALFWIDTH HANGUL LETTER KIYEOK-SIOS
FFA4 HALFWIDTH HANGUL LETTER NIEUN
FFA5 HALFWIDTH HANGUL LETTER NIEUN-CIEUC
FFA6 HALFWIDTH HANGUL LETTER NIEUN-HIEUH
FFA7 HALFWIDTH HANGUL LETTER TIKEUT
FFA8 HALFWIDTH HANGUL LETTER SSANGTIKEUT
FFA9 HALFWIDTH HANGUL LETTER RIEUL
FFAA HALFWIDTH HANGUL LETTER RIEUL-KIYEOK
FFAB HALFWIDTH HANGUL LETTER RIEUL-MIEUM
FFAC HALFWIDTH HANGUL LETTER RIEUL-PIEUP
FFAD HALFWIDTH HANGUL LETTER RIEUL-SIOS
FFAE HALFWIDTH HANGUL LETTER RIEUL-THIEUTH
FFAF HALFWIDTH HANGUL LETTER RIEUL-PHIEUPH
FFB0 HALFWIDTH HANGUL LETTER RIEUL-HIEUH
FFB1 HALFWIDTH HANGUL LETTER MIEUM
FFB2 HALFWIDTH HANGUL LETTER PIEUP
FFB3 HALFWIDTH HANGUL LETTER SSANGPIEUP
FFB4 HALFWIDTH HANGUL LETTER PIEUP-SIOS
FFB5 HALFWIDTH HANGUL LETTER SIOS
FFB6 HALFWIDTH HANGUL LETTER SSANGSIOS
FFB7 HALFWIDTH HANGUL LETTER IEUNG
FFB8 HALFWIDTH HANGUL LETTER CIEUC
FFB9 HALFWIDTH HANGUL LETTER SSANGCIEUC
FFBA HALFWIDTH HANGUL LETTER CHIEUCH
FFBB HALFWIDTH HANGUL LETTER KHIEUKH
FFBC HALFWIDTH HANGUL LETTER THIEUTH
FFBD HALFWIDTH HANGUL LETTER PHIEUPH
FFBE HALFWIDTH HANGUL LETTER HIEUH
FFC2 HALFWIDTH HANGUL LETTER A
FFC3 HALFWIDTH HANGUL LETTER AE
FFC4 HALFWIDTH HANGUL LETTER YA
FFC5 HALFWIDTH HANGUL LETTER YAE
FFC6 HALFWIDTH HANGUL LETTER EO
FFC7 HALFWIDTH HANGUL LETTER E
FFCA HALFWIDTH HANGUL LETTER YEO
FFCB HALFWIDTH HANGUL LETTER YE
FFCC HALFWIDTH HANGUL LETTER O
FFCD HALFWIDTH HANGUL LETTER WA
FFCE HALFWIDTH HANGUL LETTER WAE
FFCF HALFWIDTH HANGUL LETTER OE
FFD2 HALFWIDTH HANGUL LETTER YO
FFD3 HALFWIDTH HANGUL LETTER U
FFD4 HALFWIDTH HANGUL LETTER WEO
FFD5 HALFWIDTH HANGUL LETTER WE
FFD6 HALFWIDTH HANGUL LETTER WI
FFD7 HALFWIDTH HANGUL LETTER YU
FFDA HALFWIDTH HANGUL LETTER EU
FFDB HALFWIDTH HANGUL LETTER YI
FFDC HALFWIDTH HANGUL LETTER I
FFE0 FULLWIDTH CENT SIGN
FFE1 FULLWIDTH POUND SIGN
FFE2 FULLWIDTH NOT SIGN
FFE3 FULLWIDTH MACRON
FFE4 FULLWIDTH BROKEN BAR
FFE5 FULLWIDTH YEN SIGN
FFE6 FULLWIDTH WON SIGN
FFE8 HALFWIDTH FORMS LIGHT VERTICAL
FFE9 HALFWIDTH LEFTWARDS ARROW
FFEA HALFWIDTH UPWARDS ARROW
FFEB HALFWIDTH RIGHTWARDS ARROW
FFEC HALFWIDTH DOWNWARDS ARROW
FFED HALFWIDTH BLACK SQUARE
FFEE HALFWIDTH WHITE CIRCLE
FFFD REPLACEMENT CHARACTER
