package CXC::Data::Visitor;

# ABSTRACT: Invoke a callback on every element at every level of a data structure.

use v5.20;
use strict;
use warnings;


use feature 'current_sub';
use experimental 'signatures', 'lexical_subs', 'postderef';

our $VERSION = '0.02';

use base 'Exporter::Tiny';
use Hash::Util 'lock_hash', 'unlock_hash', 'unlock_value';
use Scalar::Util 'refaddr';
use Ref::Util 'is_plain_arrayref', 'is_plain_hashref', 'is_coderef', 'is_plain_refref';
use Feature::Compat::Defer;

use constant {
    CYCLE_DIE      => 'die',
    CYCLE_CONTINUE => 'continue',
    CYCLE_TRUNCATE => 'truncate',
};
use constant CYCLE_QR => qr /\A die|continue|truncate \z/x;
use constant { VISIT_CONTAINER => 0b01, VISIT_LEAF => 0b10, VISIT_ALL => 0b11 };

our %EXPORT_TAGS = (
    funcs     => [qw( visit )],
    constants => [ qw(
          VISIT_CONTAINER VISIT_LEAF VISIT_ALL
          CYCLE_DIE CYCLE_CONTINUE CYCLE_TRUNCATE
        ),
    ],
);

our @EXPORT_OK = map { $EXPORT_TAGS{$_}->@* } keys %EXPORT_TAGS;

my sub croak {
    require Carp;
    goto \&Carp::croak;
}


my sub _visit ( $node, $code, $context, $cycle, $visit, $meta ) {    ## no critic (Subroutines::ProhibitManyArgs)

    my $path      = $meta->{path};
    my $ancestors = $meta->{ancestors};

    my $refaddr = refaddr( $node );
    if ( exists $meta->{seen}{$refaddr} ) {

        my $lcycle
          = is_coderef( $cycle )
          ? $cycle->( $node, $context, $meta )
          : $cycle;

        $lcycle eq CYCLE_TRUNCATE and return !!1;
        $lcycle eq CYCLE_DIE
          and croak( __PACKAGE__ . '::visit: cycle detected: ', join( '->', $path->@* ) );

        $lcycle eq CYCLE_CONTINUE
          or croak( __PACKAGE__ . "::visit: unkown cycle parameter value: $lcycle" );
    }
    else {
        $meta->{seen}{$refaddr} = $node;
    }

    # after this call to _visit, will have visited all descendents of
    # $node, so don't need this any longer.
    defer { delete $meta->{seen}{$refaddr} }

    my %meta = $meta->%*;
    $meta{container} = $node;

    # deal with bare next in $code body
    use warnings FATAL => 'exiting';

    my $is_hashref = is_plain_hashref( $node );

    my @idx = $is_hashref ? sort keys $node->%* : keys $node->@*;

    push $ancestors->@*, $node;
    defer { pop $ancestors->@* };

    my $visit_leaf = !!( $visit & VISIT_LEAF );
    my $visit_node = !!( $visit & VISIT_CONTAINER );

    for my $idx ( @idx ) {

        push $path->@*, $idx;
        defer { pop $path->@* }

        my $vref = \( $is_hashref ? $node->{$idx} : $node->[$idx] );

        my $is_node = is_plain_refref( $vref );

        my $visit_element = $is_node ? $visit_node : $visit_leaf;

        return !!0 if $visit_element && !$code->( $idx, $vref, $context, \%meta );

        if ( is_plain_refref( $vref ) ) {
            my $ref = $vref->$*;
            __SUB__->( $ref, $code, $context, $cycle, $visit, \%meta ) || return !!0
              if is_plain_arrayref( $ref ) || is_plain_hashref( $ref );
        }
    }

    return !!1;
}































































































































































































sub visit ( $struct, $callback, %opts ) {

    is_coderef( $callback )
      or croak( q{parameter 'callback' must be a coderef} );

    my $context = delete $opts{context} // {};
    is_plain_hashref( $context )
      or croak( q{parameter 'context' must be a plain hash} );

    my %metadata = (
        path      => [],
        seen      => {},
        ancestors => [],
        container => undef,
    );

    my $cycle = delete $opts{cycle} // 'die';
    my $visit = delete $opts{visit} // VISIT_ALL;

    $cycle =~ CYCLE_QR
      or croak( "illegal value for cycle parameter: $cycle" );

    %opts
      and croak( 'illegal parameters: ', join( q{, }, keys %opts ) );

    lock_hash( %metadata );
    unlock_value( %metadata, 'container' );
    my $completed = _visit( $struct, $callback, $context, $cycle, $visit, \%metadata );
    unlock_hash( %metadata );

    delete $metadata{ancestors};    # should be empty, but just in case,
                                    # don't want to keep references
                                    # around.

    return ( $completed, $context, \%metadata );
}

1;

#
# This file is part of CXC-Data-Visitor
#
# This software is Copyright (c) 2024 by Smithsonian Astrophysical Observatory.
#
# This is free software, licensed under:
#
#   The GNU General Public License, Version 3, June 2007
#

__END__

=pod

=for :stopwords Diab Jerius Smithsonian Astrophysical Observatory selectable

=head1 NAME

CXC::Data::Visitor - Invoke a callback on every element at every level of a data structure.

=head1 VERSION

version 0.02

=head1 SYNOPSIS

  use CXC::Data::Visitor 'visit';

  my $hoh = { fruit => { berry => 'purple' }, };

  visit(
      $hoh,
      sub {
          my ( $key, $vref ) = @_;
          $$vref = 'blue' if $key eq 'berry';
      } );

  say $hoh->{fruit}{berry}    # 'blue'

=head1 DESCRIPTION

B<CXC::Data::Visitor> provides a means of performing a depth first
traversal of a data structure.  There are similar modules on CPAN
(L</SEE ALSO>); this module provides a few extras:

=over

=item *

The traversal may be aborted.

=item *

User selectable behavior upon detection of a traversal cycle.

=item *

The complete path from the structure route to an element (both the
ancestor containers and the keys and indexes required to traverse the
path) is available.

=back

=head1 SUBROUTINES

=head2 visit

   ( $completed, $context, $metadata ) = visit( $struct, $callback, %opts );

Perform a depth-first traversal of B<$struct>, invoke B<$callback> on
every element (hash entry or array element) in
B<$struct>. B<$callback> is invoked on a container first before being
called on its elements.  Blessed hashes or arrays are not traversed.
Cycles are detected, but nodes which are accessible from multiple
parents are visited more than once.

For example, if

  $struct = { a => { b => [ 0, 1 ], c => 2 } };

The calling order is

  $struct->{a}
  $struct->{a}{b}
  $struct->{a}{b}[0]
  $struct->{a}{b}[1]
  $struct->{a}{b}[2]
  $struct->{a}{c}

The values returned are:

=over

=item B<$completed>  => I<Boolean>

I<true> if all elements were visited, I<false> if
B<$callback> requested a premature return.

=item B<$context> => I<hash>

a hash made available to B<$callback> to stash data

=item B<$metadata> => I<hash>

collected metadata. See L</$metadata> below.

=back

B<%opts> may contain the following entries:

=over

=item B<context> => I<hashref>

A reference to a hash which is passed to L</$callback>. It defaults to
a freshly created hash.

=item B<cycle> => CYCLE_TRUNCATE | CYCLE_DIE | CYCLE_CONTINUE | <$coderef>

How to handle cycles in the data structure (the constants are
available for import; see L</EXPORTS>).  Cycles are detected upon
traversing a node a second time in a depth first search.

Note that a node can be reached multiple times without cycling, for
example, this is not a cycle:g

  %hash = ( a => { b => [ 0, 1 ] }, );
  $hash{c} = $hash{a};

Acceptable values are:

=over

=item CYCLE_DIE

Throw an exception (the default).

=item CYCLE_CONTINUE

Pretend we haven't seen it before. Will cause stack exhaustion if
B<$callback> does handle this.

=item CYCLE_TRUNCATE

Truncate before entering the cycle a second time.

=item I<$coderef>

Examine the situation and request a particular resolution.
B<$coderef> is called as

  $cycle = $coderef->( $container, $context, $metadata );

where B<$container> is the hash or array which has already been
traversed. See below for L</$context> and L</$metadata>.

B<$cycle> should be one of B<CYCLE_DIE>, B<CYCLE_CONTINUE>, or B<CYCLE_TRUNCATE>,
and indicates what should be done.

=back

=item I<visit> VISIT_CONTAINER, VISIT_LEAF, VISIT_ALL

The parts of the structure will trigger a callback
(the constants are available for import; see L</EXPORTS>).

Acceptable values are:

=over

=item VISIT_CONTAINER

Only containers get a callback.  For example, the elements in the following structure

  $struct = { a => { b => 1, c => [ 2, 3 ] } }

which will be passed to L</$callback> are:

  a => {...}  # $struct->{a}
  c => [...]  # $struct->{c}

=item VISIT_LEAF

Only containers get a callback.  For example, the elements in the following structure

  For example, the following structure

  $struct = { a => { b => 1, c => [ 2, 3 ] } }

which will be passed to L</$callback> are:

  b => 1  # $struct->{a}{b}
  0 => 2  # $struct->{a}{c}[0]
  1 => 3  # $struct->{a}{c}[1]

=item VISIT_ALL

All elements get a callback. I<Default>

=back

=back

B<$callback> will be called as:

  $continue = $callback->( $kydx, $vref, $context, $metadata );

and should return B<true> if the process of visiting elements should
I<continue>, B<false> if L</visit> should return immediately.

Its arguments are

=over

=item B<$kydx>

the key or index into B<$container> of the element being visited.

=item B<$vref>

a reference to the value of the element being visited.  Use B<$$vref>
to get the actual value.

=item B<$context>

A hash which can be used by the caller to stash data.

=item B<$metadata>

A hash of state information kept by B<CXC::Data::Visitor>, but which
may be of interest to the callback:

=over

=item B<container>

a reference to the hash or array which contains the element being visited.

=item B<path>

An array which contains the path (keys and indices) used to arrive
at the current element from B<$struct>.

=item B<ancestors>

An array contains the ancestor containers of the current element.

=back

=back

=head1 EXPORTS

This module uses L<Exporter::Tiny>, which provides enhanced import utilities.

The following symbols may be exported:

  visit
  VISIT_CONTAINER VISIT_LEAF VISIT_ALL
  CYCLE_DIE CYCLE_CONTINUE CYCLE_TRUNCATE

The following export tags are available:

=over

=item B<-all>

Import all symbols

=item B<-constants>

Import the following symbols

  VISIT_CONTAINER VISIT_LEAF VISIT_ALL
  CYCLE_DIE CYCLE_CONTINUE CYCLE_TRUNCATE

=back

=head1 SUPPORT

=head2 Bugs

Please report any bugs or feature requests to bug-cxc-data-visitor@rt.cpan.org  or through the web interface at: L<https://rt.cpan.org/Public/Dist/Display.html?Name=CXC-Data-Visitor>

=head2 Source

Source is available at

  https://gitlab.com/djerius/cxc-data-visitor

and may be cloned from

  https://gitlab.com/djerius/cxc-data-visitor.git

=head1 SEE ALSO

Please see those modules/websites for more information related to this module.

=over 4

=item *

L<Data::Rmap|Data::Rmap>

=item *

L<Data::Traverse|Data::Traverse>

=item *

L<Data::Visitor::Lite|Data::Visitor::Lite>

=item *

L<Data::Visitor::Tiny|Data::Visitor::Tiny>

=item *

L<Data::Walk|Data::Walk>

=back

=head1 AUTHOR

Diab Jerius <djerius@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2024 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut
