package Syntax::Highlight::Perl6;

# core modules & directives
use 5.010;
use strict;
use warnings;
use English;
use Carp;
use Encode;
use File::Basename qw(dirname);
use File::Spec;
use Cwd qw(realpath);
require Exporter;

# cpan modules
use YAML::Dumper;
use Term::ANSIColor;

# Larry's STD.pm
use STD;

# exports and version
our @ISA = qw(Exporter);
our @EXPORT_OK = qw();
our @EXPORT = qw();
our $VERSION = '0.01_01';

# These are needed for redspans
$::ACTIONS = 'Actions';

# my module variables
my ($src_text,$parser,@loc);
my $parsed_lazily = 0;

#find out the real path of the rsc directory
croak "Syntax::Highlight::Perl6 cannot see where it is installed"
    unless -f __FILE__;
my $SHARED = realpath(File::Spec->join(
            dirname(__FILE__),"../../rsc"));

#----------------------------------------------------------------
# Returns the syntax highlighting object. It needs a hash 
# of options.
#----------------------------------------------------------------
sub new($%) {
    my ($class, %options) = @ARG;
    $options{rule} = $options{rule} // 'comp_unit';
    $options{clean_html} = $options{clean_html} // 1;
    $options{file} = $options{file} // qq{option 'file' is not set'};
    $options{utf8_decode} = $options{utf8_decode} // 1;

    #is 'text' undefined?
    croak "'text' option is not found in $class->new" 
        if (!$options{text});
    
    return bless(\%options, $class);
}

#----------------------------------------------------------------
# Lazily parses the source string using STD.pm (only once)
# (private)
#----------------------------------------------------------------
sub _lazy_parse($) {
    my $self = shift;
    
    if(!$parsed_lazily) {
    
        # utf8-decode if required
        $src_text = $self->{utf8_decode} ? 
            decode('utf8', $self->{text} ) : 
            $self->{text};
        
        #grow the loc array while checking for empty strings 
        my $len = length($src_text);
        $src_text = " " if $len == 0;
        $loc[$len - 1] = [];

        #STD parse the text for the rule provided
        $parser = STD->parse($src_text, $self->{rule});

        #we parsed it lazily...
        $parsed_lazily = 1;
    }
}
 

#---------------------------------------------------------------------
# Returns snippet htmls which can embedded without any side effects
# on your page
#---------------------------------------------------------------------
sub snippet_html($) {
    my $self = shift;
    my $str = "";
    my %colors = ();

    $self->_lazy_parse();
    
    my $CSS = File::Spec->join($SHARED,"p6_style.css");
    open CSS_FILE, $CSS
        or die "Could not open $CSS: $OS_ERROR\n";
    my $line;
    while($line = <CSS_FILE>) {
        if($line =~ /^\s*\.(\w+)\s*{\s*(.+?)\s*}/) {
            $colors{$1} = $2;
        }
    }
    close CSS_FILE;

    $str .= "<pre>";

    local *spit_snippet_html = sub {
        my ($i, $buffer, $rule, $tree) = @ARG;
        $buffer = _escape_html($buffer);
        my $style = $colors{$rule};
        if($rule) {
            $str .= qq{<span style="$style">$buffer</span>};
        } else {
            $str .= $buffer;
        }
    };
    _redspans_traverse(\&spit_snippet_html,%colors); 

    $str .= "</pre>";

    $str;
}
#---------------------------------------------------------------
# Returns the Perl 6 highlighted HTML string 
# (without the javascript stuff).
#---------------------------------------------------------------
sub simple_html($) {
    my $self = shift;
    my $str = "";
    my %colors = ();

    $self->_lazy_parse();

    my $CSS = File::Spec->join($SHARED,"p6_style.css");
    open CSS_FILE, $CSS
        or die "Could not open $CSS: $OS_ERROR\n";
    my $line;
    while($line = <CSS_FILE>) {
        if($line =~ /^\s*\.(\w+)\s*{\s*color\s*:\s*(\w+)/) {
            $colors{$1} = $2;
        }
    }
    close CSS_FILE;

    # slurp css inline it
    my $css = qq{<link href="$CSS" rel="stylesheet" type="text/css">};
    if(!$self->{clean_html}) {
        $css = _slurp($CSS)
            or die "Error while slurping file: $OS_ERROR\n";
        $css = qq{<style type="text/css">\n$css\n</style>};
    }

    my $file = $self->{file};
    my $timestamp = localtime;
    $str .= <<"HTML";
<html>
<head>
    <title>$file</title>
<!--
     Generated by $PROGRAM_NAME at $timestamp
-->
    $css
</head>
<body>
    <pre>
HTML

    local *spit_simple_html = sub {
        my ($i, $buffer, $rule, $tree) = @ARG;
        $buffer = _escape_html($buffer);
        if($rule) {
            $str .= qq{<span class="$rule">$buffer</span>};
        } else {
            $str .= $buffer;
        }
    };

    _redspans_traverse(\&spit_simple_html,%colors); 

    $str .= <<"HTML";
    </pre>
</body>
</html>
HTML

   $str;
}

#-------------------------------------------------------------------
# Returns the Perl 6 highlighted HTML string. The HTML consists of a
# JavaScript Parse Tree Viewer along with CSS-styling.
#-------------------------------------------------------------------
sub full_html($) {
    my $self = shift;
    my $str = "";

    $self->_lazy_parse();

    # slurp libraries and javascript to inline them
    my ($JQUERY_JS,$JS,$CSS) = (
        File::Spec->join($SHARED,'jquery-1.2.6.pack.js'), 
        File::Spec->join($SHARED,'p6_style.js'),
        File::Spec->join($SHARED,'p6_style.css'));
    my %colors = ();
    my $line;
    open CSS_FILE, $CSS
        or die "Could not open $CSS: $OS_ERROR\n";
    while($line = <CSS_FILE>) {
        if($line =~ /^\s*\.(\w+)\s*{\s*color\s*:\s*(\w+)/) {
            $colors{$1} = $2;
        }
    }
    close CSS_FILE;

    my $jquery_js = qq{<script type="text/javascript" src="$JQUERY_JS"></script>};
    my $js = qq{<script type="text/javascript" src="$JS"></script>};
    my $css = qq{<link href="$CSS" rel="stylesheet" type="text/css">};
    if(!$self->{clean_html}) {
        $jquery_js = _slurp($JQUERY_JS) 
            or die "Error while slurping file: $OS_ERROR\n";    
        $js = _slurp($JS) 
            or die "Error while slurping file: $OS_ERROR\n";
        $css = _slurp($CSS)
            or die "Error while slurping file: $OS_ERROR\n";
        $jquery_js = qq{<script type="text/javascript">\n$jquery_js\n</script>};
        $js = qq{<script type="text/javascript">\n$js\n</script>};
        $css = qq{<style type="text/css">\n$css\n</style>};
    }

    my $file = $self->{file};
    my $timestamp = localtime;
    $str .= <<"HTML";
<html>
<head>
    <title>$file</title>
<!--
    Generated by $PROGRAM_NAME at $timestamp
-->
    $css
    $jquery_js
    $js
</head>
<body>
    <div id="parse_tree">
        <button id="parse_tree_expand">Show Syntax Tree</button>
        <button id="parse_tree_collapse">Hide it</button>
        <button id="parse_tree_help">Help</button>
        <div id="parse_tree_output"></div>
    </div>
    <pre>
HTML

    local *spit_full_html = sub {
        my ($i, $buffer, $rule, $tree) = @ARG;
        $buffer = _escape_html($buffer);
        $str .= qq{<span id="tree_$i" style="display:none;">$tree</span>};
        if($rule) {
            $str .= qq{<span id="node_$i" class="$rule">$buffer</span>};
        } else {
            $str .= $buffer;
        }
    };

    _redspans_traverse(\&spit_full_html,%colors); 

    $str .= <<"HTML";
    </pre>
</body>
</html>
HTML

    $str;
}

#---------------------------------------------------------------
# Returns a Perl highlighted ANSI escape color string.
#---------------------------------------------------------------
sub ansi_text($) {
    my $self = shift;
    my $str = "";
    my %colors = ();

    $self->_lazy_parse();

    my $ANSI = File::Spec->join($SHARED,"p6_style.ansi");
    open ANSI_FILE, $ANSI
        or die "Could not open $ANSI: $OS_ERROR\n";
    my $line;
    while($line = <ANSI_FILE>) {
        if($line =~ /^(\w+)=(.+)$/) {
            $colors{$1} = $2;
        }
    }
    close ANSI_FILE;

    local *spit_ansi_text = sub {
        my ($i, $buffer, $rule, $tree) = @ARG;
        if($rule) {
            my $color = $colors{$rule};
            $str .= (color $color) . $buffer. (color 'reset');
        } else {
            $str .= $buffer;
        }
    };

    _redspans_traverse(\&spit_ansi_text,%colors); 

    $str;
}

#---------------------------------------------------------------
# Returns a Perl 5 array containing parse tree records.
# The array consists of one or more of the following record:
#   ($position, $buffer, $rule_name, $parse_tree)
#---------------------------------------------------------------
sub parse_trees($) {
    my $self = shift;
    my $str = "";
    my %colors = ();

    $self->_lazy_parse();

    my $ANSI = File::Spec->join($SHARED,"p6_style.ansi");
    open ANSI_FILE, $ANSI
        or die "Could not open $ANSI: $OS_ERROR\n";
    my $line;
    while($line = <ANSI_FILE>) {
        if($line =~ /^(\w+)=(.+)$/) {
            $colors{$1} = $2;
        }
    }
    close ANSI_FILE;

    my @parse_trees = ();
    local *spit_parse_tree = sub {
        push @parse_trees, @ARG;
    };

    _redspans_traverse(\&spit_parse_tree,%colors); 

    @parse_trees;
}

#---------------------------------------------------------------
#    Helper private method that traverses STD.pm's parse 
# tree array. It needs a callback process_buffer and a 
# colors hash.
#---------------------------------------------------------------
sub _redspans_traverse($%) {
    my ($process_buffer,%colors) = @ARG;

    my ($last_tree,$buffer, $last_type) = ("","","");
    for my $i (0 .. @loc-1) {
        next unless defined $loc[$i];
        my $c = substr($src_text,$i,1);
        my $tree = "";
        for my $action_ref (@{$loc[$i]}) {
            $tree .= ${$action_ref} . " ";
        }
        if($tree ne $last_tree) {
            my $rule;
            my $rule_to_color = 0;
            $buffer = $buffer;
            my @rules = ();
            @rules = reverse(split / /,$last_tree) if $last_tree ne '';
            for $rule (@rules) {
                if($rule eq 'unv') {
                    $rule_to_color = '_comment';
                    last;
                } elsif($colors{$rule} && $buffer ne '') {
                    $rule_to_color = $rule;
                    last;
                }
            }
            if($rule_to_color) {
                if($last_tree =~ /\sidentifier/) {
                    if($last_type ne '') {
                        $rule_to_color = $last_type;
                        $last_type = '';
                    } 
                } elsif($last_tree =~ /\ssigil/) {
                    given($buffer) {
                        when ('$') { $last_type = '_scalar'; }
                        when ('@') { $last_type = '_array'; }
                        when ('%') { $last_type = '_hash'; }
                        default { $last_type = ''; }
                    }      
                    $rule_to_color = $last_type if $last_type ne ''; 
                }             
            }
            #now delegate printing to a callback
            $process_buffer->($i, $buffer, $rule_to_color, $last_tree); 
            $buffer = $c;
        } else {
            $buffer .= $c;
        }
        $last_tree = $tree;
    }
}

#------------------------------------------------------------------
# R E D S P A N S
# STD.pm calls this method when you call STD->parse(...)
# and we populate @loc with action references and parse trees...
#------------------------------------------------------------------
{ 
    package Actions;

    our $AUTOLOAD;

    my %action_refs = ();

    sub AUTOLOAD {
        my $self = shift;
        my $C = shift;
        my $F = $C->{_from};
        my $P = $C->{_pos};
        $AUTOLOAD =~ s/^Actions:://;
        $loc[$P] = [] if $loc[$P];	# in case we backtracked to here
        my $action = $AUTOLOAD;
        my $action_ref = $action_refs{$action};
        if(!$action_ref) {
            $action_refs{$action} = $action_ref = \$action;
        }
        for ($F..$P-1) {
            unshift @{$loc[$_]}, $action_ref;
        }
    }

    sub stdstopper { }
    sub terminator { }
    sub unitstopper { }
    sub comp_unit { }
}

#---------------------------------------------------------------
# Private method to converts characters to their equivalent 
# html entities.
#----------------------------------------------------------------
sub _escape_html($) {
    my $str = shift;
    my %esc = (
        '<'     => '&lt;',
        '>'     => '&gt;',
        '"'     => '&quot;',
        '&'     => '&amp;'
    );
    my $re = join '|', map quotemeta, keys %esc;
    $str =~ s/($re)/$esc{$1}/g;
    return $str;
}

#-----------------------------------------------------
# Load file into a scalar without File::Slurp
# Thanks ofcourse to this post
# http://www.perlmonks.org/?node_id=20235
#-----------------------------------------------------
sub _slurp($) {
    local $/ =<> if local @ARGV = @_
}

1;

#------------------ T H E   E N D --------------------

__END__

=head1 NAME

Syntax::Highlight::Perl6 - Perl 6 syntax highlighter

=head1 SYNOPSIS

    ### NOTE: This is needed and will be removed in future releases
    use STD;
    use Syntax::Highlight::Perl6;

    # Creates the Perl6 syntax highlighter object
    my $p = Syntax::Highlight::Perl6->new(
        text => 'my $foo;'
    );

    # Prints html that can be embedded in your pages
    print $p->snippet_html;

    # Prints html with css (useful for full pages)
    print $p->simple_html;

    # Prints html that has a JavaScript parse tree viewer
    print $p->full_html;

    # Prints ANSI escaped color sequences (useful for console and IRC output)
    print $p->ansi_text;

    # Prints the Perl 5 array of parse trees (useful for building stuff on top of it) 
    print $p->parse_trees;

=head1 DESCRIPTION

Highlights Perl 6 source code. The code is parsed using Larry Wall's STD.pm. 

The available output formats are:

=over

=item *
    HTML (snippet,simple and full)

=item *
    ANSI escaped color sequences

=item *
    Perl 5 array of parse trees

=back

=head1 METHODS

=over 4

=item new(options)

Returns the syntax highlighting object. It needs a hash of options.

=over 4

=item text

This is a B<required> option. 
This is where you should provide the Perl 6 code.

=item rule

parse rule name for STD.pm to parse against (default: B<comp_unit>)

=item clean_html

Flag to enable/disable CSS/JavaScript HTML inlining. (default: 1)

=item file

file name string in HTML modes (default: a warning message)

=item utf8_decode

Flag to enable/disable utf8 decoding. (default: 1)

=back

=item snippet_html

Returns the Perl 6 highlighted HTML string that can be embedded. 
No CSS or JavaScript is inside.

=item simple_html

Returns the Perl 6 highlighted HTML string. The HTML is the same as 
C<full_html> but lacks a JavaScript Parse Tree Viewer. 

=item full_html

Returns the Perl 6 highlighted HTML string. The HTML consists of a 
JavaScript Parse Tree Viewer along with CSS-styling. 
It can inlined if C<clean_html> option is 0. 

=item ansi_text

Returns a Perl highlighted ANSI escape color string.

=item parse_trees

Returns a Perl 5 array containing parse tree records.
The array consists of one or more of the following record:
  ($position, $buffer, $rule_name, $parse_tree)

=back

=head1 SEE ALSO

Discussion about this module is usually in #perl6 @ irc.freenode.net

=head1 AUTHOR

Ahmad Zawawi, E<lt>ahmad.zawawi@gmail.comE<gt>

Written by Ahmad M. Zawawi (azawawi), Moritz Lenz and Paweł Murias (pmurias)

The project idea was inspired by Moritz Lenz (moritz)
See http://www.nntp.perl.org/group/perl.perl6.users/2008/07/msg788.html

The initial STD tree traversal code was written by Paweł Murias (pmurias).

A C<gimme5>-generated Perl5 C<STD.pmc> is included to parse Perl 6 code.

The C<redspans> traversal code was written by Larry Wall (TimToady).

C<redspans> stands for C<red> for reductions, and C<spans> from the 
from/to span calculations".

The javascript jQuery code was written by Ahmad M. Zawawi (azawawi)

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2008 by Ahmad Zawawi

This library is free software; you can redistribute it and/or modify
it under the same terms as the Artistic License 2.0

This library also includes the following libraries:

STD.pm by Larry Wall (Artistic License 2.0 - same license)

JQuery 1.2.6 by John Resig (dual licensed under the MIT and GPL licenses).
