#!/usr/bin/env perl

use Test::More;
use Math::BigInt try => 'GMP,Pari';
use strict;
use warnings;
no strict 'refs';

use lib '../lib';

our $module;
BEGIN {
    our $module = 'Crypt::MagicSignatures::Key';
    use_ok($module, qw/b64url_encode b64url_decode/);   # 1
};

my $test_msg = 'This is a small message test.';

# The RSA Validation System (RSAVS)
# B.1.3 SigVerRSA.req
# CAVS 3.2
# "SigVer RSA (X9.31)" information for "testshas"
# Mod sizes selected: 1024 1536
# Generated on Wed Apr 28 08:35:11 2004
# sha-256


my $mod = 1024;
my ($e, $m, $s);

my $n = Math::BigInt->from_hex(
  '9ec4d483330916b69eee4e9b7614eafc4fbf60e74b5127a3ff5bd9d48c7ecf8418d94d1e60388b' .
    'b68546f8bc92deb1974b9def6748fbb4ec93029ea8b7bea36f61c5c6aeedfd512a0f765846fad5' .
      'edacb08c3d75cf1d43b48b394c94323c3f3e9ba6612f93fe2900134217433afb088b5ca33fc4e6' .
	'b270194df077d2b6592743');

# Exponent
# $e = Math::BigInt->from_hex('0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003');
$e = 3;

# Message
$m = '7c5177b2ef9ecc43c6b2048397d70f2b7dc98feabec59815aee4b49bd0a72b373fd381e94c7f3f' .
  'a6696bf74f469382e039048ceae3ef534311fcabfbe0e046932532326a0b7aa378fe8cf33ec814' .
  'e7fdfa7134278ec74113ca4f2ff468f2170cf317921d74b97f214ebc003a6781c6ec88b88f8a07' .
  '75eceea386486daf05260b';

# Signature
$s = '01da3b0936cc9e6261e80595e46ea228c93cb7f348b2cace6a5a2704eba204b96d5cb9e29cd2cb' .
  '9ba968eeab994294e5f4fa2c6d44b52bc8768a802c4bc8201f267fc9e6dfea53b98677f21a77e7' .
  '178ae0166151470f628831afa59203b6a233f133544d51669eb2e5de159ed3819ef0cc50474471' .
  '16351b78ee6831e9498746';

my $mkey = Crypt::MagicSignatures::Key->new(n => $n, e => $e);

# Key length matches modulus
is(*{"${module}::_bitsize"}->($mkey->n), $mod, 'Key length');
is(*{"${module}::_octet_len"}->($mkey->n), ($mod / 8), 'Key length');
is($mkey->size, 1024, 'Correct Key size');

done_testing;

exit;









print $mkey->verify($m, $s);


$m = '7ad2f4bfbed0dba767ec7f106f4750376f2945c4c09624fbe022fe361706f8935a7252ea6f25a1' .
  '02523c5f04d847a62f92a239cef403c467b64f65367bb26ad9b1ee5d4db8f33e1946b10fc90a2a' .
  '969e8fcb5e8464fcff447af69ffbcdd4b9cb46ed1dd0e06238560bf396494e17a5ec2f4bbcce57' .
  'aa5bfbf2beb56f55966bd0';

$s = '009f3e544f38658c3ab1af8a09623cb611167908c01eced7863a93d417d76098d5148485669c11' .
  '9adfbe0d7d0cda483e788c0c5b8186c192156a9a54d75d462f0da558978a7b12fe2baf9c07b3c4' .
  '191d4bf15d5f66a1c5f7079b8a535e95638a4dbf7095ef4e147b8fdd3e3498f13853710f44f778' .
  'ec6b79e95646cbb27414e4';

unless ($mkey->verify($m, $s)) {
  diag 'not fine.';
};

$m = 'f5de826b61d81957cc4fcf26c959f1432c4b0f4f1b7fac0b685439791e77453e5961ee4b5b219b' .
  'bcdd5ced00a392f23b53a29ce8879172c3218786e6df1aa7322fcfd7f044de00b86936e29295c1' .
  '505c40e99c6c765b50762a0b1eafcc781a321e3127a34398af1318e69824c86f736e9b28f6210f' .
  '66aceb2ae8eb1c0e180708';

$s = '35f86e0912d099298062967bb41f6d8dadba532ecc9a66f9ad51c5dbb8de8fb29b06f8a022c4d2' .
  '8a18e7a5f9515fab51b428b7a73957fd877fcaff2fe4d3a026dae0388747cb1fbd69b20df0781b' .
  '41bfd5692f0fa4033a0399479c1a6036f8d27a9bd2018ebeb736a098090bf5ed791b9cb12cb963' .
  'ebd03dd46ffcee68b95b4b';

print $mkey->verify($m, $s);

$e = '000000000000000000000000000000000000000000000000000000000000000000000000000000' .
  '000000000000000000000000000000000000000000000000000000000000000000000000000000' .
  '000000000000000000000000000000000000000000000000000000000000000000000000000000' .
  '0000000000000000010001';

$m = '489b177653809b9921b178eca7ddf8a31df19e45b9d40b02be551e46b5625f8efa7a9e7b7b64d7' .
  '24bd2259b12021272663b29f7c6abe59f63fb452b258c74a7f18576aee97ac2aaca6ea720e0e41' .
  'ffee196509b6543e23ba92e062cb34bdc108a819c4f830bf5cd6e5f30b2cfba748a446f2251afd' .
  'bd2ff5bfc096b8d3ad8ed4';

$s = '494ce82c0af37d1b222d381b4383994f60b4897b3f6314c167bf679507436fc9f5cb6d7309d9c5' .
  '0ffe0a0838c4d2874824c78cd55a8f34654b53d9bce3989d779556b51560d92b9031ffa7f8b72d' .
  'fc6f607e829e467b17affff854ba524b6df27e53d6ef605859d2e24ebdb84db49c6af92496677d' .
  'a4d173cc054b68eb065b7f';


diag $mkey->verify($m, $s);





__END__

SHAAlg = SHA256
Msg =

S =

SHAAlg = SHA256
e =
000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000003


SHAAlg = SHA256


e =
000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000003




SHAAlg = SHA256
e =

__END__





my $mod = 1024;
my ($e, $m);

my $n = Math::BigInt->from_hex('9ec4d483330916b69eee4e9b7614eafc4fbf60e74b5127a3ff5bd9d48c7ecf8418d94d1e60388bb68546f8bc92deb1974b9def6748fbb4ec93029ea8b7bea36f61c5c6aeedfd512a0f765846fad5edacb08c3d75cf1d43b48b394c94323c3f3e9ba6612f93fe2900134217433afb088b5ca33fc4e6b270194df077d2b6592743');

# Exponent
#$e = Math::BigInt->from_hex('0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003');

$e = 3;

# Message
$m = '7c5177b2ef9ecc43c6b2048397d70f2b7dc98feabec59815aee4b49bd0a72b373fd381e94c7f3fa6696bf74f469382e039048ceae3ef534311fcabfbe0e046932532326a0b7aa378fe8cf33ec814
e7fdfa7134278ec74113ca4f2ff468f2170cf317921d74b97f214ebc003a6781c6ec88b88f8a0775eceea386486daf05260b';

# Signature
my $s = Math::BigInt->from_hex('01da3b0936cc9e6261e80595e46ea228c93cb7f348b2cace6a5a2704eba204b96d5cb9e29cd2cb9ba968eeab994294e5f4fa2c6d44b52bc8768a802c4bc8201f267fc9e6dfea53b98677f21a77e7178ae0166151470f628831afa59203b6a233f133544d51669eb2e5de159ed3819ef0cc5047447116351b78ee6831e9498746');

my $mkey = Crypt::MagicSignatures->new(n => $n, e => $e);

# Key length matches modulus
is(*{"${module}::_bitsize"}->($n), $mod, 'Key length');
is(*{"${module}::_octet_len"}->($n), ($mod / 8), 'Key length');

print $mkey->verify($m, b64url_encode($s));

__END__

foreach ($n, $s, $m) { #  $e
  $_ = Math::BigInt->from_hex( "0x" . $_ );
};


warn '*'.$mkey->verify($m, b64url_encode(*{"${module}::_i2osp"}->($s, $mod))).'*';

__END__

###
# $e = '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003';
# $mod = 1536;
$m = '7ad2f4bfbed0dba767ec7f106f4750376f2945c4c09624fbe022fe361706f8935a7252ea6f25a102523c5f04d847a62f92a239cef403c467b64f65367bb26ad9b1ee5d4db8f33e1946b10fc90a2a969e8fcb5e8464fcff447af69ffbcdd4b9cb46ed1dd0e06238560bf396494e17a5ec2f4bbcce57aa5bfbf2beb56f55966bd0';
$s = '009f3e544f38658c3ab1af8a09623cb611167908c01eced7863a93d417d76098d5148485669c119adfbe0d7d0cda483e788c0c5b8186c192156a9a54d75d462f0da558978a7b12fe2baf9c07b3c4191d4bf15d5f66a1c5f7079b8a535e95638a4dbf7095ef4e147b8fdd3e3498f13853710f44f778ec6b79e95646cbb27414e4';

foreach ($s, $m) {
  $_ = Math::BigInt->from_hex( "0x" . $_ );
};
#$mkey = Crypt::MagicSignatures->new(n => $n, e => $e);

is(*{"${module}::_bitsize"}->($n), $mod, 'Key length');
is(*{"${module}::_octet_len"}->($n), ($mod / 8), 'Key length');

warn '*'.$mkey->verify($m, b64url_encode(*{"${module}::_i2osp"}->($s, $mod))).'*';

__END__
###
$e = '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003';
$m = 'f5de826b61d81957cc4fcf26c959f1432c4b0f4f1b7fac0b685439791e77453e5961ee4b5b219bbcdd5ced00a392f23b53a29ce8879172c3218786e6df1aa7322fcfd7f044de00b86936e29295c1505c40e99c6c765b50762a0b1eafcc781a321e3127a34398af1318e69824c86f736e9b28f6210f66aceb2ae8eb1c0e180708';
$s = '35f86e0912d099298062967bb41f6d8dadba532ecc9a66f9ad51c5dbb8de8fb29b06f8a022c4d28a18e7a5f9515fab51b428b7a73957fd877fcaff2fe4d3a026dae0388747cb1fbd69b20df0781b41bfd5692f0fa4033a0399479c1a6036f8d27a9bd2018ebeb736a098090bf5ed791b9cb12cb963ebd03dd46ffcee68b95b4b';

###
$e = '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010001';
$m = '489b177653809b9921b178eca7ddf8a31df19e45b9d40b02be551e46b5625f8efa7a9e7b7b64d724bd2259b12021272663b29f7c6abe59f63fb452b258c74a7f18576aee97ac2aaca6ea720e0e41ffee196509b6543e23ba92e062cb34bdc108a819c4f830bf5cd6e5f30b2cfba748a446f2251afdbd2ff5bfc096b8d3ad8ed4';
$s = '494ce82c0af37d1b222d381b4383994f60b4897b3f6314c167bf679507436fc9f5cb6d7309d9c50ffe0a0838c4d2874824c78cd55a8f34654b53d9bce3989d779556b51560d92b9031ffa7f8b72dfc6f607e829e467b17affff854ba524b6df27e53d6ef605859d2e24ebdb84db49c6af92496677da4d173cc054b68eb065b7f'
