#ifndef __GL_MPEG_H
#define __GL_MPEG_H

#include <stdio.h>
#include <malloc.h>
#include "mpeg.h"

#ifdef DEBUG
# define dprintf printf
#else
# define dprintf (void) 
#endif

#define max(x,y) (((x)>(y))?(x):(y))
#define min(x,y) (((x)<(y))?(x):(y))

#define ISMOUSEBUT(t) (((t) >= MOUSE1) && ((t) <= MOUSE3))
#define NUM_BUFFER_TYPES 3
#define BUFFER_UNIQUE_LEN 1

typedef enum { NO_BUFFER, MEMORY_BUFFER, DISK_BUFFER } BufferEnum;
typedef enum { FORWARD, BACKWARD, ROCK } StyleEnum;

typedef struct
{
   long   Handle;
   long   Height, Width;	/* window size */
   float  ZoomX, ZoomY;		/* zoom factors */
   long   ImageX, ImageY;	/* where to put image in window */
} WindowDesc;


/*
 * Note: the default values that this player uses for the fields of
 * OptionFlags are set (rather obscurely, I must admit) in the
 * function ParseCmdLine() in parseargs.c.
 */
typedef struct
{
   /* User-settable options */

   unsigned int Continuous: 1;	/* continuous or single-play? */	
   unsigned int DoubleBuff: 1;	/* double or single buffered? */
   unsigned int FullScreen: 1;	/* full screen or windowed mode? */
   unsigned int Preview   : 1;	/* play movie while decoding? */
} OptionFlags;



typedef struct
{
   /* Movie status flags */

   unsigned int Paused    : 1;	/* paused or playing? */
   unsigned int Decoding  : 1;	/* currently decoding? (can't single step) */
   unsigned int Forward   : 1;	/* currently playing forwards or back? */
} StatusFlags;


/*
 * MovieState - the all-purpose, whole-wheat, high-fibre, low-fat
 * structure containing damn near everything we could want to know
 * about a movie currently being played.  About the only stuff NOT
 * in this structure is:
 *    - anything pertaining to frame buffering (NOT to be confused
 *      with double buffering) - see buffers.c
 *    - GL-specific stuff, such as DisplayName (note that the flag
 *      DoubleBuff could easily be construed as GL-specific...so
 *      sue me)
 *
 * Also, note that MOST of the data types used are defined in this 
 * file; however, ImageDesc comes from the MPEG library (specifically,
 * the MNI front end to it - see mpeg.h).
 */
typedef struct
{
   char       *Filename;
   FILE       *InputStream;
   WindowDesc  Window;		/* window size/zoom factor/etc */
   ImageDesc   Image;		/* image size/depth/etc */
   StyleEnum   PlayStyle;	/* play forwards, backwards, or both? */
   int         FrameDelay;	/* hundredths of a second between frames */
   int         TotFrames;
   int         CurFrame;
   OptionFlags Options;		/* user-settable options */
   StatusFlags Status;		/* status of current playback */
   char       *CurrentImage;
} MovieState;


/* Global variables exported by gl_mpeg.c (the main program) */

extern char       *DisplayName;
extern BufferEnum  BufferType;
extern BufferEnum  BufferFallback;
extern int         MaxMemory;
extern Boolean     KeepTempFlag;
extern Boolean     QuietFlag;

extern char       *BufferNames [];
extern char       *StyleNames[];

/* Other global variables... */

extern int         HelpWindowID; /* from help.c, used in interface.c */


/* Prototypes for functions in gl_mpeg.c and used elsewhere: */

double ReadSysClock (void);
int qprintf (char *format, ...);
Boolean AdvanceFrame (MovieState *Movie);

/* Functions in parseargs.c: */

void ParseCmdLine (int *pargc, char *argv[], MovieState *Movie);

/* Functions in display.c: */

void CalcZoomFromWindow (WindowDesc *WinInfo, ImageDesc *ImageInfo, 
			 Boolean KeepAspect);
void CalcWindowFromZoom (WindowDesc *WinInfo, ImageDesc *ImageInfo);
void CalcImageLocation (WindowDesc *WinInfo, ImageDesc *ImageInfo);
void FullScreenMode (MovieState *Movie);
void WindowMode (MovieState *Movie, long PrefX, long PrefY);
Boolean OpenWindow (MovieState *Movie);
Boolean InitColorDisplay(MovieState *Movie);
void DisplayFrame (MovieState *Movie);


/* Functions in buffers.c: */

Boolean SetupBuffer (BufferEnum BufType, 
		     ImageDesc *ImageInfo, 
		     int MaxMemory, int MaxFrames);
void CloseBuffer (BufferEnum BufType, int NumFrames);
Boolean SaveFrame (int FrameNum, ImageDesc *ImageInfo, char *ImageData);
char *LoadFrame (int FrameNum, ImageDesc *ImageInfo);

/* Functions in interface.c: */

void InitializeQueues (void);
void SetPause (MovieState *Movie, Boolean Pause);
void HandleEvents (Boolean *Done, MovieState *Movie);


/* Functions in help.c: */

void OpenHelpWindow (MovieState *Movie);
void CloseHelpWindow ();
void WriteHelpWindow (MovieState *Movie);
void UpdateStatus (MovieState *Movie);


#endif    /* __GL_MPEG_H */
