/*
 * Copyright (c) 1992-93 by Mark Boyns (boyns@sdsu.edu)
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include "version.h"
#include "conf.h"
#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/fcntl.h>
#include <sys/time.h>
#include <sys/stat.h>
#ifdef MMAP
#include <sys/mman.h>
#endif MMAP
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <search.h>
#include <strings.h>
#include <errno.h>
#include "libst.h"
#include "rplay.h"

#define SPOOL_SIZE		12		/* maximum number of simultaneous sounds */
#define AUDIO_BUFSIZE		8000		/* size of the audio buffer */
#define AUDIO_DEVICE		"/dev/audio"	/* name of the audio device */
#define MAX_SOUNDS		4096		/* for hsearch */
#define SOUND_LIST_SIZE		256		/* maximum size of a sound list */
#define SELECT_TIMEOUT		1		/* seconds */
#define AUDIO_TIMEOUT		5		/* close audio device after 5 idle seconds */
#define RPLAYD_TIMEOUT		60		/* exit after 60 idle seconds */
#define FDSET_NBITS		5		/* number of bits needed for select */
#define ULAW_MAGIC		0x2e736e64	/* ulaw file header */
#define ULAW_HDRSIZE		24		/* ulaw minimum header size */
#define RECVFROM_ATTEMPTS	12		/* number of times to poll the socket */

typedef struct
{
	char	*name;
	char	path[MAXPATHLEN];
	char	*buf;
	char	*start;
	char	*stop;
	int	size;
} SOUND;

typedef struct spool
{
	int		state;
	int		curr_sound;
	SOUND		*sound[SOUND_LIST_SIZE];
	char		*ptr;
	char		*end;
	RPLAY		*rp;
	RPLAY_ATTRS	*curr_attrs;
} SPOOL;
SPOOL	spool[SPOOL_SIZE];

#ifdef AUTH
#define ADDR_HASH(addr)	(0x000000ff&(addr))

typedef struct addr
{
	struct addr	*next;
	u_long		addr;
} ADDR;

ADDR	*addr_table[256];
#endif AUTH

char		audio_buf[AUDIO_BUFSIZE];
char		recv_buf[MAX_PACKET];
int		sock_fd;
int		audio_fd = -1;
int		spool_size = 0;
int		audio_size = 0;
char		*progname;
int		debug = 0;
int		inetd = 1;
int		audio_timeout = AUDIO_TIMEOUT;
int		rplayd_timeout = RPLAYD_TIMEOUT;
struct timeval	select_timeout = { SELECT_TIMEOUT, 0 };
extern char	*sys_errlist[];
extern char	*optarg;
extern int	optind;

#define spool_clear(sp) \
	(sp)->curr_sound = 0; \
	if ((sp)->rp) rplay_destroy((sp)->rp); \
	(sp)->rp = NULL; \
	(sp)->ptr = NULL; \
	(sp)->end = NULL; \
	(sp)->state = RPLAY_NULL;

#ifdef __STDC__
void	conf_read(void);
void	hosts_read(void);
int	addr_lookup(u_long addr);
ENTRY	*hsearch(ENTRY, ACTION);
SOUND	*sound_lookup(char *);
SOUND	*sound_create(void);
void	spool_init(void);
SPOOL	*spool_match(RPLAY *);
SPOOL	*spool_next(void);
void	audio_open(void);
void	audio_close(void);
void	audio_write(void);
void	socket_open(void);
void	socket_close(void);
void	socket_read(void);
void	usage(void);
void	done(int x);
#else
void	conf_read();
void	hosts_read();
int	addr_lookup();
ENTRY	*hsearch();
SOUND	*sound_lookup();
SOUND	*sound_create();
void	spool_init();
SPOOL	*spool_next();
SPOOL	*spool_match();
void	audio_open();
void	audio_close();
void	audio_write();
void	socket_open();
void	socket_close();
void	socket_read();
void	usage();
void	done();
#endif

#ifdef __STDC__
main(int argc, char **argv)
#else
main(argc, argv)
int	argc;
char	**argv;
#endif
{
	int	nfds, tcount = 0, c;
	fd_set	rfds;

	progname = argv[0];

	while ((c = getopt(argc, argv, "c:dhnt:v")) != -1)
	{
		switch (c)
		{
			case 'c':
				audio_timeout = atoi(optarg);
				break;

			case 'd':
				debug = 1;
				break;

			case 'n':
				inetd = 0;
				break;

			case 't':
				rplayd_timeout = atoi(optarg);
				break;

			case 'v':
				printf("%s: %s\n", progname, rplay_version);
				done(0);
				break;

			case '?':
			case 'h':
				usage();
				done(1);
		}
	}

	conf_read();
#ifdef AUTH
	hosts_read();
#endif AUTH
	spool_init();
	socket_open();
	audio_open();
	FD_ZERO(&rfds);

	for (;;)
	{
		if (spool_size)
		{
			socket_read();
			audio_write();
		}
		else
		{
			FD_SET(sock_fd, &rfds);
			nfds = select(FDSET_NBITS, &rfds, 0, 0, &select_timeout);
			switch (nfds)
			{
				case -1:
					perror("select");
					done(1);

				case 0:
					tcount++;
					if (audio_timeout && tcount == audio_timeout)
					{
						audio_close();
					}
					if (rplayd_timeout && tcount == rplayd_timeout)
					{
						done(0);
					}
					break;

				default:
					tcount = 0;
					if (FD_ISSET(sock_fd, &rfds))
					{
						socket_read();
					}
					audio_write();
					break;
			}
		}
	}
}

/*
 * read the rplay configuration file and load the hash table
 */
#ifdef __STDC__
void	conf_read(void)
#else
void	conf_read()
#endif
{
	SOUND		*s;
	ENTRY		e;
	FILE		*fp;
	char		buf[MAXPATHLEN], *p;

	fp = fopen(RPLAY_CONF, "r");
	if (fp == NULL)
	{
		fprintf(stderr, "%s: cannot open %s\n", progname, RPLAY_CONF);
		done(1);
	}

	if (hcreate(MAX_SOUNDS) == 0)
	{
		fprintf(stderr, "%s: cannot create hash table\n", progname);
		done(1);
	}

	while (fgets(buf, sizeof(buf), fp) != NULL)
	{
		switch (buf[0])
		{
			case '#':
			case ' ':
			case '\t':
			case '\n':
				continue;
		}

		s = sound_create();
		sscanf(buf, "%s", s->path);
		p = rindex(s->path, '/');
		s->name = p == NULL ? s->path : p+1;

		e.key = s->name;
		e.data = (char *)s;
		if (hsearch(e, ENTER) == NULL)
		{
			fprintf(stderr, "%s: the hash table is full (MAX_SOUNDS=%d)\n",
				progname, MAX_SOUNDS);
			done(1);
		}
	}
	fclose(fp);
}

/*
 * read the rplay hosts file and load the addr hash table
 */
#ifdef AUTH
#ifdef __STDC__
void	hosts_read(void)
#else
void	hosts_read()
#endif
{
	FILE		*fp;
	char		buf[BUFSIZ], *p;
	u_long		addr;
	struct hostent	*hp;
	ADDR		*a;
	int		hval;

	fp = fopen(RPLAY_HOSTS, "r");
	if (fp == NULL)
	{
		fprintf(stderr, "%s: cannot open %s\n", progname, RPLAY_HOSTS);
		done(1);
	}
	
	bzero((char *)addr_table, sizeof(addr_table));

	while (fgets(buf, sizeof(buf), fp))
	{
		switch (buf[0])
		{
			case '#':
			case ' ':
			case '\t':
			case '\n':
				continue;
		}

		p = index(buf, '\n');
		*p = '\0';
		addr = inet_addr(buf);
		if (addr == 0xffffffff)
		{
			hp = gethostbyname(buf);
			if (hp == NULL)
			{
				fprintf(stderr, "%s: %s unknown host in %s\n", progname, buf, RPLAY_HOSTS);
				done(1);
			}
			bcopy(hp->h_addr, &addr, hp->h_length);
		}
		a = (ADDR *)malloc(sizeof(ADDR));
		a->addr = addr;
		hval = ADDR_HASH(a->addr);
		a->next = addr_table[hval];
		addr_table[hval] = a;
	}
	fclose(fp);
}
#endif AUTH

/*
 * lookup the given address in the addr hash table
 */
#ifdef AUTH
#ifdef __STDC__
int	addr_lookup(u_long addr)
#else
int	addr_lookup(addr)
u_long	addr;
#endif
{
	ADDR	*a;

	for (a = addr_table[ADDR_HASH(addr)]; a; a = a->next)
	{
		if (a->addr == addr)
		{
			return 1;
		}
	}

	return 0;
}
#endif AUTH

/*
 * initialize the sound spool
 */
#ifdef __STDC__
void	spool_init(void)
#else
void	spool_init()
#endif
{
	int	i;

	 for(i = 0; i < SPOOL_SIZE; i++)
	 { 
		 spool_clear(&spool[i]);
	 }
}

/*
 * return the next available spool entry
 */
#ifdef __STDC__
SPOOL	*spool_next(void)
#else
SPOOL	*spool_next()
#endif
{
	int	i;

	for (i = 0; i < SPOOL_SIZE; i++)
	{
		if (spool[i].state == RPLAY_NULL)
		{
			return &spool[i];
		}
	}

	return NULL;
}

/*
 * find the given RPLAY in the spool
 */
#ifdef __STDC__
SPOOL	*spool_match(RPLAY *match)
#else
SPOOL	*spool_match(match)
RPLAY	*match;
#endif
{
	int		i;
	RPLAY		*rp;
	RPLAY_ATTRS	*a1, *a2;

	for (i = 0; i < SPOOL_SIZE; i++)
	{
		if (spool[i].state == RPLAY_NULL)
		{
			continue;
		}

		rp = spool[i].rp;
		if (rp->nsounds == match->nsounds)
		{
			for (a1 = rp->attrs, a2 = match->attrs; a1 && a2; a1 = a1->next, a2 = a2->next)
			{
				if (strcmp(a1->sound, a2->sound))
				{
					break;
				}
				if (a1->volume != a2->volume)
				{
					break;
				}
			}
			if (!a1 && !a2)
			{
				return &spool[i];
			}
		}
	}

	return NULL;
}

/*
 * create a sound
 */
#ifdef __STDC__
SOUND	*sound_create(void)
#else
SOUND	*sound_create()
#endif
{
	SOUND	*s;

	s = (SOUND *)malloc(sizeof(SOUND));
	s->path[0] = '\0';
	s->name = NULL;
	s->buf = NULL;
	s->start = NULL;
	s->stop = NULL;
	s->size = 0;

	return s;
}

/*
 * lookup the sound name in the hash table and load
 * the sound file if necessary
 */
#ifdef __STDC__
SOUND	*sound_lookup(char *name)
#else
SOUND	*sound_lookup(name)
char	*name;
#endif
{
	ENTRY	e, *found;
	SOUND	*s;
	char	*p;

	/*
	 * check for absolute path
	 */
	if (*name == '/')
	{
		p = rindex(name, '/');
		p++;
	}
	else
	{
		p = name;
	}

	e.key = p;
	found = hsearch(e, FIND);
	if (found == NULL)
	{
		return NULL;
	}

	s = (SOUND *)found->data;

	/*
	 * if absolute path make sure path matches
	 */
	if (*name == '/' && strcmp(name, s->path))
	{
		return NULL;
	}

	if (s->buf == NULL)
	{
		struct stat     st;
		long            ulaw_hdr_size;
		int             fd;

		fd = open(s->path, O_RDONLY, 0);
		if (fd < 0)
		{
			return NULL;
		}

		if (fstat(fd, &st) < 0)
		{
			return NULL;
		}

		s->size = st.st_size;

#ifdef MMAP
		s->buf = mmap(0, s->size, PROT_READ, MAP_SHARED, fd, 0);
		if (s->buf == (caddr_t)-1)
		{
			if (debug)
			{
				fprintf(stderr, "%s: file=%s size=%s mmap: %s\n",
					progname, s->path, s->size, sys_errlist[errno]);
			}
			return NULL;
		}
#else MMAP
		s->buf = (char *)malloc(s->size);
		if (read(fd, s->buf, s->size) != s->size)
		{
			if (debug)
			{
				fprintf(stderr, "%s: %s read: %s\n", progname, s->path,
					sys_errlist[errno]);
			}
			return NULL;
		}
#endif MMAP

		/*
		 * make sure it is a ulaw audio file
		 */
		if (*((long *)s->buf) != ULAW_MAGIC)
		{
			if (debug)
			{
				fprintf(stderr, "%s: %s not a ulaw file\n", progname, s->path);
			}
			return NULL;
		}

		/*
		 * ignore the ulaw header
		 */
		ulaw_hdr_size = *((long *)s->buf + 1);
		if (ulaw_hdr_size < ULAW_HDRSIZE)
		{
			if (debug)
			{
				fprintf(stderr, "%s: %s not a ulaw file\n", progname, s->path);
			}
			return NULL;
		}

		s->start = s->buf + ulaw_hdr_size;
		s->stop = s->buf + s->size - 1;

		close(fd);
	}

	return s;
}

/*
 * open the audio device
 */
#ifdef __STDC__
void	audio_open(void)
#else
void	audio_open()
#endif
{
	if (debug)
	{
		fprintf(stderr, "%s: opening %s\n", progname, AUDIO_DEVICE);
	}
	audio_fd = open(AUDIO_DEVICE, O_WRONLY|O_NDELAY, 0);
	if (audio_fd < 0 && debug)
	{
		fprintf(stderr, "%s: %s open: %s\n", progname, AUDIO_DEVICE, sys_errlist[errno]);
	}
}

/*
 * close the audio device
 */
#ifdef __STDC__
void	audio_close(void)
#else
void	audio_close()
#endif
{
	if (audio_fd > 0)
	{
		if (debug)
		{
			fprintf(stderr, "%s: closing %s\n", progname, AUDIO_DEVICE);
		}
		close(audio_fd);
	}
	audio_fd = -1;
}

/*
 * set up the socket to receive rplay packets
 */
#ifdef __STDC__
void	socket_open(void)
#else
void	socket_open()
#endif
{
	struct sockaddr_in	s;
	int			flags;

	if (inetd)
	{
		sock_fd = 0;
	}
	else
	{
		s.sin_family = AF_INET;
		s.sin_port = RPLAY_PORT;
		s.sin_addr.s_addr = INADDR_ANY;

		sock_fd = socket(AF_INET, SOCK_DGRAM, 0);
		if (sock_fd < 0)
		{
			fprintf(stderr, "%s: socket: %s\n", progname, sys_errlist[errno]);
			done(1);
		}

		if (bind(sock_fd, &s, sizeof(s)) < 0)
		{
			fprintf(stderr, "%s: bind: %s\n", progname, sys_errlist[errno]);
			done(1);
		}
	}

	/*
	 * set the socket to non-blocking so it can be polled
	 * between selects
	 */
	flags = fcntl(sock_fd, F_GETFL, 0);
	if (flags < 0)
	{
		fprintf(stderr, "%s: F_GETFL fcntl: %s\n", progname, sys_errlist[errno]);
		done(1);
	}
	flags |= FNDELAY;
	if (fcntl(sock_fd, F_SETFL, flags) < 0)
	{
		fprintf(stderr, "%s: F_SETFL fcntl: %s\n", progname, sys_errlist[errno]);
		done(1);
	}
}

/*
 * close the socket
 */
#ifdef __STDC__
void	socket_close(void)
#else
void	socket_close()
#endif
{
	close(sock_fd);
	sock_fd = -1;
}

/*
 * read rplay packets from the socket
 */
#ifdef __STDC__
void	socket_read(void)
#else
void	socket_read()
#endif
{
	char			*p, *packet;
	int			attr;
	struct sockaddr_in	f;
	int			len, flen = sizeof(f);
	SOUND			*sound;
	SPOOL			*sp;
	int			i, j;
	RPLAY			*rp;

	for (j = 0; j < RECVFROM_ATTEMPTS; j++)
	{
		if (recvfrom(sock_fd, recv_buf, sizeof(recv_buf), 0, &f, &flen) <= 0)
		{
			usleep(1); /* believe me this helps xtank a lot */
			continue;
		}
#ifdef AUTH
		if (!addr_lookup(f.sin_addr.s_addr))
		{
			if (debug)
			{
				fprintf(stderr, "%s: %s permission denied\n", progname,
					inet_ntoa(f.sin_addr));
			}
			continue;
		}
#endif AUTH

		packet = recv_buf;

		switch (*packet)
		{
#ifdef OLD_RPLAY
			case OLD_RPLAY_PLAY:
			case OLD_RPLAY_STOP:
			case OLD_RPLAY_PAUSE:
			case OLD_RPLAY_CONTINUE:
				packet = rplay_convert(recv_buf);
#endif OLD_RPLAY
				
			case RPLAY_VERSION:
				rp = rplay_unpack(packet);
				if (rp == NULL)
				{
					rplay_perror(progname);
					break;
				}

				switch (rp->command)
				{
					case RPLAY_PLAY:
						sp = spool_next();
						if (!sp)
						{
							if (debug)
							{
								fprintf(stderr, "%s: spool full\n", progname);
							}
							break;
						}
						for (i = 0; i < rp->nsounds; i++)
						{
							if (debug)
							{
								fprintf(stderr, "%s: play sound=%s volume=%d ... from %s\n",
									progname, 
									(char *)rplay_get(rp, RPLAY_SOUND, i),
									rplay_get(rp, RPLAY_VOLUME, i),
									inet_ntoa(f.sin_addr));
							}
							sp->sound[i] = sound_lookup((char *)rplay_get(rp, RPLAY_SOUND, i));
							if (sp->sound[i] == NULL)
							{
								fprintf(stderr, "%s: sound %s not found\n",
									(char *)rplay_get(rp, RPLAY_SOUND, i));
								goto endswitch;
							}
						}
						sp->rp = rp;
						sp->curr_attrs = rp->attrs;
						sp->ptr = sp->sound[0]->start;
						sp->end = sp->sound[0]->stop;
						sp->curr_sound = 0;
						sp->state = RPLAY_PLAY;
						spool_size++;
						break;

					case RPLAY_STOP:
						if (debug)
						{
							fprintf(stderr, "%s: stop sound=%s volume=%d ... from %s\n",
								progname, 
								(char *)rplay_get(rp, RPLAY_SOUND, 0),
								rplay_get(rp, RPLAY_VOLUME, 0),
								inet_ntoa(f.sin_addr));
						}
						sp = spool_match(rp);
						if (!sp)
						{
							if (debug)
							{
								fprintf(stderr, "%s: stop sound not found\n", progname);
							}
							break;
						}
						spool_clear(sp);
						spool_size--;
						break;

					case RPLAY_PAUSE:
						if (debug)
						{
							fprintf(stderr, "%s: pause sound=%s volume=%d ... from %s\n",
								progname, 
								(char *)rplay_get(rp, RPLAY_SOUND, 0),
								rplay_get(rp, RPLAY_VOLUME, 0),
								inet_ntoa(f.sin_addr));
						}
						sp = spool_match(rp);
						if (!sp)
						{
							if (debug)
							{
								fprintf(stderr, "%s: pause sound not found\n", progname);
							}
							break;
						}
						sp->state = RPLAY_PAUSE;
						spool_size--; /* sort of remove it from the spool */
						break;

					case RPLAY_CONTINUE:
						if (debug)
						{
							fprintf(stderr, "%s: continue sound=%s volume=%d ... from %s\n",
								progname, 
								(char *)rplay_get(rp, RPLAY_SOUND, 0),
								rplay_get(rp, RPLAY_VOLUME, 0),
								inet_ntoa(f.sin_addr));
						}
						sp = spool_match(rp);
						if (!sp)
						{
							if (debug)
							{
								fprintf(stderr, "%s: continue sound not found\n", progname);
							}
							break;
						}
						sp->state = RPLAY_PLAY;
						spool_size++; /* put it back in the spool */
						break;
				}
				break;
				
			default:
				if (debug)
				{
					fprintf(stderr, "%s: unknown packet received from %s\n",
						progname, inet_ntoa(f.sin_addr));
				}
				break;
		}
endswitch:	continue;
	}
}

/*
 * write a buffer from the spool to the audio device
 */
#ifdef __STDC__
void	audio_write(void)
#else
void	audio_write()
#endif
{
	int	i, n;
	long	total, val;
	SPOOL	*sp;

	while (audio_size != AUDIO_BUFSIZE && spool_size)
	{
		total = 0;
		for (i = 0; i < SPOOL_SIZE && audio_size != AUDIO_BUFSIZE; i++)
		{
			sp = &spool[i];
			switch (sp->state)
			{
				case RPLAY_PLAY:
					val = st_ulaw_to_linear((unsigned char)*sp->ptr)*sp->curr_attrs->volume;
					total += val >> 7;
					if (sp->ptr == sp->end)
					{
						sp->curr_sound++;
						if (sp->curr_sound == sp->rp->nsounds)
						{
							spool_clear(sp);
							spool_size--;
						}
						else
						{
							sp->ptr = sp->sound[sp->curr_sound]->start;
							sp->end = sp->sound[sp->curr_sound]->stop;
							sp->curr_attrs = sp->curr_attrs->next;
						}
					}
					else
					{
						sp->ptr++;
					}
					break;

			}
		}
		audio_buf[audio_size++] = st_linear_to_ulaw(total);
	}
	if (audio_fd < 0)
	{
		audio_open();
	}
	if (audio_fd > 0)
	{
		n = write(audio_fd, audio_buf, audio_size);
		if (n < 0 && debug)
		{
			fprintf(stderr, "%s: write: %s\n", sys_errlist[errno]);
		}
	}
	audio_size = 0;
}

#ifdef __STDC__
void	usage(void)
#else
void	usage()
#endif
{
	printf("\n%s\n\n", rplay_version);
	printf("usage: %s [options]\n", progname);
	printf("\t-c n\tclose %s after n idle seconds, 0 disables timeout (%d)\n", AUDIO_DEVICE, audio_timeout);
	printf("\t-d\tdebug mode (%s)\n", debug ? "on" : "off");
	printf("\t-n\tdisable inetd mode (%s)\n", inetd ? "enabled" : "disabled");
	printf("\t-t n\texit after n idle seconds, 0 disables timeout (%d)\n", rplayd_timeout);
	printf("\t-v\tprint rplay version and exit\n");
	printf("\t-h\tthis very helpful list\n");
	printf("\n");
	printf("Compile options:\n");
	printf("\tRPLAY_CONF\t%s\n", RPLAY_CONF);
	printf("\tRPLAY_PORT\t%d\n", RPLAY_PORT);
#ifdef AUTH
	printf("\tAUTH\t\tenabled\n");
	printf("\tRPLAY_HOSTS\t%s\n", RPLAY_HOSTS);
#else AUTH
	printf("\tAUTH\t\tdisabled\n");
#endif AUTH
#ifdef MMAP
	printf("\tMMAP\t\tenabled\n");
#else MMAP
	printf("\tMMAP\t\tdisabled\n");
#endif MMAP
#ifdef OLD_RPLAY
	printf("\tOLD_RPLAY\trplay2.0 support enabled\n");
#else OLD_RPLAY
	printf("\tOLD_RPLAY\trplay2.0 support disabled\n");
#endif OLD_RPLAY
	printf("\tSPOOL_SIZE\t%d\n", SPOOL_SIZE);
	printf("\tAUDIO_BUFSIZE\t%d\n", AUDIO_BUFSIZE);
	printf("\tAUDIO_DEVICE\t%s\n", AUDIO_DEVICE);
	printf("\tMAX_SOUNDS\t%d\n", MAX_SOUNDS);
	printf("\n");
	done(1);
}

#ifdef __STDC__
void	done(int x)
#else
void	done(x)
int	x;
#endif
{
	if (debug)
	{
		fprintf(stderr, "%s: done(%d)\n", progname, x);
	}
	exit(x);
}
