/*
 * Copyright (c) 1992-93 by Mark Boyns (boyns@sdsu.edu)
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <varargs.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include "rplay.h"
#include "version.h"
#include "conf.h"

int 	rplay_errno;

char	*rplay_errlist[] =
{
	"no error",			/* RPLAY_ERROR_NONE */
	"out of memory",		/* RPLAY_ERROR_MEMORY */
	"host not found",		/* RPLAY_ERROR_HOST */
	"cannot connect to socket",	/* RPLAY_ERROR_CONNECT */
	"cannot create socket",		/* RPLAY_ERROR_SOCKET */
	"error writing to socket",	/* RPLAY_ERROR_WRITE */
	"error closing socket",		/* RPLAY_ERROR_CLOSE */ 
	"max packet size exceeded",	/* RPLAY_ERROR_PACKET_SIZE */
	"cannot enable broadcast",	/* RPLAY_ERROR_BROADCAST */
	"unknown rplay attribute",	/* RPLAY_ERROR_ATTRIBUTE */
	"unknown rplay command",	/* RPLAY_ERROR_COMMAND */
	"illegal rplay index value",	/* RPLAY_ERROR_INDEX */
	"unknown rplay modifier",	/* RPLAY_ERROR_MODIFIER */
};

#ifdef __STDC__
static RPLAY_ATTRS	*rplay_attrs_create(void)
#else
static RPLAY_ATTRS	*rplay_attrs_create()
#endif
{
	RPLAY_ATTRS	*attrs;

	attrs = (RPLAY_ATTRS *)malloc(sizeof(RPLAY_ATTRS));
	if (attrs == NULL)
	{
		return NULL;
	}
	attrs->next = NULL;
	attrs->sound = NULL;
	attrs->volume = RPLAY_DEFAULT_VOLUME;

	return attrs;
}

#ifdef __STDC__
static void	rplay_attrs_destroy(RPLAY_ATTRS *attrs)
#else
static void	rplay_attrs_destroy(attrs)
RPLAY_ATTRS	*attrs;
#endif
{
	free(attrs->sound);
	free(attrs);
}

#define COPY_SIZE	128
#define COPY(rp, p, n) \
	grow = 0; \
	while (rp->len + n > rp->size) \
	{ \
		grow++; \
		rp->size += COPY_SIZE; \
		if (rp->size > MAX_PACKET) \
		{ \
			rplay_errno = RPLAY_ERROR_PACKET_SIZE; \
			return -1; \
		} \
	} \
	if (grow) \
	{ \
		rp->buf = realloc(rp->buf, rp->size); \
		if (rp->buf == NULL) \
		{ \
			rplay_errno = RPLAY_ERROR_MEMORY; \
			return -1; \
		} \
	} \
	bcopy(p, rp->buf + rp->len, n); \
	rp->len += n; \

#ifdef __STDC__
int	rplay_pack(RPLAY *rp)
#else
int	rplay_pack(rp)
RPLAY	*rp;
#endif
{
	RPLAY_ATTRS	*attrs;
	int		i, len, grow = 0;
	unsigned char	val;
	
	rp->len = 0;
	val = RPLAY_VERSION;
	COPY(rp, &val, sizeof(val));
	val = rp->command;
	COPY(rp, &val, sizeof(val));

	for (attrs = rp->attrs; attrs; attrs = attrs->next)
	{
		val = RPLAY_SOUND;
		COPY(rp, &val, sizeof(val));
		len = strlen(attrs->sound)+1;
		COPY(rp, attrs->sound, len);

		val = RPLAY_VOLUME;
		COPY(rp, &val, sizeof(val));
		val = attrs->volume;
		COPY(rp, &val, sizeof(val));

		val = RPLAY_NULL;
		COPY(rp, &val, sizeof(val));
	}

	val = RPLAY_NULL;
	COPY(rp, &val, sizeof(val));

	return 0;
}

#ifdef __STDC__
RPLAY	*rplay_unpack(char *packet)
#else
RPLAY	*rplay_unpack(packet)
char	*packet;
#endif
{
	RPLAY		*rp;
	RPLAY_ATTRS	*attrs;
	int		still_going = 1;
	int		version;

	rplay_errno = RPLAY_ERROR_NONE;

	rp = (RPLAY *)malloc(sizeof(RPLAY));
	if (rp == NULL)
	{
		rplay_errno = RPLAY_ERROR_MEMORY;
		return NULL;
	}

	version = *packet++;
	rp->command = *packet++;
	rp->len = 0;
	rp->size = 0;
	rp->nsounds = 0;
	rp->attrs = NULL;
	rp->attrsp = &rp->attrs;

	*(rp->attrsp) = rplay_attrs_create();

	while (still_going)
	{
		switch (*packet++)
		{
			case RPLAY_SOUND:
				(*rp->attrsp)->sound = strdup(packet);
				packet += strlen(packet)+1;
				break;

			case RPLAY_VOLUME:
				(*rp->attrsp)->volume = (unsigned char)*packet++;
				break;
			
			case RPLAY_NULL:
				rp->nsounds++;
				rp->attrsp = &(*rp->attrsp)->next;
				if (*packet == RPLAY_NULL)
				{
					still_going = 0;
				}
				else
				{
					*(rp->attrsp) = rplay_attrs_create();
				}
				break;

			default:
				rplay_errno = RPLAY_ERROR_ATTRIBUTE;
				return NULL;
		}
	}

	return rp;
}

#ifdef __STDC__
RPLAY	*rplay_create(int command)
#else
RPLAY	*rplay_create(command)
int	command;
#endif
{
	RPLAY	*rp;

	rplay_errno = RPLAY_ERROR_NONE;

	rp = (RPLAY *)malloc(sizeof(RPLAY));
	if (rp == NULL)
	{
		rplay_errno = RPLAY_ERROR_MEMORY;
		return NULL;
	}

	rp->attrs = NULL;
	rp->attrsp = &rp->attrs;
	rp->buf = (char *)malloc(COPY_SIZE);
	if (rp->buf == NULL)
	{
		rplay_errno = RPLAY_ERROR_MEMORY;
		return NULL;
	}
	rp->len = 0;
	rp->size = 0;
	rp->command = RPLAY_NULL;
	rp->nsounds = 0;

	switch (command)
	{
		case RPLAY_PLAY:
		case RPLAY_STOP:
		case RPLAY_PAUSE:
		case RPLAY_CONTINUE:
			rp->command = command;
			break;

		default:
			rplay_errno = RPLAY_ERROR_COMMAND;
			return NULL;
	}

	return rp;
}

int	rplay_set(va_alist)
va_dcl
{
	va_list		args;
	RPLAY		*rp;
	RPLAY_ATTRS	*attrs = NULL, *prev = NULL, *curr;
	int		index, i, modifier;

	rplay_errno = RPLAY_ERROR_NONE;

	va_start(args);
	rp = va_arg(args, RPLAY *);
	modifier = va_arg(args, int);

	switch (modifier)
	{
		case RPLAY_APPEND:
			*(rp->attrsp) = attrs = rplay_attrs_create();
			if (attrs == NULL)
			{
				rplay_errno = RPLAY_ERROR_MEMORY;
				return -1;
			}
			rp->attrsp = &attrs->next;
			rp->nsounds++;
			break;
		
		case RPLAY_INSERT:
			index = va_arg(args, int);
			if (index < 0)
			{
				rplay_errno = RPLAY_ERROR_INDEX;
				return -1;
			}
			for (i = 0, curr = rp->attrs; i < index && curr; i++)
			{
				prev = curr;
				curr = curr->next;
			}
			if (curr == NULL && i != index)
			{
				rplay_errno = RPLAY_ERROR_INDEX;
				return -1;
			}
			attrs = rplay_attrs_create();
			if (attrs == NULL)
			{
				rplay_errno = RPLAY_ERROR_MEMORY;
				return -1;
			}
			if (prev)
			{
				prev->next = attrs;
			}
			else
			{
				*(rp->attrsp) = attrs;
			}
			attrs->next = curr;
			if (attrs->next == NULL)
			{
				rp->attrsp = &attrs->next;
			}
			rp->nsounds++;
			break;

		case RPLAY_DELETE:
			index = va_arg(args, int);
			if (index < 0)
			{
				rplay_errno = RPLAY_ERROR_INDEX;
				return -1;
			}
			for (i = 0, curr = rp->attrs; i < index && curr; i++)
			{
				prev = curr;
				curr = curr->next;
			}
			if (curr == NULL)
			{
				rplay_errno = RPLAY_ERROR_INDEX;
				return -1;
			}
			if (prev)
			{
				prev->next = curr->next;
				if (prev->next == NULL)
				{
					rp->attrsp = &prev->next;
				}
			}
			else
			{
				rp->attrs = curr->next;
				if (rp->attrs == NULL)
				{
					rp->attrsp = &rp->attrs;
				}
			}
			rplay_attrs_destroy(curr);
			rp->nsounds--;
			break;

		case RPLAY_CHANGE:
			index = va_arg(args, int);
			if (index < 0)
			{
				rplay_errno = RPLAY_ERROR_INDEX;
				return -1;
			}
			for (i = 0, attrs = rp->attrs; i < index && attrs; i++, attrs = attrs->next);
			if (attrs == NULL)
			{
				rplay_errno = RPLAY_ERROR_INDEX;
				return -1;
			}
			break;

		default:
			rplay_errno = RPLAY_ERROR_MODIFIER;
			return -1;
	}

	if (attrs)
	{
		int	attribute;

		while (attribute = va_arg(args, int))
		{
			switch (attribute)
			{
				case RPLAY_SOUND:
					attrs->sound = va_arg(args, char *);
					break;

				case RPLAY_VOLUME:
					attrs->volume = va_arg(args, int);
					break;

				default:
					rplay_errno = RPLAY_ERROR_ATTRIBUTE;
					return -1;
			}
		}
	}

	return rplay_pack(rp);
}

int	rplay_get(va_alist)
va_dcl
{
	va_list		args;
	RPLAY		*rp;
	RPLAY_ATTRS	*attrs;
	int		get, index, i;

	rplay_errno = RPLAY_ERROR_NONE;

	va_start(args);
	rp = va_arg(args, RPLAY *);
	get = va_arg(args, int);

	switch (get)
	{
		case RPLAY_NSOUNDS:
			return rp->nsounds;

		case RPLAY_COMMAND:
			return rp->command;

		case RPLAY_SOUND:
			index = va_arg(args, int);
			if (index < 0)
			{
				rplay_errno = RPLAY_ERROR_INDEX;
				return -1;
			}
			for (i = 0, attrs = rp->attrs; i < index && attrs; i++, attrs = attrs->next);
			if (attrs == NULL)
			{
				rplay_errno = RPLAY_ERROR_INDEX;
				return -1;
			}
			return (int)attrs->sound;
		
		case RPLAY_VOLUME:
			index = va_arg(args, int);
			if (index < 0)
			{
				rplay_errno = RPLAY_ERROR_INDEX;
				return -1;
			}
			for (i = 0, attrs = rp->attrs; i < index && attrs; i++, attrs = attrs->next);
			if (attrs == NULL)
			{
				rplay_errno = RPLAY_ERROR_INDEX;
				return -1;
			}
			return attrs->volume;

		default:
			rplay_errno = RPLAY_ERROR_ATTRIBUTE;
			return -1;
	}
}

#ifdef __STDC__
void	rplay_destroy(RPLAY *rp)
#else
void	rplay_destroy(rp)
RPLAY	*rp;
#endif
{
	RPLAY_ATTRS	*p, *q;

	rplay_errno = RPLAY_ERROR_NONE;

	for (p = rp->attrs; p; q = p, p = p->next, rplay_attrs_destroy(q));
	free(rp->buf);
	free(rp);
}

#ifdef __STDC__
char	*rplay_convert(char *p)
#else
char	*rplay_convert(p)
char	*p;
#endif
{
	static char	buf[MAX_PACKET];
	char		*q = buf;
	int		len;

	*q++ = RPLAY_VERSION;
	switch (*p++)
	{
		case OLD_RPLAY_PLAY:
			*q++ = RPLAY_PLAY;
			break;

		case OLD_RPLAY_STOP:
			*q++ = RPLAY_STOP;
			break;

		case OLD_RPLAY_PAUSE:
			*q++ = RPLAY_PAUSE;
			break;

		case OLD_RPLAY_CONTINUE:
			*q++ = RPLAY_CONTINUE;
			break;
	}

	do {
		*q++ = RPLAY_SOUND;
		strcpy(q, p);
		len = strlen(p)+1;
		p += len;
		q += len;
		*q++ = RPLAY_VOLUME;
		*q++ = *p++;
		*q++ = RPLAY_NULL;
	} while (*p);

	*q++ = RPLAY_NULL;

	return buf;
}

#ifdef __STDC__
int	rplay_debug(RPLAY *rp)
#else
int	rplay_debug(rp)
RPLAY	*rp;
#endif
{
	RPLAY_ATTRS	*attrs;

	printf("command=%d\n", rp->command);
	printf("nsounds=%d\n", rp->nsounds);
	printf("size=%d\n", rp->size);
	printf("len=%d\n", rp->len);
	for (attrs = rp->attrs; attrs; attrs = attrs->next)
	{
		printf("sound=%s\n", attrs->sound);
		printf("volume=%d\n", attrs->volume);
	}
}

#ifdef __STDC__
int	rplay_open(char *host)
#else
int	rplay_open(host)
char	*host;
#endif
{
	struct hostent		*hp;
	struct servent		*sp;
	int			rplay_fd;
	u_long			addr;
	struct sockaddr_in	s;
	int			on = 1;

	rplay_errno = RPLAY_ERROR_NONE;

	bzero(&s, sizeof(s));

	addr = inet_addr(host);
	if (addr == 0xffffffff)
	{
		hp = gethostbyname(host);
		if (hp == NULL)
		{
			rplay_errno = RPLAY_ERROR_HOST;
			return -1;
		}
		bcopy(hp->h_addr, &s.sin_addr.s_addr, hp->h_length);
	}
	else
	{
		bcopy(&addr, &s.sin_addr.s_addr, sizeof(addr));
	}

	sp = getservbyname("rplay", "udp");
	if (sp)
	{
		s.sin_port = sp->s_port;
	}
	else
	{
		s.sin_port = htons(RPLAY_PORT);
	}
	s.sin_family = AF_INET;

	rplay_fd = socket(AF_INET, SOCK_DGRAM, 0);
	if (rplay_fd < 0)
	{
		rplay_errno = RPLAY_ERROR_SOCKET;
		return -1;
	}

	if (connect(rplay_fd, &s, sizeof(s)) < 0)
	{
		rplay_errno = RPLAY_ERROR_CONNECT;
		return -1;
	}

	/*
	 * enable broadcasting
	 */
	if (setsockopt(rplay_fd, SOL_SOCKET, SO_BROADCAST, &on, sizeof(on)) < 0)
	{
		rplay_errno = RPLAY_ERROR_BROADCAST;
		return -1;
	}

	return rplay_fd;
}

#ifdef __STDC__
int	rplay(int rplay_fd, RPLAY *rp)
#else
int	rplay(rplay_fd, rp)
int	rplay_fd;
RPLAY	*rp;
#endif
{
	rplay_errno = RPLAY_ERROR_NONE;

	if (write(rplay_fd, rp->buf, rp->len) != rp->len)
	{
		rplay_errno = RPLAY_ERROR_WRITE;
		return -1;
	}

	return 0;
}

#ifdef __STDC__
int	rplay_close(int rplay_fd)
#else
int	rplay_close(rplay_fd)
int	rplay_fd;
#endif
{
	rplay_errno = RPLAY_ERROR_NONE;

	if (close(rplay_fd) < 0)
	{
		rplay_errno = RPLAY_ERROR_CLOSE;
		return -1;
	}

	return 0;
}

#ifdef __STDC__
void	rplay_perror(char *s)
#else
void	rplay_perror(s)
char	*s;
#endif
{
	fprintf(stderr, "%s: %s\n", s, rplay_errlist[rplay_errno]);
}
