/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.net.InetSocketAddress;
import java.util.Arrays;
import org.eclipse.californium.elements.util.DatagramReader;
import org.eclipse.californium.elements.util.DatagramWriter;
import org.eclipse.californium.elements.util.NoPublicAPI;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.AbstractMessage;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.CertificateMessage;
import org.eclipse.californium.scandium.dtls.CertificateRequest;
import org.eclipse.californium.scandium.dtls.CertificateVerify;
import org.eclipse.californium.scandium.dtls.ClientHello;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.ECDHClientKeyExchange;
import org.eclipse.californium.scandium.dtls.EcdhEcdsaServerKeyExchange;
import org.eclipse.californium.scandium.dtls.EcdhPskClientKeyExchange;
import org.eclipse.californium.scandium.dtls.EcdhPskServerKeyExchange;
import org.eclipse.californium.scandium.dtls.Finished;
import org.eclipse.californium.scandium.dtls.FragmentedHandshakeMessage;
import org.eclipse.californium.scandium.dtls.GenericHandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeParameter;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.dtls.HelloRequest;
import org.eclipse.californium.scandium.dtls.HelloVerifyRequest;
import org.eclipse.californium.scandium.dtls.PSKClientKeyExchange;
import org.eclipse.californium.scandium.dtls.PSKServerKeyExchange;
import org.eclipse.californium.scandium.dtls.ServerHello;
import org.eclipse.californium.scandium.dtls.ServerHelloDone;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NoPublicAPI
public abstract class HandshakeMessage
extends AbstractMessage {
    public static final int MESSAGE_TYPE_BITS = 8;
    public static final int MESSAGE_LENGTH_BITS = 24;
    public static final int MESSAGE_SEQ_BITS = 16;
    public static final int FRAGMENT_OFFSET_BITS = 24;
    public static final int FRAGMENT_LENGTH_BITS = 24;
    public static final int MESSAGE_HEADER_LENGTH_BYTES = 12;
    private static final Logger LOGGER = LoggerFactory.getLogger(HandshakeMessage.class);
    private int messageSeq;
    private byte[] rawMessage;
    private byte[] byteArray;
    private HandshakeMessage nextHandshakeMessage;

    protected HandshakeMessage(InetSocketAddress peerAddress) {
        super(peerAddress);
    }

    @Override
    public int size() {
        return this.getFragmentLength() + 12;
    }

    public abstract HandshakeType getMessageType();

    public abstract int getMessageLength();

    public abstract byte[] fragmentToByteArray();

    @Override
    public final ContentType getContentType() {
        return ContentType.HANDSHAKE;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("\tHandshake Protocol");
        sb.append(StringUtil.lineSeparator()).append("\tType: ").append((Object)this.getMessageType());
        sb.append(StringUtil.lineSeparator()).append("\tPeer: ").append(this.getPeer());
        sb.append(StringUtil.lineSeparator()).append("\tMessage Sequence No: ").append(this.messageSeq);
        sb.append(StringUtil.lineSeparator()).append("\tLength: ").append(this.getMessageLength()).append(StringUtil.lineSeparator());
        return sb.toString();
    }

    @Override
    public byte[] toByteArray() {
        if (this.rawMessage != null) {
            return this.rawMessage;
        }
        if (this.byteArray == null) {
            int fragmentLength = this.getFragmentLength();
            DatagramWriter writer = new DatagramWriter(fragmentLength + 12);
            this.writeTo(writer);
            this.byteArray = writer.toByteArray();
        }
        return this.byteArray;
    }

    protected void writeTo(DatagramWriter writer) {
        writer.write(this.getMessageType().getCode(), 8);
        writer.write(this.getMessageLength(), 24);
        writer.write(this.messageSeq, 16);
        writer.write(this.getFragmentOffset(), 24);
        writer.write(this.getFragmentLength(), 24);
        writer.writeBytes(this.fragmentToByteArray());
    }

    protected void fragmentChanged() {
        this.byteArray = null;
    }

    public static HandshakeMessage fromByteArray(byte[] byteArray, InetSocketAddress peerAddress) throws HandshakeException {
        try {
            int offset = 0;
            HandshakeMessage first = null;
            HandshakeMessage last = null;
            DatagramReader reader = new DatagramReader(byteArray, false);
            do {
                HandshakeMessage body;
                int code;
                HandshakeType type;
                if ((type = HandshakeType.getTypeByCode(code = reader.read(8))) == null) {
                    throw new HandshakeException(String.format("Cannot parse unsupported message type %d", code), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.UNEXPECTED_MESSAGE, peerAddress));
                }
                LOGGER.trace("Parsing HANDSHAKE message of type [{}]", (Object)type);
                int length = reader.read(24);
                int messageSeq = reader.read(16);
                int fragmentOffset = reader.read(24);
                int fragmentLength = reader.read(24);
                int left = reader.bitsLeft() / 8;
                if (fragmentLength > left) {
                    throw new HandshakeException(String.format("Message %s fragment length %d exceeds available data %d", new Object[]{type, fragmentLength, left}), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.DECODE_ERROR, peerAddress));
                }
                DatagramReader fragmentReader = reader.createRangeReader(fragmentLength);
                int start = offset;
                offset = byteArray.length - reader.bitsLeft() / 8;
                if (length != fragmentLength) {
                    if (fragmentOffset + fragmentLength > length) {
                        throw new HandshakeException(String.format("Message %s fragment %d exceeds overall length %d", new Object[]{type, fragmentOffset + fragmentLength, length}), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.DECODE_ERROR, peerAddress));
                    }
                    body = new FragmentedHandshakeMessage(type, length, messageSeq, fragmentOffset, fragmentReader.readBytesLeft(), peerAddress);
                } else {
                    if (fragmentOffset != 0) {
                        throw new HandshakeException(String.format("Message %s unexpected fragment offset", new Object[]{type}), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.DECODE_ERROR, peerAddress));
                    }
                    try {
                        body = HandshakeMessage.fromReader(type, fragmentReader, null, peerAddress);
                    }
                    catch (MissingHandshakeParameterException ex) {
                        body = GenericHandshakeMessage.fromByteArray(type, peerAddress);
                    }
                    body.rawMessage = Arrays.copyOfRange(byteArray, start, offset);
                    body.setMessageSeq(messageSeq);
                }
                if (first == null) {
                    first = body;
                } else {
                    last.setNextHandshakeMessage(body);
                }
                last = body;
            } while (reader.bytesAvailable());
            return first;
        }
        catch (IllegalArgumentException ex) {
            LOGGER.debug("Handshake message from peer [{}] malformed", (Object)peerAddress, (Object)ex);
            throw new HandshakeException("Handshake message malformed, " + ex.getMessage(), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.DECODE_ERROR, peerAddress));
        }
    }

    public static HandshakeMessage fromGenericHandshakeMessage(GenericHandshakeMessage message, HandshakeParameter parameter) throws HandshakeException {
        InetSocketAddress peerAddress = message.getPeer();
        try {
            HandshakeType type = message.getMessageType();
            LOGGER.trace("Parsing HANDSHAKE message of type [{}]", (Object)type);
            byte[] byteArray = message.toByteArray();
            DatagramReader reader = new DatagramReader(message.fragmentToByteArray(), false);
            HandshakeMessage body = HandshakeMessage.fromReader(type, reader, parameter, peerAddress);
            body.rawMessage = byteArray;
            body.setMessageSeq(message.getMessageSeq());
            body.setNextHandshakeMessage(message.getNextHandshakeMessage());
            return body;
        }
        catch (IllegalArgumentException ex) {
            LOGGER.debug("Handshake message from peer [{}] malformed", (Object)peerAddress, (Object)ex);
            throw new HandshakeException("Handshake message malformed, " + ex.getMessage(), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.DECODE_ERROR, peerAddress));
        }
    }

    private static HandshakeMessage fromReader(HandshakeType type, DatagramReader reader, HandshakeParameter parameter, InetSocketAddress peerAddress) throws HandshakeException {
        HandshakeMessage body;
        switch (type) {
            case HELLO_REQUEST: {
                body = new HelloRequest(peerAddress);
                break;
            }
            case CLIENT_HELLO: {
                body = ClientHello.fromReader(reader, peerAddress);
                break;
            }
            case SERVER_HELLO: {
                body = ServerHello.fromReader(reader, peerAddress);
                break;
            }
            case HELLO_VERIFY_REQUEST: {
                body = HelloVerifyRequest.fromReader(reader, peerAddress);
                break;
            }
            case CERTIFICATE: {
                if (parameter == null) {
                    throw new MissingHandshakeParameterException("HandshakeParameter must not be null!");
                }
                body = CertificateMessage.fromReader(reader, parameter.getCertificateType(), peerAddress);
                break;
            }
            case SERVER_KEY_EXCHANGE: {
                if (parameter == null) {
                    throw new MissingHandshakeParameterException("HandshakeParameter must not be null!");
                }
                body = HandshakeMessage.readServerKeyExchange(reader, parameter.getKeyExchangeAlgorithm(), peerAddress);
                break;
            }
            case CERTIFICATE_REQUEST: {
                body = CertificateRequest.fromReader(reader, peerAddress);
                break;
            }
            case SERVER_HELLO_DONE: {
                body = new ServerHelloDone(peerAddress);
                break;
            }
            case CERTIFICATE_VERIFY: {
                body = CertificateVerify.fromReader(reader, peerAddress);
                break;
            }
            case CLIENT_KEY_EXCHANGE: {
                if (parameter == null) {
                    throw new MissingHandshakeParameterException("HandshakeParameter must not be null!");
                }
                body = HandshakeMessage.readClientKeyExchange(reader, parameter.getKeyExchangeAlgorithm(), peerAddress);
                break;
            }
            case FINISHED: {
                body = Finished.fromReader(reader, peerAddress);
                break;
            }
            default: {
                throw new HandshakeException(String.format("Cannot parse unsupported message type %s", new Object[]{type}), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.UNEXPECTED_MESSAGE, peerAddress));
            }
        }
        if (reader.bytesAvailable()) {
            byte[] bytesLeft = reader.readBytesLeft();
            throw new HandshakeException(String.format("Too many bytes, %d left, message not completely parsed! message type %s", new Object[]{bytesLeft.length, type}), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.DECODE_ERROR, peerAddress));
        }
        return body;
    }

    private static HandshakeMessage readServerKeyExchange(DatagramReader reader, CipherSuite.KeyExchangeAlgorithm keyExchange, InetSocketAddress peerAddress) throws HandshakeException {
        switch (keyExchange) {
            case EC_DIFFIE_HELLMAN: {
                return EcdhEcdsaServerKeyExchange.fromReader(reader, peerAddress);
            }
            case PSK: {
                return PSKServerKeyExchange.fromReader(reader, peerAddress);
            }
            case ECDHE_PSK: {
                return EcdhPskServerKeyExchange.fromReader(reader, peerAddress);
            }
        }
        throw new HandshakeException("Unsupported key exchange algorithm", new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.ILLEGAL_PARAMETER, peerAddress));
    }

    private static HandshakeMessage readClientKeyExchange(DatagramReader reader, CipherSuite.KeyExchangeAlgorithm keyExchange, InetSocketAddress peerAddress) throws HandshakeException {
        switch (keyExchange) {
            case EC_DIFFIE_HELLMAN: {
                return ECDHClientKeyExchange.fromReader(reader, peerAddress);
            }
            case PSK: {
                return PSKClientKeyExchange.fromReader(reader, peerAddress);
            }
            case ECDHE_PSK: {
                return EcdhPskClientKeyExchange.fromReader(reader, peerAddress);
            }
        }
        throw new HandshakeException("Unknown key exchange algorithm", new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.ILLEGAL_PARAMETER, peerAddress));
    }

    public int getFragmentOffset() {
        return 0;
    }

    public int getFragmentLength() {
        return this.getMessageLength();
    }

    public int getMessageSeq() {
        return this.messageSeq;
    }

    public void setMessageSeq(int messageSeq) {
        if (this.byteArray != null) {
            throw new IllegalStateException("message is already serialized!");
        }
        this.messageSeq = messageSeq;
    }

    public void setNextHandshakeMessage(HandshakeMessage message) {
        this.nextHandshakeMessage = message;
    }

    public HandshakeMessage getNextHandshakeMessage() {
        return this.nextHandshakeMessage;
    }

    protected final byte[] getRawMessage() {
        return this.rawMessage;
    }

    private static class MissingHandshakeParameterException
    extends IllegalArgumentException {
        private static final long serialVersionUID = -5365688530126068164L;

        private MissingHandshakeParameterException(String message) {
            super(message);
        }
    }
}

