cmake_minimum_required(VERSION 3.16)

include(RunCMake)

function(run_install_test case)
  set(RunCMake_TEST_BINARY_DIR ${RunCMake_BINARY_DIR}/${case}-build)
  set(RunCMake_TEST_NO_CLEAN 1)
  set(RunCMake_TEST_OPTIONS -DCMAKE_BUILD_TYPE:STRING=Debug "-DCMAKE_INSTALL_PREFIX:PATH=${RunCMake_TEST_BINARY_DIR}/fake_install")
  file(REMOVE_RECURSE "${RunCMake_TEST_BINARY_DIR}")
  file(MAKE_DIRECTORY "${RunCMake_TEST_BINARY_DIR}")
  run_cmake(${case})
  run_cmake_command(${case}-build ${CMAKE_COMMAND} --build . --config Debug)
  run_cmake_command(${case}-install ${CMAKE_COMMAND} --install . --config Debug --prefix "${RunCMake_TEST_BINARY_DIR}/real_install")
endfunction()

find_program(OTOOL_COMMAND otool)

function(check_install_name_dir file expected)
  execute_process(COMMAND ${OTOOL_COMMAND} -l ${file} RESULT_VARIABLE _result OUTPUT_VARIABLE _output)
  if(_result)
    string(APPEND RunCMake_TEST_FAILED "Could not run otool on ${file}\n")
  elseif(_output MATCHES "cmd LC_ID_DYLIB\n[^\n]*\n *name ([^\n]*) \\(offset [0-9]+\\)\n")
    set(_install_name "${CMAKE_MATCH_1}")
    if(NOT _install_name MATCHES "${expected}")
      string(APPEND RunCMake_TEST_FAILED "Install name of ${file} did not match ${expected} (actual: ${_install_name})\n")
    endif()
  else()
    string(APPEND RunCMake_TEST_FAILED "otool did not print install name for ${file}\n")
  endif()

  set(RunCMake_TEST_FAILED "${RunCMake_TEST_FAILED}" PARENT_SCOPE)
endfunction()

function(check_imported_soname contents target expected)
  if(contents MATCHES "set_target_properties\\(${target} PROPERTIES\n[^\n]*\n *IMPORTED_SONAME_DEBUG \"([^\n]*)\"\n")
    set(_soname "${CMAKE_MATCH_1}")
    set(_regex "^${expected}lib${target}\\.dylib$")
    if(NOT _soname MATCHES "${_regex}")
      string(APPEND RunCMake_TEST_FAILED "Target ${target}'s IMPORTED_SONAME_DEBUG did not match ${_regex} (actual: ${_soname})\n")
    endif()
  else()
    string(APPEND RunCMake_TEST_FAILED "Could not find IMPORTED_SONAME_DEBUG for target ${target} in package config file\n")
  endif()

  set(RunCMake_TEST_FAILED "${RunCMake_TEST_FAILED}" PARENT_SCOPE)
endfunction()

function(check_libraries fake_install real_install soname_prefix)
  file(STRINGS "${RunCMake_TEST_BINARY_DIR}/targets.txt" _targets)
  list(GET _targets 0 _build_dir)
  list(GET _targets 1 _install_dir)
  check_install_name_dir("${_build_dir}" "^@rpath/libbuild_dir\\.dylib$")
  check_install_name_dir("${_install_dir}" "^${fake_install}libinstall_dir\\.dylib$")
  check_install_name_dir("${RunCMake_TEST_BINARY_DIR}/real_install/lib/libbuild_dir.dylib" "^${real_install}libbuild_dir\\.dylib$")
  check_install_name_dir("${RunCMake_TEST_BINARY_DIR}/real_install/lib/libinstall_dir.dylib" "^${real_install}libinstall_dir\\.dylib$")

  file(READ "${RunCMake_TEST_BINARY_DIR}/real_install/lib/cmake/InstallNameDirTest/InstallNameDirTest-targets-debug.cmake" _targets)
  check_imported_soname("${_targets}" build_dir "${soname_prefix}")
  check_imported_soname("${_targets}" install_dir "${soname_prefix}")

  set(RunCMake_TEST_FAILED "${RunCMake_TEST_FAILED}" PARENT_SCOPE)
endfunction()

run_install_test(none)
run_install_test(empty)
run_install_test(simple)
run_install_test(simple_genex)
run_install_test(prefix_genex)
run_install_test(empty_genex)
