/*
 * Decompiled with CFR 0.152.
 */
package fi.csc.microarray.filebroker;

import fi.csc.microarray.config.Configuration;
import fi.csc.microarray.config.DirectoryLayout;
import fi.csc.microarray.filebroker.AuthorisedUrlRepository;
import fi.csc.microarray.filebroker.ChecksumInputStream;
import fi.csc.microarray.filebroker.ChecksumParseException;
import fi.csc.microarray.filebroker.DerbyMetadataServer;
import fi.csc.microarray.filebroker.DiskCleanUp;
import fi.csc.microarray.filebroker.Md5FileUtils;
import fi.csc.microarray.filebroker.WelcomePage;
import fi.csc.microarray.util.IOUtils;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.sql.SQLException;
import java.text.DecimalFormat;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.eclipse.jetty.servlet.DefaultServlet;
import org.eclipse.jetty.util.IO;
import org.eclipse.jetty.util.URIUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.ReadableInstant;

public class RestServlet
extends DefaultServlet {
    private static final String UPLOAD_FILE_EXTENSION = ".upload";
    private Logger logger = Log.getLogger((String)null);
    private String cachePath;
    private String publicPath;
    private String storagePath;
    private String rootUrl;
    private AuthorisedUrlRepository urlRepository;
    private DerbyMetadataServer metadataServer;
    private DiskCleanUp cacheCleanUp;
    private boolean useChecksums;
    private boolean logRest = false;

    public RestServlet(String rootUrl, AuthorisedUrlRepository urlRepository, DerbyMetadataServer metadataServer, DiskCleanUp cacheCleanUp) {
        this.rootUrl = rootUrl;
        this.urlRepository = urlRepository;
        this.metadataServer = metadataServer;
        this.cacheCleanUp = cacheCleanUp;
        Configuration configuration = DirectoryLayout.getInstance().getConfiguration();
        this.cachePath = "cache";
        this.storagePath = "storage";
        this.publicPath = configuration.getString("filebroker", "public-path");
        this.useChecksums = configuration.getBoolean("messaging", "use-checksums");
        if (configuration.getBoolean("filebroker", "log-rest")) {
            this.logRest = true;
        }
        this.logger.info("logging rest requests: " + this.logRest, new Object[0]);
    }

    public void service(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        if (this.isValidRequest(request)) {
            super.service(request, response);
        } else {
            response.sendError(403);
        }
    }

    private boolean isValidRequest(HttpServletRequest request) throws ServletException {
        if (this.isWelcomePage(request)) {
            return true;
        }
        File file = this.locateFile(request);
        if (file.isDirectory()) {
            return false;
        }
        return this.isCacheRequest(request) || this.isStorageRequest(request) || this.isPublicRequest(request);
    }

    private boolean isWelcomePage(HttpServletRequest request) {
        return "/".equals(request.getPathInfo());
    }

    private boolean isCacheRequest(HttpServletRequest request) {
        return this.checkRequest(request, this.cachePath);
    }

    private boolean isStorageRequest(HttpServletRequest request) {
        return this.checkRequest(request, this.storagePath);
    }

    private boolean checkRequest(HttpServletRequest request, String prefix) {
        String path = request.getPathInfo();
        if (path == null) {
            return false;
        }
        if (!path.startsWith("/" + prefix + "/")) {
            return false;
        }
        return AuthorisedUrlRepository.checkFilenameSyntax(path.substring(("/" + prefix + "/").length()));
    }

    private boolean isPublicRequest(HttpServletRequest request) {
        String path = request.getPathInfo();
        return path != null && path.startsWith("/" + this.publicPath + "/");
    }

    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("RESTful file access: GET request for " + request.getRequestURI(), new Object[0]);
        }
        if (this.isWelcomePage(request)) {
            new WelcomePage(this.rootUrl).print(response);
        } else {
            File file = this.locateFile(request);
            file.setLastModified(System.currentTimeMillis());
            try {
                String checksum = Md5FileUtils.readMd5(file);
                if (checksum != null) {
                    response.setHeader("Etag", checksum);
                }
            }
            catch (ChecksumParseException e) {
                this.logger.info("reading checksum file failed", (Throwable)e);
            }
            if (this.isStorageRequest(request)) {
                String uuid = AuthorisedUrlRepository.stripCompressionSuffix(IOUtils.getFilenameWithoutPath(request));
                try {
                    this.metadataServer.markFileAccessed(uuid);
                }
                catch (SQLException e) {
                    throw new ServletException((Throwable)e);
                }
            }
            DateTime before = new DateTime();
            super.doGet(request, response);
            DateTime after = new DateTime();
            Duration duration = new Duration((ReadableInstant)before, (ReadableInstant)after);
            double rate = this.getTransferRate(file.length(), duration);
            if (this.logRest) {
                this.logger.info("GET " + file.getName() + " " + "from " + request.getRemoteHost() + " | " + FileUtils.byteCountToDisplaySize((long)file.length()) + " | " + DurationFormatUtils.formatDurationHMS((long)duration.getMillis()) + " | " + new DecimalFormat("###.##").format(rate * 8.0) + " Mbit/s" + " | " + new DecimalFormat("###.##").format(rate) + " MB/s", new Object[0]);
            }
        }
    }

    protected void doPut(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        AuthorisedUrlRepository.Authorisation authorisation;
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("RESTful file access: PUT request for " + request.getRequestURI(), new Object[0]);
        }
        if ((authorisation = this.urlRepository.getAuthorisation(this.constructUrl(request))) == null) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("PUT denied for " + this.constructUrl(request), new Object[0]);
            }
            response.sendError(401);
            return;
        }
        File targetFile = this.locateFile(request);
        File tmpFile = this.getUploadTempFile(targetFile);
        if (tmpFile == null) {
            this.logger.info("PUT denied for " + this.constructUrl(request) + ": too many upload temp files for same uuid", new Object[0]);
            response.sendError(500);
            return;
        }
        long maxBytes = authorisation.getFileSize() + 1L;
        FileOutputStream out = new FileOutputStream(tmpFile);
        ChecksumInputStream in = new ChecksumInputStream((InputStream)request.getInputStream(), this.useChecksums);
        try {
            DateTime before = new DateTime();
            IO.copy((InputStream)in, (OutputStream)out, (long)maxBytes);
            DateTime after = new DateTime();
            Duration duration = new Duration((ReadableInstant)before, (ReadableInstant)after);
            double rate = this.getTransferRate(authorisation.getFileSize(), duration);
            if (this.logRest) {
                this.logger.info("PUT " + targetFile.getName() + " " + "from " + request.getRemoteHost() + " | " + FileUtils.byteCountToDisplaySize((long)authorisation.getFileSize()) + " | " + DurationFormatUtils.formatDurationHMS((long)duration.getMillis()) + " | " + new DecimalFormat("###.##").format(rate * 8.0) + " Mbit/s" + " | " + new DecimalFormat("###.##").format(rate) + " MB/s", new Object[0]);
            }
        }
        catch (IOException e) {
            this.logger.warn("EXCEPTION ", (Throwable)e);
            tmpFile.delete();
            throw e;
        }
        finally {
            IOUtils.closeIfPossible(in);
            IOUtils.closeIfPossible(out);
        }
        if (tmpFile.length() < authorisation.getFileSize()) {
            this.logger.info("PUT denied for " + this.constructUrl(request) + ": stream was shorter than authorised file size", new Object[0]);
            response.sendError(500);
            return;
        }
        if (tmpFile.length() > authorisation.getFileSize()) {
            this.logger.info("PUT denied for " + this.constructUrl(request) + ": stream was longer than authorised file size", new Object[0]);
            response.sendError(500);
            return;
        }
        String uuid = AuthorisedUrlRepository.stripCompressionSuffix(IOUtils.getFilenameWithoutPath(request));
        long size = tmpFile.length();
        String checksum = in.getChecksum();
        if (this.useChecksums) {
            Md5FileUtils.writeMd5(checksum, targetFile);
        }
        if (targetFile.exists()) {
            this.logger.debug("uploaded file exists already, keeping the old one", new Object[0]);
            tmpFile.delete();
        } else {
            this.logger.debug("rename uploaded file to make it visible", new Object[0]);
            boolean success = tmpFile.renameTo(targetFile);
            if (!success) {
                if (targetFile.exists()) {
                    tmpFile.delete();
                } else {
                    this.logger.debug("rename failed", new Object[0]);
                    response.sendError(500);
                    return;
                }
            }
        }
        if (this.isStorageRequest(request)) {
            try {
                this.metadataServer.addFile(uuid, size);
            }
            catch (SQLException e) {
                throw new ServletException((Throwable)e);
            }
        }
        if (this.cacheCleanUp != null) {
            this.cacheCleanUp.spaceRequest(0L, false, "Schedule clean up after PUT");
        }
        if (this.useChecksums) {
            response.setHeader("Etag", checksum);
        }
        response.setStatus(204);
    }

    private double getTransferRate(long fileSize, Duration duration) {
        double rate = duration.getMillis() != 0L ? (double)fileSize / (double)duration.getMillis() * 1000.0 / 1024.0 / 1024.0 : 0.0;
        return rate;
    }

    private File getUploadTempFile(File targetFile) {
        File tmpFile = null;
        for (int i = 1; i < 100 && (tmpFile = new File(targetFile.getAbsolutePath() + "." + i + UPLOAD_FILE_EXTENSION)).exists(); ++i) {
        }
        return tmpFile;
    }

    protected void doDelete(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        response.setStatus(405);
    }

    protected void doHead(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        response.setStatus(405);
    }

    protected void doOptions(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        response.setStatus(405);
    }

    protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        response.setStatus(405);
    }

    protected void doTrace(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        response.setStatus(405);
    }

    private File locateFile(HttpServletRequest request) throws ServletException {
        String requestPath = URIUtil.addPaths((String)request.getServletPath(), (String)request.getPathInfo());
        String realPath = this.getServletContext().getRealPath(requestPath);
        if (realPath == null) {
            throw new ServletException("Servlet context refused to convert a request path to a real path. Make sure that serving files through symlinks is enabled");
        }
        return new File(realPath);
    }

    private URL constructUrl(HttpServletRequest request) throws MalformedURLException {
        return new URL(this.rootUrl + request.getPathInfo());
    }
}

