/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.threading;

import com.google.java.contract.Ensures;
import com.google.java.contract.Invariant;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadFactory;
import org.apache.log4j.Logger;
import org.apache.log4j.Priority;
import org.broadinstitute.sting.utils.AutoFormattingTime;

@Invariant(value={"activeThreads.size() <= nThreadsToCreate", "countDownLatch.getCount() <= nThreadsToCreate", "nThreadsToCreated <= nThreadsToCreate"})
public class StateMonitoringThreadFactory
implements ThreadFactory {
    protected static final boolean DEBUG = false;
    private static Logger logger = Logger.getLogger(StateMonitoringThreadFactory.class);
    public static final List<Thread.State> TRACKED_STATES = Arrays.asList(Thread.State.BLOCKED, Thread.State.RUNNABLE, Thread.State.WAITING);
    final int nThreadsToCreate;
    final List<Thread> activeThreads;
    final EnumMap<Thread.State, Long> times = new EnumMap(Thread.State.class);
    int nThreadsToCreated = 0;
    final ThreadMXBean bean;
    final CountDownLatch countDownLatch;
    static final EnumMap<Thread.State, String> PRETTY_NAMES = new EnumMap(Thread.State.class);

    public StateMonitoringThreadFactory(int nThreadsToCreate) {
        if (nThreadsToCreate <= 0) {
            throw new IllegalArgumentException("nThreadsToCreate <= 0: " + nThreadsToCreate);
        }
        this.nThreadsToCreate = nThreadsToCreate;
        this.activeThreads = new ArrayList<Thread>(nThreadsToCreate);
        for (Thread.State state : Thread.State.values()) {
            this.times.put(state, 0L);
        }
        this.bean = ManagementFactory.getThreadMXBean();
        if (this.bean.isThreadContentionMonitoringSupported()) {
            this.bean.setThreadContentionMonitoringEnabled(true);
        } else {
            logger.warn("Thread contention monitoring not supported, we cannot track GATK multi-threaded efficiency");
        }
        this.countDownLatch = new CountDownLatch(nThreadsToCreate);
    }

    @Ensures(value={"result >= 0", "TRACKED_STATES.contains(state)"})
    public synchronized long getStateTime(Thread.State state) {
        return this.times.get((Object)state);
    }

    @Ensures(value={"result >= 0"})
    public synchronized long getTotalTime() {
        long total = 0L;
        for (long time : this.times.values()) {
            total += time;
        }
        return total;
    }

    @Ensures(value={"result >= 0.0", "result <= 1.0", "TRACKED_STATES.contains(state)"})
    public synchronized double getStateFraction(Thread.State state) {
        return (double)this.getStateTime(state) / (1.0 * (double)Math.max(this.getTotalTime(), 1L));
    }

    @Ensures(value={"result >= 0"})
    public int getNThreadsCreated() {
        return this.nThreadsToCreated;
    }

    public void waitForAllThreadsToComplete() throws InterruptedException {
        this.countDownLatch.await();
    }

    public synchronized String toString() {
        StringBuilder b = new StringBuilder();
        b.append("total ").append(this.getTotalTime()).append(" ");
        for (Thread.State state : TRACKED_STATES) {
            b.append((Object)state).append(" ").append(this.getStateTime(state)).append(" ");
        }
        return b.toString();
    }

    public synchronized void printUsageInformation(Logger logger) {
        this.printUsageInformation(logger, Priority.INFO);
    }

    public synchronized void printUsageInformation(Logger logger, Priority priority) {
        logger.log(priority, "Number of activeThreads used: " + this.getNThreadsCreated());
        logger.log(priority, "Total runtime " + new AutoFormattingTime((double)this.getTotalTime() / 1000.0));
        for (Thread.State state : TRACKED_STATES) {
            logger.log(priority, String.format("  Fraction of time spent %s is %.2f (%s)", this.prettyName(state), this.getStateFraction(state), new AutoFormattingTime((double)this.getStateTime(state) / 1000.0)));
        }
        logger.log(priority, String.format("Efficiency of multi-threading: %.2f%% of time spent doing productive work", this.getStateFraction(Thread.State.RUNNABLE) * 100.0));
    }

    private String prettyName(Thread.State state) {
        return PRETTY_NAMES.get((Object)state);
    }

    @Override
    @Ensures(value={"activeThreads.size() > old(activeThreads.size())", "activeThreads.contains(result)", "nThreadsToCreated == old(nThreadsToCreated) + 1"})
    public synchronized Thread newThread(Runnable runnable) {
        if (this.activeThreads.size() >= this.nThreadsToCreate) {
            throw new IllegalStateException("Attempting to create more activeThreads than allowed by constructor argument nThreadsToCreate " + this.nThreadsToCreate);
        }
        ++this.nThreadsToCreated;
        TrackingThread myThread = new TrackingThread(runnable);
        this.activeThreads.add(myThread);
        return myThread;
    }

    @Ensures(value={"activeThreads.size() < old(activeThreads.size())", "! activeThreads.contains(thread)", "getTotalTime() >= old(getTotalTime())", "countDownLatch.getCount() < old(countDownLatch.getCount())"})
    private synchronized void threadIsDone(Thread thread, long runtimeInMilliseconds) {
        ThreadInfo info = this.bean.getThreadInfo(thread.getId());
        if (info != null) {
            this.incTimes(Thread.State.BLOCKED, info.getBlockedTime());
            this.incTimes(Thread.State.WAITING, info.getWaitedTime());
            this.incTimes(Thread.State.RUNNABLE, runtimeInMilliseconds - info.getWaitedTime() - info.getBlockedTime());
        }
        if (!this.activeThreads.remove(thread)) {
            throw new IllegalStateException("Thread " + thread + " not in list of active activeThreads");
        }
        this.countDownLatch.countDown();
    }

    private synchronized void incTimes(Thread.State state, long by) {
        this.times.put(state, this.times.get((Object)state) + by);
    }

    static {
        PRETTY_NAMES.put(Thread.State.RUNNABLE, "running");
        PRETTY_NAMES.put(Thread.State.BLOCKED, "blocked");
        PRETTY_NAMES.put(Thread.State.WAITING, "waiting");
    }

    private class TrackingThread
    extends Thread {
        private TrackingThread(Runnable runnable) {
            super(runnable);
        }

        @Override
        public void run() {
            long startTime = System.currentTimeMillis();
            super.run();
            long endTime = System.currentTimeMillis();
            StateMonitoringThreadFactory.this.threadIsDone(this, endTime - startTime);
        }
    }
}

