/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.gatk.downsampling;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import net.sf.samtools.SAMRecord;
import org.broadinstitute.sting.gatk.downsampling.ReadsDownsampler;
import org.broadinstitute.sting.gatk.downsampling.ReservoirDownsampler;
import org.broadinstitute.sting.utils.MathUtils;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;

public class PositionalDownsampler<T extends SAMRecord>
implements ReadsDownsampler<T> {
    private int targetCoverage;
    private ReservoirDownsampler<T> reservoir;
    private int currentContigIndex;
    private int currentAlignmentStart;
    private LinkedList<PositionalReadGrouping> pendingReads;
    private ArrayList<T> finalizedReads;

    public PositionalDownsampler(int targetCoverage) {
        this.targetCoverage = targetCoverage;
        this.clear();
    }

    @Override
    public void submit(T newRead) {
        if (this.readIsPastCurrentPosition(newRead)) {
            this.updateAndDownsamplePendingReads();
        }
        this.reservoir.submit(newRead);
        this.updateCurrentPosition(newRead);
    }

    @Override
    public void submit(Collection<T> newReads) {
        for (SAMRecord read : newReads) {
            this.submit((T)read);
        }
    }

    @Override
    public boolean hasDownsampledItems() {
        return this.finalizedReads.size() > 0;
    }

    @Override
    public List<T> consumeDownsampledItems() {
        ArrayList<T> toReturn = this.finalizedReads;
        this.finalizedReads = new ArrayList();
        return toReturn;
    }

    @Override
    public boolean hasPendingItems() {
        return this.pendingReads.size() > 0;
    }

    @Override
    public void signalEndOfInput() {
        this.updateAndDownsamplePendingReads();
        for (PositionalReadGrouping group : this.pendingReads) {
            group.finalizeAllActiveReads();
            this.finalizedReads.addAll(group.getFinalizedReads());
        }
        this.pendingReads.clear();
    }

    @Override
    public void clear() {
        this.reservoir = new ReservoirDownsampler(this.targetCoverage);
        this.pendingReads = new LinkedList();
        this.finalizedReads = new ArrayList();
    }

    @Override
    public boolean requiresCoordinateSortOrder() {
        return true;
    }

    private void updateCurrentPosition(T read) {
        this.currentContigIndex = ((SAMRecord)read).getReferenceIndex();
        this.currentAlignmentStart = ((SAMRecord)read).getAlignmentStart();
    }

    private boolean readIsPastCurrentPosition(T read) {
        return ((SAMRecord)read).getReferenceIndex() != this.currentContigIndex || ((SAMRecord)read).getAlignmentStart() > this.currentAlignmentStart;
    }

    private void updateAndDownsamplePendingReads() {
        this.finalizeOutOfScopeReads();
        List<T> oldLocusReads = this.reservoir.consumeDownsampledItems();
        this.pendingReads.add(new PositionalReadGrouping(oldLocusReads, this.currentContigIndex, this.currentAlignmentStart));
        this.downsampleOverlappingGroups();
    }

    private void finalizeOutOfScopeReads() {
        Iterator iter = this.pendingReads.iterator();
        boolean noPrecedingUnfinalizedGroups = true;
        while (iter.hasNext()) {
            PositionalReadGrouping currentGroup = (PositionalReadGrouping)iter.next();
            currentGroup.finalizeActiveReadsBeforePosition(this.currentContigIndex, this.currentAlignmentStart);
            if (currentGroup.isFinalized() && noPrecedingUnfinalizedGroups) {
                iter.remove();
                this.finalizedReads.addAll(currentGroup.getFinalizedReads());
                continue;
            }
            noPrecedingUnfinalizedGroups = false;
        }
    }

    private void downsampleOverlappingGroups() {
        int[] groupReadCounts = new int[this.pendingReads.size()];
        int totalCoverage = 0;
        int numActiveGroups = 0;
        int currentGroup = 0;
        for (PositionalReadGrouping group : this.pendingReads) {
            groupReadCounts[currentGroup] = group.numActiveReads();
            totalCoverage += groupReadCounts[currentGroup];
            if (groupReadCounts[currentGroup] > 0) {
                ++numActiveGroups;
            }
            ++currentGroup;
        }
        if (totalCoverage <= this.targetCoverage) {
            return;
        }
        int numReadsToRemove = Math.min(totalCoverage - this.targetCoverage, totalCoverage - numActiveGroups);
        currentGroup = 0;
        while (numReadsToRemove > 0) {
            if (groupReadCounts[currentGroup] > 1) {
                int n = currentGroup;
                groupReadCounts[n] = groupReadCounts[n] - 1;
                --numReadsToRemove;
            }
            currentGroup = (currentGroup + 1) % groupReadCounts.length;
        }
        currentGroup = 0;
        for (PositionalReadGrouping group : this.pendingReads) {
            if (!group.isFinalized()) {
                group.downsampleActiveReads(groupReadCounts[currentGroup]);
            }
            ++currentGroup;
        }
    }

    private class PositionalReadGrouping {
        private List<T> activeReads;
        private List<T> finalizedReads;
        private int contig;
        private int alignmentStart;

        public PositionalReadGrouping(Collection<T> reads, int contig, int alignmentStart) {
            this.activeReads = new LinkedList(reads);
            this.finalizedReads = new ArrayList();
            this.contig = contig;
            this.alignmentStart = alignmentStart;
        }

        public int numActiveReads() {
            return this.activeReads.size();
        }

        public boolean isFinalized() {
            return this.activeReads.size() == 0;
        }

        public List<T> getFinalizedReads() {
            return this.finalizedReads;
        }

        public void finalizeActiveReadsBeforePosition(int contig, int position) {
            if (this.contig != contig) {
                this.finalizeAllActiveReads();
                return;
            }
            Iterator iter = this.activeReads.iterator();
            while (iter.hasNext()) {
                SAMRecord read = (SAMRecord)iter.next();
                if (read.getAlignmentEnd() >= position) continue;
                iter.remove();
                this.finalizedReads.add(read);
            }
        }

        public void finalizeAllActiveReads() {
            this.finalizedReads.addAll(this.activeReads);
            this.activeReads.clear();
        }

        public void downsampleActiveReads(int numReadsToKeep) {
            if (numReadsToKeep > this.activeReads.size() || numReadsToKeep < 0) {
                throw new ReviewedStingException(String.format("Cannot retain %d reads out of %d total reads", numReadsToKeep, this.activeReads.size()));
            }
            BitSet itemsToKeep = new BitSet(this.activeReads.size());
            for (Integer selectedIndex : MathUtils.sampleIndicesWithoutReplacement(this.activeReads.size(), numReadsToKeep)) {
                itemsToKeep.set(selectedIndex);
            }
            int currentIndex = 0;
            Iterator iter = this.activeReads.iterator();
            while (iter.hasNext()) {
                SAMRecord read = (SAMRecord)iter.next();
                if (!itemsToKeep.get(currentIndex)) {
                    iter.remove();
                }
                ++currentIndex;
            }
        }
    }
}

