/**
 * @file zscore.c
 *
 * This module computes the z-score by supplying the population
 * from which the sample mean and deviation is determined
 *
 * author: Fabian Buske
 * version: 1.0 (26.06.2008)
 *
*/

#include <assert.h>
#include <stdio.h>
#include "zscore.h"
#include <math.h>
#include <macros.h>
#include "utils.h"

/**********************************************************************
  zscore

  this struct contains the results of a ranksum calculation
**********************************************************************/
struct zscore{
	double mean;  	/* the mean of the population */
	double var;		/* the variance of the population */
	double sd;		/* the standard deviation of the population */
};

/**********************************************************************
  allocate_ranksum_result()

  Constructor for the ranksum result data structure
**********************************************************************/
ZSCORE_T *allocate_zscore(
		double* population,	/* the population */
		int size			/* the size of the population */
){
	assert(population != NULL);
	assert(size >= 1);

	// Allocate memory and initialize fields
	ZSCORE_T *score = (ZSCORE_T *) mm_malloc(sizeof(ZSCORE_T));

	double sum = 0.0;
	int i;
	for (i=0; i<size; ++i) {
		sum += population[i];
	}
	score->mean = sum/size;
	double dev = 0.0;
	for (i=0; i<size; ++i) {
		dev += (population[i]-score->mean)*(population[i]-score->mean);
	}
	score->var = dev / size;
	score->sd = sqrt(score->var);

	return score;
};

/**********************************************************************
  get_population_details()

  Returns the population details for the given zscore
**********************************************************************/
void get_population_details(
		ZSCORE_T* zscore,	// population of interest (IN)
		double* mean, 		// mean of population (OUT)
		double* sd			// standard deviation of population (OUT)
){
	assert(zscore != NULL);
	*mean = zscore->mean;
	*sd = zscore->sd;
}

/**********************************************************************
  get_zscore()

  Compute the z-score from the mean and standard deviation.
  Returns the z-score for x.
**********************************************************************/
extern double get_zscore(
		double x,	 	/* the value for which a z-score is requested */
		double mean, 	/* the mean */
		double sd		/* the standard deviation */
){
	return (x-mean)/sd;
}

/**********************************************************************
  get_zscore_z()

  Compute the z-score for x given a zscore struct and returns it.
**********************************************************************/
double get_zscore_z(
		ZSCORE_T* zscore, 	/* zscore struct holding mean and sd IN */
		double x			/* the value for which a z-score is requested IN */
){
	return (x - zscore->mean)/zscore->sd;
}

/**********************************************************************
  destroy_zscore()

  Frees the memory of a ZSCORE_T* data structure
**********************************************************************/
void destroy_zscore(ZSCORE_T *zscore){
	myfree(zscore);
}


