/***************************************************************************
 * FILE: params.c
 * AUTHOR: William Stafford Noble
 * CREATE DATE: 8/23/01
 * PROJECT: MHMM
 * COPYRIGHT: 2001-2008, WSN
 * DESCRIPTION: Type for keeping track of different kinds of model parameters.
 ***************************************************************************/
#include "params.h"
#include "utils.h"
#include "mhmm-state.h"
#include <string.h>

/* The parameter sets selected for training by the user. */
BOOLEAN_T trans_train = FALSE;
BOOLEAN_T length_train = FALSE;
BOOLEAN_T spacer_train = FALSE;
BOOLEAN_T motif_train = FALSE;

char*  TRAIN_STRS[] = {"invalid", "none", "trans", "length", "spacer",
		       "motif", "all"};
int NUM_TRAIN_T = 7;

/***************************************************************************
 * Add to the global list of parameters to train.
 ***************************************************************************/
void add_train_type
  (TRAIN_T new_train_type) 
{
  switch (new_train_type) {
  case INVALID_TRAIN :
    die("Invalid train type.\n");
  case NONE_TRAIN :
    trans_train = FALSE;
    length_train = FALSE;
    spacer_train = FALSE;
    motif_train = FALSE;
    break;
  case TRANS_TRAIN :
    trans_train = TRUE;
    break;
  case LENGTH_TRAIN :
    length_train = TRUE;
    break;
  case SPACER_TRAIN :
    spacer_train = TRUE;
    break;
  case MOTIF_TRAIN :
    motif_train = TRUE;
    break;
  case ALL_TRAIN :
    trans_train = TRUE;
    length_train = TRUE;
    spacer_train = TRUE;
    motif_train = TRUE;
    break;
  }
}

/***************************************************************************
 * Are any of the emission distributions being trained?
 ***************************************************************************/
BOOLEAN_T train_emissions()
{
  return(spacer_train || motif_train);
}

/***************************************************************************
 * Are any of the transition distributions being trained?
 ***************************************************************************/
BOOLEAN_T train_transitions()
{
  return(trans_train || length_train);
}

/***************************************************************************
 * Is a particular parameter subset being trained?
 ***************************************************************************/
BOOLEAN_T do_train
  (TRAIN_T train_what) 
{
  switch (train_what) {
  case INVALID_TRAIN :
    die("Invalid train type.\n");
  case NONE_TRAIN :
    die("Invalid train type.\n");
  case TRANS_TRAIN :
    return(trans_train);
  case LENGTH_TRAIN :
    return(length_train);
  case SPACER_TRAIN :
    return(spacer_train);
  case MOTIF_TRAIN :
    return(motif_train);
  case ALL_TRAIN :
    die("Invalid train type.\n");
  }
  return(FALSE); /* Unreachable. */
}  
  

/***************************************************************************
 * Get a string representing all the parameters currently being trained.
 ***************************************************************************/
char*  train_string()
{
  static char return_string[100];

  return_string[0] = '\0';

  if (trans_train) {
    strcat(return_string, "trans ");
  }
  if (length_train) {
    strcat(return_string, "length ");
  }
  if (spacer_train) {
    strcat(return_string, "spacer ");
  }
  if (motif_train) {
    strcat(return_string, "motif ");
  } 

  if (return_string[0] == '\0') {
    return("none");
  }

  /* Get rid of the trailing space. */
  return_string[strlen(return_string) - 1] = '\0';

  return(return_string);
}


/***************************************************************************
 * Determine whether transitions from a given state should be trained.
 ***************************************************************************/
BOOLEAN_T train_state_transition
  (MHMM_STATE_T * this_state)
{
  switch(this_state->type) {
  case START_MOTIF_STATE :
  case MID_MOTIF_STATE :
    /* Internal motif transitions are always 1.0. */
    return(FALSE);
  case END_STATE :
    /* The end state has no outgoing transitions. */
    return(FALSE);
  case START_STATE :
  case END_MOTIF_STATE :
    /* Only train motif-to-motif transitions if it was asked for. */
    return(do_train(TRANS_TRAIN));
  case SPACER_STATE :
    /* Only train spacer self-loops if it was asked for. */
    return(do_train(LENGTH_TRAIN));
  case INVALID_STATE :
    die("Invalid state type.\n");
  }
  return(FALSE); /* Unreachable. */
}

/***************************************************************************
 * Determine whether emissions from a given state should be trained.
 ***************************************************************************/
BOOLEAN_T train_state_emissions
  (MHMM_STATE_T * this_state)
{
  switch(this_state->type) {
  case END_STATE :
  case START_STATE :
    /* The start and end states have no emissions. */
    return(FALSE);
  case START_MOTIF_STATE :
  case MID_MOTIF_STATE :
  case END_MOTIF_STATE :
    /* Only train motif emissions if it was asked for. */
    return(do_train(MOTIF_TRAIN));
  case SPACER_STATE :
    /* Only train spacer self-loops if it was asked for. */
    return(do_train(SPACER_TRAIN));
  case INVALID_STATE :
    die("Invalid state type.\n");
  }
  return(FALSE); /* Unreachable. */
}

