/* @source enstable ***********************************************************
**
** Ensembl Table functions
**
** @author Copyright (C) 1999 Ensembl Developers
** @author Copyright (C) 2006 Michael K. Schuster
** @version $Revision: 1.32 $
** @modified 2009 by Alan Bleasby for incorporation into EMBOSS core
** @modified $Date: 2012/07/14 14:52:40 $ by $Author: rice $
** @@
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Lesser General Public
** License as published by the Free Software Foundation; either
** version 2.1 of the License, or (at your option) any later version.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Lesser General Public License for more details.
**
** You should have received a copy of the GNU Lesser General Public
** License along with this library; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
** MA  02110-1301,  USA.
**
******************************************************************************/

/* ========================================================================= */
/* ============================= include files ============================= */
/* ========================================================================= */

#include "enstable.h"




/* ========================================================================= */
/* =============================== constants =============================== */
/* ========================================================================= */




/* ========================================================================= */
/* =========================== global variables ============================ */
/* ========================================================================= */




/* ========================================================================= */
/* ============================= private data ============================== */
/* ========================================================================= */




/* ========================================================================= */
/* =========================== private constants =========================== */
/* ========================================================================= */




/* ========================================================================= */
/* =========================== private variables =========================== */
/* ========================================================================= */




/* ========================================================================= */
/* =========================== private functions =========================== */
/* ========================================================================= */

static AjBool tableuintliststrMatch(AjPStr *Pstr, void *cl);




/* ========================================================================= */
/* ======================= All functions by section ======================== */
/* ========================================================================= */




/* @filesection enstable ******************************************************
**
** @nam1rule ens Function belongs to the Ensembl library
**
******************************************************************************/




/* @datasection [AjPTable] AJAX Table *****************************************
**
** @nam2rule Table Functions for manipulating AJAX Table objects
**
******************************************************************************/




/* @datasection [AjPTable] AJAX Table *****************************************
**
** @nam2rule Tableuintliststr Functions for manipulating AJAX Table objects of
**                            AJAX unsigned integer key data and
**                            AJAX String-List value data
**
******************************************************************************/




/* @section constructors ******************************************************
**
** These constructors return a new AJAX Table by pointer.
** It is the responsibility of the user to first destroy any previous
** Table. The target pointer does not need to be initialised to
** NULL, but it is good programming practice to do so anyway.
**
** @fdata [AjPTable]
**
** @nam3rule New Constructor
** @nam4rule Len Constructor with an initial length
**
** @argrule Len size [ajuint] Number of key-value pairs
**
** @valrule * [AjPTable] AJAX Table or NULL
**
** @fcategory new
******************************************************************************/




/* @func ensTableuintliststrNewLen ********************************************
**
** Creates, initialises, and returns a new, empty AJAX Table that can hold an
** arbitrary number of key-value pairs.
** The AJAX Table is initialised with comparison, hash and key deletion
** functions suitable for AJAX unsigned integer keys.
**
** @param [r] size [ajuint] Number of key-value pairs
**
** @return [AjPTable] AJAX Table or NULL
**
** @release 6.4.0
** @@
**
******************************************************************************/

AjPTable ensTableuintliststrNewLen(ajuint size)
{
    AjPTable table = NULL;

    table = ajTableuintNew(size);

    ajTableSetDestroyvalue(table, (void (*)(void **)) &ajListstrFreeData);

    return table;
}




/* @section destructors *******************************************************
**
** Destruction destroys all internal data structures and frees the memory
** allocated for an AJAX Table object.
**
** @fdata [AjPTable]
**
** @nam3rule Clear  Clear
** @nam3rule Delete Destroy (free) an AJAX Table of
**                  AJAX unsigned integer key data and
**                  AJAX String-List value data
**
** @argrule Clear table [AjPTable] AJAX Table
** @argrule Delete Ptable [AjPTable*] AJAX Table address
**
** @valrule * [void]
**
** @fcategory delete
******************************************************************************/




/* @func ensTableuintliststrClear *********************************************
**
** Utility function to clear an AJAX Table of
** AJAX unsigned integer key data and
** AJAX String-List value data.
**
** @param [u] table [AjPTable] AJAX Table
**
** @return [void]
**
** @release 6.4.0
** @@
******************************************************************************/

void ensTableuintliststrClear(AjPTable table)
{
    ajTableClearDelete(table);

	return;
}




/* @func ensTableuintliststrDelete ********************************************
**
** Utility function to clear and delete an AJAX Table of
** AJAX unsigned integer key data and
** AJAX String-List value data.
**
** @param [d] Ptable [AjPTable*] AJAX Table address
**
** @return [void]
**
** @release 6.4.0
** @@
******************************************************************************/

void ensTableuintliststrDelete(AjPTable *Ptable)
{
    ajTableDel(Ptable);

	return;
}




/* @section register **********************************************************
**
** Registers AJAX String objects in an AJAX Table of
** AJAX unsigned integer key data and
** AJAX String-List value data.
**
** @fdata [AjPTable]
**
** @nam3rule Register Register an AJAX String
**
** @argrule Register table [AjPTable] AJAX Table
** @argrule Register identifier [ajuint] AJAX unsigned integer identifier
** @argrule Register str [AjPStr] AJAX String
**
** @valrule * [AjBool] ajTrue upon success, ajFalse otherwise
**
** @fcategory use
******************************************************************************/




/* @funcstatic tableuintliststrMatch ******************************************
**
** An ajListstrMapfind "apply" function to match AJAX String objects in an
** AJAX String-List.
**
** @param [u] Pstr [AjPStr*] AJAX String address
** @param [u] cl [void*] AJAX String
** @see ajListstrMapfind
**
** @return [AjBool] ajTrue:  A particular AJAX String has already been indexed.
**                  ajFalse: A particular AJAX String has not been indexed
**                           before, but has been added now.
**
** @release 6.4.0
** @@
******************************************************************************/

static AjBool tableuintliststrMatch(AjPStr *Pstr, void *cl)
{
    if (!Pstr)
        return ajFalse;

    if (!cl)
        return ajFalse;

    return ajStrMatchS(*Pstr, (AjPStr) cl);
}




/* @func ensTableuintliststrRegister ******************************************
**
** For an AJAX Table of AJAX unsigned integer key data and AJAX String-List
** value data check, whether an AJAX String has already been indexed for a
** particular AJAX unsigned integer.
**
** @param [u] table [AjPTable] AJAX Table
** @param [rN] identifier [ajuint] AJAX unsigned integer identifier
** @param [u] str [AjPStr] AJAX String
**
** @return [AjBool] ajTrue:  A particular AJAX String has already been indexed.
**                  ajFalse: A particular AJAX String has not been indexed
**                           before, but has been added now.
**
** @release 6.4.0
** @@
******************************************************************************/

AjBool ensTableuintliststrRegister(AjPTable table,
                                   ajuint identifier,
                                   AjPStr str)
{
    ajuint *Pidentifier = NULL;

    AjPList list = NULL;

    if (ajDebugTest("ensTableuintliststrRegister"))
        ajDebug("ensTableuintliststrRegister\n"
                "  table %p\n"
                "  identifier %u\n"
                "  str '%S'\n",
                table,
                identifier,
                str);

    if (!table)
        return ajFalse;

    if (!str)
        return ajFalse;

    list = (AjPList) ajTableFetchmodV(table, (const void *) &identifier);

    if (list)
    {
        if (ajListstrMapfind(list, &tableuintliststrMatch, (void *) str))
            return ajTrue;
        else
        {
            ajListstrPushAppend(list, ajStrNewS(str));

            return ajFalse;
        }
    }
    else
    {
        AJNEW0(Pidentifier);

        *Pidentifier = identifier;

        list = ajListstrNew();

        ajTablePut(table, (void *) Pidentifier, (void *) list);

        ajListstrPushAppend(list, ajStrNewS(str));

        return ajFalse;
    }
}
