/* trig.c
 * Trigonometric functions for RLaB */

/*  This file is a part of RLaB ("Our"-LaB)
    Copyright (C) 1992  Ian R. Searle

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    See the file ./COPYING
 ***********************************************************************/

#include "rlab.h"
#include "bltin.h"
#include "code.h"
#include "symbol.h"

#include "complex.h"
#include "scop1.h"
#include "matop1.h"
#include "matop2.h"
#include "list.h"
#include "listnode.h"
#include "r_string.h"
#include "util.h"
#include "mathl.h"

#include <math.h>

#define TARG_DESTROY(arg, targ)   if (targ.u.ent != arg.u.ent) \
                                    remove_tmp_destroy (targ.u.ent);

/* **************************************************************
 * Arc-Tangent. Only compute arc-tan-2 for scalars.
 * ************************************************************** */
void
Atan2 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, arg2, targ1, targ2;

  /* Check n_args */
  if (n_args != 2)
    error_1 ("Wrong number of args to atan2", 0);

  /* get args from list */

  arg1 = get_bltin_arg ("atan2", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("atan2", d_arg, 2, NUM);

  arg1 = convert_to_scalar (arg1);
  arg2 = convert_to_scalar (arg2);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  *return_ptr = (VPTR) matrix_Atan2 (e_data (targ1.u.ent),
				     e_data (targ2.u.ent));

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);

  return;
}

/* **************************************************************
 * sine function.
 * ************************************************************** */
void
Sin (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d;
  Complex z;
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to sin()", 0);

  /* get args from list */

  arg = get_bltin_arg ("sin", d_arg, 1, NUM);

  switch (arg.type)
  {
  case CONSTANT:
    d = errcheck (sin (arg.u.val), "sin");
    *return_ptr = (VPTR) scalar_Create (d);
    break;
  case iCONSTANT:
    z = complex_sin (0.0, arg.u.val);
    *return_ptr = (VPTR) scalar_CreateC (z.r, z.i);
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Sin (e_data (arg.u.ent));
      break;
    case MATRIX:
      *return_ptr = (VPTR) matrix_Sin (e_data (arg.u.ent));
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for sin()");
      break;
    }
    break;
  }
}

/* **************************************************************
 * arc-sine function.
 * ************************************************************** */
void
Asin (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d;
  Complex z;
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to asin()", 0);

  /* get args from list */

  arg = get_bltin_arg ("asin", d_arg, 1, NUM);

  switch (arg.type)
  {
  case CONSTANT:
    if (arg.u.val >= 1. || arg.u.val <= -1.)
    {
      z.r = arg.u.val;
      z.i = 0.0;
      z = complex_Asin (z);
      *return_ptr = (VPTR) scalar_CreateC (z.r, z.i);
    }
    else
    {
      d = errcheck (asin (arg.u.val), "asin");
      *return_ptr = (VPTR) scalar_Create (d);
    }
    break;
  case iCONSTANT:
    z.r = 0.0;
    z.i = arg.u.val;
    z = complex_Asin (z);
    *return_ptr = (VPTR) scalar_CreateC (z.r, z.i);
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Asin (e_data (arg.u.ent));
      break;
    case MATRIX:
      *return_ptr = (VPTR) matrix_Asin (e_data (arg.u.ent));
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for asin()");
      break;
    }
    break;
  }
}

/* **************************************************************
 * cosine function.
 * ************************************************************** */
void
Cos (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d;
  Complex z;
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to cos()", 0);

  /* get args from list */

  arg = get_bltin_arg ("cos", d_arg, 1, NUM);

  switch (arg.type)
  {
  case CONSTANT:
    d = errcheck (cos (arg.u.val), "cos");
    *return_ptr = (VPTR) scalar_Create (d);
    break;
  case iCONSTANT:
    z = complex_cos (0.0, arg.u.val);
    *return_ptr = (VPTR) scalar_CreateC (z.r, z.i);
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Cos (e_data (arg.u.ent));
      break;
    case MATRIX:
      *return_ptr = (VPTR) matrix_Cos (e_data (arg.u.ent));
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for cos()");
      break;
    }
    break;
  }
}

/* **************************************************************
 * arc-cosine function.
 * ************************************************************** */
void
Acos (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d;
  Complex z;
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to acos()", 0);

  /* get args from list */

  arg = get_bltin_arg ("acos", d_arg, 1, NUM);

  switch (arg.type)
  {
  case CONSTANT:
    if (arg.u.val >= 1. || arg.u.val <= -1.)
    {
      z.r = arg.u.val;
      z.i = 0.0;
      z = complex_Acos (z);
      *return_ptr = (VPTR) scalar_CreateC (z.r, z.i);
    }
    else
    {
      d = errcheck (acos (arg.u.val), "acos");
      *return_ptr = (VPTR) scalar_Create (d);
    }
    break;
  case iCONSTANT:
    z.r = 0.0;
    z.i = arg.u.val;
    z = complex_Acos (z);
    *return_ptr = (VPTR) scalar_CreateC (z.r, z.i);
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Acos (e_data (arg.u.ent));
      break;
    case MATRIX:
      *return_ptr = (VPTR) matrix_Acos (e_data (arg.u.ent));
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for acos()");
      break;
    }
    break;
  }
}

/* **************************************************************
 * tangent function.
 * ************************************************************** */
void
Tan (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d;
  Complex z;
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to tan()", 0);

  /* get args from list */

  arg = get_bltin_arg ("tan", d_arg, 1, NUM);

  switch (arg.type)
  {
  case CONSTANT:
    d = errcheck (tan (arg.u.val), "tan");
    *return_ptr = (VPTR) scalar_Create (d);
    break;
  case iCONSTANT:
    z = complex_tan (0.0, arg.u.val);
    *return_ptr = (VPTR) scalar_CreateC (z.r, z.i);
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Tan (e_data (arg.u.ent));
      break;
    case MATRIX:
      *return_ptr = (VPTR) matrix_Tan (e_data (arg.u.ent));
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for tan()");
      break;
    }
    break;
  }
}

/* **************************************************************
 * arc-tangent function.
 * ************************************************************** */
void
Atan (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d;
  Complex z;
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to atan()", 0);

  /* get args from list */

  arg = get_bltin_arg ("atan", d_arg, 1, NUM);

  switch (arg.type)
  {
  case CONSTANT:
    d = errcheck (atan (arg.u.val), "atan");
    *return_ptr = (VPTR) scalar_Create (d);
    break;
  case iCONSTANT:
    z.r = 0.0;
    z.i = arg.u.val;
    z = complex_Atan (z);
    *return_ptr = (VPTR) scalar_CreateC (z.r, z.i);
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Atan (e_data (arg.u.ent));
      break;
    case MATRIX:
      *return_ptr = (VPTR) matrix_Atan (e_data (arg.u.ent));
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for atan()");
      break;
    }
    break;
  }
}
