/* relation.c 
   Take care of the object relational comparisons */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "code.h"

#include "util.h"
#include "scop1.h"
#include "matop1.h"
#include "matop2.h"
#include "r_string.h"

#define SCALAR_SCALAR 1111
#define SCALAR_MATRIX 1113
#define SCALAR_STRING 1120

#define MATRIX_MATRIX 1133
#define MATRIX_SCALAR 1131
#define MATRIX_STRING 1140

#define STRING_STRING 1210
#define STRING_SCALAR 1201
#define STRING_MATRIX 1203

/* **************************************************************
 * equal-to operation.
 * d1 == d2
 * ************************************************************** */

static Datum ent_ent_eq _PROTO ((ListNode * e1, ListNode * e2));

Datum
eq (d1, d2)
     Datum d1, d2;
{
  Datum new;

  d1 = convert_to_scalar (d1);
  d2 = convert_to_scalar (d2);

  new = ent_ent_eq (d1.u.ent, d2.u.ent);

  return (new);
}

/* **************************************************************
 * Entity == Entity operation.
 * ************************************************************** */
static Datum
ent_ent_eq (d1, d2)
     ListNode *d1, *d2;
{
  Datum new;

  switch (STYPE (e_type (d1), e_type (d2)))
  {
  case SCALAR_SCALAR:
    new.u.val = scalar_eq (e_data (d1), e_data (d2));
    new.type = CONSTANT;
    break;

  case SCALAR_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_eq (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_eq (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_eq (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_STRING:
    if (string_Compare (e_data (d1), e_data (d2)) == 0)
      new.u.val = 1.0;
    else
      new.u.val = 0.0;
    new.type = CONSTANT;
    break;

  case SCALAR_STRING:
  case STRING_SCALAR:
    new.u.val = 0.0;  /* This can never be TRUE */
    new.type = CONSTANT;
    break;

  case MATRIX_STRING:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_eq (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_eq (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (d1), e_name (d2), "invalid types for `=='");
  }
  remove_tmp_destroy (d1);
  remove_tmp_destroy (d2);
  return (new);
}

/* **************************************************************
 * not-equal-to operation.
 * ************************************************************** */

static Datum ent_ent_ne _PROTO ((ListNode * e1, ListNode * e2));

Datum
ne (d1, d2)
     Datum d1, d2;
{
  Datum new;

  d1 = convert_to_scalar (d1);
  d2 = convert_to_scalar (d2);

  new = ent_ent_ne (d1.u.ent, d2.u.ent);

  return (new);
}

/* **************************************************************
 * Entity != Entity operation.
 * ************************************************************** */
static Datum
ent_ent_ne (d1, d2)
     ListNode *d1, *d2;
{
  Datum new;

  switch (STYPE (e_type (d1), e_type (d2)))
  {
  case SCALAR_SCALAR:
    new.u.val = scalar_ne (e_data (d1), e_data (d2));
    new.type = CONSTANT;
    break;

  case SCALAR_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_ne (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_ne (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_ne (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_STRING:
    if (string_Compare (e_data (d1), e_data (d2)) == 0)
      new.u.val = 0.0;
    else
      new.u.val = 1.0;
    new.type = CONSTANT;
    break;

  case SCALAR_STRING:
  case STRING_SCALAR:
    new.u.val = 1.0;  /* This can never be FALSE */
    new.type = CONSTANT;
    break;

  case MATRIX_STRING:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_ne (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_ne (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (d1), e_name (d2), "invalid types for `!='");
  }
  remove_tmp_destroy (d1);
  remove_tmp_destroy (d2);
  return (new);
}

/* **************************************************************
 * less-than-or-equal-to operation.
 * ************************************************************** */

static Datum ent_ent_le _PROTO ((ListNode * e1, ListNode * e2));
static Datum ent_ent_ge _PROTO ((ListNode * e1, ListNode * e2));

Datum
le (d1, d2)
     Datum d1, d2;
{
  Datum new;

  d1 = convert_to_scalar (d1);
  d2 = convert_to_scalar (d2);

  new = ent_ent_le (d1.u.ent, d2.u.ent);

  return (new);
}

/* **************************************************************
 * Entity <= Entity operation.
 * ************************************************************** */
static Datum
ent_ent_le (d1, d2)
     ListNode *d1, *d2;
{
  Datum new;

  switch (STYPE (e_type (d1), e_type (d2)))
  {
  case SCALAR_SCALAR:
    new.u.val = scalar_le (e_data (d1), e_data (d2));
    new.type = CONSTANT;
    break;

  case SCALAR_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_le1 (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_le2 (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_le (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_STRING:
    if (string_Compare (e_data (d1), e_data (d2)) <= 0)
      new.u.val = 1.0;
    else
      new.u.val = 0.0;
    new.type = CONSTANT;
    break;

  case MATRIX_STRING:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_le1 (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_le2 (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (d1), e_name (d2), "invalid types for `<='");
  }
  remove_tmp_destroy (d1);
  remove_tmp_destroy (d2);

  return (new);
}

/* **************************************************************
 * greater-than-or-equal-to operation.
 * ************************************************************** */

Datum
ge (d1, d2)
     Datum d1, d2;
{
  Datum new;

  d1 = convert_to_scalar (d1);
  d2 = convert_to_scalar (d2);

  new = ent_ent_ge (d1.u.ent, d2.u.ent);

  return (new);
}

/* **************************************************************
 * Entity >= Entity operation.
 * ************************************************************** */
static Datum
ent_ent_ge (d1, d2)
     ListNode *d1, *d2;
{
  Datum new;

  switch (STYPE (e_type (d1), e_type (d2)))
  {
  case SCALAR_SCALAR:
    new.u.val = scalar_ge (e_data (d1), e_data (d2));
    new.type = CONSTANT;
    break;

  case SCALAR_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_ge1 (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_ge2 (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_ge (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_STRING:
    if (string_Compare (e_data (d1), e_data (d2)) >= 0)
      new.u.val = 1.0;
    else
      new.u.val = 0.0;
    new.type = CONSTANT;
    break;

  case MATRIX_STRING:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_ge1 (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_ge2 (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (d1), e_name (d2), "invalid types for `>='");
  }
  remove_tmp_destroy (d1);
  remove_tmp_destroy (d2);
  return (new);
}

/* **************************************************************
 * less-than operation.
 * ************************************************************** */

static Datum ent_ent_lt _PROTO ((ListNode * e1, ListNode * e2));
static Datum ent_ent_gt _PROTO ((ListNode * e1, ListNode * e2));

Datum
lt (d1, d2)
     Datum d1, d2;
{
  Datum new;

  d1 = convert_to_scalar (d1);
  d2 = convert_to_scalar (d2);

  new = ent_ent_lt (d1.u.ent, d2.u.ent);

  return (new);
}

/* **************************************************************
 * Entity < Entity operation.
 * ************************************************************** */
static Datum
ent_ent_lt (d1, d2)
     ListNode *d1, *d2;
{
  Datum new;

  switch (STYPE (e_type (d1), e_type (d2)))
  {
  case SCALAR_SCALAR:
    new.u.val = scalar_lt (e_data (d1), e_data (d2));
    new.type = CONSTANT;
    break;

  case SCALAR_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_lt1 (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_lt2 (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_lt (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_STRING:
    if (string_Compare (e_data (d1), e_data (d2)) < 0)
      new.u.val = 1.0;
    else
      new.u.val = 0.0;
    new.type = CONSTANT;
    break;

  case MATRIX_STRING:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_lt1 (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_lt2 (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (d1), e_name (d2), "invalid types for `<'");
  }
  remove_tmp_destroy (d1);
  remove_tmp_destroy (d2);
  return (new);
}

/* **************************************************************
 * greater-than operation.
 * ************************************************************** */

Datum
gt (d1, d2)
     Datum d1, d2;
{
  Datum new;

  d1 = convert_to_scalar (d1);
  d2 = convert_to_scalar (d2);

  new = ent_ent_gt (d1.u.ent, d2.u.ent);

  return (new);
}

/* **************************************************************
 * Entity > Entity operation.
 * ************************************************************** */
static Datum
ent_ent_gt (d1, d2)
     ListNode *d1, *d2;
{
  Datum new;

  switch (STYPE (e_type (d1), e_type (d2)))
  {
  case SCALAR_SCALAR:
    new.u.val = scalar_gt (e_data (d1), e_data (d2));
    new.type = CONSTANT;
    break;

  case SCALAR_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_gt1 (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_gt2 (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_gt (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_STRING:
    if (string_Compare (e_data (d1), e_data (d2)) > 0)
      new.u.val = 1.0;
    else
      new.u.val = 0.0;
    new.type = CONSTANT;
    break;

  case MATRIX_STRING:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_gt1 (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_gt2 (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (d1), e_name (d2), "invalid types for `>'");
  }
  remove_tmp_destroy (d1);
  remove_tmp_destroy (d2);
  return (new);
}

/* **************************************************************
 * AND operation.
 * ************************************************************** */

static Datum ent_ent_and _PROTO ((ListNode * e1, ListNode * e2));

Datum
and (d1, d2)
     Datum d1, d2;
{
  Datum new;

  d1 = convert_to_scalar (d1);
  d2 = convert_to_scalar (d2);

  new = ent_ent_and (d1.u.ent, d2.u.ent);

  return (new);
}

/* **************************************************************
 * Entity && Entity operation.
 * ************************************************************** */
static Datum
ent_ent_and (d1, d2)
     ListNode *d1, *d2;
{
  Datum new;

  switch (STYPE (e_type (d1), e_type (d2)))
  {
  case SCALAR_SCALAR:
    new.u.val = scalar_and (e_data (d1), e_data (d2));
    new.type = CONSTANT;
    break;

  case SCALAR_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_and (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_and (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_and (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_STRING:
    {
      String *str_tmp = string_Create (0);
      if (string_Compare (e_data (d1), str_tmp) != 0
	  && string_Compare (e_data (d2), str_tmp) != 0)
      {
	new.u.val = 1.0;
      }
      else
      {
	new.u.val = 0.0;
      }
      new.type = CONSTANT;
      string_Destroy (str_tmp);
      break;
    }

  case MATRIX_STRING:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_and (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_and (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (d1), e_name (d2), "invalid types for `&&'");
  }
  remove_tmp_destroy (d1);
  remove_tmp_destroy (d2);
  return (new);
}

/* **************************************************************
 * OR operation.
 * ************************************************************** */

static Datum ent_ent_or _PROTO ((ListNode * e1, ListNode * e2));

Datum
or (d1, d2)
     Datum d1, d2;
{
  Datum new;

  d1 = convert_to_scalar (d1);
  d2 = convert_to_scalar (d2);

  new = ent_ent_or (d1.u.ent, d2.u.ent);

  return (new);
}

/* **************************************************************
 * Entity || Entity operation.
 * ************************************************************** */
static Datum
ent_ent_or (d1, d2)
     ListNode *d1, *d2;
{
  Datum new;

  switch (STYPE (e_type (d1), e_type (d2)))
  {
  case SCALAR_SCALAR:
    new.u.val = scalar_or (e_data (d1), e_data (d2));
    new.type = CONSTANT;
    break;

  case SCALAR_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_or (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:
    new.u.ent = install_tmp (MATRIX,
			     matrix_scalar_or (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_or (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_STRING:
    {
      String *str_tmp = string_Create (0);

      if (string_Compare (e_data (d1), str_tmp) != 0
	  || string_Compare (e_data (d2), str_tmp) != 0)
      {
	new.u.val = 1.0;
      }
      else
      {
	new.u.val = 0.0;
      }
      new.type = CONSTANT;
      string_Destroy (str_tmp);
      break;
    }

  case MATRIX_STRING:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_or (e_data (d1), e_data (d2)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING_MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_string_or (e_data (d2), e_data (d1)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (d1), e_name (d2), "invalid types for `||'");
  }
  remove_tmp_destroy (d1);
  remove_tmp_destroy (d2);
  return (new);
}

/* **************************************************************
 * NOT operation.
 * ************************************************************** */

static Datum ent_not _PROTO ((ListNode * e));

Datum
not (d)
     Datum d;
{
  Datum new;

  switch (d.type)
  {
  case CONSTANT:		/* !constant */
    new.u.val = (double) (d.u.val == 0.0);
    new.type = CONSTANT;
    break;
  case iCONSTANT:		/* !i_constant */
    new.u.val = (double) (d.u.val == 0.0);
    new.type = CONSTANT;
  case ENTITY:			/* !entity */
    new = ent_not (d.u.ent);
    break;
  }
  return (new);
}

/* **************************************************************
 * ! Entity operation.
 * ************************************************************** */
static Datum
ent_not (d)
     ListNode *d;
{
  Datum new;

  switch (e_type (d))
  {
  case SCALAR:
    new.u.val = scalar_not (e_data (d));
    new.type = CONSTANT;
    break;

  case MATRIX:
    new.u.ent = install_tmp (MATRIX,
			     matrix_not (e_data (d)),
			     matrix_Destroy);
    new.type = ENTITY;
    break;

  case STRING:
    {
      String *str_tmp = string_Create (0);

      if (string_Compare (e_data (d), str_tmp) == 0)
      {
	new.u.val = 1.0;
      }
      else
      {
	new.u.val = 0.0;
      }
      new.type = CONSTANT;
      string_Destroy (str_tmp);
      break;
    }

  default:
    error_1 (e_name (d), "invalid types for `!'");
  }
  remove_tmp_destroy (d);
  return (new);
}
