/*
 * random.c
 * RLaB interface to RANLIB functions
 */

/*  This file is a part of RLaB ("Our"-LaB)
    Copyright (C) 1992  Ian R. Searle

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    See the file ./COPYING
 ***********************************************************************/

#include "rlab.h"
#include "bltin.h"
#include "util.h"
#include "scalar.h"
#include "matop1.h"
#include "r_string.h"

#include "random.h"

static F_INT iseed1 = 1;
static F_INT iseed2 = 1;
static int once = 1;

#define DRANF     1
#define DBETA     2
#define DCHI      3
#define DEXP      4
#define DF        5
#define DGAMMA    6
#define DMULN     7		/* not used yet */
#define DNONCCHI  8
#define DNONCF    9
#define DNORM    10
#define DUNIF    11
#define DBIN     12
#define DPOIS    13

static int dist_type;
static F_REAL dp1, dp2, dp3;

/*
 * We have our own cover package for rand() functionality.
 * This allows us to generate random numbers based upon
 * stored distribution type and parameters.
 */

double
rrand ()
{
  F_INT ip1;
  double d = 0.0;

  switch (dist_type)
  {
  case DRANF:
    d = (F_DOUBLE) RANF ();
    break;
  case DBETA:
    d = (F_DOUBLE) GENBET (&dp1, &dp2);
    break;
  case DCHI:
    d = (F_DOUBLE) GENCHI (&dp1);
    break;
  case DEXP:
    d = (F_DOUBLE) GENEXP (&dp1);
    break;
  case DF:
    d = (F_DOUBLE) GENF (&dp1, &dp2);
    break;
  case DGAMMA:
    d = (F_DOUBLE) GENGAM (&dp1, &dp2);
    break;
  case DNONCCHI:
    d = (F_DOUBLE) GENCHI (&dp1);
    break;
  case DNONCF:
    d = (F_DOUBLE) GENF (&dp1, &dp2, &dp3);
    break;
  case DNORM:
    d = (F_DOUBLE) GENNOR (&dp1, &dp2);
    break;
  case DUNIF:
    d = (F_DOUBLE) GENUNF (&dp1, &dp2);
    break;
  case DBIN:
    ip1 = (F_INT) dp1;
    d = (F_DOUBLE) IGNBIN (&ip1, &dp2);
    break;
  case DPOIS:
    d = (F_DOUBLE) IGNPOI (&dp1);
    break;
  }
  return (d);
}

void
Rand (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double d, d1;
  Datum arg, arg1, arg2, arg3;
  Matrix *m, *mtmp;

  /*
   * Initialize random number generators.
   * Do this only once. User can reset via srand().
   */

  if (once)
  {
    SETALL (&iseed1, &iseed2);
    dist_type = DRANF;
    dp1 = dp2 = dp3 = 0.0;
    once = 0;
  }

  /* get args from list */

  if (n_args == 0)
  {
    /* return a randomly generated scalar */
    d = rrand ();
    *return_ptr = (VPTR) scalar_Create (d);
    return;
  }
  else if (n_args == 1)
  {
    arg = get_bltin_arg ("rand", d_arg, 1, 0);

    if (arg.type != ENTITY)
      error_1 ("must supply a MATRIX, or STRING to rand()", 0);

    if (e_type (arg.u.ent) == MATRIX)
    {
      mtmp = (Matrix *) e_data (arg.u.ent);

      if (MTYPE (mtmp) == COMPLEX)
	error_1 (e_name (arg.u.ent), "COMPLEX MATRIX not allowed");
      if (MNR (mtmp) != 1 || MNC (mtmp) != 2)
	error_1 (e_name (arg.u.ent),
		 "must supply rand with 1-by-2 MATRIX");

      m = matrix_Rand ((int) MAT (mtmp, 1, 1), (int) MAT (mtmp, 1, 2));
      *return_ptr = (VPTR) m;
    }
    else if (e_type (arg.u.ent) == STRING)
    {
      if (!strcmp ("default", string_GetString (e_data (arg.u.ent))))
      {
	dist_type = DRANF;
	*return_ptr = (VPTR) scalar_Create (1.0);
      }
      else
	error_1 (e_name (arg.u.ent), "invalid STRING for rand()");
    }
    else
    {
      error_1 (e_name (arg.u.ent), "invalid type for rand()");
    }
  }
  else if (n_args >= 2)
  {
    /*
       * The 1st arg can either be a SCALAR, or a STRING. If the
       * 1st arg is a SCALAR, the 2nd arg must be a SCALAR.
       * If the 1st arg is a STRING. The string describes the
       * distribution. The remaining args are the parameters
       * required to define the distribution.
       */

    arg = get_bltin_arg ("rand", d_arg, 1, 0);
    if (arg.type == ENTITY)
    {
      if (e_type (arg.u.ent) == STRING)
      {
	if (!strcmp ("beta", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 3)
	    error_1 ("rand() requires 3 args when arg 1 is \"beta\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  arg2 = get_bltin_arg ("rand", d_arg, 3, NUM);
	  dist_type = DBETA;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  dp2 = (F_REAL) get_num_scalar_val (arg2);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else if (!strcmp ("chi", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 2)
	    error_1 ("rand() requires 2 args when arg 1 is \"chi\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  dist_type = DCHI;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else if (!strcmp ("exp", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 2)
	    error_1 ("rand() requires 2 args when arg 1 is \"exp\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  dist_type = DEXP;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else if (!strcmp ("f", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 3)
	    error_1 ("rand() requires 3 args when arg 1 is \"f\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  arg2 = get_bltin_arg ("rand", d_arg, 3, NUM);
	  dist_type = DF;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  dp2 = (F_REAL) get_num_scalar_val (arg2);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else if (!strcmp ("gamma", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 3)
	    error_1 ("rand() requires 3 args when arg 1 is \"gamma\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  arg2 = get_bltin_arg ("rand", d_arg, 3, NUM);
	  dist_type = DGAMMA;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  dp2 = (F_REAL) get_num_scalar_val (arg2);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else if (!strcmp ("nchi", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 3)
	    error_1 ("rand() requires 3 args when 1st arg is \"nchi\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  arg2 = get_bltin_arg ("rand", d_arg, 3, NUM);
	  dist_type = DNONCCHI;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  dp2 = (F_REAL) get_num_scalar_val (arg2);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else if (!strcmp ("nf", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 4)
	    error_1 ("rand() requires 4 args when 1st arg is \"nf\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  arg2 = get_bltin_arg ("rand", d_arg, 3, NUM);
	  arg3 = get_bltin_arg ("rand", d_arg, 4, NUM);
	  dist_type = DNONCF;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  dp2 = (F_REAL) get_num_scalar_val (arg2);
	  dp3 = (F_REAL) get_num_scalar_val (arg3);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else if (!strcmp ("normal", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 3)
	    error_1 ("rand() requires 3 args when 1st arg is \"normal\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  arg2 = get_bltin_arg ("rand", d_arg, 3, NUM);
	  dist_type = DNORM;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  dp2 = (F_REAL) get_num_scalar_val (arg2);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else if (!strcmp ("uniform", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 3)
	    error_1 ("rand() requires 3 args when arg 1 is \"uniform\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  arg2 = get_bltin_arg ("rand", d_arg, 3, NUM);
	  dist_type = DUNIF;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  dp2 = (F_REAL) get_num_scalar_val (arg2);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else if (!strcmp ("bin", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 3)
	    error_1 ("rand() requires 3 args when 1st arg is \"bin\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  arg2 = get_bltin_arg ("rand", d_arg, 3, NUM);
	  dist_type = DBIN;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  dp2 = (F_REAL) get_num_scalar_val (arg2);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else if (!strcmp ("poisson", string_GetString (e_data (arg.u.ent))))
	{
	  if (n_args != 2)
	    error_1 ("rand() requires 2 args when arg 1 is \"poissom\"",
		     (char *) 0);
	  arg1 = get_bltin_arg ("rand", d_arg, 2, NUM);
	  dist_type = DPOIS;
	  dp1 = (F_REAL) get_num_scalar_val (arg1);
	  *return_ptr = (VPTR) scalar_Create (1.0);
	  return;
	}
	else
	{
	  error_1 ("invalid distribution string, see `help rand'", 0);
	}
      }
    }

    arg2 = get_bltin_arg ("rand", d_arg, 2, 0);
    d = get_num_scalar_val (arg);
    d1 = get_num_scalar_val (arg2);
    m = matrix_Rand ((int) d, (int) d1);
    *return_ptr = (VPTR) m;
    return;
  }
}


#include <sys/types.h>

#ifdef __STDC__
#include <time.h>
#endif /* __STDC__ */

#ifdef HAVE_TIME_H
#include <time.h>
#endif

void
SRand (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  long int init;
  Datum arg;

  /* Check n_args */
  if (!(n_args == 0 || n_args == 1))
    error_1 ("srand() takes 0 or 1 arg", (char *) 0);

  if (once)
  {
    SETALL (&iseed1, &iseed2);
    dist_type = DRANF;
    dp1 = dp2 = dp3 = 0.0;
    once = 0;
  }

  if (n_args == 0)
  {
    /* Reset seed to defaults */
    init = -1;
    INITGN (&init);
  }
  else if (n_args == 1)
  {
    arg = get_bltin_arg ("srand", d_arg, 1, 0);
    switch (arg.type)
    {
    case CONSTANT:
      iseed1 = (long int) arg.u.val;
      if (iseed1 == 0)
	error_1 ("cannot call srand() with argument = 0", (char *) 0);
      SETALL (&iseed1, &iseed2);
      break;
    case ENTITY:
      switch (e_type (arg.u.ent))
      {
      case SCALAR:
	iseed1 = (long int) SVALr (e_data (arg.u.ent));
	if (iseed1 == 0)
	  error_1 ("cannot call srand() with argument = 0", (char *) 0);
	SETALL (&iseed1, &iseed2);
	break;
      case STRING:
	if (!strcmp ("clock", string_GetString (e_data (arg.u.ent))))
	{
#ifdef THINK_C
	  /* mac uses 1904 as the stating year for counting, 
	     it will give negative value here, so... */
	  iseed1 = (long int) (time (0)) - 2081376000L;
#else	  	  
	  iseed1 = (long int) time (0);
#endif
	  SETALL (&iseed1, &iseed2);
	}
	else
	  error_1 ("\"clock\" is the only valid string-arg to srand()",
		   (char *) 0);
	break;
      default:
	error_1 (e_name (arg.u.ent), "invalid type for srand()");
      }
      break;
    default:
      error_1 ("invalid type for srand()", (char *) 0);
    }
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
}
