/* math_3.c 
 * Miscellaneous math functions for RLaB */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mem.h"
#include "bltin.h"
#include "code.h"
#include "symbol.h"
#include "util.h"
#include "scop1.h"
#include "matop1.h"
#include "matop2.h"
#include "listnode.h"
#include "btree.h"
#include "r_string.h"
#include "fi_1.h"
#include "fi_2.h"
#include "mathl.h"

#include <math.h>

#define TARG_DESTROY(arg, targ)   if (targ.u.ent != arg.u.ent) \
                                    remove_tmp_destroy (targ.u.ent);

/* **************************************************************
 * Return the REAL part of a complex entity.
 * ************************************************************** */
void
Real (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ;
  Matrix *m;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to real()", (char *) 0);

  /* get arg from list */
  arg = get_bltin_arg ("real", d_arg, 1, NUM);

  targ = convert_to_scalar (arg);

  switch (e_type (targ.u.ent))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Create (SVALr (e_data (targ.u.ent)));
    break;
  case MATRIX:
    m = matrix_Real (e_data (targ.u.ent));
    *return_ptr = (VPTR) m;
    break;
  default:
    error_1 (e_name (targ.u.ent), "invalid type for real()");
    break;
  }
  TARG_DESTROY (arg, targ);
  return;
}

void
Imag (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ;
  Matrix *m;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to imag()", (char *) 0);

  /* get arg from list */
  arg = get_bltin_arg ("imag", d_arg, 1, NUM);

  targ = convert_to_scalar (arg);

  switch (e_type (targ.u.ent))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Create (SVALi (e_data (targ.u.ent)));
    break;
  case MATRIX:
    m = matrix_Imag (e_data (targ.u.ent));
    *return_ptr = (VPTR) m;
    break;
  default:
    error_1 (e_name (targ.u.ent), "invalid type for imag()");
    break;
  }
  TARG_DESTROY (arg, targ);
  return;
}

void
Conj (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ;
  Matrix *m;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to conj()", (char *) 0);

  /* get arg from list */
  arg = get_bltin_arg ("conj", d_arg, 1, NUM);

  targ = convert_to_scalar (arg);

  switch (e_type (targ.u.ent))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_CreateC (SVALr (e_data (targ.u.ent)),
					 -SVALi (e_data (targ.u.ent)));
    break;
  case MATRIX:
    m = matrix_Conj (e_data (targ.u.ent));
    *return_ptr = (VPTR) m;
    break;
  default:
    error_1 (e_name (targ.u.ent), "invalid type for conj()");
    break;
  }
  TARG_DESTROY (arg, targ);
  return;
}

#include "fftp.h"

void
FFT (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  F_INT n;
  int i, N, vtype;
  Datum arg, arg2, targ;
  ListNode *tmp1, *tmp2;
  Matrix *m, *mfft, *wsave;

  /* Check n_args */
  if (n_args != 1 && n_args != 2)
    error_1 ("Wrong number of args to fft()", (char *) 0);

  /* get arg from list */
  arg = get_bltin_arg ("fft", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  if (n_args == 2)
  {
    arg2 = get_bltin_arg ("fft", d_arg, 2, NUM);
    N = get_num_scalar_val (arg2);
  }
  else
    N = 0;
  
  m = (Matrix *) e_data (targ.u.ent);
  if (MNR (m) == 1 || MNC (m) == 1)
  {
    /*
     * Vector fft
     */
    
    if (MNR (m) > MNC (m))     /* Set n, and figure out vector type */
    {
      n = (F_INT) MNR (m);
      vtype = 1;
    }
    else
    {
      n = (F_INT) MNC (m);
      vtype = 2;
    }
    
    tmp1 = install_tmp (MATRIX, mfft = matrix_copy_complex (m),
			matrix_Destroy);
    
    if (N != 0)
    {
      /* Set length of vector to N */
      if (N > (int) n)
      {
	if (vtype == 1)                    /* column vector */
	  matrix_Extend (mfft, N, 1);
	else                               /* row vector */
	  matrix_Extend (mfft, 1, N);
	n = (F_INT) N;
      }
      else if (N < (int) n)
      {
	if (vtype == 1)                    /* column vector */
	  matrix_Truncate (mfft, N, 1);
	else                               /* row vector */
	  matrix_Truncate (mfft, 1, N);
	n = (F_INT) N;
      }
    }
    
    tmp2 = install_tmp (MATRIX, wsave = matrix_Create (1, 4 * n + 15),
			matrix_Destroy);
    
    signal (SIGINT, intcatch);
    CFFTI (&n, MDPTRr (wsave));
    CFFTF (&n, (double *) MDPTRc (mfft), MDPTRr (wsave));
    signal (SIGINT, intcatch_wait);
  }
  else
  {
    /* MAtrix fft, one column at a time */
    n = (F_INT) MNR (m);
    
    tmp1 = install_tmp (MATRIX, mfft = matrix_copy_complex (m),
			matrix_Destroy);
    
    if (N != 0)
    {
      /* Set row dimension of [mfft] to N */
      if (N > (int) n)
      {
	matrix_Extend (mfft, N, MNC (mfft));
	n = (F_INT) N;
      }
      else if (N < (int) n)
      {
	matrix_Truncate (mfft, N, MNC (mfft));
	n = (F_INT) N;
      }
    }
    
    tmp2 = install_tmp (MATRIX, wsave = matrix_Create (1, 4 * n + 15),
			matrix_Destroy);

    for (i = 0; i < MNC (m); i++)
    {
      signal (SIGINT, intcatch);
      CFFTI (&n, MDPTRr (wsave));
      CFFTF (&n, (double *) (mfft->val.mc + (i * n)), MDPTRr (wsave));
      signal (SIGINT, intcatch_wait);
    }
  }
  remove_tmp (tmp1);
  remove_tmp_destroy (tmp2);
  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) mfft;
}

void
IFFT (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  F_INT n;
  int i, N, vtype;
  Datum arg, arg2, targ;
  ListNode *tmp1, *tmp2;
  Matrix *m, *mfft, *wsave;

  /* Check n_args */
  if (n_args != 1 && n_args != 2)
    error_1 ("Wrong number of args to ifft()", (char *) 0);

  /* get arg from list */
  arg = get_bltin_arg ("ifft", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);

  if (n_args == 2)
  {
    arg2 = get_bltin_arg ("ifft", d_arg, 2, NUM);
    N = get_num_scalar_val (arg2);
  }
  else
    N = 0;

  m = (Matrix *) e_data (targ.u.ent);
  if (MNR (m) == 1 || MNC (m) == 1)
  {
    /*
     * Vector fft
     */
    
    if (MNR (m) > MNC (m))     /* Set n, and figure out vector type */
    {
      n = (F_INT) MNR (m);
      vtype = 1;
    }
    else
    {
      n = (F_INT) MNC (m);
      vtype = 2;
    }
    
    tmp1 = install_tmp (MATRIX, mfft = matrix_copy_complex (m),
			matrix_Destroy);
    
    if (N != 0)
    {
      /* Set length of vector to N */
      if (N > (int) n)
      {
	if (vtype == 1)                    /* column vector */
	  matrix_Extend (mfft, N, 1);
	else                               /* row vector */
	  matrix_Extend (mfft, 1, N);
	n = (F_INT) N;
      }
      else if (N < (int) n)
      {
	if (vtype == 1)                    /* column vector */
	  matrix_Truncate (mfft, N, 1);
	else                               /* row vector */
	  matrix_Truncate (mfft, 1, N);
	n = (F_INT) N;
      }
    }
    
    tmp2 = install_tmp (MATRIX, wsave = matrix_Create (1, 4 * n + 15),
			matrix_Destroy);
    
    signal (SIGINT, intcatch);
    CFFTI (&n, MDPTRr (wsave));
    CFFTB (&n, (double *) MDPTRc (mfft), MDPTRr (wsave));
    signal (SIGINT, intcatch_wait);
  }
  else
  {
    /* Matrix ifft, one column at a time */
    n = MNR (m);

    tmp1 = install_tmp (MATRIX, mfft = matrix_copy_complex (m),
			matrix_Destroy);

    if (N != 0)
    {
      /* Set row dimension of [mfft] to N */
      if (N > (int) n)
      {
	matrix_Extend (mfft, N, MNC (mfft));
	n = (F_INT) N;
      }
      else if (N < (int) n)
      {
	matrix_Truncate (mfft, N, MNC (mfft));
	n = (F_INT) N;
      }
    }
    
    tmp2 = install_tmp (MATRIX, wsave = matrix_Create (1, 4 * n + 15),
			matrix_Destroy);

    for (i = 0; i < MNC (m); i++)
    {
      signal (SIGINT, intcatch);
      CFFTI (&n, MDPTRr (wsave));
      CFFTB (&n, (double *) (mfft->val.mc + (i * n)), MDPTRr (wsave));
      signal (SIGINT, intcatch_wait);
    }
  }

  /*
   * Scale the result by 1/N
   */
  for (i = 0; i < MNR (mfft) * MNC (mfft); i++)
  {
    MATcvr (mfft, i) = MATcvr (mfft, i) / ((double) n);
    MATcvi (mfft, i) = MATcvi (mfft, i) / ((double) n);
  }
  
  remove_tmp (tmp1);
  remove_tmp_destroy (tmp2);
  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) mfft;
}

/*
 * Sum the elements of a matrix
 */

void
Sum (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to sum()", (char *) 0);

  /* get arg from list */
  arg = get_bltin_arg ("sum", d_arg, 1, NUM);

  switch (arg.type)
  {
  case CONSTANT:
    *return_ptr = (VPTR) scalar_Create (arg.u.val);
    return;
  case iCONSTANT:
    *return_ptr = (VPTR) scalar_CreateC (0.0, arg.u.val);
    return;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Copy (e_data (arg.u.ent));
      return;
    case MATRIX:
      *return_ptr = (VPTR) matrix_Sum (e_data (arg.u.ent));
      return;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for sum()");
      break;
    }
  default:
    error_1 ("invalid type for sum", (char *) 0);
  }
}

void
Prod (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to prod()", (char *) 0);

  /* get arg from list */
  arg = get_bltin_arg ("prod", d_arg, 1, NUM);

  switch (arg.type)
  {
  case CONSTANT:
    *return_ptr = (VPTR) scalar_Create (arg.u.val);
    return;
  case iCONSTANT:
    *return_ptr = (VPTR) scalar_CreateC (0.0, arg.u.val);
    return;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      *return_ptr = (VPTR) scalar_Copy (e_data (arg.u.ent));
      return;
    case MATRIX:
      *return_ptr = (VPTR) matrix_Prod (e_data (arg.u.ent));
      return;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for prod()");
      break;
    }
  default:
    error_1 ("invalid type for prod", (char *) 0);
  }
}

/* **************************************************************
 * Factor, and Backsub functions.
 * ************************************************************** */

void
Factor (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double rcond;
  Datum arg, targ;
  Btree *rlist;
  Matrix *lu, *pvt;
  Scalar *stmp;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to factor()", 0);

  /* get arg from list */
  arg = get_bltin_arg ("factor", d_arg, 1, NUM);
  targ = convert_to_matrix (arg);
  matrix_Factor (e_data (targ.u.ent), &lu, &pvt, &rcond);

  rlist = btree_Create ();
  install (rlist, cpstr ("lu"), MATRIX, lu);
  install (rlist, cpstr ("pvt"), MATRIX, pvt);
  install (rlist, cpstr ("rcond"), SCALAR, stmp = scalar_Create (rcond));
  matrix_SetName (lu, cpstr ("lu"));
  matrix_SetName (pvt, cpstr ("pvt"));
  scalar_SetName (stmp, cpstr ("rcond"));

  *return_ptr = (VPTR) rlist;
  TARG_DESTROY (arg, targ);
  return;
}

void
Backsub (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, arg2, targ2;
  Matrix *sol;

  /* Check n_args */
  if (n_args != 2)
    error_1 ("Wrong number of args to backsub()", 0);

  /* get arg from list */
  arg1 = get_bltin_arg ("backsub", d_arg, 1, BTREE);
  arg2 = get_bltin_arg ("backsub", d_arg, 2, NUM);
  targ2 = convert_to_matrix (arg2);
  sol = matrix_Backsub (e_data (arg1.u.ent), e_data (targ2.u.ent));

  *return_ptr = (VPTR) sol;
  TARG_DESTROY (arg2, targ2);
  return;
}

void
Schur (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Btree *rlist;
  Datum arg1, targ1;
  Matrix *t, *z;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to schur()", 0);

  /* get arg from list */
  arg1 = get_bltin_arg ("schur", d_arg, 1, NUM);
  targ1 = convert_to_matrix (arg1);
  matrix_Schur (e_data (targ1.u.ent), &t, &z);

  rlist = btree_Create ();
  install (rlist, cpstr ("t"), MATRIX, t);
  install (rlist, cpstr ("z"), MATRIX, z);
  matrix_SetName (t, cpstr ("t"));
  matrix_SetName (z, cpstr ("z"));

  *return_ptr = (VPTR) rlist;
  TARG_DESTROY (arg1, targ1);
  return;
}

void
Sylv (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg1, targ1, arg2, targ2, arg3, targ3;
  Matrix *a, *b, *c, *x;

  /* Check n_args */
  if (!(n_args == 2 || n_args == 3))
    error_1 ("sylv: 2 or 3 arguments are required", 0);

  /* get arg from list */
  if (n_args == 2)
  {
    /* Lyapunov */
    arg1 = get_bltin_arg ("sylv", d_arg, 1, NUM);
    targ1 = convert_to_matrix (arg1);
    arg2 = get_bltin_arg ("sylv", d_arg, 2, NUM);
    targ2 = convert_to_matrix (arg2);

    a = (Matrix *) e_data (targ1.u.ent);
    c = (Matrix *) e_data (targ2.u.ent);

    matrix_Sylv (a, (Matrix *) 0, c, &x);

    *return_ptr = (VPTR) x;

    TARG_DESTROY (arg1, targ1);
    TARG_DESTROY (arg2, targ2);
    return;
  }
  else if (n_args == 3)
  {
    /* Sylvester */
    arg1 = get_bltin_arg ("sylv", d_arg, 1, NUM);
    targ1 = convert_to_matrix (arg1);
    arg2 = get_bltin_arg ("sylv", d_arg, 2, NUM);
    targ2 = convert_to_matrix (arg2);
    arg3 = get_bltin_arg ("sylv", d_arg, 3, NUM);
    targ3 = convert_to_matrix (arg3);

    a = (Matrix *) e_data (targ1.u.ent);
    b = (Matrix *) e_data (targ2.u.ent);
    c = (Matrix *) e_data (targ3.u.ent);

    matrix_Sylv (a, b, c, &x);

    *return_ptr = (VPTR) x;

    TARG_DESTROY (arg1, targ1);
    TARG_DESTROY (arg2, targ2);
    TARG_DESTROY (arg3, targ3);
    return;
  }
}

/* **************************************************************
 * Functions for determining symmetry
 * ************************************************************** */

int matrix_is_symm _PROTO ((Matrix *m));

void
Issymm (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int retval;
  Datum arg, targ;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("Wrong number of args to issym()", 0);

  /* get arg from list */
  arg = get_bltin_arg ("issym", d_arg, 1, 0);
  targ = convert_all_to_matrix (arg);
  retval = matrix_is_symm (e_data (targ.u.ent));

  *return_ptr = (VPTR) scalar_Create ((double) retval);

  TARG_DESTROY (arg, targ);
  return;
}

/*
 * Return TRUE (1) if the input matrix is symmetric (Hermitian),
 * FALSE (0) otherwise. Check the matrix colum-wise.
 * Return FALSE (0) at the 1st failed test.
 */

int
matrix_is_symm (m)
     Matrix *m;
{
  int i, j;

  if (MNR (m) != MNC (m))
    return (0);

  if (MTYPE (m) == REAL)
  {
    for (j = 0; j < MNC (m); j++)
    {
      for (i = j+1; i < MNR (m); i++)
      {
	if (MAT0 (m, i, j) != MAT0 (m, j, i))
	  return (0);
      }
    }
  }
  else if (MTYPE (m) == COMPLEX)
  {
    for (j = 0; j < MNC (m); j++)
    {
      for (i = j+1; i < MNR (m); i++)
      {
	if (MATr0 (m, i, j) != MATr0 (m, j, i)) return (0);
	if (MATi0 (m, i, j) != - MATi0 (m, j, i)) return (0);
      }
    }
  }
  else if (MTYPE (m) == STRING)
  {
    for (j = 0; j < MNC (m); j++)
    {
      for (i = j+1; i < MNR (m); i++)
      {
	if (strcmp (MATs0 (m, i, j), MATs0 (m, j, i)))
	  return (0);
      }
    }
  }

  return (1);
}

/* **************************************************************
 * Symmetric Eigenvalue solvers
 * ************************************************************** */
void
EigS (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int sol = 0;
  Datum arg1, arg2, targ1, targ2;
  Btree *rlist;
  Matrix *vec, *val;

  /* Check n_args */
  if ((n_args != 1) && (n_args != 2))
    error_1 ("eigs: Wrong number of arguments", 0);

  if (n_args == 1)
  {
    arg1 = get_bltin_arg ("eigs", d_arg, 1, NUM);
    targ1 = convert_to_matrix (arg1);
    sol = 1;        /* Standard symmetric problem */
  }
  else if (n_args == 2)
  {
    arg1 = get_bltin_arg ("eigs", d_arg, 1, NUM);
    targ1 = convert_to_matrix (arg1);
    arg2 = get_bltin_arg ("eigs", d_arg, 2, NUM);
    targ2 = convert_to_matrix (arg2);
    sol = 2;       /* Generalized symmetric problem */
  }
  else
    error_1 ("eig: invalid number of arguments", 0);

  /* Create list for returning results */
  rlist = btree_Create ();

  if (sol == 1)
  {
    /* Standard Eigenvalue Problem, Symmetric */
    matrix_Eig_SEP (e_data (targ1.u.ent), &val, &vec);
    install (rlist, cpstr ("val"), MATRIX, val);
    install (rlist, cpstr ("vec"), MATRIX, vec);
    matrix_SetName (val, cpstr ("val"));
    matrix_SetName (vec, cpstr ("vec"));
    TARG_DESTROY (arg1, targ1);
  }
  else if (sol == 2)
  {
    /* Generalized problem, symmetric */
    matrix_Eig_GSEP (e_data (targ1.u.ent), e_data (targ2.u.ent),
		       &val, &vec);
    install (rlist, cpstr ("val"), MATRIX, val);
    install (rlist, cpstr ("vec"), MATRIX, vec);
    matrix_SetName (val, cpstr ("val"));
    matrix_SetName (vec, cpstr ("vec"));
    TARG_DESTROY (arg1, targ1);
    TARG_DESTROY (arg2, targ2);
  }
  *return_ptr = (VPTR) rlist;
}

/* **************************************************************
 * Non-Symmetric Eigenvalue solvers
 * ************************************************************** */
void
EigN (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int sol = 0;
  Datum arg1, arg2, targ1, targ2;
  Btree *rlist;
  Matrix *val, *lvec, *rvec, *alpha, *beta;

  /* Check n_args */
  if ((n_args != 1) && (n_args != 2))
    error_1 ("eign: Wrong number of arguments", 0);

  if (n_args == 1)
  {
    arg1 = get_bltin_arg ("eign", d_arg, 1, NUM);
    targ1 = convert_to_matrix (arg1);
    sol = 1;        /* Standard non-symmetric problem */
  }
  else if (n_args == 2)
  {
    arg1 = get_bltin_arg ("eign", d_arg, 1, NUM);
    targ1 = convert_to_matrix (arg1);
    arg2 = get_bltin_arg ("eign", d_arg, 2, NUM);
    targ2 = convert_to_matrix (arg2);
    sol = 2;       /* Generalized non-symmetric problem */
  }
  else
    error_1 ("eign: invalid number of arguments", 0);

  /* Create list for returning results */
  rlist = btree_Create ();

  if (sol == 1)
  {
    /* Standard Eigenvalue Problem, Non-Symmetric */
    matrix_Eig_NEP (e_data (targ1.u.ent), &val, &rvec, &lvec, 1);

    TARG_DESTROY (arg1, targ1);
    install (rlist, cpstr ("val"), MATRIX, val);
    install (rlist, cpstr ("rvec"), MATRIX, rvec);
    install (rlist, cpstr ("lvec"), MATRIX, lvec);
    matrix_SetName (val, cpstr ("val"));
    matrix_SetName (rvec, cpstr ("rvec"));
    matrix_SetName (lvec, cpstr ("lvec"));    
  }
  else if (sol == 2)
  {
    /* Generalized problem, Non-symmetric */
    matrix_Eig_GNEPa (e_data (targ1.u.ent), e_data (targ2.u.ent),
		      &alpha, &beta, &rvec, &lvec);

    install (rlist, cpstr ("alpha"), MATRIX, alpha);
    install (rlist, cpstr ("beta"), MATRIX, beta);
    install (rlist, cpstr ("rvec"), MATRIX, rvec);
    install (rlist, cpstr ("lvec"), MATRIX, lvec);
    matrix_SetName (alpha, cpstr ("alpha"));
    matrix_SetName (beta, cpstr ("beta"));
    matrix_SetName (rvec, cpstr ("rvec"));
    matrix_SetName (lvec, cpstr ("lvec"));
    TARG_DESTROY (arg1, targ1);
    TARG_DESTROY (arg2, targ2);
  }

  *return_ptr = (VPTR) rlist;
}

void
SchOrd (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Btree *rlist;
  Datum arg1, arg2, targ2, arg3, targ3;
  ListNode *tmp;
  Matrix *t, *z, *T, *Z;
  int ifst, ilst;

  /* Check n_args */
  if (n_args != 3)
    error_1 ("Wrong number of args to schord()", 0);

  /* get arg from list */
  arg1 = get_bltin_arg ("schord", d_arg, 1, BTREE);
  arg2 = get_bltin_arg ("schord", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("schord", d_arg, 3, NUM);

  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);

  if (!(tmp = btree_FindNode (e_data (arg1.u.ent), "t")))
    error_1 ("List must contain \"t\" element for schord()", 0);
  t = (Matrix *) e_data (tmp);

  if (!(tmp = btree_FindNode (e_data (arg1.u.ent), "z")))
    error_1 ("List must contain \"z\" element for schord()", 0);
  z = (Matrix *) e_data (tmp);

  ifst = (int) MAT (e_data (targ2.u.ent), 1, 1);
  ilst = (int) MAT (e_data (targ3.u.ent), 1, 1);

  /* Check correctness of ifst, and ilst arguments */
  if (ifst < 1 || ilst < 1)
    error_1 ("schord: IFST, and ILST must be > 0", 0);
  if (ifst > MNR (z) || ilst > MNR (z))
    error_1 ("schord: IFST and ILST must be <= z.n", 0);
  
  matrix_Schur_Reorder (t, z, ifst, ilst, &T, &Z);

  rlist = btree_Create ();
  install (rlist, cpstr ("t"), MATRIX, T);
  install (rlist, cpstr ("z"), MATRIX, Z);
  matrix_SetName (t, cpstr ("t"));
  matrix_SetName (z, cpstr ("z"));

  *return_ptr = (VPTR) rlist;
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  return;
}

/* **************************************************************
 * Filter: 
 * ************************************************************** */

static void filter _PROTO ((double *b, double *a, double *x, 
			    double *y, double *v, double *vi,
			    int *ntotal, int *M));

void
Filter (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int M, N, NN, i, ntotal;
  double *v, *vi;
  Btree *rlist;
  Datum arg1, arg2, arg3, arg4;
  Datum targ1, targ2, targ3, targ4;
  ListNode *tmp1, *tmp2;
  Matrix *a, *b, *x, *y, *zi, *zf;
  Matrix *aa, *bb;

  /* Check n_args */
  if (n_args < 3)
    error_1 ("filter: requires at least 3 arguments", 0);

  /*
   * Get args from list and error-check.
   */

  arg1 = get_bltin_arg ("filter", d_arg, 1, NUM);
  arg2 = get_bltin_arg ("filter", d_arg, 2, NUM);
  arg3 = get_bltin_arg ("filter", d_arg, 3, NUM);

  targ1 = convert_to_matrix (arg1);
  targ2 = convert_to_matrix (arg2);
  targ3 = convert_to_matrix (arg3);

  a = (Matrix *) e_data (targ2.u.ent);
  b = (Matrix *) e_data (targ1.u.ent);
  x = (Matrix *) e_data (targ3.u.ent);

  if (MTYPE (a) != REAL)
    error_1 ("filter: A must be type: REAL", 0);
  if (MTYPE (b) != REAL)
    error_1 ("filter: B must be type: REAL", 0);
  if (MTYPE (x) != REAL)
    error_1 ("filter: X must be type: REAL", 0);

  if (MNR (x) != 1 && MNC (x) != 1)
    error_1 ("filter: X must be a row or column matrix", 0);
  if (MNR (a) != 1 && MNC (a) != 1)
    error_1 ("filter: A must be a row or column matrix", 0);
  if (MNR (b) != 1 && MNC (b) != 1)
    error_1 ("filter: B must be a row or column matrix", 0);

  ntotal = MNR (x) * MNC (x);
  M = MNR (b) * MNC (b);
  N = MNR (a) * MNC (a);
  NN = max (M, N);
  y = matrix_Create (MNR (x), MNC (x));

  /*
   * Fix up pole and zero vectors.
   * Make them the same length, this makes
   * filter's job much easier.
   */

  tmp1 = install_tmp (MATRIX, aa = matrix_Create (NN, 1),
		      matrix_Destroy);
  tmp2 = install_tmp (MATRIX, bb = matrix_Create (NN, 1),
		      matrix_Destroy);
  matrix_Zero (aa);
  matrix_Zero (bb);
  
  if (MATrv (a, 0) == 0)
    error_1 ("filter: 1st A term must be non-zero", 0);

  for (i = 0; i < M; i++)
    MATrv (bb, i) = MATrv (b, i) / MATrv (a, 0);
  for (i = 1; i < N; i++)
    MATrv (aa, i) = MATrv (a, i) / MATrv (a, 0);
  
  /*
   * Create delay vectors and load inital delays.
   * Add an extra term to vi[] to make filter's 
   * job a little easier. This extra term will
   * always be zero.
   */

  v = (double *) MALLOC (sizeof (double) * NN);
  vi = (double *) MALLOC (sizeof (double) * (NN+1));

  if (n_args == 4)
  {
    arg4 = get_bltin_arg ("filter", d_arg, 4, NUM);
    targ4 = convert_to_matrix (arg4);
    zi = (Matrix *) e_data (targ4.u.ent);
    if (MTYPE (zi) != REAL)
      error_1 ("filter: Zi must be type: REAL", 0);
    if (MNR (zi) * MNC (zi) != NN)
      error_1 ("filter: Zi must be max(size(A), size(B))", 0);
    for (i = 0; i < NN; i++)
      vi[i] = MATrv (zi, i);
    vi[NN] = 0.0;
  }
  else
  {
    for (i = 0; i < NN; i++)
      vi[i] = 0.0;
    vi[NN] = 0.0;
  }

  /*
   * Do the work...
   */

  filter (MDPTRr (bb), MDPTRr (aa), MDPTRr (x), MDPTRr (y),
	  v, vi, &ntotal, &NN);
  
  zf = matrix_Create (NN, 1);
  for (i = 0; i < NN; i++)
    MATrv (zf, i) = v[i];
  
  rlist = btree_Create ();
  install (rlist, cpstr ("y"), MATRIX, y);
  install (rlist, cpstr ("zf"), MATRIX, zf);
  matrix_SetName (y, cpstr ("y"));
  matrix_SetName (zf, cpstr ("zf"));

  *return_ptr = (VPTR) rlist;
  
  FREE (v);
  FREE (vi);

  remove_tmp_destroy (tmp1);
  remove_tmp_destroy (tmp2);

  TARG_DESTROY (arg1, targ1);
  TARG_DESTROY (arg2, targ2);
  TARG_DESTROY (arg3, targ3);
  if (n_args == 4)
    TARG_DESTROY (arg4, targ4);

  return;
}

/*
 * This is the function that acutally computes the
 * response. There is no error checking done here,
 * because Filter() does it all.
 */

static void
filter (b, a, x, y, v, vi, ntotal, N)
     double *b, *a, *x, *y, *v, *vi;
     int *ntotal, *N;
{
  int k, n;
 
  for (n = 0; n < *ntotal; n++)
  {
    v[0] = b[0]*x[n] + vi[1];
    y[n] = v[0];
    for (k = 1; k < *N; k++)
    {
      v[k] = b[k]*x[n] - a[k]*v[0] + vi[k+1];
      vi[k] = v[k];
    }
  }
}
