/* main.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mem.h"
#include "code.h"
#include "util.h"
#include "version.h"
#include "getopt.h"

#include <stdio.h>
#include <sys/types.h>

#ifdef __STDC__
#include <stdlib.h>
#else
extern char *getenv ();
#endif

#ifdef unix
static char PATH_DELIM[] ="/";
#endif

#ifdef THINK_C

static char PATH_DELIM[] = ":";

/* ColorQuickDraw Windows */
#include <myconsole16.h>
/* Preference */
char    *PreferenceName = "RLaB-Preference";
char*	getpref(char*);
#define getenv(env_name) getpref(env_name)
int     mac_rows = 25;             /* no. of rows */
char    *txFont, *txSize, *nrows;  /* console font and size */
int     txFontNum = 4, txSizeNum = 9;  

static void think_c_setup _PROTO ((void));

#endif  /* THINK_C */

#ifdef HAVE_DIRENT
#include <dirent.h>
#else
#include <sys/dir.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

/* Scanner declarations */
extern void set_rlab_input _PROTO ((int type));
extern int yyparse ();
extern FILE *yyin;
extern int lineno;
extern int flush_line;
extern char *curr_file_name;

/* print.c */
extern close_file_ds _PROTO ((char *fn));

/* rfile.c */
extern int rfile_test _PROTO ((char *name));
int rfile_dir _PROTO ((char *dirname));

/* code.c */
extern void init_frame _PROTO ((void));
extern void init_stack _PROTO ((void));

void init_misc _PROTO ((void));
void init_file_list _PROTO ((void));
void init_static_tree _PROTO ((void));
int run_program _PROTO ((char *input));
void run _PROTO ((Program * p));
void run_no_clean _PROTO ((Program * p));
void warning_1 _PROTO ((char *s, char *t));
void warning_2 _PROTO ((char *s1, char *s2, char *t));
void print_greeting _PROTO ((void));

#ifdef unix
void init_environment _PROTO ((void));
#endif

#ifdef THINK_C
void init_tc_environment _PROTO ((void));
#endif

extern void diss_assemble _PROTO ((Inst * p, int progoff));

char *progname;

/*
 * These hold the env for longjmp()s back to the prompt, 
 * and error exit
 */

#define NJMP 40			/* Depth of load() recursion */
jmp_buf jmp[NJMP];
static int ijmp;		/* Counter for tracking jmp[] */
int inc_buff _PROTO ((void));
int dec_buff _PROTO ((void));

#ifdef THINK_C
char *DEFAULT_RC0   = ".rlab";
char *DEFAULT_HELP  = "help";
char *DEFAULT_LIB   = "rlib";
char *DEFAULT_PAGER = " ";
char *DEFAULT_SEARCH_PATH = "toolbox;examples";
char rlab_dir[100];     /* absolute rlab directory path */
#endif

char *rlab_rc0;			/* RLAB_RC0, DEFAULT_RC0 */
char *help_dir;			/* RLAB_HELP_DIR, DEFAULT_HELP */
char *lib_dir;			/* RLAB_LIB_DIR, DEFAULT_LIB */
char *pager;			/* RLAB_PAGER, DEFAULT_PAGER */
char *search_path;		/* RLAB_SEARCH_PATH, DEFAULT_SEARCH_PATH */

/* Functions for signal handling */
void fpecatch _PROTO ((int));
void pipecatch _PROTO ((int));

int yydebug = 0;		/* set to `1' for parser debugging */

int print_machine = 0;		/* If TRUE, print contents of machine queue */
int use_readline = 1;		/* If FALSE DO NOT use GNU readline */
int use_rc0 = 1;		/* If TRUE run rlab_rc0 on start-up */
int use_pager = 1;		/* if TRUE use default pager (more) */
int load_lib = 1;		/* if TRUE load libraries */
int line_nos = 1;		/* If TRUE use line #s in op-codes */
int message = 1;                /* If TRUE print the start-up message */

char no_file[] = "no file info available";

/* If TRUE we will go interactive after start-up */
static int go_interactive = 1;
/* If TRUE we are running interactively */
static int interactive = 0;

/* Control output print formats */
int fwidth, fprec;
int FWIDTH_DEFAULT = 9;
int FPREC_DEFAULT = 3;

int write_diary = 0;		/* If TRUE echo input/output to diary file */
char *diary_filenm = 0;
FILE *diary_file_ptr = 0;
char *null_string = "NULL";

Program *mprog;			/* Main program array */

static char usage_string[] = "rlab -Vdhlmnpqr [file(s)] [-]";

/* **************************************************************
 * main, RLaB
 * ************************************************************** */
int
main (argc, argv)
     int argc;
     char *argv[];
{
  int c, r;

#ifdef THINK_C
  think_c_setup ();
  argc = ccommand(&argv);
#endif 

  /* Process command line args */
  progname = argv[0];
  while ((c = getopt (argc, argv, "Vdhlmnpqr")) != EOF)
  {
    switch (c)
    {
    case 'V':
      fprintf (stderr, "%s version: %s\n", progname, version_string);
      return (0);
      break;
    case 'd':
      print_machine = 1;
      break;
    case 'h':
      fprintf (stderr, "usage: %s\n", usage_string);
      return (1);
      break;
    case 'l':
      load_lib = 0;
      break;
    case 'm':
      message = 0;
      break;
    case 'n':
      line_nos = 0;
      break;
    case 'p':
      use_pager = 0;
      break;
    case 'q':
      use_rc0 = 0;
      break;
    case 'r':
      use_readline = 0;
      break;
    default:
      fprintf (stderr, "usage: %s\n", usage_string);
      return (1);
      break;
    }
  }

  /* Perform initialization */
  init_misc ();

#ifdef unix
  init_environment ();
#endif
#ifdef THINK_C
  init_tc_environment ();
#endif

  init_symbol_table ();
  init_file_list ();
  init_static_tree ();

  /* Process .rlab file */
  if (use_rc0)
  {
    if ((r = run_program (rlab_rc0)) == 0)
    {
      fprintf (stderr, "\nCould not open RLaB init script\n");
      fprintf (stderr, "try setting environment variable RLAB_RC0 ");
      fprintf (stderr, "and re-run RLaB\n\n\n");
    }
    else if (r < 0)
    {
      fprintf (stderr, "ERROR in rlab_rc0 file\n");
      return (0);
    }
  }

  /* Process rfiles in library */
  if (load_lib)
    rfile_dir (lib_dir);

  if (optind < argc)
    go_interactive = 0;

  /* Process files on the command line */
  while ((c = optind++) < argc)
  {
    if (!strcmp (argv[c], "-"))
      go_interactive = 1;
    else
      run_program (argv[c]);
  }

  /* Finally, go interactive */
  if (go_interactive)
  {
    if (message)  print_greeting ();
    interactive = 1;
    run_program (0);
  }
  return (0);
}

/* **************************************************************
 * Misc initialization.
 * ************************************************************** */
void
init_misc ()
{
  /* Set the input function pointer */
  set_rlab_input (0);

  /* Initialize the interpreter stacks */
  init_frame ();
  init_stack ();

  /* Initialize printf format string specifiers */
  fwidth = FWIDTH_DEFAULT;
  fprec = FPREC_DEFAULT;

  /* Probabaly a vain attempt to deal with fpe exceptions */
  FP_SET_MASK;
}

/* **************************************************************
 * Get all RLaB environment/default variables.
 * ************************************************************** */

#ifdef unix
void
init_environment ()
{

  if ((rlab_rc0 = getenv ("RLAB_RC0")) == 0)
    rlab_rc0 = DEFAULT_RC0;
  
  if ((help_dir = getenv ("RLAB_HELP_DIR")) == 0)
    help_dir = DEFAULT_HELP;
  
  if ((lib_dir = getenv ("RLAB_LIB_DIR")) == 0)
    lib_dir = DEFAULT_LIB;
  
  if (use_pager)
  {
    if ((pager = getenv ("RLAB_PAGER")) == 0)
    {
      if ((pager = getenv ("PAGER")) == 0)
        pager = DEFAULT_PAGER;
    }
  }
  else
    pager = cpstr ("cat");
  
  if ((search_path = getenv ("RLAB_SEARCH_PATH")) == 0)
    search_path = DEFAULT_SEARCH_PATH;
}
#endif  /* unix */

#ifdef THINK_C
void
init_tc_environment ()
{
  char *tmp;
  int  counter = 1;
  getcwd(rlab_dir, 99);

  if ((rlab_rc0 = getenv ("RLAB_RC0")) == 0)
    rlab_rc0 = DEFAULT_RC0;

  if ((help_dir = getenv ("RLAB_HELP_DIR")) == 0)
  {
    help_dir = (char *) malloc(strlen(rlab_dir)+strlen(DEFAULT_HELP)+8);
    sprintf(help_dir,"%s%s",rlab_dir,DEFAULT_HELP);
  }
  
  if ((lib_dir = getenv ("RLAB_LIB_DIR")) == 0)
  {
    lib_dir = (char *) malloc(strlen(rlab_dir)+strlen(DEFAULT_LIB)+8); 
    sprintf(lib_dir,"%s%s",rlab_dir,DEFAULT_LIB);         
  }
  
  if (use_pager)
  {
    if ((pager = getenv ("RLAB_PAGER")) == 0)
    {
      if ((pager = getenv ("PAGER")) == 0)
        pager = DEFAULT_PAGER;
    }
  }
  else
    pager = cpstr ("cat");
  
  if ((search_path = getenv ("RLAB_SEARCH_PATH")) == 0)
  {
    tmp = strchr(DEFAULT_SEARCH_PATH,';');
    while (tmp) 
    {
      counter++; tmp++;
      tmp = strchr(tmp,';');
    }    
    search_path = (char *) malloc (strlen (rlab_dir)*counter+
				   strlen(DEFAULT_SEARCH_PATH)+8);
    search_path[0] = '\0';
    tmp = strtok(DEFAULT_SEARCH_PATH,";");
    while (tmp)
    {
      strcat(search_path,rlab_dir);
      strcat(search_path,tmp);       
      tmp = strtok('\0',";");
      if (tmp) strcat(search_path,";");
    }
  }
}
#endif  /* THINK_C */

/* **************************************************************
 * Run the parser/machine.
 * ************************************************************** */

extern Program *program_Get _PROTO ((void));
extern Inst *get_program_counter _PROTO ((void));
extern void set_program_counter _PROTO ((Inst * prgm));
extern int reset_frame_ptr _PROTO ((void));
extern int reset_stack_ptr _PROTO ((void));
extern Datum *get_stackp _PROTO ((void));
extern void set_stackp _PROTO ((Datum *new_stackp));

int
run_program (input)
     char *input;
{
  int retval;
  Inst *oldpc;
  Program *old_program, *program;

  if (input == 0)
  {
    if (!new_file ("stdin"))
      return (0);
  }
  else
  {
    close_file_ds (input);         /* In case it was open by accident */
    if (!new_file (input))
      return (0);
  }

  program = program_Create (500);	/* Create a new program array */
  old_program = program_Get ();	        /* Save the old program array */
  oldpc = get_program_counter ();	/* Save the old counter */

  /* Point the parser at the new program array */
  program_Set (program);

  /* Run the parser/machine */
  while (1)
  {
    if (!setjmp (jmp[inc_buff ()]))
    {
      /* Normal operation */
      signal (SIGFPE, fpecatch);
      signal (SIGINT, intcatch_wait);
#ifdef HAVE_PIPE
      signal (SIGPIPE, pipecatch);
#endif
      if (input == 0)
	run (program);
      else
	run_no_clean (program);

      /* Decrement the jmp buffer counter */
      dec_buff ();
      retval = 1;
      break;
    }
    else
    {
      /* An error (longjmp) has occurred */
      reset_frame_ptr ();
      reset_stack_ptr ();
      retval = -1;
      if (input == 0)
	continue;
      else
      {
	/*
	 * We error'ed, keep longjmp'ing until we hit
	 * input == 0, or we run out of jmps.
	 */

	new_file (0); flush_line = 0;
	program_Destroy (program);
	program_Set (old_program);
	set_program_counter (oldpc);

	if (ijmp > 0)
	  longjmp( jmp[dec_buff()], 1 );
	else
	  return (retval);
      }
    }
  }

  /* Reset the old program array, etc, ... */
  program_Destroy (program);
  program_Set (old_program);
  set_program_counter (oldpc);

  return (retval);
}

/*
 * Run the parser / machine on a character string.
 */

extern int do_eval;   /* Signal the parser when we are doing an eval() */

int
run_program_eval ()
{
  int retval;
  Datum *old_stackp;
  Inst *oldpc;
  Program *old_program, *program;

  /*
   * Set up new program space, and save current state
   * so we can get back.
   */

  program = program_Create (50);        /* Create a new program array */
  old_program = program_Get ();	        /* Save the old program array */
  oldpc = get_program_counter ();	/* Save the old counter */
  old_stackp = get_stackp ();           /* Save current stackp */

  /* Point the parser at the new program array */
  program_Set (program);

  /* Run the parser/machine */
  while (1)
  {
    if (!setjmp (jmp[inc_buff ()]))
    {
      /* Normal operation */
      signal (SIGFPE, fpecatch);
      signal (SIGINT, intcatch_wait);
#ifdef HAVE_PIPE
      signal (SIGPIPE, pipecatch);
#endif

      initcode ();
      yyparse ();
      if (print_machine)
	diss_assemble (program->prog, progoff);
      execute (program->prog);

      /* Decrement the jmp buffer counter */
      dec_buff ();
      retval = 1;   /* Successfull return */
      break;
    }
    else
    {
      /*
       * An error (longjmp) has occurred.
       * Do not reset the frame and stack pointers
       * since we will no longjmp back to the prompt.
       * Instead we will return to the caller.
       */
      
      retval = 0;
      program_Destroy (program);
      program_Set (old_program);
      set_program_counter (oldpc);
      set_stackp (old_stackp);

      return (retval);
    }
  }

  /* Reset the old program array, etc, ... */
  program_Destroy (program);
  program_Set (old_program);
  set_program_counter (oldpc);

  return (retval);
}

void
run (program)
     Program *program;
{
  for (initcode (); yyparse (); initcode ())
  {
    if (print_machine)
      diss_assemble (program->prog, progoff);
    execute (program->prog);
    clean_list ();
  }
}

void
run_no_clean (program)
     Program *program;
{
  for (initcode (); yyparse (); initcode ())
  {
    if (print_machine)
      diss_assemble (program->prog, progoff);
    execute (program->prog);
  }
}

/* **************************************************************
 * Run a whole directory of rfiles through the machine.
 * ************************************************************** */

int
rfile_dir (dirname)
     char *dirname;
{
  /* FIX */
  char tmp[100];
  DIR *dirp;
#ifdef HAVE_DIRENT
  struct dirent *direntp;
#else
  struct direct *direntp;
#endif

  mprog = program_Create (1000);
  program_Set (mprog);
  if ((dirp = opendir (dirname)) != 0)
  {
    while ((direntp = readdir (dirp)) != NULL)
    {
      if (rfile_test (direntp->d_name))
      {
	strcpy (tmp, dirname);
	strcat (tmp, PATH_DELIM);
	strcat (tmp, direntp->d_name);
	run_program (tmp);
      }
    }
    closedir (dirp);
    return (1);
  }
  else
  {
    fprintf (stderr, "ERROR opening directory: %s\n", dirname);
  }
  return (0);
}

/* **************************************************************
 * Recover from a run-time error. 1 name, 1 message
 * ************************************************************** */
void
error_1 (s, t)
     char *s, *t;
{
  warning_1 (s, t);
  longjmp (jmp[dec_buff ()], 1);
}

/* **************************************************************
 * Recover from a run-time error. 2 names, 1 message.
 * ************************************************************** */
void
error_2 (s1, s2, t)
     char *s1, *s2, *t;
{
  warning_2 (s1, s2, t);
  longjmp (jmp[dec_buff ()], 1);
}

/* **************************************************************
 * Recover from a run-time error. 3 names, 1 message.
 * ************************************************************** */
void
error_3 (s1, s2, s3, t)
     char *s1, *s2, *s3, *t;
{
  warning_3 (s1, s2, s3, t);
  longjmp (jmp[dec_buff ()], 1);
}

/* **************************************************************
 * Print warning messages. We are not worried about speed, since
 * an error has occurred. We must check the string arguments,
 * because under some circumstances they may be NULL. If the input
 * is not coming from the command line (stdin) then print out
 * file and line number information.
 * ************************************************************** */
void
warning_1 (s, t)
     char *s, *t;
{
  char *fn;
  int lineno;

  /* 
   * Check s, if errors cascade, it is possible that s may be NULL.
   * Or s may be something obscure, like an argument tag.
   */
  if (s == 0)
  {
    s = null_string;
  }
  else if (!strncmp (s, "-", 1))
  {
    s = null_string;
  }

  /* Print file and line info if available */
  if (line_nos)
  {
    fn = find_file_name ();
    lineno = find_lineno ();
  }
  else
  {
    fn = no_file;
    lineno = 0;
  }
  if (write_diary)
  {
    fprintf (diary_file_ptr, "%s: %s", progname, s);
    if (t)
      fprintf (diary_file_ptr, ", %s", t);
    fprintf (diary_file_ptr, "\n");

    if (strcmp ("stdin", fn))
      fprintf (diary_file_ptr, "near line %d, file: %s\n",
	       lineno, fn);
    fflush (diary_file_ptr);
  }
  fprintf (stderr, "%s: %s", progname, s);

  if (t)
    fprintf (stderr, ", %s", t);
  fprintf (stderr, "\n");

  if (strcmp ("stdin", fn))
    fprintf (stderr, "near line %d, file: %s\n", lineno, fn);
  fflush (stderr);
}

void
warning_2 (s1, s2, t)
     char *s1, *s2, *t;
{
  char *fn;
  int lineno;

  /* 
   * Check s, if errors cascade, it is possible that s may be NULL.
   * Or s may be something obscure, like an argument tag.
   */
  if (s1 == 0)
    s1 = null_string;
  else if (!strncmp (s1, "-", 1))
    s1 = null_string;
  if (s2 == 0)
    s2 = null_string;
  else if (!strncmp (s2, "-", 1))
    s2 = null_string;

  /* Print file and line info if available */
  if (line_nos)
  {
    fn = find_file_name ();
    lineno = find_lineno ();
  }
  else
  {
    fn = no_file;
    lineno = 0;
  }
  if (write_diary)
  {
    fprintf (diary_file_ptr, "%s: %s, %s", progname, s1, s2);
    if (t)
      fprintf (diary_file_ptr, ", %s", t);
    fprintf (diary_file_ptr, "\n");

    if (strcmp ("stdin", fn))
      fprintf (diary_file_ptr, "near line %d, file: %s\n", lineno, fn);

    fflush (diary_file_ptr);
  }
  fprintf (stderr, "%s: %s, %s", progname, s1, s2);

  if (t)
    fprintf (stderr, ", %s", t);
  fprintf (stderr, "\n");

  if (strcmp ("stdin", fn))
    fprintf (stderr, "near line %d, file: %s\n", lineno, fn);
  fflush (stderr);
}

void
warning_3 (s1, s2, s3, t)
     char *s1, *s2, *s3, *t;
{
  char *fn;
  int lineno;

  /* 
   * Check s, if errors cascade, it is possible that s may be NULL.
   * Or s may be something obscure, like an argument tag.
   */
  if (s1 == 0)
    s1 = null_string;
  else if (!strncmp (s1, "-", 1))
    s1 = null_string;
  if (s2 == 0)
    s2 = null_string;
  else if (!strncmp (s2, "-", 1))
    s2 = null_string;
  if (s3 == 0)
    s3 = null_string;
  else if (!strncmp (s3, "-", 1))
    s3 = null_string;

  /* Print file and line info if available */
  if (line_nos)
  {
    fn = find_file_name ();
    lineno = find_lineno ();
  }
  else
  {
    fn = no_file;
    lineno = 0;
  }
  if (write_diary)
  {
    fprintf (diary_file_ptr, "%s: %s, %s, %s", progname, s1, s2, s3);
    if (t)
      fprintf (diary_file_ptr, ", %s", t);
    fprintf (diary_file_ptr, "\n");

    if (strcmp ("stdin", fn))
      fprintf (diary_file_ptr, "near line %d, file: %s\n", lineno, fn);

    fflush (diary_file_ptr);
  }
  fprintf (stderr, "%s: %s, %s, %s", progname, s1, s2, s3);

  if (t)
    fprintf (stderr, ", %s", t);
  fprintf (stderr, "\n");

  if (strcmp ("stdin", fn))
    fprintf (stderr, "near line %d, file: %s\n", lineno, fn);
  fflush (stderr);
}

/* **************************************************************
 * Signal catching functions.
 * ************************************************************** */
void
fpecatch (tmp)
     int tmp;
{
  error_1 ("floating point exception", (char *) 0);
}

void
pipecatch (tmp)
     int tmp;
{
  /* Don't print anything, just jump */
  longjmp (jmp[dec_buff ()], 1);
}

/* **************************************************************
 * Print a greeting to RLaB users.
 * ************************************************************** */

void
print_greeting ()
{
  printf ("Welcome to RLaB. New users type `help INTRO'\n");
  printf ("RLaB version %s ", version_string);
  printf ("Copyright (C) 1992, 93, 94 Ian Searle\n");
  printf ("RLaB comes with ABSOLUTELY NO WARRANTY; for details type");
  printf (" `help WARRANTY'\n");
  printf ("This is free software, and you are welcome to redistribute it");
  printf (" under\ncertain conditions; type `help CONDITIONS' for details\n");
}

/* **************************************************************
 * Cover for pclose() so we don't get messed up by pclose() that
 * cause a SIGPIPE, when closing an already broken pipe.
 * ************************************************************** */

#ifdef HAVE_PIPE

void
rpclose (fp)
     FILE *fp;
{
  signal (SIGPIPE, SIG_IGN);
  pclose (fp);
  signal (SIGPIPE, pipecatch);
}

#else

void
rpclose (fp)
     FILE *fp;
{
  fclose (fp);
}

#endif  /* HAVE_PIPE */

/* **************************************************************
 * Jump buffer handling routines. These routines merely inc, or 
 * an integer, whilst checking to see that the bounds of the array 
 * have not been exceeded .
 * ************************************************************** */

int
inc_buff ()
{
  if (ijmp >= NJMP)
    fprintf (stderr, "NJMP too small\n");

  return (ijmp++);
}

int
dec_buff ()
{
  if (ijmp <= 0)
    fprintf (stderr, "error while decrementing ijmp\n");

  return (--ijmp);
}



/* **************************************************************
 * Platform specific setup, in this case THINK-C.
 * ************************************************************** */

#ifdef THINK_C

static void
think_c_setup ()
{
  if (txFont = getenv ("RLAB_FONT"))
    txFontNum = atoi(txFont);
  if (txSize = getenv ("RLAB_SIZE"))
    txSizeNum = atoi(txSize);
  if (nrows = getenv ("RLAB_ROWS"))
    mac_rows = atoi(nrows);
  console_options.pause_atexit = 1;
  console_options.procID = 0;    
  console_options.title = "\pMacRLaB";    
  console_options.nrows = mac_rows;
  console_options.txFont = txFontNum;
  console_options.txSize = txSizeNum;    
  cgotoxy(1,1,stdout);
}

#endif  /* THINK_C */

