/* listnode.c */

/*  This file is a part of RLaB ("Our"-LaB)
    Copyright (C) 1992  Ian R. Searle

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    See the file ./COPYING
 ***********************************************************************/

#include "rlab.h"
#include "mem.h"
#include "listnode.h"
#include <stdio.h>

static int listNode_isValid _PROTO ((ListNode *));

/* ************************************************************************
 * PURPOSE: Initialize the members of a ListNode structure.
 * ************************************************************************ */
static int
listNode_isValid (listNode)
     ListNode *listNode;
{
  return (listNode != NULL &&
	  listNode->next != BAD_LIST_NODE &&
	  listNode->prev != BAD_LIST_NODE);
}

/* ************************************************************************
 * PURPOSE: To dynamically allocate and initialize a ListNode.
 * ************************************************************************ */
ListNode *
listNode_Create ()
{
  ListNode *new = (ListNode *) MALLOC (sizeof (ListNode));
  new->type = 0;
  new->key = 0;
  new->data = 0;
  new->dtor = 0;
  new->next = 0;
  new->prev = 0;
  return (new);
}

/* ************************************************************************
 * PURPOSE: Free all of the dynamically allocated memory associated
 *	    with a ListNode.  Notice that this function checks to see
 *	    if listNode->dtor is non-NULL.  If so, the pointer
 *	    to function is executed, and passed the data that resides
 *	    on the ListNode.  Clients must write and supply the
 *	    function that dtor points to in order to have any
 *          dynamically allocated memory within the data freed.
 * ************************************************************************ */
void
listNode_Destroy (listNode)
     ListNode *listNode;
{
  ASSERT (listNode_isValid (listNode));
  {
    if (listNode->dtor)
      (*(listNode->dtor)) (listNode->data);

    listNode->type = 0;
    FREE (listNode->key);
    listNode->dtor = NULL;
    listNode->data = NULL;
    listNode->next = BAD_LIST_NODE;
    listNode->prev = BAD_LIST_NODE;

    FREE (listNode);
  }
}

/* ************************************************************************
 * PURPOSE: Free all of dynamic memory ascociated with a list-node,
 *          but NOT the attached data.
 * ************************************************************************ */
int
listNode_DestroyNodeOnly (listNode)
     ListNode *listNode;
{
  ASSERT (listNode_isValid (listNode));
  {
    FREE (listNode->key);
    listNode->type = 0;
    listNode->next = BAD_LIST_NODE;
    listNode->prev = BAD_LIST_NODE;
    FREE (listNode);
    return (1);
  }
}

/* **************************************************************
 * Destroy a listNodes DATA only.
 * Set the data and dtor() pointers to NULL.
 * ************************************************************** */
int
listNode_DestroyDataOnly (listNode)
     ListNode *listNode;
{
  ASSERT (listNode);
  {
    if (listNode_GetData (listNode))
    {
      if (listNode->dtor)
      {
	(*(listNode->dtor)) (listNode->data);
	listNode->dtor = 0;
	listNode->data = 0;
      }
      return (1);
    }
    return (0);
  }
}

/* **************************************************************
 * Attach a ListNode ahead of another ListNode.
 * listNode is put ahead of nodeBehind.  If nodeBehind already
 * had a ListNode ahead of it, then we hook up pointers
 * as necessary.
 *
 * Note that a ListNode has no concept as to what a list is.
 * All it knows how to do is to attach and detach itself from
 * other ListNode's.  Lists will use ListNode's to build
 * and destroy lists.
 * ************************************************************** */
int
listNode_AttachAhead (listNode, nodeBehind)
     ListNode *listNode, *nodeBehind;
{
  ASSERT (listNode_isValid (listNode));
  ASSERT (listNode_isValid (nodeBehind));
  {
    ListNode *nodeAhead = listNode_GetNodeAhead (nodeBehind);

    listNode->prev = nodeBehind;
    nodeBehind->next = listNode;

    if (nodeAhead)
    {
      nodeAhead->prev = listNode;
      listNode->next = nodeAhead;
    }
    return (1);
  }
}

/* **************************************************************
 * Attach a ListNode behind another ListNode.
 * listNode is put behind of nodeAhead.  If nodeAhead already
 * had a ListNode behind it, then we hook up pointers
 * as necessary.
 *
 * Note that a ListNode has no concept as to what a list is.
 * All it knows how to do is to attach and detach itself from
 * other ListNode's.  Lists will use ListNode's to build
 * and destroy lists.
 * ************************************************************** */
int
listNode_AttachBehind (listNode, nodeAhead)
     ListNode *listNode, *nodeAhead;
{
  ASSERT (listNode_isValid (listNode));
  ASSERT (listNode_isValid (nodeAhead));
  {
    ListNode *nodeBehind = listNode_GetNodeBehind (nodeAhead);

    listNode->next = nodeAhead;
    nodeAhead->prev = listNode;

    if (nodeBehind)
    {
      nodeBehind->next = listNode;
      listNode->prev = nodeBehind;
    }
    return (1);
  }
}

/* ************************************************************************
 * PURPOSE: Detach a ListNode from a list of ListNodes.
 *          The list-node is detached ONLY, not destroyed. The adjoining
 *          nodes (if there are any) are updated to reflect the
 *          detachment.
 * ************************************************************************ */
int
listNode_Detach (listNode)
     ListNode *listNode;
{
  ASSERT (listNode_isValid (listNode));
  {
    ListNode *nodeAhead = listNode_GetNextNode (listNode);
    ListNode *nodeBehind = listNode_GetPrevNode (listNode);

    if (listNode_IsNodeAttached (listNode))
    {
      if (nodeAhead)
	nodeAhead->prev = nodeBehind ? nodeBehind : NULL;

      if (nodeBehind)
	nodeBehind->next = nodeAhead ? nodeAhead : NULL;

      listNode->prev = listNode->next = NULL;
    }
    return (1);
  }
}

/* **************************************************************
 * Get the next node in the list.
 * ************************************************************** */
ListNode *
listNode_GetNodeAhead (listNode)
     ListNode *listNode;
{
  return listNode->next;
}

/* **************************************************************
 * Get a pointer to the ListNode that is behind a given
 * ListNode.  If there is no ListNode behind then NULL
 * is returned.
 * ************************************************************** */
ListNode *
listNode_GetNodeBehind (listNode)
     ListNode *listNode;
{
  return (listNode->prev);
}

/* ************************************************************************
 * PURPOSE:     Attach some data to a ListNode.  data may be anything at
 *		all (data) that the users of ListNode's want to store.
 *              A NULL data pointer may be specified since we may want
 *              to create listNode, and install data later.
 *		A NULL freeFn is perfectly acceptable if the data has
 *		no dynamically allocated memory associated with it.
 *
 *              One may think it strange that this function returns one
 *		of the arguments that were passed into it - namely the
 *		pointer to ListNode.  The reason it does so is so
 *		that one may dynamically allocate ListNode's on a List  .
 *		See the macros in list.h for further clarification.
 * ************************************************************************ */
ListNode *
listNode_AttachData (listNode, type, data, freeFn)
     ListNode *listNode;
     int type;
     VPTR data;
     void (*freeFn) ();
{
  ASSERT (listNode_isValid (listNode));
  {
    listNode->type = type;
    listNode->data = data;
    listNode->dtor = freeFn;
    return (listNode);
  }
}

/* **************************************************************
 * To determine if a ListNode is currently attached to another
 * ListNode.  Actually, the best we can do is to check and
 * see that either the prev or next pointers are non-NULL,
 * and, if so, assume that the ListNode is attached.  Note
 * that one of either the prev or next pointers may very well
 * be NULL, if the listNode is the first node in a list or
 * the last node in a list, respectively.
 * ************************************************************** */
/* int
 * listNode_IsNodeAttached (listNode)
 * ListNode *listNode;
 * {
 * ASSERT (listNode_isValid (listNode));
 * {
 * if (listNode->next || listNode->prev)
 * return (1);
 * else
 * return (0);
 * }
 *}
 */

/* ************************************************************************
 * PURPOSE:
 *          Return a pointer to the character string Key of a listNode.
 * ************************************************************************ */
/*
 * char *
 * listNode_GetKey (listNode)
 * ListNode *listNode;
 * {
 * return (listNode->key);
 * }
 */

/*
 * char *
 * e_key (lnode)
 * ListNode *lnode;
 * {
 * return (lnode->key);
 * }
 */

/* ************************************************************************
 * Set the "key" identifier in a list-node to the value of string.
 * ************************************************************************ */
int
listNode_SetKey (listNode, string_ptr)
     ListNode *listNode;
     char *string_ptr;
{
  ASSERT (listNode_isValid (listNode));
  {
    FREE (listNode->key);
    listNode->key = string_ptr;
    return (1);
  }
}

int
listNode_SetNum (lnode, num)
     ListNode *lnode;
     int num;
{
  ASSERT (lnode);
  {
    return (lnode->num = num);
  }
}

int
listNode_SetScope (lnode, scope)
     ListNode *lnode;
     int scope;
{
  ASSERT (lnode);
  {
    return (lnode->scope = scope);
  }
}

/* **************************************************************
 * Set the type member
 * ************************************************************** */
int
listNode_SetType (listNode, type)
     ListNode *listNode;
     int type;
{
  ASSERT (listNode);
  listNode->type = type;
  return (1);
}
