#ifndef lint
static char SCCSid[] = "@(#) ./xtools/solid/shapes.c 07/23/93";
#endif

#include <math.h>
#ifndef M_PI
#define	M_PI	3.14159265358979323846
#endif
#include "tools.h"
#include "xtools/basex11.h"
#include "xtools/base3d.h"

/* 
    Routines for creating basic shapes as collections of triangles.
    Note that for the best performance, we'd like to ADD these triangles
    and vertices to an existing list (so that the object could be drawn
    using the "back-to-front" triangle fill algorithm.

    Additional routines (or an outer wrapper) could do the allocation
    of storage.
 */
void XB3dAddCylinder( r, h, nr, nh, cyl )
double      r, h;
int         nr, nh;
XBGraphObject *cyl;
{
double       theta, *xx, *yy, *zz, zh;
XBTxyz       *v;
XBTriangle   *tris;
int          i, j, b;

/* Generate the points */
v  = cyl->v.txyz + cyl->v.nv;
zh = 0.0;
for (j=0; j<nh; j++) {
    theta = 0.0;
    for (i=0; i<nr; i++) {
        theta = i * 2.0 * M_PI / nr;
        XBTX(v) = r * sin(theta);
	XBTY(v) = r * cos(theta);
        XBTZ(v) = zh;
	v++;
        }
    zh += 1/nh;
    }

/* Generate the triangles */
tris = cyl->t + cyl->nt;
b    = cyl->v.nv;
for (j=0; j<nh-1; j++) {
    for (i=0; i<nr; i++) {
	XBVertexIndex1(tris) = b + i;
	XBVertexIndex2(tris) = b + (i+1) % nr;
	XBVertexIndex3(tris) = b + i + nr;
	tris++;
	XBVertexIndex1(tris) = b + ((i+1) % nr);
	XBVertexIndex2(tris) = b + i + nr;
	XBVertexIndex3(tris) = b + ( (i+1) % nr ) + nr;
	}
    b += nr;
    }
}

void XB3dAddCircle( r, nr, cir )
double      r;
int         nr;
XBGraphObject *cir;
{
int        i; 
double     theta;
XBTriangle *tris;
XBVertices *v;
XBTxyz     *vv;

v    = &cir->v;
tris = cir->t;
vv   = v->txyz;
for (i=0; i<nr; i++) {
    theta = i * 2 * M_PI / nr;
    XBTX(vv) = r * sin(theta);
    XBTY(vv) = r * cos(theta);
    XBTZ(vv) = 0.0;
    vv++;
    XBVertexIndex1(tris) = i;
    XBVertexIndex2(tris) = (i+1) % nr;
    XBVertexIndex3(tris) = nr;
    tris++;
    }
/* Center point */
XBTX(vv) = 0.0;
XBTY(vv) = 0.0;
XBTZ(vv) = 0.0;
}

void XB3dAddSphere( r, nr, sphere, color )
double      r;
int         nr;
XBGraphObject *sphere;
Color       color;
{
int        nrv;
double     dphi, theta, dtheta, phi;
double     x[1000], y[1000], z[1000];
int        vidx, tidx, i, j, vlidx, left;
XBTriangle *tris;
XBVertices *v;
XBTxyz     *vv;

tris = sphere->t + sphere->nt;
tidx = 0;
/*
 We create a sequence of rings of constant phi, and connect them
 with triangles.  The poles are special cases.  For this first try,
 each ring will have a fixed number of vertices nrv.
 */
nrv     = 12;
if (nr < 1) nr      = 12;
r       = 1;      /* radius, center = (0,0,0) */
dphi    = 3.141592654 / (nr + 2);
dtheta  = 3.141592654 * 2 / nrv;

/* Check that there is enough space */
if (sphere->ntmax - sphere->nt < 2 * nr * nrv) {
    SETERRC( 1, "Insufficient triangles in object" );
    return;
    }

phi     = dphi;
/* north polar cap */
theta   = 0.0;
vidx = 0;
tidx = 0;
vlidx= 0;
x[vlidx] = 0.0;
y[vlidx] = 0.0;
z[vlidx] = r;
vlidx++;
vidx++;
for (i=0; i<nrv; i++) {
    x[vlidx]    = r*cos(theta)*sin(phi);
    y[vlidx]    = r*sin(theta)*sin(phi);
    z[vlidx]    = r*cos(phi);
    theta       += dtheta;
    if (i == nrv - 1) {
	XBSetVertexIndex( tris + tidx, 0, vidx, 1 );}
    else {
	XBSetVertexIndex( tris + tidx, 0, vidx, vidx+1 );}
    tris[tidx].discolor = color;
    tidx++;
    vidx++;
    vlidx++;
    }
XB3dAddVerticesFromXYZ( sphere, vlidx, x, y, z );
vlidx = 0;
/* annular rings */
for (i=1; i<=nr; i++) {
    phi     += dphi;
    for (j=0; j<nrv; j++) {
        x[vlidx]    = r*cos(theta)*sin(phi);
	y[vlidx]    = r*sin(theta)*sin(phi);
	z[vlidx]    = r*cos(phi);
	theta       += dtheta;
	if (j == nrv - 1) left = -nrv + 1;
	else              left = 1;
	XBSetVertexIndex( tris + tidx, vidx - nrv, vidx, vidx + left );
	tris[tidx].discolor = color;
	tidx ++;
	XBSetVertexIndex( tris + tidx, 
			  vidx - nrv, vidx + left, vidx - nrv + left );
	tris[tidx].discolor = color;
	tidx ++;
	vidx++;
	vlidx++;
	}
    XB3dAddVerticesFromXYZ( sphere, vlidx, x, y, z );
    vlidx = 0;
    }

/* south polar cap */
/* We already have the points of this annulus on the last nrv points */
vlidx = 0;
x[vlidx] = 0.0;
y[vlidx] = 0.0;
z[vlidx] = -r;
vlidx++;
XB3dAddVerticesFromXYZ( sphere, vlidx, x, y, z );
vlidx = vidx++;
vidx -= nrv;
for (i=0; i<nrv; i++) {
    if (i == nrv - 1) {
	XBSetVertexIndex( tris + tidx, vlidx, vidx-nrv+1, vidx ); }
    else {
	XBSetVertexIndex( tris + tidx, vlidx, vidx+1, vidx ); }
    tris[tidx].discolor = color;
    tidx++;
    vidx++;
    }
sphere->nt   += tidx;
sphere->v.nv += vidx;
}

/* @
   XB3dCreateGraphObject - Create a graphic object 

   Input Parameters:
.  nt   - number of triangles
.  nv   - number of vertices   
  @ */
XBGraphObject *XB3dCreateGraphObject( nt, nv )
int nt, nv;
{
XBGraphObject *g;

g          = NEW(XBGraphObject);   CHKPTRN(g);
g->t       = (XBTriangle *)MALLOC( nt * sizeof(XBTriangle) );   CHKPTRN(g->t);
g->nt      = 0;
g->ntmax   = nt;

g->v.xyz      = (XBxyz *) MALLOC( nv * sizeof(XBxyz) );   CHKPTRN(g->v.xyz);
g->v.txyz     = (XBTxyz *)MALLOC( nv * sizeof(XBTxyz) ); CHKPTRN(g->v.txyz);
g->v.color    = (Color *)MALLOC( nv * sizeof(Color) );  CHKPTRN(g->v.color);
g->v.discolor = (PixVal *)MALLOC( nv * sizeof(PixVal) ); 
                CHKPTRN(g->v.discolor);
g->v.nvmax    = nv;
g->v.nv       = 0;

g->ts         = (XBTriangle **)MALLOC( nt * sizeof(XBTriangle*) );
                CHKPTRN(g->ts);

return g;
}

/* @
   XB3dDestroyGraphObject - Destroy a graphic object 

   Input Parameters:
   g    - graph object to destroy
  @ */
void XB3dDestroyGraphObject( g )
XBGraphObject *g;
{
FREE( g->t );
FREE( g->v.xyz );
FREE( g->v.txyz );
FREE( g->v.color );
FREE( g->v.discolor );
FREE( g->ts );
FREE( g );
}

