/* @(#) ./xtools/base3d.h 07/23/93 */

/*
   Definitions for support of 3-d objects.

   In order to provide for a simple interface for back-to-front 
   drawing, the 3-d object support contains an entire COLLECTION
   of objects (this assumes that we do not make frequent inserts
   or deletions).  To allow for generalized colors, both
   vertices and triangles have colors.

   Eventually, we'll want these to be accessed in a data-structure neutral
   form so that we can switch between different representations AND
   use user-defined representations.
 */

#ifndef _XB3d
#define _XB3d

typedef unsigned long Color;

/* 
   In order to maximize the use of aggregate operations, rather than
   define a vertex and a triangle, we define vertices and triangles.
 */
typedef int XBTCoord;

/* Arrays for the triples of coordinates */
typedef double XBxyz[3];
typedef XBTCoord XBTxyz[3];

/* The following is one possible vertex structure.  It has the advantage that
   the individual fields are contiguous.  This may or may not be advantageous
 */
typedef struct {
    int       nv,               /* Number of vertices */
              nvmax;            /* Allocated number of vertices */
    XBxyz     *xyz;             /* Location in user space (3 components are
				   adjacent) */
    XBTxyz    *txyz;            /* Location in display space (3 components
				   are adjactent) */
    Color     *color;           /* Color value */
    PixVal    *discolor;        /* Corresponding display color value */
    } XBVertices;

/* 
  The following is one possible triangle structure; it contains references
  by OFFSET into a related XBVertices structure.
 */
typedef struct {
    int    vt1, vt2, vt3;       /* Offsets to the vertex
				   in the vertices structure */
    Color  color;
    PixVal discolor;
    int    depth;               /* computed depth of triangle, for sorting */
    /* ? maintain the surface normal? */
    } XBTriangle;
typedef struct {
    int nv, *vt;                /* Offsets to the vertex in the vertex struc */
    Color   color;
    PixVal  discolor;
    int     depth;
    } XBPolygon;
typedef struct {
    int     vt1, vt2, vt3, vt4;  
    Color  color;
    PixVal discolor;
    int    depth;              
    } XBQuadrilateral;          /* Special case polygon */
/*
  The following is one possible graphical object structure.
 */
typedef struct {
    int         nt,               /* Number of triangles */
                ntmax,            /* Maximum number of triangles */
                ntvis;            /* Number of visable triangles */
    XBVertices  v;                /* Associated vertices */
    XBTriangle  *t;               /* Triangles */
    XBTriangle  **ts;             /* Sorted triangles */
    } XBGraphObject;

typedef struct {
    double mat[16], permat[16];
    int    xoff, yoff;
    } XBTrans3d;

typedef enum { LIGHT_POINT, LIGHT_DIFFUSE, LIGHT_AMBIENT } XBLightType; 
typedef struct {
    XBxyz        xyz;                /* location of light */
    double       intens;             /* intensity of light */
    Color        red, green, blue;   /* color of light */
    XBLightType  type;
    } XBLightSource;


/*
   Here are macros to access the elements of these structures.  If the
   structures are changed, change these and the rest of the codes will
   continue to function.
 */
#define XBVertexIndex1( triangle ) ((triangle)->vt1)
#define XBVertexIndex2( triangle ) ((triangle)->vt2)
#define XBVertexIndex3( triangle ) ((triangle)->vt3)

#define XBSetVertexIndex( triangle, v1, v2, v3 ) { \
   (triangle)->vt1 = v1; (triangle)->vt2 = v2; (triangle)->vt3 = v3; }
/* These return a structure (XBTxyz or XBxyz) that contain the coords of the
   vertex */
#define XBVertexTXYZ(vidx, vertices)  ((vertices)->txyz+(vidx))
#define XBVertexXYZ(vidx, vertices)   ((vertices)->xyz+(vidx))

#define XBTX(v) (*(v))[0]
#define XBTY(v) (*(v))[1]
#define XBTZ(v) (*(v))[2]

#define XBVX(v) (*(v))[0]
#define XBVY(v) (*(v))[1]
#define XBVZ(v) (*(v))[2]

#define XBVXA(v) (v)[0]
#define XBVYA(v) (v)[1]
#define XBVZA(v) (v)[2]

extern XBGraphObject *XB3dCreateGraphObject();
extern XBTrans3d *XB3dCreateTransform( );

#endif
