/*
    This file contains routines to convert between symmetric matrices
  in SpMat format and diagonal, Aij format, i.e. the YSMP format with 
  the diagonal stored separately.
*/
#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "sparse/util/usparse.h"

/*@
   SpToDiagonalAijSymmetric - Given a symmetric SpMat, puts it in
  diagonal, Aij format. Mallocs the space for diagonal, Aij format.
  
  Notes: 
  Column oriented results not row oriented!

   The matrix must be symmetric or it breaks down.

   Input Parameters:
.  mat - matrix to convert

   Output Parameters:
.  diag  - pointer to array containing diagonal
.  off   - pointer to array containing the off-diagonal members
.  ai,aj - pointers to arrays containing i,j indices
@*/
void SpToDiagonalAijSymmetric( mat, nrow, diag, off, ai, aj )
SpMat  *mat;
int    *nrow,**ai,**aj;
double **diag,**off;
{
  double     *p,*d,*a;
  int        *j, nz, row, i,n ,*ia,*ja,*nzl,*nzlstart;
  SpVec      *r;
  SpRowMat   *R;

  nzl = nzlstart = SpiDiagPtrs(mat); 

  R  = GETROWMAT(mat);
 
  /* determine number of nonzeros in symmetric part */
  *nrow = n = mat->rows; nz = 0;
  for ( i=0; i<n; i++ ) {
     r    = R->rs[i];
     nz  += r->nz - nzl[i] - 1;
  }

  /* get the storages space */
  *diag = d  = (double *) MALLOC(n*sizeof(double));  CHKPTR(d);
  *ai   = ia = (int *)    MALLOC((n+1)*sizeof(int)); CHKPTR(ia);
  *off  = a  = (double *) MALLOC(nz*sizeof(double)); CHKPTR(a);
  *aj   = ja = (int *)    MALLOC(nz*sizeof(int));    CHKPTR(ja);

  ia[0] = 1;
  for (row = 0; row < n; row++) {
    r    = R->rs[row];
    nz   = r->nz - *nzl - 1;
    p    = r->v + *nzl;
    j    = r->i + *nzl++;
    *d++ = *p;

    if ( *j != row ) {
        SETERRC(1,"Diagonal is zero in SpToDiagonalAijSymmetric");
        return ;}
    p++; j++;
    for ( i=0; i<nz; i++) {

      *ja++ = 1 + *j++;   
      *a++  = *p++;
    }
    ia[row+1] = ia[row] + nz;
  }
  FREE(nzlstart); 
}
/* -------------------------------------------------------------*/
/*@
   SpFromDiagonalAijSymmetric - Given a symmetric matrix in
  diagonal, Aij format, returns a SpMat copy of it.
   
  Notes:
  column oriented not row oriented! This is a very bad 
  implementation, since we only have the lower triangular
  part of the matrix it is not easy to allocate the entire new
  matrix at once, we therefore add elements one at the time 
  which leads to lots of pointer chasing and means we allocate 
  small chunks at a time so much of the storage is devoted to 
  keeping track of what we have allocated rather then the matrix
  entries.

   Input Parameters:
.   nrow - number of rows in matrix
.   diag - Diagonal entries
.   off  - off-diagonal entries
.   ai,aj - row, column indices

   Returns:
   matrix
@*/
SpMat *SpFromDiagonalAijSymmetric( nrow, diag, off, ia, ja)
int    nrow,*ia,*ja;
double *diag,*off;
{
  SpMat    *mat;
  SpVec    *vs,**nb;
  SpRowMat *R;
  int      row, nz, *rows,i;

  /* determine how many elements in each row */
  rows = (int *) MALLOC(nrow*sizeof(int)); CHKPTRN(rows);
  for ( i=0; i<nrow; i++ ) rows[i] = 1 + ia[i+1] - ia[i];
  nz = ia[nrow] - 1; for ( i=0; i<nz; i++ ) {rows[ja[i]-1]++;}

#ifdef FOO
  /* allocate the space for the matrix */
  mat              = NEW(SpMat); CHKPTRV(mat,0);
  R                = NEW(SpRowMat); CHKPTRV(R,0);
  mat->data = (void *)R;
  R->blks = 0;
  R->rblks = 0;
  nb               = (SpVec **) SPMalloc( nrow*(sizeof(SpVec *) +
                                          sizeof(SpVec)) ); CHKPTRV(nb,0);
  mat->rows  = nrow;
  mat->cols  = nrow;
  mat->map   = 0;
  mat->type  = MATROW;
  mat->is_sorted = 0;
  SPiInitPool( &mat->pool ); CHKERRV(0,0);
  R->rs          = nb;
  R->blks        = 0;

  vs = (SpVec *) (R->rs + nrow);
  for (i=0; i<nrow; i++) {
    R->rs[i] = vs;
    vs->maxn   = rows[i];
    vs->nz     = 0;
    vs->blki   = -1;
    SPMallocNV( mat, rows[i], &vs->v, &vs->i ); CHKERRV(0,0);
    vs++;
  }
#else
  mat = SpCreate( nrow, nrow, 0 );   CHKPTRV(mat,0);
  R   = (SpRowMat *)mat->data;
  for (i=0; i<nrow; i++) {
    vs = R->rs[i];
    vs->maxn   = rows[i];
    vs->nz     = 0;
    vs->blki   = -1;
    SPMallocNV( mat, rows[i], &vs->v, &vs->i ); CHKERRV(0,0);
  }
#endif
  /* stick the values into the matrix */
  R   = (SpRowMat *)mat->data;
  for (row = 0; row < nrow; row++) {
    nz   = ia[row+1] - ia[row];

    SpAddValue(mat,diag[row],row,row);

    while (nz--) {
        SpAddValue(mat,*off,*ja - 1,row);
        SpAddValue(mat,*off++,row,*ja++ - 1);
    }      
  } 
  FREE(rows);
  return mat;
}
