/*
   A program to test solves.  This one finds part of an inverse
 */

#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "sparse/spmat.h"

/* Forward references */
void CompareSolution();

#define max(a,b) ( ((a)>(b))?a:b )
#define min(a,b) ( ((a)<(b))?a:b )

SpMat *BuildLaplacian();

double MegaFlopsFactor( BB, time )
SpMatSplit *BB;
double     time;
{
return (SpCostFactor( BB ) / time) * 1.0e-6;
}
double MegaFlopsSolve( BB, time )
SpMatSplit *BB;
double     time;
{
return (SpCostSolve( BB ) / time) * 1.0e-6;
}

main( argc, argv )
int  argc;
char **argv;
{
int        n, m;

if (argc > 1) m = atoi( argv[1] );
else m   = 16;
n   = m * m;

TestInverse( 3 );

{int s, f;
TRSPACE(&s,&f);
printf( "Memory allocated = %d in %d blocks\n", s, f );
printf( "List of blocks is\n" );
TRDUMP(stdout);
 }
return 0;
}

SpMat *BuildLaplacian( m, vrhs )
int    m;
double *vrhs;
{
SpMat       *f;
double sum;
int    n, i, err=0;

n   = m * m;
/* This would be faster if the last parameter was 5, but using 1 gives
   us more testing (on-the-fly allocation of space) */
f   = SpCreate( n, n, 1 );

for (i=0; i<n; i++) {
    sum = 0.0;
    if (i - m >= 0) {
	err += SpAddValue( f, -1.0, i, i - m );
	sum += -1.0;
	}
    if (i - 1 >= 0) {
	err += SpAddValue( f, -1.0, i, i - 1 );
	sum += -1.0;
	}
    err += SpAddValue( f, 4.0, i, i );
    sum += 4.0;
    if (i + 1 < n) {
	err += SpAddValue( f, -1.0, i, i + 1 );
	sum += -1.0;
	} 
    if (i + m < n) {
	err += SpAddValue( f, -1.0, i, i + m );
	sum += -1.0;
	}
    vrhs[i] = sum;
    }
return f;
}

void CompareSolution( v, n )
register double *v;
register n;
{
register double sum = 0.0;
while (n--) {
    sum += (*v - 1.0) * (*v - 1.0);
    v++;
    }
printf( "Difference in 2-norm is %f\n", sum );
}

/* Add 1 to perm (change from zero origin to one origin */
void ShiftPerm( perm, n )
register int    *perm, n;
{
while (n--) 
    *perm++ += 1;
}

void InverseOrderingOne( perm, iperm, n )
int *perm, *iperm, n;
{
int i;

for (i=1; i<=n; i++) {
    iperm[*perm++ - 1] = i;
    }
}

/* This works by computing the inverse in the "usual" way, and checking
   that against the value computed by the rows of inverse routine */
TestInverse( n )
int n;
{
SpMat      *mat;
SpMatSplit *factor;
int        i, j, err;
double     *x, *y, *xt, *b;
int        *IndexOfRows;

mat = SpCreate( n, n, 0 );

for (i=0; i<n; i++)
    for (j=0; j<n; j++) 
	SpAddValue( mat, (double)(i+j), i, j );
SpAddValue( mat, 3.0, 1, 0 );
SpAddValue( mat, 1.0, 0, 0 );
factor = SpCreateSplit( mat, 0 );
err    = SpComputeFill( mat, factor );
err    = SpComputeFactor( mat, factor );
b      = (double *)MALLOC( n * sizeof(double) );
x      = (double *)MALLOC( n * n * sizeof(double) );
xt     = (double *)MALLOC( n * n * sizeof(double) );
IndexOfRows = (int *)MALLOC( n * sizeof(int) );

/* Compute the inverse matrix of the transpose */
for (i=0; i<n; i++) IndexOfRows[i] = i;
SpRowsOfInverse( mat, n, IndexOfRows, xt, (void *)0 );

/* Compute the inverse matrix using the usual technique */
for (i=0; i<n; i++) {
    for (j=0; j<n; j++) b[j] = 0.0;
    b[i] = 1.0;
    SpSolve( factor, b, x + (n*i) );
    }

/* Now that we have the inverse, print it */
printf( "Inverse matrix\n" );
DisplayMat( x, n, n );
printf( "Inverse computed by rows\n" );
DisplayMat( xt, n, n );

/* Check that the two versions match up.  Note that both are stored in
   COLUMN-major order. */
for (i=0; i<n; i++)
    for (j=0; j<n; j++) {
    	if (fabs(x[i+j*n] - xt[i+j*n]) > 1.0e-8) {
    	    printf( "a^-1(%d,%d)[%e] != (a^-T)^T[%e]\n", i, j,
    	    	    x[i+j*n], xt[i+j*n] );
    	    }
        }

SpDestroy( mat );
SpDestroySplit( factor );
FREE(b); FREE(x); FREE(xt); FREE( IndexOfRows );
}

DisplayMat( x, nr, nc )
double *x;
int    nr, nc;
{
int    i, j;
double v;  /* Used to clamp value */

for (i=0; i<nr; i++) {
    for (j=0; j<nc; j++) {
	v = x[i+j*nr];
	if (fabs(v) < 1.e-300) v = 0.0;
    	printf( "%13e ", v );
        }
    printf( "\n" );
    } 
}
