#ifndef lint
static char SCCSid[] = "@(#) ./sparse/print.c 07/23/93";
#endif

#if defined(cray)
#include <errno.h>
#else
extern int errno;
#endif
#include <sys/errno.h>
#include <stdio.h>                 /*I <stdio.h> I*/
#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "inline/setval.h"
#include "inline/spops.h"

/*@
   SpPrint - Prints a sparse matrix to a given FILE.

   Input parameters:
.  B    - matrix to print
.  fp   - FILE pointer. Use stdout for standard output.

   Note:
   The format of the output is 
   <row nn>
   [column] = value
 @*/
void SpPrint( fp, B )
FILE  *fp;
SpMat *B;
{
int    i, j, *c;
double *p;

if (fp == 0) fp  = stdout;
SPLITTOMAT(B)
for (i=0; i<B->rows; i++) {
    fprintf( fp, "<row %d>\n", i );
    SpScatterFromRow( B, i, &j, &c, &p );
    while (j--) 
	fprintf( fp, "[%d] = %e\n", *c++, *p++ ); 
    }
}

/*@ 
   SpNz - Returns the number of non-zeros in a sparse matrix.

   Input Parameter:
.  B - matrix
 @*/
int SpNz( B )
SpMat *B;
{
int     row, nz, n, nnz;

SPLITTOMAT(B);

if (B->type == MATDENSE) return B->rows * B->cols;

/* We should try to use the nz field; if we don't, we should set it */    
if (B->nz > 0) return B->nz;
nz = 0;
n  = B->rows;
for (row=0; row<n; row++) {
    SpScatterFromRow( B, row, &nnz, (int **)0, (double **)0 );
    nz += nnz;
    }
/* Update the nz field */    
B->nz = nz;
return nz;
}

/*@
   SpGetSpaceUsed - Returns the amount of space used by the sparse matrix pool
                    and any other data areas.

   Input Parameter:
.  B - matrix

   Returns:
   Number of bytes used by the matrix pool for a given matrix.
 @*/
int SpGetSpaceUsed( B )
SpMat *B;
{
SPLITTOMAT(B);
return SPiPoolSize( &B->pool );
}


/*@
  SpPrintPattern - Prints the non-zero pattern of a sparse matrix.

  Input Parameters:
. fp  - FILE pointer to write to
. mat - matrix to print out

  Note:
  The format of the output is blanks for zeros and +'s for non-zeros
 @*/
void SpPrintPattern( fp, mat )
FILE  *fp;
SpMat *mat;
{
int      i, mcol;
char     *rbuf;
register char *p;
int      nz, *xi;

SPLITTOMAT(mat);
mcol = mat->cols;
rbuf = (char *)MALLOC( mcol+1 ); CHKPTR(rbuf);
rbuf[mcol] = '\0';
p    = rbuf; 
nz   = mcol;
SET(p,nz,' ');

for (i=0; i<mat->rows; i++) {
    SpScatterFromRow( mat, i, &nz, &xi, (double **)0 );

    /* Set the indicated column locations to "+" */
    p  = rbuf;
    /* Must NOT use version of SCATTERVAL that does not preserve args */
    SCATTERVAL(p,xi,nz,'+');
    fputs( p, fp );
    fputc( '\n', fp );

    /* Restore blanks */
    p  = rbuf;
    SCATTERVAL(p,xi,nz,' ');
    }
FREE( rbuf );
}

/*@
   SpPrintSparseMatlab - Prints a sparse matrix to a given FILE, in
   sparse MATLAB format.

   Input parameters:
.  B    - matrix to print
.  fp   - FILE pointer. Use stdout for standard output.
.  name - name to give matrix in matlab
 @*/
void SpPrintSparseMatlab( fd, B, name )
FILE  *fd;
SpMat *B;
char  *name;
{
int      i, nz, *ix;
double   *p;

SPLITTOMAT(B);

if (fd == 0) fd = stdout;
nz = SpNz( B );
fprintf(fd,"%% Size = %d %d \n",B->rows,B->cols);
fprintf(fd,"%% Nonzeros = %d \n",nz);
fprintf(fd,"zzz = zeros(%d,3);\n",nz);
fprintf(fd,"zzz = [\n");

for (i=0; i<B->rows; i++) {
    SpScatterFromRow( B, i, &nz, &ix, &p );
    while (nz--) {
	fprintf(fd,"%d %d  %18.16e\n", i+1, 1 + *ix++, *p++ );
	}
    }     

fprintf(fd,"];\n %s = spconvert(zzz);\n",name);
fflush(fd);
}

/*@
   SpPrintSparseMatlabPermuted - Prints a sparse matrix to a given FILE,
   in sparse MATLAB format.

   Input parameters:
.  B    - matrix to print
.  fp   - FILE pointer. Use stdout for standard output.
.  name - name to give matrix in matlab
.  rperm - row permutation
.  cperm - column permutation 
 @*/
void SpPrintSparseMatlabPermuted( fd, B, name, rperm, cperm )
FILE  *fd;
SpMat *B;
char  *name;
int   *rperm,*cperm;
{
int      i, nz, *ix, *irperm, *icperm;
double   *p;

SPLITTOMAT(B);

irperm = (int *) MALLOC( B->rows*sizeof(int) ); CHKPTR(irperm);
icperm = (int *) MALLOC( B->cols*sizeof(int) ); CHKPTR(icperm);
SpInverse(B->rows, rperm, irperm);
SpInverse(B->cols, cperm, icperm);

if (fd == 0) fd = stdout;
nz = SpNz( B );
fprintf(fd,"%% Size = %d %d \n",B->rows,B->cols);
fprintf(fd,"%% Nonzeros = %d \n",nz);
fprintf(fd,"zzz = zeros(%d,3);\n",nz);
fprintf(fd,"zzz = [\n");
 
for (i=0; i<B->rows; i++) {
    SpScatterFromRow( B, i, &nz, &ix, &p );
    while (nz--) {
        fprintf(fd,"%d %d  %18.16e\n", irperm[i]+1, icperm[*ix++]+1, *p++ );
	}
    }     

fprintf(fd,"];\n %s = spconvert(zzz);\n",name);
fflush(fd);
FREE(irperm); FREE(icperm);
}
/*@
   SpPrintMatlab - Prints a sparse matrix to a given FILE, in MATLAB format.

   Input parameters:
.  B    - matrix to print
.  fp   - FILE pointer. Use stdout for standard output.
.  name - name to give matrix in matlab
 @*/
void SpPrintMatlab( fp, B, name )
FILE  *fp;
SpMat *B;
char  *name;
{
int    i, j, k, *c;
double   *p;

if (fp == 0) fp = stdout;

SPLITTOMAT(B);
fprintf( fp, "%s = [", name );
for (i=0; i<B->rows; i++) {
    SpScatterFromRow( B, i, &j, &c, &p );
    for (k=0; k<B->cols; k++) {
    	if (j && k == *c) { fprintf( fp, "%e ", *p++ ); c++; j--; }
    	else           fprintf( fp, "0.0 " );
        }
    if (i < B->rows - 1)
        fprintf( fp, ";\n" );
    }
fprintf( fp, "]\n" );
fflush(fp);
}

/*@
  SpPrintAsDense - Prints the sparse matrix as if it were dense

  Input Parameters:
. fp  - FILE pointer to write to
. mat - matrix to print out

  Note:
  The format of the output is blanks for zeros and +'s for non-zeros
 @*/
void SpPrintAsDense( fp, mat )
FILE  *fp;
SpMat *mat;
{
int      i, j, mcol;
double   *rbuf;
double   *p, *xv;
int      nz, *xi;

SPLITTOMAT(mat);
mcol = mat->cols;
rbuf = (double *) MALLOC( mcol * sizeof(double) ); CHKPTR(rbuf);
p    = rbuf; 
nz   = mcol;
SET(p,nz,0.0);

for (i=0; i<mat->rows; i++) {
    /* Set the indicated column locations to "+" */
    SpScatterFromRow( mat, i, &nz, &xi, &xv );
    SCATTER(xv,xi,rbuf,nz);
    for (j=0; j<mcol; j++) 
	fprintf( fp, "%12.5e ", rbuf[j] );
    fputc( '\n', fp );

    /* Restore values */
    SCATTERVAL(rbuf,xi,nz,0.0);
    }
FREE( rbuf );
}

/* This is really a private structure for SpRead and SpWrite.
   Eventually, there will be a routine for each data type. */
typedef struct {
    int type, nrows, ncols, nz, bsize;
    } SPDUMP;

/* Eventually, these read/write routines should go to system with 
   some better error handling */
/* This is a routine that writes data out, checking for errors and
   writing multiple blocks as required. Returns 0 on success, != 0 on failure
 */
int SpiSafeWrite( fd, p, n )
int  fd, n;
char *p;
{
int err, maxblock, wsize;

maxblock = 65536;
/* Write the data in blocks of 65536 (some systems don't like large writes;
   SunOS is a particular example) */
while (n) {
    wsize = (n < maxblock) ? n : maxblock;
    err = write( fd, p, wsize );
    if (err < 0 && errno == EINTR) continue;
    if (err != wsize) {
	SETERRC( 1, "Error in writing to file" );
	return 1;
	}
    n -= wsize;
    p += wsize;
    }
return 0;
}

/* This is a routine that reads data in, checking for errors and
   reading multiple blocks as required. Returns 0 on success, != 0 on failure
 */
int SpiSafeRead( fd, p, n )
int  fd, n;
char *p;
{
int maxblock, wsize, err;

maxblock = 65536;
/* Read the data in blocks of 65536 (some systems don't like large reads;
   SunOS is a particular example) */
while (n) {
    wsize = (n < maxblock) ? n : maxblock;
    err = read( fd, p, wsize );
    if (err < 0 && errno == EINTR) continue;
    if (err != wsize) {
	SETERRC( 1, "Error in reading from file" );
	return 1;
	}
    n -= wsize;
    p += wsize;
    }
return 0;
}

/* 
   This is a raw, aij format routine to write the data out
 */
void SpiWrite( fd, type, rows, cols, nz, ia, ja, a )
int    fd, type, rows, cols, nz, *ia, *ja;
double *a;
{
SPDUMP header;

header.type = type;
header.nrows= rows;
header.ncols= cols;
header.nz   = nz;
header.bsize= 1;
if (SpiSafeWrite( fd, (char *)&header, sizeof(SPDUMP) ))
    return;
if (SpiSafeWrite( fd, (char *)ia, (rows + 1) * sizeof(int) ))
    return;
if (SpiSafeWrite( fd, (char *)ja, nz * sizeof(int) ))
    return;
if (SpiSafeWrite( fd, (char *)a,  nz * sizeof(double) ))
    return;
}

/*@
    SpWrite - Writes a matrix out in binary form. Use SpRead() to read 
              it back in.

    Input parameters:
.   fd   - file descriptor (not FILE pointer).  Use open() for this.
.   mat  - matrix to write out

@*/    
void SpWrite( fd, mat )
int   fd;
SpMat *mat;
{
int    *ia, *ja, nz, nr;
double *a;

nr = mat->rows;
/* If the matrix is ALREADY in AIJ format, then don't reformat it */
if (mat->type == MATAIJ) {
    SpAIJ    *amat = (SpAIJ *)mat->data;
    SpiWrite( fd, mat->type, mat->rows, mat->cols, 
	      amat->ia[mat->rows]-1, amat->ia, amat->ja, amat->a );
    }
else {
    /* Get work spaces */	
    nz = SpNz( mat );
    a  = (double *)MALLOC( nz * sizeof(double) );   CHKPTR(a);
    ia = (int *)MALLOC( (nr+1 + nz) * sizeof(int) );CHKPTR(ia);
    ja = ia + nr + 1;
    
    /* Convert the matrix */
    SpToAIJ( mat, ia, ja, a, nz );
    SpiWrite( fd, mat->type, mat->rows, mat->cols, nz, ia, ja, a );
    
    FREE( ia );
    FREE( a );
    }
}


/*@
    SpRead - Reads a matrix in binary form. Use SpWrite() to write the 
             matrix out.

    Input parameters:
.   fd   - file descriptor (not FILE pointer).  Use open() for this.

    Returns:
.   matrix
    
@*/    
SpMat *SpRead( fd )
int   fd;
{
int    *ia, *ja, nz, nr;
double *a;
SPDUMP header;
SpMat  *mat;

if (SpiSafeRead( fd, (char *)&header, sizeof(SPDUMP) ) )
    return 0;
nr  = header.nrows;
nz  = header.nz;

/* Get work spaces */	
a  = (double *)MALLOC( nz * sizeof(double) );   CHKPTRV(a,0);
ia = (int *)MALLOC( (nr+1 + nz) * sizeof(int) );CHKPTRV(ia,0);
ja = ia + nr + 1;

/* read the rest of the matrix */
if (SpiSafeRead( fd, (char *)ia, (nr+1+nz)*sizeof(int) ) )
    return 0;
if (SpiSafeRead( fd, (char *)a,  nz*sizeof(double) ) )
    return 0;

/* This ALWAYS produces a ROW format matrix.  We'll eventually need to 
   fix this */
mat = SpFromAIJ( ia, ja, a, nr );
FREE( ia );
FREE( a );

return mat;
}

#ifndef SEEK_SET
#define	SEEK_SET	0	/* Set file pointer to "offset" */
#define	SEEK_CUR	1	/* Set file pointer to current plus "offset" */
#define	SEEK_END	2	/* Set file pointer to EOF plus "offset" */
#endif

/*@
    SpReadRowsAIJ - Reads a matrix in binary form, reading only selected rows.

    Input parameters:
.   fd   - file descriptor (not FILE pointer).  Use open() for this.
.   rows - indices of rows
.   nnr  - number of rows

    Returns:
.   matrix
    
    Notes:
    This routine is intended for the case where a large matrix is stored on 
    disk and only part of it is desired.  For example, a parallel solver may
    access only some of the rows from each processor.  The algorithm used here
    reads relatively small blocks of data rather than reading the entire 
    matrix and then subsetting it.  

    For this version, storage proportional to the number of rows in the full
    matrix is used.

    The file is positioned at the first byte after the matrix record on 
    successful exit.

    This may be a temporary until a more general read-into-specified-form
    routine is created.
@*/    
SpMat *SpReadRowsAIJ( fd, rows, nnr )
int   fd, *rows, nnr;
{
int    *ia, *ja, nz, nr, *iia, nnz, i, j, jj;
double *a;
SPDUMP header;
SpMat  *mat;
long   startloc;


if (SpiSafeRead( fd, (char *)&header, sizeof(SPDUMP) ) )
    return 0;
nr  = header.nrows;
nz  = header.nz;

/* Get the location of the beginning of the matrix data (in case the file
   contains multiple elements) */
startloc = lseek( fd, 0L, SEEK_CUR );  

/* Read the ia array and collect the necessary data */
/* Get work spaces */	
ia = (int *)MALLOC( (nr+1) * sizeof(int) );  CHKPTRV(ia,0);
/* iia is the compressed row storage */
iia= (int *)MALLOC( (nnr+1) * sizeof(int) ); CHKPTRV(iia,0);

if (SpiSafeRead( fd, (char *)ia, (nr+1)*sizeof(int) ) )
    return 0;
/* count the number of nonzeros that we actually need */
nnz    = 0;
j      = 0;
iia[0] = 1;
for (i=0; i<nr; i++) {
    if (rows[j] == i) {
	nnz += (ia[i+1] - ia[i]);
	j++;
	iia[j] = nnz + 1;
	if (j >= nnr) break;
	}
    }

/* read the rest of the matrix */
a  = (double *)MALLOC( nnz * sizeof(double) );   CHKPTRV(a,0);
ja = (int *)MALLOC( nnz * sizeof(int) );         CHKPTRV(ja,0);

/* Inorder to maximize the use of a disk cache, we read the ja vector first,
   then the a vector */
j   = 0;
SpiReadBuffer( -1, 0, 0, (char *)0 );
i   = 0;
while (i < nr) {
    if (rows[j] == i) {
	/* Position and read the columns for this row.
	   We refer to a routine that tries to buffer the data */
	/* Try to read multiple lines */
	jj = 1;
	while (jj < nnr && rows[j+jj] == i + jj) jj++;
	if (SpiReadBuffer( fd, startloc + (nr+1+(ia[i]-1)) * sizeof(int), 
			   (ia[i+jj]-ia[i]) * sizeof(int), 
			   (char *)(ja + iia[j]-1)) )
	    return 0;
	j += jj;
	i += jj;
	if (j >= nnr) break;
	}
    else 
	i++;
    }
j = 0;
SpiReadBuffer( -1, 0, 0, (char *)0 );
i = 0;
while (i < nr) {
    if (rows[j] == i) {
	/* Position and read the values for this row.
	   We refer to a routine that tries to buffer the data */
	/* Try to read multiple lines */
	jj = 1;
	while (jj < nnr && rows[j+jj] == i + jj) jj++;
	if (SpiReadBuffer( fd, startloc + (nr+1+nz) * sizeof(int) + 
			   (ia[i]-1) * sizeof(double), 
			   (ia[i+jj]-ia[i]) * sizeof(double), 
			   (char *)(a + iia[j]-1)) )
	    return 0;
	j += jj;
	i += jj;
	if (j >= nnr) break;
	}
    else 
	i++;
    }

/* This ALWAYS produces a ROW format matrix.  We'll eventually need to 
   fix this */
mat       = SpAIJCreateFromData( nnr, nr, iia, ja, a, nz );
((SpAIJ *)(mat->data))->quser = 0;     /* Change to PETSc allocated */

FREE( ia );

lseek( fd, startloc + (nr + 1 + nz) * sizeof(int) + nz * sizeof(double), 
       SEEK_SET );  

return mat;
}

/*@
    SpReadRows - Reads a matrix in binary form, reading only selected rows.

    Input parameters:
.   fd   - file descriptor (not FILE pointer).  Use open() for this.
.   rows - indices of rows
.   nnr  - number of rows

    Returns:
.   matrix
    
    Notes:
    This routine is intended for the case where a large matrix is stored on 
    disk and only part of it is desired.  For example, a parallel solver may
    access only some of the rows from each processor.  The algorithm used here
    reads relatively small blocks of data rather than reading the entire 
    matrix and then subsetting it.  

    For this version, storage proportional to the number of rows in the full
    matrix is used.

    The file is positioned at the first byte after the matrix record on 
    successful exit.
@*/    
SpMat *SpReadRows( fd, rows, nnr )
int   fd, *rows, nnr;
{
int    *ia, *ja, nz, nr, *iia, nnz, i, j;
double *a;
SPDUMP header;
SpMat  *mat;
long   startloc;


if (SpiSafeRead( fd, (char *)&header, sizeof(SPDUMP) ) )
    return 0;
nr  = header.nrows;
nz  = header.nz;

/* Get the location of the beginning of the matrix data (in case the file
   contains multiple elements) */
startloc = lseek( fd, 0L, SEEK_CUR );  

/* Read the ia array and collect the necessary data */
/* Get work spaces */	
ia = (int *)MALLOC( (nr+1) * sizeof(int) );  CHKPTRV(ia,0);
/* iia is the compressed row storage */
iia= (int *)MALLOC( (nnr+1) * sizeof(int) ); CHKPTRV(iia,0);

if (SpiSafeRead( fd, (char *)ia, (nr+1)*sizeof(int) ) )
    return 0;
/* count the number of nonzeros that we actually need */
nnz    = 0;
j      = 0;
iia[0] = 1;
for (i=0; i<nr; i++) {
    if (rows[j] == i) {
	nnz += (ia[i+1] - ia[i]);
	j++;
	iia[j] = nnz + 1;
	if (j >= nnr) break;
	}
    }

/* read the rest of the matrix */
a  = (double *)MALLOC( nnz * sizeof(double) );   CHKPTRV(a,0);
ja = (int *)MALLOC( nnz * sizeof(int) );         CHKPTRV(ja,0);

/* Inorder to maximize the use of a disk cache, we read the ja vector first,
   then the a vector */
j   = 0;
SpiReadBuffer( -1, 0, 0, (char *)0 );
for (i=0; i<nr; i++) {
    if (rows[j] == i) {
	/* Position and read the columns for this row.
	   We refer to a routine that tries to buffer the data */
	if (SpiReadBuffer( fd, startloc + (nr+1+(ia[i]-1)) * sizeof(int), 
			   (ia[i+1]-ia[i]) * sizeof(int), 
			   (char *)(ja + iia[j]-1)) )
	    return 0;
	j++;
	if (j >= nnr) break;
	}
    }
j = 0;
SpiReadBuffer( -1, 0, 0, (char *)0 );
for (i=0; i<nr; i++) {
    if (rows[j] == i) {
	/* Position and read the values for this row.
	   We refer to a routine that tries to buffer the data */
	if (SpiReadBuffer( fd, startloc + (nr+1+nz) * sizeof(int) + 
			   (ia[i]-1) * sizeof(double), 
			   (ia[i+1]-ia[i]) * sizeof(double), 
			   (char *)(a + iia[j]-1)) )
	    return 0;
	j++;
	if (j >= nnr) break;
	}
    }

/* This ALWAYS produces a ROW format matrix.  We'll eventually need to 
   fix this */
mat       = SpFromAIJ( iia, ja, a, nnr );
mat->cols = nr;
FREE( iia );
FREE( ia );
FREE( ja );
FREE( a );

lseek( fd, startloc + (nr + 1 + nz) * sizeof(int) + nz * sizeof(double), 
       SEEK_SET );  

return mat;
}

/* Routine to read data from a file fd at:

   startloc + (nr+1) * sizeof(int) + (so-1)*sizeof(int)        into ja
   startloc + (nr+1 + nz)* sizeof(int) + (so-1)*sizeof(double) into a

   For a first pass, we just seek and read.  Eventually, we want to buffer 
   and reuse (should make buffers lie on 4k boundaries relative to 0, not
   to startloc)

   An fd = -1 clears the existing buffers 
 */
int SpiReadBuffer( fd, startloc, n, p )
int    fd, n;
long   startloc;
char   *p;
{
if (fd < 0) {
    return 0;
    }

lseek( fd, startloc, SEEK_SET );
if (SpiSafeRead( fd, p, n ))
    return 1;
return 0;
}

