/*
     Contains the main routines for the SV package. Creates and 
  sets up the SVctx variable for a linear system solve.
*/
#include "solvers/svctx.h"
#include "solvers/svpriv.h"
#include <math.h>
#include "system/nreg.h"
extern NRList *__SVList;

/* Default routine to build the solution */
void  *SViStandardBuildSolution( itP, ctx, v )
ITCntx *itP;
SVctx   *ctx;
void    *v;
{
  return (*itP->BuildSolution)(itP, (void *) ctx, v);
}

/* Initialize fields in a context */
void SViInitCtx( ctx, mat )
SVctx *ctx;
SpMat *mat;
{
  ctx->size        = mat->rows;
  ctx->mat         = mat;
  ctx->amat        = mat;
  ctx->setupcalled = 0;
  ctx->solvecalled = 0;
  ctx->BuildSolution = SViStandardBuildSolution;

  ctx->method      = -1;
  ctx->symmetric   = 0;
  ctx->itctx       = 0;

  ctx->flops       = 0;
  ctx->memory      = 0;
  ctx->nz          = 0;
  ctx->t_setup     = 0;
  ctx->t_solve     = 0;
  ctx->namult      = 0;
  ctx->nbinv       = 0;
  ctx->is_iter     = 0;

  ctx->natural_blocksize = 0;

  ctx->use_initial_guess = 0;

  ctx->ns          = 0;      /* null space */
}

/*@
   SVCreate - Given a sparse matrix, creates as SVctx structure;
   this structure will then be used in solving linear systems. 
  
   Input Parameters:
.   mat - matrix
.   name - name of method to use
$       SVLU - direct factorization 
$       SVJacobi - Jacobi preconditioning
$       SVSSOR - SSOR preconditioning using the Eisenstat trick 
$       SVILU - incomplete LU preconditioning  
$       SCICC, SVICCJP - 2 different incomplete Choleski precondioners  
                         (for symmetric problems only)      
$       SVBDD - block diagonal preconditioning
$       SVOSM - overlapping Schwarz preconditioning
$       SVNOPRE - no preconditioning at all

   Returns:
   Solver context
@*/
SVctx *SVCreate(mat,name)
SpMat    *mat;
SVMETHOD name;
{
  SVctx   *ctx;
  void    (*r)();

  ctx = NEW(SVctx); CHKPTRV(ctx,0); 

  /* Get the function pointers for the solver name requested */
  if (!__SVList) {SVRegisterAll();} 
  if (!__SVList) {SETERRC(1,"Could not acquire list of solvers"); return 0;}
  r =  NRFindRoutine( __SVList, (int)name, (char *)0 );
  if (!r) {SETERRC(1,"Unknown SV method"); return 0;}

  SViInitCtx( ctx, mat );
  ctx->type        = name;

  (*r)(ctx,mat);
  return ctx;
}

/*
   This routine may be used by all of the iterative methods to 
   handle the setup of the initial guess.  For simplicity, we 
   set as the default that the initial guess is zero and we set it
   to that just to make sure.  The iterative routines are told
   that the initial guess is zero; they may use this to speed up
   the solution process (for example, the initial residual is computed
   directly from the right-hand-side, without using "x" at all).

   This is provided because people using the Solver's interface will
   USUALLY be expecting a zero initial guess; in trials, this has
   been a problem.  For historical reasons, the iterative code
   uses the opposite interpretation.
 */
#include "inline/setval.h"

void SViManageInitialGuess( ctx, x )
SVctx  *ctx;
double *x;
{
if (ctx->itctx) {
    if (!ctx->use_initial_guess) {
	SET( x, ctx->size, 0.0 );
	ITSetInitialGuessZero( ctx->itctx );
	}
    else ctx->itctx->guess_zero = 0;
    }
}

