#ifndef LINT
static char SCCSid[] = "@(#) ./iter/template/cg.c 07/23/93";
#endif

/*                       
       This implements BiCG-Stab.
       This is a simple template that may be used to create new methods
       You will need to add the method to 

       iter/itctx.h

       create a new context file (if there is a private context; 
       many methods do not require one)

       iter/<method>ctx.h

       and insert that into 

       iter/itall.h

       and add the extern declarations of the create, solve, and setup 
       routines to 

       iter/itfunc.h

*/
#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "iter/itctx.h"
#include "iter/itfunc.h"
#include "iter/itpriv.h"

/* @
ITBCGSCreate - Creates an ITCntx variable for the BiCG-Stab method.
The prefered calling sequence is ITCreate(ITBCGS);
@ */
ITCntx *ITBCGSCreate()
{
ITCntx    *itP;

itP = NEW(ITCntx);  CHKPTRV(itP,0);
ITSetDefaults( itP );
itP->MethodPrivate = (void *) 0;

itP->method               = ITBCGS;
itP->max_it               = 50;
itP->right_inv            = 0;
itP->calc_res             = 1;
itP->use_pres             = 0;
itP->guess_zero           = 0;
itP->rtol                 = 1.e-5;
itP->atol                 = 1.e-50;
itP->usr_monitor          = ITDefaultMonitor;
itP->residual_history     = NULL;

itP->setup                = ITBCGSSetUp;
itP->solver               = ITBCGSSolve;
itP->adjustwork           = ITDefaultAdjustWork;
itP->closedown            = ITDefaultDestroy;
return(itP);
}

/* @
ITBCGSSetUp - Called after a call to ITBCGSCreate() or ITCREATE(ITBCGS)
allocates space needed in the conjugate gradient solution. Preferred
calling sequence is ITSetUp(itP,usrP).

Input Parameters: 
.   itP - the iterative context
.   usrP - the user context
@ */
void ITBCGSSetUp(itP,usrP)
ITCntx *itP;
void   *usrP;
{
if (itP->method != ITBCGS) {
   SETERRC(1,"Attempt to use BCGS Setup on wrong context"); return;}

/* check user parameters and functions */
if (ITCheckDef( itP )) return;

/* get work vectors from user code */
ITDefaultGetWork( itP, usrP, 7 );
}

/* @
ITBCGSSolve - Called after a call to ITBCGSCreate() or ITCreate(ITBCGS)
and the call to ITBCGSSetUp() or ITSetUp().
Actually solves the linear system using the BiCG-stab  
method. Prefered calling sequence ITSolve(itP,usrP).

Input Parameters: 
.   itP  - the iterative context
.   usrP - the user context

Returns:
the number of iterations required or -1 on error.
@ */
int  ITBCGSSolve(itP, usrP)
void   *usrP;
ITCntx *itP;
{
int       i = 0, maxit, res, pres, cerr;
double    rho, rhoold, alpha, beta, omega, omegaold, *history, dp, d1, d2; 
void      *X,*B,*V,*P,*R,*RP,*T,*S, *TT, *BINVF;

res     = itP->calc_res;
pres    = itP->use_pres;
maxit   = itP->max_it;
history = itP->residual_history;
X       = itP->vec_sol;
B       = itP->vec_rhs;
R       = itP->work[0];
RP      = itP->work[1];
V       = itP->work[2];
T       = itP->work[3];
S       = itP->work[4];
P       = itP->work[5];
BINVF   = itP->work[6];

/* Compute initial preconditioned residual */
ITResidual(usrP,itP,X,V,T, R, BINVF, B );

/* Test for nothing to do */
NORM(R,&dp);
if (CONVERGED(dp,0)) return RCONV(0);
MONITOR(dp,0);
if (history) history[0] = dp;

/* Make the initial Rp == R */
COPY(R,RP);

rhoold   = 1.0;
alpha    = 1.0;
omegaold = 1.0;
SET(0.0,P);
SET(0.0,V);

for (i=0; i<maxit; i++) {
    DOT(R,RP,&rho);                       /*   rho <- rp' r     */
    beta = (rho/rhoold) * (alpha/omegaold);
    DAXPY(-omegaold,V,P);                 /*     p <- p - w v   */
    DAYPX(beta,R,P);                      /*     p <- r + p beta */
    MATOP(P,V,T);                         /*     v <- K p       */
    DOT(RP,V,&d1);
    alpha = rho / d1;                     /*     a <- rho / (rp' v) */
    COPY(R,S);
    DAXPY(-alpha,V,S);                    /*     s <- r - a v   */
    MATOP(S,T,R);                         /*     t <- K s       */
    DOT(S,T,&d1);
    DOT(T,T,&d2);
    omega = d1 / d2;                      /*     w <- (s't) / (t't) */
    DAXPY(alpha,P,X);                     /*     x <- x + a p   */
    DAXPY(omega,S,X);                     /*     x <- x + w s   */
    COPY(S,R);
    DAXPY(-omega,T,R);                    /*     r <- s - w t   */
    NORM(R,&dp);

    rhoold   = rho;
    omegaold = omega;

    if (history) history[i+1] = dp;
    MONITOR(dp,i+1);
    if (CONVERGED(dp,i+1)) break;
    }
if (i == maxit) i--;

ITUnwindPre( usrP, itP, X, T );
return RCONV(i+1);
}
