#ifndef lint
static char SCCSid[] = "@(#) ./iter/lsqr/lsqr.c 07/23/93";
#endif

#define SWAP(a,b,c) { c = a; a = b; b = c; }

/*                       
       This implements LSQR (Paige and Saunders, ACM Transactions on
       Mathematical Software, Vol 8, pp 43-71, 1982).

       and add the extern declarations of the create, solve, and setup 
       routines to 

       iter/itfunc.h

*/
#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "iter/itctx.h"
#include "iter/itfunc.h"
#include "iter/itpriv.h"

/*+
ITLSQRCreate - Creates an ITCntx variable for the LSQR method.
The prefered calling sequence is ITCreate(ITLSQR);
+*/
ITCntx *ITLSQRCreate()
{
ITCntx    *itP;

itP = NEW(ITCntx);  CHKPTRV(itP,0);
ITSetDefaults( itP );
itP->MethodPrivate = (void *) 0;

itP->method               = ITLSQR;
itP->max_it               = 50;
itP->right_inv            = 0;
itP->calc_res             = 1;
itP->use_pres             = 0;
itP->guess_zero           = 0;
itP->rtol                 = 1.e-5;
itP->atol                 = 1.e-50;
itP->usr_monitor          = ITDefaultMonitor;
itP->residual_history     = NULL;

itP->setup                = ITLSQRSetUp;
itP->solver               = ITLSQRSolve;
itP->adjustwork           = ITDefaultAdjustWork;
itP->closedown            = ITDefaultDestroy;
return(itP);
}

/*+
ITLSQRSetUp - Called after a call to ITLSQRCreate() or ITCREATE(ITLSQR)
allocates space needed in the conjugate gradient solution. Preferred
calling sequence is ITSetUp(itP,usrP).

Input Parameters: 
.   itP - the iterative context
.   usrP - the user context
+*/
void ITLSQRSetUp(itP,usrP)
ITCntx *itP;
void   *usrP;
{
if (itP->method != ITLSQR) {
   SETERRC(1,"Attempt to use LSQR Setup on wrong context"); return;}

/* check user parameters and functions */
if (ITCheckDef( itP )) return;
if (!itP->tamult) {
    SETERRC(1,"LSQR requires matrix-transpose * vector"); return;}

/* get work vectors from user code */
ITDefaultGetWork( itP, usrP, 6 );
}

/*+
ITLSQRSolve - Called after a call to ITLSQRCreate() or ITCreate(ITLSQR)
and the call to ITLSQRSetUp() or ITSetUp().
Actually solves the linear system using the LSQR
method. Prefered calling sequence ITSolve(itP,usrP).

Input Parameters: 
.   itP  - the iterative context
.   usrP - the user context

Returns:
the number of iterations required or -1 on error.

Note:
This method does not currently accept any preconditioners OR initial
guesses.
+*/
int ITLSQRSolve(itP, usrP)
void   *usrP;
ITCntx *itP;
{
int       i = 0, maxit, res, pres, hist_len, cerr;
double    rho, rhobar, phi, phibar, theta, c, s, beta, alpha, rnorm, *history;
void      *X,*B,*V,*V1,*U,*U1,*TMP,*W,*BINVF;

res     = itP->calc_res;
pres    = itP->use_pres;
maxit   = itP->max_it;
history = itP->residual_history;
hist_len= itP->res_hist_size;
X       = itP->vec_sol;
B       = itP->vec_rhs;
U       = itP->work[0];
U1      = itP->work[1];
V       = itP->work[2];
V1      = itP->work[3];
W       = itP->work[4];
BINVF   = itP->work[5];

/* Compute initial preconditioned residual */
ITResidual(usrP,itP,X,V,U, W, BINVF, B );

/* Test for nothing to do */
NORM(W,&rnorm);
if (CONVERGED(rnorm,0)) return RCONV(0);
MONITOR(rnorm,0);
if (history) history[0] = rnorm;

COPY(B,U);
NORM(U,&beta);
SCALE( 1.0/beta, U );

(*itP->tamult)( usrP, U, V );
NORM(V,&alpha);
SCALE( 1.0/alpha, V );

COPY(V,W);
SET(0.0,X);

phibar = beta;
rhobar = alpha;
for (i=0; i<maxit; i++) {
    MM(V,U1);
    DAXPY(-alpha,U,U1);
    NORM(U1,&beta);
    SCALE( 1.0 / beta, U1 );

    MTM(U1,V1);
    DAXPY(-beta,V,V1);
    NORM(V1,&alpha);
    SCALE( 1.0 / alpha, V1 );

    rho   = sqrt(rhobar*rhobar + beta*beta);
    c     = rhobar / rho;
    s     = beta / rho;
    theta = s * alpha;
    rhobar= - c * alpha;
    phi   = c * phibar;
    phibar= s * phibar;

    DAXPY(phi/rho,W,X);                   /*    x <- x + (phi/rho) w   */
    DAYPX(-theta/rho,V1,W);               /*    w <- v - (theta/rho) w */

    rnorm = phibar;

    if (history && hist_len > i + 1) history[i+1] = rnorm;
    MONITOR(rnorm,i+1);
    if (CONVERGED(rnorm,i+1)) break;
    SWAP( U1, U, TMP );
    SWAP( V1, V, TMP );
    }
if (i == maxit) i--;
if (history) itP->res_act_size = (hist_len < i + 1) ? hist_len : i + 1;

ITUnwindPre( usrP, itP, X, W );
return RCONV(i+1);
}
