/* @(#) ./iter/itfunc.h 07/23/93 */

/* --------------Routines to Initialize and run a Solver-----------------*/

extern ITCntx *ITCreate(/* int NameofSolver */);
extern ITCntx *ITRichardsonCreate();
extern ITCntx *ITChebychevCreate();
extern ITCntx *ITCGCreate();
extern ITCntx *ITGMRESCreate();
extern ITCntx *ITTCQMRCreate();
extern ITCntx *ITBCGSCreate();
extern ITCntx *ITCGSCreate();
extern ITCntx *ITTFQMRCreate();
extern ITCntx *ITLSQRCreate();
extern ITCntx *ITSimpleCreate();
extern ITCntx *ITPREONLYCreate();

extern void ITRegisterAll();
extern void ITRegister();
extern void ITSetDefaults();
extern void *ITDefaultBuildSolution();
extern void *ITDefaultBuildResidual();

extern char *ITGetMethodName();
int  ITResidual();

/* Default routines to acquire work vectors */
extern void ITDefaultGetWork();
extern void ITDefaultAdjustWork();
extern void ITDefaultFreeWork();
extern void ITDefaultDestroy();

extern void ITGetWorkCounts(), ITClearWorkCounts();

/*M
   ITSetUp - Sets up the internal data structures for the
   later use of an iterative solver.

   Input Parameters:
.   itP   - iterative context obtained from ITCreate()
.   usrP  - user context

   Synopsis:
   void ITSetUp( itP, usrP )
   ITCntx *itP;
   void   *usrP;
M*/
#define ITSetUp(itP,usrP)      (*(itP)->setup)(itP,usrP)

/*M
   ITSolve - Solves linear system, call it after calling 
   ITCreate(), ITSetup(), and ITSet*().

   Input Parameters:
.   itP - Iterative context obtained from ITCreate
.   usrP - User context

   Returns:
   The number of iterations required.  If the return is negative, the 
   iterations were aborted by the convergence tester (if the default
   convergence test is used, this happens when the residual grows to more
   than 10000 times the initial residual).

   Synopsis:
   int ITSolve( itP, usrP )
   ITCntx *itP;
   void   *usrP;
M*/
#define ITSolve(itP,usrP)      (*(itP)->solver)(itP,usrP)

/*M
   ITDestroy Destroys ITCntx created with ITCreate().

   Input Parameters:
.   itP   - iterative context obtained from ITCreate
.   usrP  - user context

   Synopsis:
   void ITDestroy( itP, usrP )
   ITCntx *itP;
   void   *usrP;
M*/
#define ITDestroy(itP,usrP)    (*(itP)->closedown)(itP,usrP)
#define ITAdjustWork(itP,usrP) (*(itP)->adjustwork)(itP,usrP)

void ITCGSetUp( /* itP, usrP */ );
int  ITCGSolve( /* itP, usrP */ );
void ITCGGetEigenvalues(  );
void ITCGDestroy( /* itP, usrP */ );
void ITCGAdjustWork( /* itP, usrP */ );
int  ITCGDefaultConverged();

void ITRichardsonSetUp( /* itP, usrP */ );
int  ITRichardsonSolve( /* itP, usrP */ );
void ITRichardsonDestroy( /* itP, usrP */ );
void ITRichardsonAdjustWork( /* itP, usrP */ );

void ITChebychevSetUp( /* itP, usrP */ );
int  ITChebychevSolve( /* itP, usrP */ );
void ITChebychevDestroy( /* itP, usrP */ );
void ITChebychevAdjustWork( /* itP, usrP */ );

void ITGMRESSetUp();
int  ITGMRESSolve();
void ITGMRESDestroy();
void ITGMRESAdjustWork();
void ITGMRESSetDirections();
int  ITGMRESDefaultConverged();

void ITBCGSSetUp( /* itP, usrP */ );
int  ITBCGSSolve( /* itP, usrP */ );

void ITCGSSetUp( /* itP, usrP */ );
int  ITCGSSolve( /* itP, usrP */ );

void ITTFQMRSetUp( /* itP, usrP */ );
int  ITTFQMRSolve( /* itP, usrP */ );

void ITLSQRSetUp( /* itP, usrP */ );
int  ITLSQRSolve( /* itP, usrP */ );

void ITPREONLYSetUp( /* itP, usrP */ );
int  ITPREONLYSolve( /* itP, usrP */ );

/* ------------Routines (or Macros) to set Solver Parameters------------*/

/*M
   ITSetIterations - Sets the maximum number of iterations to use.

   Input Parameters:
.   itP  - iterative context obtained from ITCreate()
.   maxits - maximum iterations to use

   Synopsis:
   void ITSetIterations( itP, its )
   ITCntx *itP;
   int    its;
M*/
#define ITSetIterations(itP, its)         (itP)->max_it     = its

/*M
    ITSetRightPreconditioner - Set right preconditioning

    Input Parameter:
.   itP - Iterative context obtained from ITCreate()

    Synopsis:
    void ITSetRightPreconditioner( itP )
    ITCntx *itP;

    Note:
    Left preconditioning is used by default.  Symmetric preconditioning is
    not currently available (note that it can be emulated by using either
    right or left preconditioning and a pre or post processing step).
M*/
#define ITSetRightPreconditioner(itP)     (itP)->right_inv  = 1

/*M
    ITGetPreconditionerSide - Get preconditioning side

    Input Parameter:
.   itP - Iterative context obtained from ITCreate()

    Synopsis:
    int ITGetPreconditionerSide( itP )
    ITCntx *itP;

    Returns:
    One for right preconditioning and zero for left preconditioning.
M*/
#define ITGetPreconditionerSide(itP)     (itP)->right_inv

/*M
    ITMethodFromCtx - Return the chosen method type

    Input Parameter:
.   itP - Iterative context obtained from ITCreate()

    Synopsis:
    int ITMethodFromCtx( itP )
    ITCntx *itP;

    Note:
    ITGetMethod gets the method from the command line.
M*/
#define ITMethodFromCtx( itP )               (itP)->method

/*M
   ITSetRelativeTolerance - Sets the convergence tolerance as a relative 
   decrease in the residual of tol.

   Input Parameters:
.   itP - Iterative context obtained from ITCreate()
.   tol - tolerance

   Synopsis:
   void ITSetRelativeTolerance( itP, tol )
   ITCntx *itP;
   double tol;
M*/
#define ITSetRelativeTolerance(itP, r)    (itP)->rtol       = r

/*M
   ITSetAbsoluteTolerance - Sets the convergence tolerance as an absolute 
   decrease in the residual of tol. Use ITSetRelativeTolerance() for 
   relative tolerance.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()
.   tol - tolerance

   Synopsis:
   void ITSetAbsoluteTolerance( itP, a )
   ITCntx *itP;
   double a;
M*/
#define ITSetAbsoluteTolerance(itP, a)    (itP)->atol       = a

/*M
   ITSetCalculateResidual - Sets a flag so that the two norm of the 
   residual is calculated at each iteration.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()

   Synopsis:
   void ITSetCalculateResidual( itP )
   ITCntx *itP;
M*/
#define ITSetCalculateResidual(itP)       (itP)->calc_res   = 1

/*M
   ITSetDoNotCalculateResidual - Sets a flag so that the two norm of the 
   residual is not calculated at each iteration.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()

   Synopsis:
   void ITSetDoNotCalculateResidual( itP )
   ITCntx *itP;

   Notes:
   Most Krylov methods do not yet take advantage of this flag.
M*/
#define ITSetDoNotCalculateResidual(itP)       (itP)->calc_res   = 0

/*M
   ITSetUsePreconditionedResidual - Sets a flag so that the two norm of the 
   preconditioned residual is used rather then the true residual.

   Input Parameters:
.   itP  - iterative context obtained from ITCreate()

   Synopsis:
   void ITSetUsePreconditionedResidual( itP )
   ITCntx *itP;

   Notes:
   This is ignored in methods where it is expensive to compute.  Currently, 
   only CG, CHEBYCHEV, and RICHARDSON use this.  The other methods always
   compute the preconditioned residual (or an approximation to it).
M*/
#define ITSetUsePreconditionedResidual(itP)       (itP)->use_pres   = 1

/*M
   ITSetInitialGuessZero - Tells the iterative solver
   that the initial guess is zero, otherwise it assumes it is 
   non-zero. If the initial guess is zero, this saves one matrix 
   multiply in the calculation of the initial residual.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()

   Synopsis:
   void ITSetInitialGuessZero( itP )
   ITCntx *itP;
M*/
#define ITSetInitialGuessZero(itP)       (itP)->guess_zero   = 1

/*M
   ITSetCalculateEigenvalues - Sets a flag so that the the method will
   calculate the extreme eigenvalues via a Lanczo of Arnoldi process
   as it solves the linear system.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()

   Synopsis:
   void ITSetCalculateEigenvalues( itP )
   ITCntx *itP;
M*/
#define ITSetCalculateEigenvalues(itP)    (itP)->calc_eigs  = 1

/*M
   ITSetRhs - Sets the right hand side for the linear system to
   be solved.

   Input Parameters:
.   itP - Iterative context obtained from ITCreate()
.   x   - the right hand side vector

   Synopsis:
   void ITSetRhs( itP, x )
   ITCntx *itP;
   void   *x;
M*/
#define ITSetRhs(itP,b)                   (itP)->vec_rhs    = (void*)(b)

/*M
   ITGetRhs - Gets the right hand side for the linear system to
   be solved.

   Input Parameter:
.   itP - Iterative context obtained from ITCreate()

   Synopsis:
   void *ITGetRhs( itP )
   ITCntx *itP;
M*/
#define ITGetRhs(itP)                   (itP)->vec_rhs

/*M
  ITVectorCtx - Return a pointer to the vector context in an iterative context

  Input Parameter:
.   itP - Iterative context obtained from ITCreate()

  Synopsis:
  VECntx *ITVectorCtx( itP )
  ITCntx *itP;
 M*/
#define ITVectorCtx(itP)                  (itP)->vc

/*M
   ITSetSolution - Sets the location of the solution for the 
   linear system to be solved.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()
.   x   - the solution vector

   Synopsis:
   void ITSetSolution( itP, b )
   ITCntx *itP;
   void   *b;
M*/
#define ITSetSolution(itP,b)              (itP)->vec_sol    = (void*)(b)

/*M
   ITGetSolution - Gets the location of the solution for the 
   linear system to be solved.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()

   Synopsis:
   void *ITGetSolution( itP )
   ITCntx *itP;
M*/
#define ITGetSolution(itP)              (itP)->vec_sol

/*M
   ITSetAmult - Sets the function which is to be used to calculate the 
   matrix vector product.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()
.   amult - pointer to void function

   Synopsis:
   void ITSetAmult( itP, amult )
   ITCntx *itP;
   void   (*amult)();
M*/
#define ITSetAmult(itP,a)                 (itP)->amult      = a

/*M
   ITSetBinv - Sets the function which is to be used to calculate the 
   application of the preconditioner on a vector.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()
.   binv - pointer to void function

   Synopsis:
   void ITSetBinv( itP, binv )
   ITCntx *itP;
   void   (*binv)();
M*/
#define ITSetBinv(itP,a)                  (itP)->binv       = a

/*M
   ITSetMatop - Sets the function which is to be used to calculate the 
   application of the preconditioner followed by the application of the 
   matrix multiplier on a vector.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()
.   matop - pointer to void function

   Synopsis:
   void ITSetMatop( itP, matop )
   ITCntx *itP;
   void   (*matop)();
M*/
#define ITSetMatop(itP,a)                 (itP)->matop      = a

/*M
   ITSetMonitor - Sets the function which is to be used at every
   iteration of the iterative solution. 

   Input Parameters:
.   itP - iterative context obtained from ITCreate()
.   monitor - pointer to void function
.   mctx    - context for private data for the monitor routine (may be null)

   Synopsis:
   void ITSetMonitor( itP, monitor, mctx )
   ITCntx *itP;
   void   (*monitor)(), *mctx;

   Notes:
   The default is to print the residual, or preconditioned residual if
   ITSetUsePreconditionedResidual was called.

M*/
#define ITSetMonitor(itP,a,b)      {(itP)->usr_monitor = a;(itP)->monP = b;}

/*M
   ITGetMonitorContext - Gets the context provided by ITSetMonitor.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()

   Synopsis:
   void *ITGetMonitorContext( itP )
   ITCntx *itP;
M*/
#define ITGetMonitorContext(itP)      ((itP)->monP)

/*M
   ITSetResidualHistory - Sets the array used to hold the residual history.

   Input Parameters:
.   itP - iterative context obtained from ITCreate()
.   a   - array to hold history
.   na  - size of a

   Synopsis:
   void ITSetResidualHistory( itP, a, na )
   ITCntx *itP;
   double *a;
   int    na;
M*/
#define ITSetResidualHistory(itP,a,na)  {(itP)->residual_history = a; \
					 (itP)->res_hist_size    = na; }

/*M
   ITSetConvergenceTest - Sets the function which is to be used to determine
   convergence.  

   Input Parameters:
.   itP - iterative context obtained from ITCreate()
.   converge - pointer to void function
.   cctx    - context for private data for the convergence routine (may be 
              null)

   Synopsis:
   void ITSetConvergenceTest( itP, converge, cctx )
   ITCntx *itP;
   int   (*converge)();
   void   *cctx;

   Notes:
   The test should return 0 for not converged, 1 for converged, and -1 for
   abort on failure to converge.  The default convergence tester aborts 
   if the residual grows to more than 10000 times the initial residual.

   The default is a combination of relative and absolute
   tolerances.  The residual value that is tested my be an approximation;
   routines that need exact values should compute them.
M*/
#define ITSetConvergenceTest(itP,a,b) {(itP)->converged = a;(itP)->cnvP = b;}

/*M
  ITGetConvergenceContext - Gets the convergence context set with 
  ITSetConvergenceTest.  

   Input Parameters:
.   itP - iterative context obtained from ITCreate()

   Synopsis:
   void *ITGetConvergenceContext( itP )
   ITCntx *itP;
M*/
#define ITGetConvergenceContext(itP) ((itP)->cnvP)

void ITRichardsonSetScale(/* ITCntx *, double  */);
void ITChebychevSetEigenvalues(/* ITCntx *, double, double */); 
void ITGMRESSetDirections(/* ITCntx *, int */);

/* ---------------------Some Default Routines --------------------------*/

void ITDefaultCGMonitor(/* itP, usrP, n, rnorm */);
int  ITDefaultCGConverged(/* itP, usrP, n, rnorm */);

void ITDefaultMonitor(), ITXMonitor();
int  ITDefaultConverged();

   
