#ifndef lint
static char SCCSid[] = "@(#) ./iter/cheby/cheby.c 07/23/93";
#endif

/*
    This is a first attempt at a Chebychev Routine, it is not 
    necessarily well optimized.
*/
#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "iter/itctx.h"
#include "iter/itfunc.h"
#include "iter/chebctx.h"
#include "iter/itpriv.h"

/*+
  ITChebychevCreate - Creates an ITCntx variable for the Chebychev
    method. The prefered calling sequence is ITCreate( ITCHEBYCHEV ).
 +*/
ITCntx *ITChebychevCreate()
{
  ITCntx        *itP;
  ChebychevCntx *chebychevP;

  itP = NEW(ITCntx); CHKPTRN(itP);
  ITSetDefaults( itP );
  chebychevP = NEW(ChebychevCntx); CHKPTRN(chebychevP);
  itP->MethodPrivate = (void *) chebychevP;

  itP->method               = ITCHEBYCHEV;
  itP->max_it               = 50;
  itP->right_inv            = 0;
  itP->calc_res             = 1;
  itP->use_pres             = 0;
  itP->guess_zero           = 0;
  itP->rtol                 = 1.e-5;
  itP->atol                 = 1.e-50;
  chebychevP->emin          = 1.e-2;
  chebychevP->emax          = 1.e+2;
  itP->residual_history     = NULL;

  itP->setup      = ITChebychevSetUp;
  itP->solver     = ITChebychevSolve;
  itP->adjustwork = ITDefaultAdjustWork;
  itP->closedown  = ITChebychevDestroy;
  return(itP);
}

/*+
    ITChebychevSetUp - Called after a call to ITChebychevCreate() or
    ITCreate(ITChebychev), allocates space
    needed in the Chebychev solution. Preferred call sequence is
    ITSetUP(itP,usrP).

    Input Parameters:
.   itP - the iterative context
.   usrP - the user context
+*/
void ITChebychevSetUp(itP,usrP)
ITCntx *itP;
void   *usrP;
{
  if (itP->method != ITCHEBYCHEV) {
     SETERRC(1,"Attempt to use Chebychev Setup on wrong context"); return;}
 
  /* check user parameters and functions */
  if (ITCheckDef(itP)) return;
 
  /* get work vectors from user code */
  ITDefaultGetWork( itP, usrP, 3 );
}

/*@
    ITChebychevSetEigenvalues - Called after a call to 
    ITCreate( ITCHEBYCHEV ) sets the extreme eigenvalues used by 
    the Chebychev method. 

    Input Parameters:
.   itP - the iterative context
.   emax,emin - the extreme eignevalues
@*/
void ITChebychevSetEigenvalues(itP,emax,emin)
ITCntx *itP;
double emax,emin;
{
  ChebychevCntx *chebychevP;
  if (itP->method != ITCHEBYCHEV) return;
  chebychevP = (ChebychevCntx *) itP->MethodPrivate;
  chebychevP->emax = emax;
  chebychevP->emin = emin;
}

/*+
    ITChebychevSolve - Called after a call to ITChebychevCreate() or
    ITCreate(ITCHEBYCHEV) and the call to ITChebychevSetUp() or ITSetUp().
    Actually solves the linear system using the Chebychev method.
    Preferred calling sequence ITSolve(itP,usrP).
 
    Input Parameters:
.   itP - the iterative context
.   usrP - the user context

    Returns:
    the number of iterations required or -1 on error.
+*/
int  ITChebychevSolve(itP,usrP)
void   *usrP;
ITCntx *itP;
{
  int              k,kp1,km1,maxit,ktmp,i = 0,pres,brokeout = 0, hist_len,cerr;
  double           alpha,rnorm,omegaprod;
  double           mu,omega,Gamma,c[3],scale,*history;
  void             *x,*b,*p[3],*r;
  ChebychevCntx    *chebychevP;
  chebychevP = (ChebychevCntx *) itP->MethodPrivate;

  history = itP->residual_history;
  hist_len= itP->res_hist_size;
  maxit   = itP->max_it;
  pres    = itP->use_pres;
  cerr    = 1;

  /* These three point to the three active solutions, we
     rotate these three at each solution update */
  km1 = 0; k = 1; kp1 = 2;
  x = itP->vec_sol;
  b = itP->vec_rhs;
  p[km1] = x;
  p[k]   = itP->work[0];
  p[kp1] = itP->work[1];
  r      = itP->work[2];

  /* use scale*B as our preconditioner */
  scale = 2.0/( chebychevP->emax + chebychevP->emin );

  /*   -alpha <=  scale*lambda(B^{-1}A) <= alpha   */
  alpha = 1.0 - scale*(chebychevP->emin); ;
  Gamma = 1.0;
  mu = 1.0/alpha; 
  omegaprod = 2.0/alpha;

  c[km1] = 1.0;
  c[k] = mu;

  if (!itP->guess_zero) {
    MM(x,r);                              /*  r = b - Ax     */
    DAYPX(-1.0,b,r);       
  }
  else COPY(b,r);
                  
  PRE(r,p[k]);                          /*  p[k] = scale B^{-1}r  + x */
  DAYPX(scale,x,p[k]);                        

  for ( i=0; i<maxit; i++) {
    c[kp1] = 2.0*mu*c[k] - c[km1];
    omega = omegaprod*c[k]/c[kp1];

    MM(p[k],r);                                /*  r = b - Ap[k]    */
    DAYPX(-1.0,b,r);                        
    PRE(r,p[kp1]);                            /*  p[kp1] = B^{-1}z  */

    /* calculate residual norm if requested */
    if (itP->calc_res) {
      if (!pres) NORM(r,&rnorm);
      else NORM(p[kp1],&rnorm);
      if (history && hist_len > i) history[i] = rnorm;
      itP->vec_sol = p[k]; 
      MONITOR(rnorm,i);
      if (CONVERGED(rnorm,i)) {brokeout = 1; break;}
    }

    /* y^{k+1} = omega( y^{k} - y^{k-1} + Gamma*r^{k}) + y^{k-1} */
    SCALE(omega*Gamma*scale,p[kp1]);
    DAXPY((1.0-omega),p[km1],p[kp1]);
    DAXPY(omega,p[k],p[kp1]);

    ktmp = km1;
    km1  = k;
    k    = kp1;
    kp1  = ktmp;
  }
  if (!brokeout && itP->calc_res) {
    MM(p[k],r);                                /*  r = b - Ap[k]    */
    DAYPX(-1.0,b,r);                        
    if (!pres) NORM(r,&rnorm);
    else {
      PRE(r,p[kp1]);                           /*  p[kp1] = B^{-1}z  */
      NORM(p[kp1],&rnorm);
    }
    if (history && hist_len > i) history[i] = rnorm;
    itP->vec_sol = p[k]; 
    MONITOR(rnorm,i);
  }
if (history) itP->res_act_size = (hist_len < i) ? hist_len : i;

  /* make sure solution is in vector x */
  itP->vec_sol = x;
  if (k != 0) {
    COPY(p[k],x);
  }

  /* Get floating point work */
  itP->namult   += (i+1);
  itP->nbinv    += (i+1);
  itP->nvectors += (i+1)*7;

  return RCONV(i+1);
}

/*+
    ITChebychevDestroy - Destroys a iterative context variable obtained
    by a call to ITChebychevCreate() or ITCreate(ITITCHEBYCHEV).
    Preferred calling sequence ITDestroy().

    Input Parameters:
.   itP  - the iterative context
.   usrP - the user context
+*/
void ITChebychevDestroy(itP,usrP)
ITCntx *itP;
void   *usrP;
{
  ChebychevCntx *chebychevP;
  chebychevP = (ChebychevCntx *) itP->MethodPrivate;

  ITDefaultFreeWork( itP, usrP ); 
  
  /* free the context variables */
  FREE(chebychevP); FREE(itP);
}
