/* Primality certificate specifications for use in extended integer package */

struct prime_cert_type
{
	integer_big      prime;
	integer_big      div_of_prime_minus_1;
	t_int    base;
};

typedef
	struct
	{
		t_block_header		block_header;
		prime_cert_length	prime_cert_sz;
		struct prime_cert_type	prime_cert_elt[VARIABLE_LENGTH];
	}
	prime_cert_header;

/*
The following gives the length of the header measured in units of one t_word.
It must be checked when porting to new machines
*/

#define PRIME_CERT_HDR_LEN	(BLOCK_HEADER_WORDS + 1)

/*
Tracked block type for blocks storing primality certificate - the value of this may be
defined by the calling package, for example, in defs.h or trbl.e which 
are included before this file:
*/

#ifndef PRIME_CERT_BLOCK_TYPE

#endif

/*
The following gives the number of bh_ints required to store a block containing
a certificate of n triples
*/

#define prime_cert_required_space(n)	(PRIME_CERT_HDR_LEN + 3*(n))

/*
Macro for accessing individual primality certificate elements, numbered from 0 to length-1
*/

#define	prime_cert_prime(h,k) \
	(((prime_cert_header *)mem_access(h))->prime_cert_elt[k].prime)

#define	prime_cert_div(h,k) \
	(((prime_cert_header *)mem_access(h))->prime_cert_elt[k].div_of_prime_minus_1)

#define	prime_cert_base(h,k) \
	(((prime_cert_header *)mem_access(h))->prime_cert_elt[k].base)

/*
The following is the CURRENT number of elements stored in the primality certificate
*/

#define prime_cert_curr_length(h)	(((prime_cert_header *)mem_access(h))->prime_cert_sz)

/*
The following is the maximum number of elements the primality certificate can hold
*/

#define prime_cert_avail_length(h)	((mem_words(h) - PRIME_CERT_HDR_LEN)/3)

/*
The following allows reduction of the allocated size of the certificate
*/

#define prime_cert_reduce(h,k)	{mem_reduce_words(h,PRIME_CERT_HDR_LEN+3*(k));}

/*
The following macro assures that there is enough room for at least req elements
in the sequence with t_handle hdl, by extending it as necessary; if required, at
least minext extra elements will be allocated space
 WARNING!!! - Beware of the "dangling else" problem if used in an if statement
 NOTE:		- Arguments are only evaluated once
*/

#define prime_cert_assure_space( hdl, req, minext ) \
	{	prime_cert_handle	h = hdl; \
		prime_cert_length	r = (req), m = (minext), \
			a = prime_cert_avail_length(h); \
		if (r > a) \
			mem_extend_words_zero( h, prime_cert_required_space( (r > (a+m)) ? r : (a+m) ) ); \
	}

/*
A pointer to the first element, for use in tight loops.
 WARNING!!!	- ANY Blockhandler operation might cause a garbage collection,
	moving the block and invalidating the pointer
*/

#define prime_cert_elt0_ptr(h)	(& (((prime_cert_header *)mem_access(h))->prime_cert_elt[0]) )

/*
The following macro will copy sequence elements from sequence with t_handle srchdl,
starting at element srcpos, into space referenced by desthdl, starting at element
dstpos.  A total of len digits are copied.

Both srcpos, dstpos are numbered from 0.

It is permissible for dsthdl to equal srchdl ONLY IF dstpos is equal to srcpos
*/

#define prime_cert_copy_elements( srchdl, srcpos, len, dsthdl, dstpos ) \
{ \
	register prime_cert_type	*p = prime_cert_elt0_ptr( srchdl ) + srcpos; \
	register prime_cert_type	*q = prime_cert_elt0_ptr( dsthdl ) + dstpos; \
	register prime_cert_length	k = len; \
	if (q != p) \
	while (k-- > 0) \
		*q++ = *p++; \
}

/*
The following macro will set all elements of the given t_handle to zero up to the
supplied length
*/

#define prime_cert_set_zero( h, len ) \
{ \
	register prime_cert_type	*p = prime_cert_elt0_ptr( h ); \
	register prime_cert_length	k = len; \
	while (k-- > 0) \
		*p++ = 0; \
}
