/*******************************************************************************
-----------------------------------------------------------------------------

    ACHTUNG:
    Zur Vermeidung von zu grossen Kompatibilitaetsproblemen duerfen
    Aenderungen in den Strukturdefinitionen nur nach gemeinsamer
    Eroerterung bzw. Einigung zwischen der Cayley- und der Kant-Gruppe
    durchgefuehrt werden.
    Sydney, 25.9.1992             John Cannon, Johannes Gf. Schmettow

-----------------------------------------------------------------------------

    ATTENTION:

    To avoid major problems of incompatibility, changes to the structure
    definitions should only be made after common discussion and agreement
    between the Cayley and Kant groups.

    Sydney, 25/9/1992             John Cannon, Johannes Ct. Schmettow

-----------------------------------------------------------------------------
                   
anf_anf.h
   
Header file for algebraic number fields.
This file should only be #included by anf.h.
    
An algebraic number field ("anf") is usually defined via a 
generating polynomial whose coefficients are algebraic integers
of a subfield. There might also be several generating polynomials.
The C-structure defining a number field also contains handles which
can be used to access data of orders in this field. So a Q-Basis
for the number field is given by the Z-Basis of any of these orders.
 
 
History:
        92-09-17 JS        Move to Cayley V4
        92-06-04 JS        split off from anf.h

*******************************************************************************/


/*******************************************************************************
/
/  Structure definition for algebraic number fields
/
*******************************************************************************/
  
typedef struct
	{
                t_ring_common   ring_hdr;
 
		integer_small	abs_degree;
		integer_small	r1;
		integer_small	r2;
		integer_small	r;

		dyn_arr_handle	polys_z;

		dyn_arr_handle	polys_order;

		dyn_arr_handle	order_info;
	}
	t_anf_table;
 
  
/*******************************************************************************
/
/   Macros for accessing structure items of number fields
/
*******************************************************************************/
 
/*
    access macro for table element (internal use)
*/
  
#define	anf_access(h)		((t_anf_table *) mem_access(h))
 
 
/*
    absolute degree of field extension 
    number of real embeddings 
    half number of complex embeddings 
*/
 
#define anf_abs_degree(h)	(anf_access(h) -> abs_degree)
#define anf_r1(h)		(anf_access(h) -> r1)
#define anf_r2(h)		(anf_access(h) -> r2)
#define anf_r(h)		(anf_access(h) -> r)
  
 
/*
    array of minimal polynomials over Z (internal use)
    array of minimal polynomials over any order (internal use)
*/
 
#define anf_polys_z(h)		(anf_access(h) -> polys_z)
#define anf_polys_order(h)	(anf_access(h) -> polys_order)
 
 
/*
    counts for polynomials over Z:
       ... accessing number of polynomials given
       ... setting number of polynomials and allocating/assuring space
*/
  
#define anf_poly_z_count(h) \
		(anf_polys_z(h) == MEM_NH \
		  ? 0 \
		  : (dyn_arr_curr_length(anf_polys_z(h)) / 7) \
		)
#define anf_poly_z_count_set(h, len) \
		((anf_polys_z(h) == MEM_NH \
		  ? (void) (anf_polys_z(h) = dyn_arr_alloc((len) * 7)) \
		  : (void) (dyn_arr_assure_space_fun(anf_polys_z(h), (len) * 7, 7)) \
		), \
		(void) (dyn_arr_curr_length(anf_polys_z(h)) = (len) * 7))
/*
    a certain minimal polynomial over Z:
       ... coefficients
       ... zeroes
       ... the real field over which the zeroes are given
       ... discriminant 
       ... factorization of discriminant
       ... reduced discriminant 
       ... factorization of reduced discriminant
       ... are the zeroes known?
       ... is the discriminant known? 
       ... ... and its factorization?
       ... is the reduced discriminant known? 
       ... ... and its factorization?
*/
  
#define anf_poly_z_poly(h, i) \
		dyn_arr_element(anf_polys_z(h), (i)*7 - 7)
  
#define anf_poly_z_zeroes(h, i) \
		dyn_arr_element(anf_polys_z(h), (i)*7 - 6)
   
#define anf_poly_z_zeroes_reals(h, i) \
		dyn_arr_element(anf_polys_z(h), (i)*7 - 5)
   
#define anf_poly_z_disc(h, i) \
		dyn_arr_element(anf_polys_z(h), (i)*7 - 4)
#define anf_poly_z_disc_fac(h, i) \
		dyn_arr_element(anf_polys_z(h), (i)*7 - 3)
  
#define anf_poly_z_rdisc(h, i) \
		dyn_arr_element(anf_polys_z(h), (i)*7 - 2)
#define anf_poly_z_rdisc_fac(h, i) \
		dyn_arr_element(anf_polys_z(h), (i)*7 - 1)
 
#define anf_poly_z_zeroes_known(h, i) \
		(anf_poly_z_zeroes(h, i) != MEM_NH)
#define anf_poly_z_disc_known(h, i) \
		(anf_poly_z_disc(h, i) != MEM_NH)
#define anf_poly_z_disc_fac_known(h, i) \
		(anf_poly_z_fac_disc(h, i) != MEM_NH)
#define anf_poly_z_rdisc_known(h, i) \
		(anf_poly_z_rdisc(h, i) != MEM_NH)
#define anf_poly_z_rdisc_fac_known(h, i) \
		(anf_poly_z_fac_rdisc(h, i) != MEM_NH)
 
 
/*
    allocating space for zeroes of polynomial
    accessing a particular zero of a particular polynomial
*/
 
#define anf_poly_z_zeroes_alloc(h, i) \
		anf_poly_z_zeroes(h, i) = dyn_arr_alloc(anf_abs_degree(h))
#define anf_poly_z_zero(h, i, j) \
	        dyn_arr_element(anf_poly_z_zeroes(h, i), (j)-1)
    
  
/*
    counts for polynomials over orders:
       ... accessing number of polynomials given
       ... setting number of polynomials and allocating/assuring space
*/
 
#define anf_poly_order_count(h) \
		(anf_polys_order(h) == MEM_NH \
		  ? 0 \
		  : (dyn_arr_curr_length(anf_polys_order(h)) / 2) \
		)
#define anf_poly_order_count_set(h, len) \
		((anf_polys_order(h) == MEM_NH \
		  ? (void) (anf_polys_order(h) = dyn_arr_alloc((len) * 2)) \
		  : (void) (dyn_arr_assure_space_fun(anf_polys_order(h), (len) * 2, 2)) \
		), \
		(void) (dyn_arr_curr_length(anf_polys_order(h)) = (len) * 2))
  
/*
    a certain minimal polynomial over an order
    the coefficient order 
*/                                            
 
#define anf_poly_order_poly(h, i) \
		dyn_arr_element(anf_polys_order(h), (i)*2 - 2)
#define anf_poly_order_order(h, i) \
		dyn_arr_element(anf_polys_order(h), (i)*2 - 1)
 
 
/*  
    array with information of orders of the actual field (internal use)
    accessing the t_handle of a certain order
    accessing number of orders given
    setting number of orders and allocating/assuring space
*/
  
#define anf_order_info(h)	(anf_access(h) -> order_info)
#define anf_order(h, i) 	dyn_arr_element(anf_order_info(h), (i) - 1)
#define anf_order_count(h) \
		(anf_order_info(h) == 0 \
		  ? 0 \
		  : (dyn_arr_curr_length(anf_order_info(h))) \
		)
#define anf_order_count_set(h, len) \
		((anf_order_info(h) == 0 \
		  ? (void) (anf_order_info(h) = dyn_arr_alloc((len))) \
		  : (void) (dyn_arr_assure_space_fun(anf_order_info(h), (len), 1)) \
		), \
		(void) (dyn_arr_curr_length(anf_order_info(h)) = (len)))
 
 
/*  
    incrementing the reference counter of a number field
*/
  
#define anf_incref(h)		block_incref(h)
