
#include "defs.h"
#include "integer.e"
#include "poly.h"

t_poly
poly_z_ith_deriv WITH_3_ARGS(
    t_handle,       pring,
    t_poly,      apoly,
    t_int,  i
)
/*
*    IPOLY_ITH_DERIV integral polynomial derivative.
*     apoly is an integral polynomial in r variables 1 <= i <= r
*    returns the derivative of apoly with respect to the ith var
*/
{
    t_handle           aph;          /*    t_handle to apoly             */
    t_handle           resph;        /*    t_handle to result polynomial */
    t_int     prvar;        /*    most principal var of apoly */
    t_int     lpvar;        /*    least princ var of apoly    */
    t_int    atermno;      /*    term counter for apoly      */
    t_int    restermno;    /*    term counter for respoly    */
    t_int    nterms;       /*    no. terms in apoly          */
    t_poly        temppoly;     /*    intermediate polynomial     */
    t_poly        respoly;      /*    result polynomial           */

    if (m_poly_const( apoly ) )
    {
        /* constant poly */
        return( 0 );
    }

    /* general case : non-trivial polynomial */

    aph = m_poly_poly_to_handle( apoly );
    prvar = m_poly_princvar( aph );
    lpvar = m_poly_least_pvar( aph );

    if ( (i < prvar) || (i > lpvar) )
    {
        /* poly constant w.r.t. ith var */
        return( poly_z_zero_poly (pring, apoly) );
    }

    if ( i == prvar )
    {
        /* i is the principal variable of apoly */
        return  poly_z_deriv_princvar( pring, apoly );
    }

    /* i > m_poly_princvar( aph )    */

    nterms = m_poly_nterms( aph );
    m_poly_create_empty(&resph, prvar, lpvar, nterms );
    respoly = m_poly_handle_to_poly( resph );

    for ( atermno = 0, restermno = 0; atermno < nterms; atermno++ )
    {
        /* take ith derivative of each term */
        temppoly = poly_z_ith_deriv( pring, m_poly_coefft( aph, atermno ), i );

        if ( temppoly != 0 )
        {
            /* do not  enter terms with zero coeffts */

            m_poly_coefft( resph, restermno ) = temppoly;
            m_poly_expt( resph, restermno ) = m_poly_expt( aph, atermno );
            restermno++;
        }
    }

    m_poly_nterms( resph ) = restermno;

    return  poly_z_clean( pring, respoly );

} /* poly_z_ith_deriv() */



t_poly
poly_z_deriv_princvar WITH_2_ARGS(
    t_handle,    pring,
    t_poly,   apoly
)
/*
*    IPOLY_DERIV_PRINCVAR : derivative w.r.t. principal variable
*    apoly is an integral polynomial. poly_z_deriv_princvar
*    returns the derivative of apoly with respect to its main
*    variable.
*/
{
    t_handle           resph;        /*    t_handle to respoly        */
    t_handle           aph;        /*    t_handle to apoly            */
    t_poly        respoly;   /*  result polynomial       */
    t_poly        coefft;
    t_poly        temp;
    t_int    expt;
    t_int    nterms;    /*    no. terms in apolt        */
    t_int    termno;    /*    term i for apoly        */
    t_int    restermno;    /*    term i for respoly        */

    if ( m_poly_const( apoly ) )
    {
        /* constant polynomial */
        return( 0 );
    }

    aph = m_poly_poly_to_handle( apoly );
    nterms = m_poly_nterms( aph );

    if ( ( nterms == 0 ) || ( poly_deg( apoly ) == 0 ) )
    {
        return( 0 );
    }

    /* not constant in main variable */

    m_poly_create_empty(&resph, m_poly_princvar(aph),
                                    m_poly_least_pvar(aph), nterms );
    respoly = m_poly_handle_to_poly( resph );
    

    for ( restermno = 0, termno = 0; termno < nterms; termno++ )
    {
        coefft = m_poly_coefft( aph, termno );
        expt = m_poly_expt( aph, termno );

        if ( expt != 0 )    /* ignore terms with expt == 0 */
        {
            /* multiply coefficient by exponent */

            if ( m_poly_not_const( coefft ) )
            {
                temp = poly_z_integer_mult( pring, coefft, expt );
            }
            else
            {
                temp = integer_mult( coefft, expt );
            }

            if ( temp != 0 )
            {
                /* ignore zero terms */

		m_poly_coefft( resph, restermno ) = temp;

                /* decrement exponent */
                m_poly_expt( resph, restermno ) = expt - 1;

                restermno++;
            }
        }
    }

    m_poly_nterms( resph ) = restermno;

    return  poly_z_clean( pring, respoly );

} /* poly_z_deriv_princvar() */



t_poly
poly_z_high_deriv_princvar WITH_3_ARGS(
    t_handle,       pring,
    t_poly,      apoly,
    t_int,   k
)
/*
*    IPOLY_HIGH_DERIV_PRINCVAR : integral polynomial higher derivative
*    principal variable. apoly is a poly in several variables.
*    k is a non-negative beta-digit
*    Returns k-th derivative of apoly w.r.t. main var, as a poly.
*/
{
    block_declarations;
    t_int    i;            /*    loop counter        */
    t_poly        respoly;    /*    result polynomial    */
    t_poly        temp;

    if (poly_deg( apoly) < k)
    {
        return (poly_z_zero_poly (pring, apoly));
    }
    respoly = m_poly_z_incref( pring, apoly );

    for (i = k; i > 0; i--)
    {
        /* continually take derivative until kth derivative */

	temp = respoly;
        respoly = poly_z_deriv_princvar( pring, temp );
	m_poly_z_delref( pring, temp );
    }

    return  respoly;

} /* poly_z_high_deriv_princvar() */


t_poly
poly_z_high_deriv_ith_var WITH_4_ARGS(
    t_handle,       pring,
    t_poly,      apoly,
    t_int,   	    k,
    t_int, 	    i 
)
/*
*    IPOLY_HIGH_DERIV_ITH_VAR : integral polynomial higher derivative
*    ith variable. apoly is a poly in several variables.
*    k is a non-negative beta-digit
*    Returns k-th derivative of apoly w.r.t. ith var, as a poly.
*/
{
    block_declarations;
    t_int    j;            /*    loop counter        */
    t_poly        respoly;    /*    result polynomial    */
    t_poly        temp;

    respoly = m_poly_z_incref( pring, apoly );

    for (j = k; j > 0; j--)
    {
        /* continually take derivative until kth derivative */

	temp = respoly;
        respoly = poly_z_ith_deriv( pring, temp, i );
	m_poly_z_delref( pring, temp );
    }

    return  respoly;

} /* poly_z_high_deriv_ith_var() */

