#include "defs.h"
#include "debug.e"
#include "integer.e"
#include "poly.h"
#include "poly_z_faclst.h"
#include "poly_mat.h"
#include "zm.e"
#include "error.e"

t_handle
poly_u_zm_fact_b2_small WITH_3_ARGS(
    t_handle,      pring,
    integer_big,   pdig,     /* field modulus */
    t_poly,     apoly
)
/*
** poly_u_zm_fact_b2_small: Modular univariate polynomial Berlekamp
** Q matrix construction - Step 2 .
** pdig: prime beta-integer.
** apoly: uivariate polynomial over Zpdig.
** deg( apoly ) >= 2 .
** Returns the Q matrix for apoly as defined in Knuth p 425.
*/
{
    t_int    adeg;          /* degree of polynomial apoly          */
    t_int    anterms;
    t_int    atermno;
    t_int    aexpt;
    t_handle           aph;           /* t_handle for apoly                    */
    t_handle           Qhdl;          /* t_handle for the Q matrix             */
    t_handle           calchdl;       /* t_handle for the calculations array   */
    t_int    termno;        /* term counter                        */
    t_int    qindex;        /* Q entry counter                     */
    t_int    total_rows;    /* total number of rows calculated     */
    t_int    k;             /* calculation row counter             */
                                    /* ( not Q row counter )               */
    t_poly        t;             /* temporary storage for A(n-1)        */
                                    /* of previous row                     */
    t_int    index;         /* counter for elements in A row       */
    t_int    ctermno;
    t_poly        temp2;

    adeg = poly_deg( apoly );

     DENY( adeg < 2 );

    Qhdl = poly_mat_new( adeg, adeg );

    calchdl = m_poly_z_calc_new( adeg );

    aph = m_poly_poly_to_handle( apoly );
    anterms = m_poly_nterms( aph );

    for ( atermno=0, ctermno=0; atermno<anterms-1; atermno++, ctermno++ )
    {
	aexpt = m_poly_expt( aph, atermno );

	for ( ; ctermno <= aexpt; ctermno++ )
	{
	    m_poly_z_calc_u( calchdl, ctermno ) = 0;
	}

	m_poly_z_calc_u( calchdl, aexpt+1 ) = m_poly_coefft( aph, atermno );
    }

    for ( ; ctermno <= adeg; ctermno++ )
    {
	m_poly_z_calc_u( calchdl, ctermno ) = 0;
    }

    /* Set up A0,i */

    for ( termno = 0; termno <= adeg; termno++ )
    {
	m_poly_z_calc_a( calchdl, termno ) = 0;
    }

    /* Initialise for k = 0 */

    m_poly_z_calc_a( calchdl, 1 ) = 1;

    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B2_SMALL,
    {
	cay_print( "initial calchdl = \n" );
	poly_z_calc_write( pring, calchdl, adeg );

	cay_print( "poly_u_zm_fact_b2(): writing into matrix, k = 0\n" );
	for ( qindex = 1; qindex <= adeg; qindex++ )
	{
	    cay_print( " %d,", m_poly_z_calc_a( calchdl, qindex ));
	}
    }
    );

    for ( qindex = 1; qindex <= adeg; qindex++ )
    {
	m_poly_mat_elt( Qhdl, 1, qindex ) = m_poly_z_calc_a( calchdl, qindex );
    }

    total_rows = pdig * (adeg - 1 );

    /* Calculate all rows */

    for ( k = 1; k <= total_rows; k++ )
    {
	/* Calculate A_k */
	t = m_poly_z_calc_a( calchdl, adeg );        

	for ( index = adeg; index > 0; index-- )
	{
	    temp2 = modint_mult( pdig, t, m_poly_z_calc_u( calchdl, index ));

	    m_poly_z_calc_a( calchdl, index ) = modint_subtract( pdig, m_poly_z_calc_a( calchdl, index-1 ), temp2 );
	}

	IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B2_SMALL,
	{
	    cay_print( "poly_u_zm_fact_b2(): k = %d\n", k );
	    for ( qindex = 1; qindex <= adeg; qindex++ )
	    {
		cay_print( " %d,", m_poly_z_calc_a( calchdl, qindex ));
	    }
	}
	);

	if ( modint_hom( pdig, k ) == 0 )
	{
	    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B2_SMALL,
	    {
		cay_print( "writing into matrix, k = %d\n", k );
	    }
	    );

	    for ( qindex = 1; qindex <= adeg; qindex++ )
	    {
		m_poly_mat_elt( Qhdl, integer_div( k, pdig ) + 1, qindex ) = m_poly_z_calc_a( calchdl, qindex );
	    }
	}
    }

    mem_delete_hptr( &calchdl );

    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B2_SMALL,
    {
	cay_print( "poly_u_zm_fact_b2() returns Q =\n" );
	poly_z_Qmatrix_write( Qhdl );
	dmplin();
    }
    );
    return  Qhdl;
}

