#include "defs.h"
#include "debug.e"
#include "integer.e"
#include "poly.h"
#include "poly_z_faclst.h"
#include "error.e"

t_handle
poly_u_zm_fact_b1 WITH_4_ARGS(
    t_handle,       pring,
    integer_big,    pdig,
    t_poly,      apoly,
    t_poly,      gcdpoly
)
/*
** Modupoly_fact_b1: Modular univariate polynomial factorisation using
** the Berlekamp algorithm - Step b1 ( MUPFBL ).
** Eliminate non-square free polynomials.
**
** pdig: prime beta-integer
** apoly: univariate polynomial over Zpdig.
** gcdpoly: gcd( apoly, aderivpoly ) != 1 .
** Returns: a list of the factors of apoly in polynomial form.
*/
{
    block_declarations;
    t_handle           aph;              /* Handle for apoly                 */
    t_int    aterms;           /* number of terms in apoly         */
    t_poly        temppoly;
    t_poly        upoly;            /* polynomial used in determining   */
                                       /* factors                          */
    t_handle           uph;              /* t_handle for upoly                 */
    t_handle           resmh;            /* result t_handle                    */
    t_handle           otherfactshdl;    /* t_handle for list of factors       */
                                       /* of apoly/gcd                     */
    t_int    factorno;         /* factor counter                   */
    t_int    termno;           /* term counter                     */
    t_int    nfactors;         /* number of factors determined     */
                                       /* so far                           */
    t_poly        monupoly;
    t_poly        ratpoly;
    t_int    temp;

    aph = m_poly_poly_to_handle( apoly );

    if ( !poly_u_zm_different( pring, pdig, apoly, gcdpoly ) )
    {
        /* apoly = gcdpoly                                    */
        /* aderivpoly must be zero and the coefficient u_k of */
        /* x^k is non-zero only if k is a multiple of pdig    */
        /* apoly(x) = u(x^pdig) = (u(x))^pdig                 */
        /* Establish u(x)                                     */

        aterms = m_poly_nterms( aph );

        m_poly_create_empty(&uph, m_poly_princvar (aph),
                                       m_poly_least_pvar (aph), aterms );
        upoly = m_poly_handle_to_poly( uph );

        aph = m_poly_poly_to_handle( apoly );

        for ( termno = 0; termno < aterms; termno++ )
        {
	    temp = m_poly_coefft( aph, termno );
            m_poly_coefft( uph, termno ) = temp;

            /* expect each exponent to be a multiple of pdig */
	    ASSERT ( m_poly_expt( aph, termno ) % pdig == 0 );

            m_poly_expt( uph, termno ) = m_poly_expt( aph, termno ) / pdig;
        }

	monupoly = modpoly_monic( pring, pdig, upoly );
	m_modpoly_delref( pring, upoly );

        resmh = poly_u_zm_fact( pring, pdig, monupoly );
	m_modpoly_delref( pring, monupoly );

        nfactors = m_poly_z_faclst_len( resmh );

        /* The factors of apoly are factors of upoly to the power pdig */

        for ( factorno = 0; factorno < nfactors; factorno++ )
        {
	    m_poly_z_faclst_power( resmh, factorno ) *= pdig;
        }

        return  resmh;
    }

    /* apoly != gcdpoly, i.e. gcd is a proper factor of apoly */

    temppoly = modpoly_monic( pring, pdig, gcdpoly );
    resmh = poly_u_zm_fact( pring, pdig, temppoly );
    m_modpoly_delref( pring, temppoly );

    /* Get factors of apoly/gcdpoly */

    ratpoly = modpoly_div( pring, pdig, apoly, gcdpoly );
    temppoly = modpoly_monic( pring, pdig, ratpoly );
    m_modpoly_delref( pring, ratpoly );

    otherfactshdl = poly_u_zm_fact( pring, pdig, temppoly );
    m_modpoly_delref( pring, temppoly );

    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B1,
    {
	cay_print( "poly_u_zm_fact_b1(): resmh and other are :\n" );
	poly_z_faclst_write( pring, resmh );
	dmplin();
	poly_z_faclst_write( pring, otherfactshdl );
	dmplin();
    }
    );

    poly_z_faclst_join( pring, resmh, otherfactshdl );

    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B1,
    {
	cay_print( "poly_u_zm_fact_b1(): after combination, resmh is\n" );
	poly_z_faclst_write( pring, resmh );
	cay_print( "other is\n" );
	poly_z_faclst_write( pring, otherfactshdl );
    }
    );

    poly_z_faclst_delete( pring, &otherfactshdl );

    return  resmh;
}

