#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_priv_log_235		WITH_4_ARGS(
	mp_int,		i,
	mp_int,		j,
	mp_int,		k,
	mp_float,	x
)
/*
Returns x = log(2^i * 3^j * 5^k).  x must be already allocated.  The method
requires time O(t^2).  It is assumed that i, j, and k are not too large.
Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_int		c[3], q, n;
    mp_round_type	save_round = round;
    mp_acc_float	temp1, temp2;
    mp_ptr_type		temp1_ptr, temp2_ptr;
    mp_base_type	b;
    mp_length		t;


    DEBUG_BEGIN(DEBUG_LOG);
    DEBUG_PRINTF_1("+log235 {\n");
    DEBUG_PRINTF_4("i = %d, j = %d, k = %d\n", i, j, k);


#define limit	(MAX_INT / 68)

    if (int_abs(i) >= limit || int_abs(j) >= limit || int_abs(k) >= limit)
	mp_error("mp_log_235: i, j, or k too large");

#undef limit

    mp_set_sign(xp, 0);

    b = mp_b(xp);
    t = mp_t(xp);

    /*
    Use truncation internally.
    */

    round = MP_TRUNC;

    mp_change_up();

    mp_acc_float_alloc_2(b, t, temp1, temp2);
    
    c[0] = 3 * i + 5 * j + 7 * k;
    c[1] = 5 * i + 8 * j + 12 * k;
    c[2] = 7 * i + 11 * j + 16 * k;

    for (q = 0; q < 3; q++)
    {

	static mp_int	data[] = { 161, 49, 31 };

	mp_q_to_mp(2 * c[q], data[q], temp1);
	mp_add_eq(x, temp1);


	/*
	Set pointers on the first iteration (q == 0) and whenever a block
	may have changed.
	*/

	if (!q || mp_has_changed())
	{
	    temp1_ptr = mp_acc_float_ptr(temp1);
	    temp2_ptr = mp_acc_float_ptr(temp2);
	}


#define fix_pointers()		if (mp_has_changed()) {			    \
				    xp = mp_ptr(x);	    		    \
				    temp1_ptr = mp_acc_float_ptr(temp1);    \
				    temp2_ptr = mp_acc_float_ptr(temp2);    \
				}

	n = 1;

	for (;;)
	{
	    register mp_length	mul_t;

	    n += 2;

	    if (mp_is_zero(temp1_ptr))
		break;

	    /*
	    Reduce t if possible for multiply/divide operations.
	    */

	    mul_t = t + mp_expt(temp1_ptr) + 2 - mp_expt(xp);

	    if (mul_t <= 2)
		break;

	    if (mul_t > t)
		mul_t = t;

	    mp_t(temp1_ptr) = mp_t(temp2_ptr) = mul_t;


	    mp_mul_double_q(temp1, 1, 1, data[q], data[q]);
	    mp_div_int(temp1, n, temp2);


	    /*
	    Restore t of temp2 for add and t of temp1 for next iteration.
	    */

	    fix_pointers();
	    mp_t(temp2_ptr) = mp_t(temp1_ptr) = t;

	    mp_add_eq(x, temp2);

	    fix_pointers();
	}
    }

    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);

    mp_change_down();
    round = save_round;

    DEBUG_1("-} x = ", xp);
    DEBUG_END();

    return x;

#undef fix_pointers
}


mp_float
mp_log_int	WITH_2_ARGS(
	mp_int,	n,
	mp_float,	x
)
/*
Returns x = log(n) for small positive integer n (2 * n must be representable).
The time taken is O(t^2).  Accumulator operations are performed.
*/
{
    /*
    The method is to use a rapidly converging series and mp_priv_log_235().
    */

    mp_ptr_type		xp = mp_ptr(x);
    mp_base_type	b;
    mp_length		t, new_t;
    mp_round_type	save_round = round;
    mp_int		i, ia, closest, min_dist, ip, ip2, iq, iq2, j, n1, n2;
    mp_acc_float	sum; 

    DEBUG_BEGIN(DEBUG_LOG);
    DEBUG_PRINTF_1("+log_int {\n");
    DEBUG_PRINTF_2("n = %d\n\n", n);


    if (n <= 0 || n > MAX_INT / 2)
	mp_error("mp_log_int: n (%d) not positive or too large", n);

    if (n == 1)
    {
	/*
	log(1) == 0.
	*/

	mp_set_sign(xp, 0);

	DEBUG_1("-} x = ", xp);
	DEBUG_END();

	return x;
    }

    b = mp_b(xp);
    t = mp_t(xp);

    new_t = t + 1 + mp_extra_guard_digits(mp_times_log2_b(t, b), b);


    /*
    Use truncated arithmetic internally.
    */

    round = MP_TRUNC;

    mp_acc_float_alloc(b, new_t, sum);


    /*
    Check for special case.
    */

    if (n == 2)
	mp_log_235(1, 0, 0, sum);

    else
    {
	/*
	Find j = 3 * 2^ia such that j <= n < 2j.
	*/

	j = 3;
	ia = 0;

	n2 = n / 2;

	while (j <= n2)
	{
	    ia++;
	    j <<= 1;
	}

	/*
	Now 3 * 2^ia <= n < 6 * 2^ia.
	*/

	j /= 3;

	min_dist = n;
	closest = 0;

	/*
	Compute closest integer to n/2^ia.
	*/

	for (i = 3; i <= 6; i++)
	{
	    mp_int	dist = int_abs(n - i * j);

	    if (dist <= min_dist)
	    {
		min_dist = dist;
		closest = i;
	    }
	}
	
	n1 = closest * j;

	/*
	Now n is close to n1 = closest * 2^ia and closest = 3, 4, 5, or 6,
	so mp_log_235 gives log(n1).
	*/

	DEBUG_PRINTF_3("closest = %d, n1 = %d\n", closest, n1);

	if (closest == 3)
	    mp_log_235(ia, 1, 0, sum);

	else if (closest == 4)
	    mp_log_235(ia + 2, 0, 0, sum);

	else if (closest == 5)
	    mp_log_235(ia, 0, 1, sum);

	else if (closest == 6)
	    mp_log_235(ia + 1, 1, 0, sum);

	
	if (n != n1)
	{
	    /*
	    We now need log(n/n1).  First we reduce n/n1 by taking out any
	    common divisors (which may speed up what follows and reduces
	    chance of integer overflow).  Write n/n1 = y = (1 + x)/(1 - x).
	    Re-arranging, we find x = (y - 1)/(y + 1) = (n - n1)/(n + n1).
	    Set ip = (n - n1) and iq = (n + n1).  The ratio n/n1 should always
	    be in the interval [7/8, 7/6) which implies that iq should be at
	    least 15.  Now with x = ip/iq, we use the power series:

		log((1 + x)/(1 - x)) = 2(x + x^3/3 + x^5/5 + ...).

	    This converges faster than the series for log(1 + y) because
	    x is roughly half as large as y and there are only odd terms in
	    the series.
	    */

	    mp_acc_float	temp, term;
	    mp_ptr_type		temp_ptr, term_ptr;


	    mp_int_gcd(&n1, &n);

	    ip = n - n1;
	    iq = n + n1;

	    DEBUG_PRINTF_3("ip = %d, iq = %d\n", ip, iq);

	    if (iq <= 14)
		mp_error("mp_log_int: n too large");

	    mp_int_gcd(&ip, &iq);


	    mp_change_up();
	    mp_acc_float_alloc_2(b, new_t, temp, term);


	    /*
	    Start with term = 2ip/iq and add into sum.
	    */

	    mp_q_to_mp(2 * ip, iq, term);
	    mp_add_eq(sum, term);


	    /*
	    Compute ip^2 and iq^2 if possible.
	    */

	    if (iq <= MAX_INT / iq)
	    {
		iq2 = iq * iq;
		ip2 = ip * ip;
	    }
	    else
		iq2 = 0;


	    /*
	    Set pointers to temporary floats.
	    */

	    term_ptr = mp_acc_float_ptr(term);
	    temp_ptr = mp_acc_float_ptr(temp);

	    /*
	    Power series loop - for each iteration, we multiply each term
	    by ip^2/iq^2 and divide by i (i = 3, 5, ...) before adding in.
	    After n iterations, term = (ip/iq)^(2n+1)
	    */


#define fix_pointers()		if (mp_has_changed()) {			\
				    temp_ptr = mp_acc_float_ptr(temp);	\
				    term_ptr = mp_acc_float_ptr(term);	\
				}


	    for (i = 3; ; i += 2)
	    {
		register mp_length	mul_t;


		if (mp_is_zero(term_ptr))
		    break;

		/*
		Reduce t for multiplications if possible.  If we can reduce
		it below 3 we are done.
		*/

		mul_t = new_t + mp_expt(term_ptr) + 2;

		if (mul_t > new_t)
		    mul_t = new_t;

		else if (mul_t <= 2)
		    break;


		mp_t(term_ptr) = mp_t(temp_ptr) = mul_t;


		/*
		Multiply term by ip^2/iq^2 and set temp = term/i.
		*/

		if (iq2)
		    mp_mul_q_eq(term, ip2, iq2);

		else
		{
		    /*
		    iq is too large for one call to mp_mul_q().
		    */

		    mp_mul_q_eq(term, ip, iq);
		    mp_mul_q_eq(term, ip, iq);
		}

		mp_div_int(term, i, temp);


		/*
		Restore t of temp for addition.
		*/

		fix_pointers();
		mp_t(temp_ptr) = new_t;


		/*
		Add in the next term.
		*/

		mp_add_eq(sum, temp);


		/*
		Fix pointers for next iteration if necessary.
		*/

		fix_pointers();
	    }

	    mp_acc_float_delete(temp);
	    mp_acc_float_delete(term);
	    mp_change_down();
	}
    }


    /*
    Restore rounding type and round result.
    */

    round = save_round;

    mp_move_round(sum, x);
    mp_acc_float_delete(sum);

    DEBUG_1("-} x = ", xp);
    DEBUG_END();

    return x;
}



mp_float
mp_logs		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = log(1 + x) if x satisfies the condition that int_abs(x) < 1/b.
Otherwise there is an error.  The time taken is O(sqrt(t) * M(t))
as for mp_exp1().  Accumulator operations are performed.
*/
{
    /*
    Newton's method is used to solve the equation x = exp1(-y), then the sign
    of y is reversed.  exp1(y) (== exp(y) - 1) is calculated by mp_exp1().
    */


    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, new_t, it0;
    mp_acc_float	x_temp, y_temp, temp1, temp2;
    mp_ptr_type		x_temp_ptr, y_temp_ptr, temp1_ptr, temp2_ptr;


    DEBUG_BEGIN(DEBUG_LOG);
    DEBUG_PRINTF_1("+logs {\n");
    DEBUG_1("x = ", xp);


    /*
    Check whether x and y have compatible parameters.
    */

    mp_check_2("mp_logs", xp, yp);

    if (mp_is_zero(xp))
    {
	mp_set_sign(yp, 0);

	DEBUG_PRINTF_1("-} y = 0\n");
	DEBUG_END();

	return y;
    }

    /*
    Check whether x is in the legal range.
    */

    if (mp_expt(xp) >= 0)
	mp_error("mp_logs: int_abs(x) >= 1/b");
    

    /*
    Use truncation internally.  Calculate sufficient guard digits and allocate
    space for temporary floats.
    */

    round = MP_TRUNC;

    b = mp_b(xp);
    t = mp_t(xp);

    new_t = t + 1 + mp_guard_digits(100, b);
    if (new_t < 5)
	new_t = 5;

    mp_change_up();
    mp_acc_float_alloc_4(b, new_t, x_temp, y_temp, temp1, temp2);


    /*
    Get starting approximation:

    Set		y = first four terms of power series of -log(1 + x),
    i.e.:	y = -(x - x^2/2 + x^3/3 - x^4/4),
    or:		y = ([(x/4 - 1/3) * x + 1/2] * x - 1) * x  (by Horner's Rule).
    */

    mp_move(x, x_temp);


    /*
    temp1 = x/4 - 3.
    */

    mp_div_int(x_temp, 4, temp1);
    mp_add_q_eq(temp1, -1, 3);


    /*
    temp1 = (x * temp1) + 1/2.
    */

    mp_mul_eq(temp1, x_temp);
    mp_add_q_eq(temp1, 1, 2);


    /*
    temp1 = (x * temp1) - 1.
    */

    mp_mul_eq(temp1, x_temp);
    mp_add_int_eq(temp1, -1);


    /*
    Finally, y = x * temp1.
    */

    mp_mul(x_temp, temp1, y_temp);


    
    /*
    Start newton iterations with small t - it will be increased later.
    */

    x_temp_ptr = mp_acc_float_ptr(x_temp);
    y_temp_ptr = mp_acc_float_ptr(y_temp);
    temp1_ptr = mp_acc_float_ptr(temp1);
    temp2_ptr = mp_acc_float_ptr(temp2);

    t = 13 - 2 * b;

    if (t < 5)
	t = 5;

    it0 = (t + 5) / 2;

    mp_t(x_temp_ptr) = mp_t(y_temp_ptr) = mp_t(temp1_ptr) = mp_t(temp2_ptr) = t;


    /*
    Main loop - we use Newton's method on f(y) = 0, where f(y) = exp1(-y) - x.
    This gives the iterative assignment (remember exp1(z) = exp(z) - 1):

    New y  =  y - f(y)/f'(y)
	   =  y - [exp(-y) - 1 - x]/[-exp(-y)]
	   =  y + [1 - exp(y) - x * exp(y)]
	   =  y - [x + x * exp(y) - x + exp(y) - 1]
	   =  y - [x + x * exp1(y) + exp1(y)].
    */


    for (;;)
    {
	mp_length	last_t, save_t;


	/*
	temp1 = exp1(y).
	*/

	mp_exp1(y_temp, temp1);


	/*
	temp2 = x * exp1(y).
	*/

	mp_mul(x_temp, temp1, temp2);


	/*
	temp1 += temp2, so temp1 == x * exp1(y) + exp1(y).
	*/

	mp_add_eq(temp1, temp2);


	/*
	temp1 += x, so temp1 == x + x * exp1(y) + exp1(y).
	*/

	mp_add_eq(temp1, x_temp);


	/*
	Finally, y -= temp1.
	*/

	mp_sub_eq(y_temp, temp1);


	/*
	Fix pointers if they have changed.
	*/

	if (mp_has_changed())
	{
	    x_temp_ptr = mp_acc_float_ptr(x_temp);
	    y_temp_ptr = mp_acc_float_ptr(y_temp);
	    temp1_ptr = mp_acc_float_ptr(temp1);
	    temp2_ptr = mp_acc_float_ptr(temp2);
	}


	if (t >= new_t)
	    break;


	/*
	The following loop computes the next value of t to use.  Because
	Newton's method has 2nd order convergence, we can almost double
	t each time.
	*/

	last_t = t;
	t = new_t;

	do
	{
	    save_t = t;
	    t = (t + it0) / 2;
	} while (t > last_t);

	t = save_t;

	mp_t(x_temp_ptr) = mp_t(y_temp_ptr) =
		mp_t(temp1_ptr) = mp_t(temp2_ptr) = t;

    }

    /*
    Check that Newton iteration was converging as expected.
    temp1 is the last quantity which was subtracted from y.
    */

    if (!mp_is_zero(temp1_ptr) && 2 * mp_expt(temp1_ptr) > it0 - t)
	mp_bug("mp_logs: Newton iteration not converging");
    

    /*
    Reverse sign, round, and copy final result in y.
    */

    mp_set_sign(y_temp_ptr, -mp_sign(y_temp_ptr));

    round = save_round;
    mp_move_round(y_temp, y);


    /*
    Restore accumulator stack pointer and level.
    */

    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);
    mp_acc_float_delete(y_temp);
    mp_acc_float_delete(x_temp);
    mp_change_down();

    DEBUG_1("-} y = ", yp);
    DEBUG_END();

    return y;
}



mp_float
mp_log		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = log(x) with the restriction that the integer part of log(x)
must be representable as an integer.  The time taken is O(sqrt(t)*M(t)
+ t^2).  For small integer x, mp_log_int() is faster.  Asymptotically
faster methods exist (e.g. the gauss-salamin method, see mp_log_gs()),
but are not useful unless t is large.  See the comments to mp_exp1(),
and mp_pi_gl().  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_acc_float	log, temp;
    mp_base_type	b;
    mp_length		t, new_t;
    mp_ptr_type		temp_ptr;


    DEBUG_BEGIN(DEBUG_LOG);
    DEBUG_PRINTF_1("+log {\n");
    DEBUG_1("x = ", xp);

    /*
    Check whether x and y have compatible parameters.
    */

    mp_check_2("mp_log", xp, yp);


    /*
    Check whether x is in the legal range.
    */

    if (!mp_is_pos(xp))
	mp_error("mp_log: x non-positive");


    round = MP_TRUNC;

    b = mp_b(xp);
    t = mp_t(xp);


    /*
    Allocate temporary floats with sufficient guard digits and move x into log.
    */

    new_t = t + 1 + mp_extra_guard_digits(1, b);
    mp_acc_float_alloc_2(b, new_t, log, temp);

    mp_move(x, log);


    /*
    Check whether x is close to 1.
    */

    mp_add_int(log, -1, temp);
    temp_ptr = mp_acc_float_ptr(temp);

    if (mp_is_zero(temp_ptr) || mp_expt(temp_ptr) < 0)

	/*
	Here x is close to (or equal to) 1, so use mp_logs() directly.
	*/

	mp_logs(temp, log);


    else
    {
	mp_int		j, k;
	mp_ptr_type	log_ptr = mp_acc_float_ptr(log);


	/*
	Take off the exponent and first two digits.
	*/

	j = b * mp_digit(log_ptr, 0) + mp_digit(log_ptr, 1);
	k = mp_expt(log_ptr) - 2;

	DEBUG_PRINTF_3("j = %d, k = %d\n", j, k);

	mp_expt(log_ptr) = 2;

	mp_div_int_eq(log, j);
	mp_add_int_eq(log, -1);


	/*
	It should now be safe to call mp_logs().
	*/

	mp_logs(log, log);


	/*
	Now use mp_log_int() to correct the result: add log(j * b^k) to log,
	i.e. set log += k * log(b) + log(j).
	*/

	mp_log_int(b, temp);
	mp_mul_int_eq(temp, k);
	mp_add_eq(log, temp);

	mp_log_int(j, temp);
	mp_add_eq(log, temp);
    }


    /*
    Restore rounding type and round result.
    */

    round = save_round;
    mp_move_round(log, y);

    mp_acc_float_delete(temp);
    mp_acc_float_delete(log);

    DEBUG_1("-} y = ", yp);
    DEBUG_END();

    return y;
}
