#include "defs.h"
#include "mp.e"
#include "mp.h"

#ifdef MP_USE_RECALL
mp_float	mp_eul_precise_val;
mp_int		mp_eul_precise_len;
#endif MP_USE_RECALL

mp_float
mp_eul		WITH_1_ARG(	mp_float,	x
)
/*
Returns x = Euler's constant (gamma = 0.57721566...) to almost full
multi-precision accuracy.  The time taken is O(t^2).
*/
{
    /*
    The method is based on Bessel function identities and was discovered by
    Edwin M.C Millan and R. Brent.  It is faster than the method of Sweeney
    (Math. Comp. 17, 1963, 170).
    */

    mp_ptr_type		xp = mp_ptr(x);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, new_t;
    mp_acc_float	gamma, temp2, temp3, temp4;
    mp_ptr_type		gamma_ptr, temp2_ptr, temp3_ptr, temp4_ptr;
    mp_int		n, k;


    b = mp_b(xp);
    t = mp_t(xp);

#ifdef MP_USE_RECALL
    if (mp_eul_precise_len >= 0)
    {
	if (mp_eul_precise_len >= t)
	{
	    /*
	    ** Return imprecise copy of recalled value of eul.
	    */

	    mp_priv_move(mp_eul_precise_val, x);
	    return x;
	}

	/*
	** Delete copy of eul, needed to greater accuracy.
	*/

	mp_delete_float(mp_eul_precise_val);
    }

    /*
    ** Can assume that no precise value is remembered at this moment in
    ** time. (Re)calculate eul and store result in mp_eul_precise_val.
    */

    mp_eul_precise_val = mp_alloc(b, t);
    mp_eul_precise_len = t;
#endif MP_USE_RECALL

    /*
    Increase t as necessary to get accurate result and use truncation
    internally.
    */

    n = mp_times_log2_b(t, b);
    new_t = t + 1 + mp_extra_guard_digits(n, b);
    round = MP_TRUNC;


    /*
    Compute n so truncation error (bounded by pi * exp(-4n)) is
    less than .01 * (b^-t).  Note that 1/8 + 1/20 = 7/40 > (log2)/4
    */

    n = n / 8 + n / 20 + 4;

    mp_acc_float_alloc_4(b, new_t, gamma, temp2, temp3, temp4);

    mp_log_int(n, gamma);

    gamma_ptr = mp_acc_float_ptr(gamma);
    temp2_ptr = mp_acc_float_ptr(temp2);
    temp3_ptr = mp_acc_float_ptr(temp3);
    temp4_ptr = mp_acc_float_ptr(temp4);

    /*
    Set up main loop.
    */

    mp_set_sign(gamma_ptr, -mp_sign(gamma_ptr));

    mp_copy_ptr(gamma_ptr, temp4_ptr);
    mp_int_to_mp(1, temp3);
    mp_copy_ptr(temp3_ptr, temp2_ptr);

    k = 0;


#define fix_pointers()		if (mp_has_changed())			    \
				{					    \
				    temp2_ptr = mp_acc_float_ptr(temp2);    \
				    temp3_ptr = mp_acc_float_ptr(temp3);    \
				    temp4_ptr = mp_acc_float_ptr(temp4);    \
				}

    do
    {
	k++;

	if (k > n)
	{
	    /*
	    Reduce t if possible.
	    */

	    mp_length	t = new_t + mp_expt(temp3_ptr) - mp_expt(temp2_ptr);
	    
	    /*
	    Test for convergence.
	    */

	    if (t < 2)
		break;
	    
	    if (t > new_t)
		t = new_t;

	    /*
	    Set t of temp3 and temp4 for multiply/divide/add.
	    */

	    mp_t(temp3_ptr) = mp_t(temp4_ptr) = t;
	}

	mp_mul_double_q(temp3, n, n, k, k);
	mp_mul_double_q(temp4, n, n, k, 1);

	mp_add_eq(temp4, temp3);
	mp_div_int_eq(temp4, k);


	/*
	Restore t of temp3 and temp4 to add into temp2 and gamma.
	*/

	fix_pointers();

	mp_t(temp3_ptr) = mp_t(temp4_ptr) = new_t;

	mp_add_eq(temp2, temp3);
	mp_add_eq(gamma, temp4);


	/*
	Fix pointers for test of while and next iteration.
	*/

	fix_pointers();

    } while (!mp_is_zero(temp3_ptr));


    /*
    Compute final quotient.
    */

    mp_div_eq(gamma, temp2);


    /*
    Restore rounding type, round result and restore accumulator.
    */

    round = save_round;
    mp_move_round(gamma, x);

    mp_acc_float_delete(temp4);
    mp_acc_float_delete(temp3);
    mp_acc_float_delete(temp2);
    mp_acc_float_delete(gamma);

#ifdef MP_USE_RECALL
    /*
    ** Record value for later recall.
    */

    mp_copy(x, mp_eul_precise_val);
#endif MP_USE_RECALL

    return x;
}
