#include "defs.h"
#include "mp.e"
#include "mp.h"

void
mp_priv_divl	WITH_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
)
/*
Sets z = x / y.  Only called by mp_div() (q.v.).
Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y), zp = mp_ptr(z);
    mp_base_type	b;
    mp_length		t, guard, total, i;
    mp_acc_index	acc_index;
    mp_digit_ptr_type	x_dig_p, y_dig_p_minus_1, z_dig_p;
    mp_int		q, j, c, k, b1, limit;


    DEBUG_BEGIN(DEBUG_DIV);
    DEBUG_PRINTF_1("+divl {\n");
    DEBUG_1("x = ", xp);
    DEBUG_1("y = ", yp);


    /*
    Extract base and number of digits from z.
    */

    t = mp_t(zp);
    b = mp_b(zp);

    guard = (round == MP_RND)? t + 2: 1 + (round != MP_TRUNC);
    total = t + guard;


    /*
    Allocate space for digits.  We need t + 1 for a copy of x, and
    total for z.
    */

    mp_change_up();
    mp_acc_digit_alloc(t + 1 + total, acc_index);

    if (mp_has_changed())
    {
	xp = mp_ptr(x);
	yp = mp_ptr(y);
	zp = mp_ptr(z);
    }

    x_dig_p = mp_acc_digit_ptr(acc_index);
    y_dig_p_minus_1 = mp_digit_ptr(yp, -1);
    z_dig_p = x_dig_p + t + 1;

#define x_dig(i)	(mp_int)x_dig_p[i]
#define x_ptr(i)	(x_dig_p + (i))
#define set_x_dig(i, d)	(x_dig_p[i] = d)
#define y_dig(i)	(mp_int)((i)? y_dig_p_minus_1[i]: 0)
#define z_dig(i)	z_dig_p[i]
#define z_ptr(i)	(z_dig_p + (i))

/*#define y_dig(i)	(mp_int)((i)? y_dig_p_minus_1[(i) - 1]: 0)*/

    mp_dig_copy(x_ptr(1), mp_digit_ptr(xp, 0), t);
    set_x_dig(0, 0);

    b1 = b * (b - 1);
    limit = (MAX_DIGIT / b) * b - 1;


    /*
    Main loop - for an explanation, see Knuth, Vol. 2, pages 255-260.
    */

    for (j = 0; j < total; j++)
    {
	z_dig(j) = 0;

	for (;;)
	{
	    /*
	    Compare x and y (usually only once for each j).
	    */

	    i = 0;

	    while (i <= t && x_dig(i) == y_dig(i))
		i++;

	    if (i > t || x_dig(i) >= y_dig(i))
	    {
		/*
		Now b * y > x >= y.  Compute lower bound q on quotient x/y.
		*/

		q = (b * x_dig(0) + x_dig(1)) / (y_dig(1) + 1);
		if (q < 1)
		    q = 1;


		/*
		Compute sharper lower bound if overflow impossible.
		*/

		if (x_dig(0) < limit)
		{
		    register mp_int	newq =
		   
		       (b * (b * x_dig(0) + x_dig(1)) + x_dig(2)) /
			   (b * y_dig(1) + y_dig(2) + 1);
		    
		    if (newq > q)
			q = newq;
		}

		/*
		Increment jth quotient digit.
		*/

		z_dig(j) += q;


		/*
		k is (b + signed carry), 0 < k <= b.
		*/

		k = b;

		for (i = t; i >= 0; i--)
		{
		    c = b1 + k + x_dig(i) - q * y_dig(i);
		    k = c / b;
		    set_x_dig(i, c - b * k);	/* c - b * k == c % b */
		}

		/*
		There should be no carry off the end.  Check just in case.
		*/

		if (k != b)
		    mp_bug("mp_divl: integer overflow");

	    }
	    else
		break;

	}

	mp_shift_l(x_ptr(1), 1, t);
	set_x_dig(t, 0);
    }


    /*
    If the remainder is non-zero make sure a guard digit is non-zero
    for directed rounding.
    */

    if ((round == MP_RND_UP || round == MP_RND_DOWN) &&
					!mp_all_zero(x_ptr(0), t))
	z_dig(total) = 1;


    /*
    Normalize & round result.
    */

    mp_nzr(mp_sign(xp) * mp_sign(yp), mp_expt(xp) - mp_expt(yp) + 1, zp,
				    z_ptr(0), guard);
    

    /*
    Restore stack pointer.
    */

    mp_acc_delete(acc_index);
    mp_change_down();

    DEBUG_1("-} z = ", zp);
    DEBUG_END();
}


mp_float
mp_div		WITH_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
)
/*
Sets z = x / y and returns z.  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y), zp = mp_ptr(z);
    mp_base_type	b;
    mp_length		t, new_t;


    DEBUG_BEGIN(DEBUG_DIV);
    DEBUG_PRINTF_1("+divl {\n");
    DEBUG_1("x = ", xp);
    DEBUG_1("y = ", yp);

    /*
    Check that x, y, & z have compatible parameters.
    */

    mp_check_3("mp_div", xp, yp, zp);


    if (mp_is_zero(yp))
	mp_error("mp_div: division by 0");
    

    if (mp_is_zero(xp))
    {
	mp_set_sign(zp, 0);
	DEBUG_PRINTF_1("div: z = 0\n");

	return z;
    }


    /*
    See if t is small enough so that mp_divl() is faster than mp_rec() or
    we must use mp_divl() to ensure the correct rounded result.
    */

    b = mp_b(xp);
    t = mp_t(xp);

    if (round != MP_TRUNC || (new_t = t + 1 + mp_guard_digits(100, b)) < 80)
	mp_divl(x, y, z);
    
    else
    {
	mp_round_type		save_round = round;
	mp_acc_float		temp1, temp2;


	/*
	Use mp_rec() and mp_mul().
	*/

	mp_acc_float_alloc_2(b, new_t, temp1, temp2);
	mp_move(y, temp1);


	/*
	Take care for directed rounding (but we can only get here if
	round == MP_TRUNC! - AKS).
	*/

	round = mp_fix_directed(-mp_sign(mp_ptr(x)), round);
	mp_rec(temp1, temp1);
	round = save_round;

	mp_move(x, temp2);
	mp_mul_eq(temp1, temp2);

	mp_move(temp1, z);

	mp_acc_float_delete(temp2);
	mp_acc_float_delete(temp1);
    }

    DEBUG_1("-} z = ", zp);
    DEBUG_END();

    return z;
}


mp_float
mp_rec		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Sets y = 1/x and returns y.  The time taken is O(M(t)).
Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_acc_float	one;


    DEBUG_BEGIN(DEBUG_DIV);
    DEBUG_PRINTF_1("+rec {\n");
    DEBUG_1("x = ", xp);

    mp_check_2("mp_rec", xp, yp);

    if (mp_is_zero(xp))
	mp_error("mp_rec: attempted division by 0");


    /*
    The following constant will be (!) determined empirically.
    */

    if (mp_t(xp) >= 50 && round == MP_TRUNC)
	mp_root(x, -1, y);

    else
    {
	mp_acc_float_alloc(mp_b(xp), mp_t(xp), one);
	mp_int_to_mp(1, one);

	mp_divl(one, x, y);

	mp_acc_float_delete(one);
    }

    DEBUG_1("-} y = ", yp);
    DEBUG_END();

    return y;
}
