#include "defs.h"
#include "ring.h"
#include "mat.h"
#include "integer.e"
#include "zm.e"


/*
 * This file includes
 *
 * 	mat_zm_row_add WITH_6_ARGS()         - Adds n times row i to row j
 * 	mat_zm_row_mult WITH_5_ARGS()        - Replaces row i by n times row i
 * 	mat_zm_col_add WITH_5_ARGS()         - Adds n times col i to col j
 * 	mat_zm_col_mult WITH_4_ARGS()        - Replaces col i by n times col i
 */



void
mat_zm_row_add WITH_7_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  arow,
	integer_small,  brow,
	integer_big,    mult,
	integer_small,  from,
	integer_small,	to
)
/*
** Adds (mult * row arow) of a to row brow of a.
** Does the addition left to right from column "from".
*/
{
	block_declarations;
	register integer_big      temp1;
	register integer_big      temp2;
	register integer_small    i;
	register integer_small    len;
	register integer_small    numrows;
	register integer_big      modulus;
	matrix           mat = 0;

	numrows = mat_row( a );
	len = mat_col( a );
	if (arow < 1 || arow > numrows || brow < 1 || brow > numrows || from < 1 || to < from || to > len )
		error_internal("out of bounds row nos. in mat_zm_row_add");

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));
	modulus = zm_modulus( ring );

	for ( i=from; i<=to; ++i )
	{
		temp1 = mat_elt( mat, arow, i );
		if (temp1 == 0)
			continue;
		temp1 = modint_mult( modulus, temp1, mult );
		temp2 = mat_elt( mat, brow, i );
		mat_elt( mat, brow, i ) = modint_add( modulus, temp1, temp2 );

		integer_delref( temp1 );
		integer_delref( temp2 );
	}

	mat_free_unpkd(a, mat);
}





void
mat_zm_row_mult WITH_6_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  arow,
	integer_big,    umult,
	integer_small,  from,
	integer_small,	to
)
/*
** replaces row arow of a with (umult * row arow) of a, where
** umult is a unit of the ring.
** Only does the replacing left to right from column "from".
*/
{
	block_declarations;
	register integer_big      temp1;
	register integer_small    i;
	register integer_small    len;
	register integer_small    numrows;
	register integer_big      modulus;
	matrix           mat = 0;

	numrows = mat_row( a );
	len = mat_col( a );
	if ( arow < 1 || arow > numrows || from < 1 || to < from || to > len )
		error_internal("out of bounds row no. in mat_zm_row_mult");

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));
	modulus = zm_modulus( ring );

	for ( i=from; i<=to; ++i )
	{
		temp1 = mat_elt( mat, arow, i );
		mat_elt( mat, arow, i ) = modint_mult( modulus, umult, temp1 );
		integer_delref( temp1 );
	}

	mat_free_unpkd(a, mat);
}




void
mat_zm_col_add WITH_7_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  acol,
	integer_small,  bcol,
	integer_big,    mult,
	integer_small,	from,
	integer_small,	to
)
/*
** adds (mult * column acol) of a to column bcol of a.
*/
{
	block_declarations;
	register integer_big      temp1;
	register integer_big      temp2;
	register integer_small    i;
	register integer_small    len;
	register integer_small    numcols;
	register integer_big      modulus;
	matrix           mat = 0;

	len = mat_row( a );
	numcols = mat_col( a );
	if (acol < 1 || acol > numcols || bcol < 1 || bcol > numcols || from < 1 || to < from || to > len )
		error_internal("out of bounds column nos. in mat_zm_col_add");

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));
	modulus = zm_modulus( ring );

	for ( i=from; i<=to; ++i )
	{
		temp1 = modint_mult( modulus, mat_elt( mat, i, acol ), mult );
		temp2 = mat_elt( mat, i, bcol );
		mat_elt( mat, i, bcol ) = modint_add( modulus, temp1, temp2 );
		integer_delref( temp1 );
		integer_delref( temp2 );
	}

	mat_free_unpkd(a, mat);
}





void
mat_zm_col_mult WITH_6_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  acol,
	integer_big,    umult,
	integer_small,	from,
	integer_small,	to
)
/*
** replaces column acol of a with (umult * column acol) of a, where
** umult is a unit of the ring.
*/
{
	block_declarations;
	register integer_big      temp1;
	register integer_small    i;
	register integer_small    len;
	register integer_small    numcols;
	register integer_big      modulus;
	matrix           mat = 0;

	len = mat_row( a );
	numcols = mat_col( a );
	if ( acol < 1 || acol > numcols || from < 1 || to < from || to > len )
		error_internal("out of bounds column nos. in mat_zm_col_mult");

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));
	modulus = zm_modulus( ring );

	for ( i=from; i<=to; ++i )
	{
		temp1 = mat_elt( mat, i, acol );
		mat_elt( mat, i, acol ) = modint_mult( modulus, umult, temp1 );
		integer_delref( temp1 );
	}

	mat_free_unpkd(a, mat);
}
