#include "defs.h"
#include "ring.h"
#include "error.e"
#include "zm.e"
#include "mat.h"

t_void
mat_zm_echelon WITH_10_ARGS(
	t_handle,	cring,
	matrix,	a,
	t_int	*,	det1,
	t_int	*,	rank,
	matrix *,	echelon,
	Logical *,	unit,
	Logical,	istoreduce,
	matrix	*,	transform,
	Logical,	istoinverse,
	t_int,		start_row
)
/*
 * this procedure will find out determinant, rank, standard echelon form, 
 * reduced echelon form, standard echelon transform, reduced echelon transform,
 * inverse and check whether a is unit.
 */
{
	block_declarations;

	register t_int	i; /* integer_small */
	register t_int	j; /* integer_small */
	register t_int	m; /* integer_small */
	register t_int	p;
	register t_int	col; /* integer_small */
	t_int				tem;
	t_int				mul;
	register t_int	pivrow; /* integer_small */
	t_int				value;
	register t_int	start; /* integer_small */
	t_int 			temp1; 
	t_int				temp2; 
	t_int				modulus;
	Logical			respack;
	Logical			respack1;
	Logical			isdiag;
	t_int				det;
	t_int				currow;
	t_int				nrow;
	t_int				ncol;
	matrix			adash;
	matrix			edash;
	matrix			tdash;
	Logical 		stop;
	t_int				*aptr;
	t_int				*eptr;

	adash = edash = tdash = 0;
	modulus = zm_modulus(cring);
	nrow = mat_row(a);
	ncol = mat_col(a);

	/* 
	 * if looking for determinant, unit or inverse
	 * but a is not square matrix
	 */
	if (
		((det1 != 0) || (unit != 0) || istoinverse)
				&&
		(nrow != ncol)
	)
	{
		error_internal("Non-squared matrix is found in mat_zm_echelon");
	}

	if (echelon != 0)
	{
		respack = mat_result_pkd(cring, *echelon);
		edash = 0;
		mat_alloc_result_unpkd(respack, *echelon, edash, nrow, ncol);
	}
	else
	{
		edash = mat_buff_alloc(nrow, ncol);
	}

	if (transform != 0)
	{
		respack1 = mat_result_pkd(cring, *transform);
		tdash = 0;
		mat_alloc_result_unpkd(respack1, *transform, tdash, nrow, nrow);
		mat_ring_create_id_sub(cring, tdash);
	}

	adash = 0;
	mat_create_unpkd(cring, a, adash, nrow, ncol);

	/*
	 * copy all elements from adash to edash
	 */

	aptr = mat_elt0_ptr(adash);
	eptr = mat_elt0_ptr(edash);
	
	for (i = 1; i <= (nrow * ncol); i++)
		eptr[i] = aptr[i];

	mat_incref_entries(cring, adash);

	currow = 0;
	isdiag = TRUE;
	
	if (unit != 0)
		*unit = TRUE;

	det = 1;

	for (col = 1; col <= ncol && currow < nrow; col++)
	{
		++currow;
		pivrow = 0;

		/*
		 * find a non-zero element of colume col
		 */

		for (j = currow; j <= nrow; j++)
		{
			value = mat_elt(edash, j, col);
			if (integer_sign(value))
			{
				integer_incref(value);
				pivrow = j;
				break;
			}
		}

		/*
		 * ...find element of smallest non-zero value
		 * in lower part of current column
		 */

		for (i = j; i <= nrow; i++)
		{
			tem = mat_elt(edash,i, col);
			if ((!integer_sign(tem)) || (integer_compare(tem, value)>=0))
				continue;
			integer_delete(&value);
			value = integer_incref(tem);
			pivrow = i;
		}
		/*
		 * ...stop if column was all zero
		 */
		if (pivrow == 0)
		{
			--currow;
			continue;
		}
		/*
		 * ...start to perform integral pivot on the (pivrow,col) 
		 * we do not bother to alter the above-diagonal part until
		 * the last time, when the pivot is + or - one.
		 */

		if (pivrow >= start_row)
			start = currow;
		else 
			start = start_row;

		stop = FALSE;
		while (!stop)
		{
			stop = TRUE;
			/*
			 * row reduction to form upper triangular
			 */
			for (i = start; i <= nrow; i++)
			{
				if (i == pivrow || (!integer_sign(mat_elt(edash, i, col))))
					continue;
				temp1 = integer_div(mat_elt(edash, i, col), value);
				tem = modint_negate( modulus, temp1 );
				integer_delete( &temp1 );

				mat_zm_row_add( cring, edash, pivrow, i, tem, col, ncol );

				if (transform != 0)
				{
					mat_zm_row_add( cring, tdash, pivrow, i, tem, 1, nrow );
				}

				integer_delete(&tem);
				/*
				 * ...have we found a smaller entry to pivot on
				 */
				if (integer_sign(mat_elt(edash, i, col)))
				{
					integer_delete(&value);
					value = integer_incref(mat_elt(edash, i, col));
					pivrow = i;
					stop = FALSE;
					break;
				}
			}
		}

		/*
		 * ...no smaller pivot was available
		 */
		temp1 = modint_mult(modulus, det, value);
		integer_delete(&det);
		det = temp1;
		temp1 = modint_negate(modulus, 1);
		if (
			integer_compare(value, 1)
				&&
			integer_compare(value, temp1)
		)
		{
			integer_delete(&temp1);
			temp2 = integer_gcd(modulus, value);
			if (!integer_compare(temp2, 1))
			{
				integer_delete(&temp2);
				temp1 = modint_invert( modulus, value );

				mat_zm_row_mult( cring, edash, pivrow, temp1, col, ncol );
				if (transform != 0)
				{
					mat_zm_row_mult( cring, edash, pivrow, temp1, 1, ncol );
				}

				integer_delete( &temp1 );
				integer_delete(&value);
				value = 1;
			}
			else  
			{
				integer_delete(&temp2);
				if (unit != 0)
				*unit = FALSE;
				DENY (istoinverse);
			}
		}
		else 
		{
			integer_delete(&temp1);
		}
		/*
		 * ...the column must now be all zero, except for the pivotal
		 * entry, which is plus or minus one.
		 */
		if (pivrow != currow)
		{
			/*
			 * ...swap rows pivrow and col to bring the pivotal entry to
			 * the main diagonal, and also make it plus one.
			 */
			temp1 = modint_negate(modulus, det);
			integer_delete(&det);
			det = temp1;

			mat_ring_row_swap( cring, edash, pivrow, currow, col, ncol );
			if (transform != 0)
			{
				mat_ring_row_swap( cring, tdash, pivrow, currow, 1, nrow );
			}
		}

		temp1 = modint_negate(modulus, value);
		if (integer_compare(temp1, value) < 0)
		{
			integer_delete( &temp1 );
			temp1 = ring_minus1( cring );
			mat_zm_row_mult( cring, edash, pivrow, temp1, col, ncol );
			if (transform != 0)
			{
				mat_zm_row_mult( cring, tdash, pivrow, temp1, 1, nrow );
			}
		}
		integer_delete(&temp1);
		integer_delete(&value);
	}

	if (istoreduce)
	{
		/*
		 * to reduce the matrix edash to be reduced echelon form
	 	 */
		if (istoinverse)
		{
			col = currow;
			for (i = currow; i >= 2; i--)
			{
				for (m = 1; m < i; m++)
				{
					if (!integer_sign(mat_elt(edash, m, col)))
					continue;
					mul = integer_incref(mat_elt(edash, m, col));
					if (echelon)
					{
						tem = mat_elt(edash, m, col);
						mat_elt(edash, m, col) = 0;
						integer_delete(&tem);
					}
					for (p = 1; p <= nrow; p++)
					{
						tem = mat_elt(tdash, m, p);
						temp1 = modint_mult(modulus, mul, mat_elt(tdash, i, p));
						mat_elt(tdash, m, p) = modint_subtract(modulus, tem, temp1);
						integer_delete(&tem);
						integer_delete(&temp1);
					}
					integer_delete(&mul);
				}
				col--;
			}
		}
		else
		{
			col = 1;
			for (i = 2; i <= currow; i++)
			{
				for (j = col + 1; j <= ncol; j++)
					if (integer_sign(mat_elt(edash, i, j)))
						break;
				if ( j > ncol )
					break;
				col = j;
				for (m = 1; m < i; m++)
				{
					if (!integer_sign(mat_elt(edash, m, j)))
						continue;
					else if (
						integer_compare(mat_elt(edash, m, j), mat_elt(edash, i, j)) < 0)
					continue;
					mul = integer_div(mat_elt(edash, m, j), mat_elt(edash, i, j));

					temp1 = mul;
					mul = modint_negate( modulus, temp1 );
					integer_delete( &temp1 );

					mat_zm_row_add( cring, edash, i, m, mul, j, ncol );
					if (transform != 0)
					{
						mat_zm_row_add( cring, edash, i, m, mul, 1, ncol );
					}
					integer_delete(&mul);
				}
			}
		}
	}

	if (isdiag)
	{
		if (det1 != 0)
			*det1 = det;
		else 
		{
			integer_delete(&det);
		}
	}
	else 
	{
		integer_delete(&det);
	}
	if (rank != 0)
		*rank = currow;
	if (echelon != 0)
	{
		mat_create_result(cring, respack, *echelon, edash);
	}
	else 
	{
		mat_delete_entries(cring, edash);
		mat_buff_free(&edash);
	}
	if (transform != 0)
	{
		mat_create_result(cring, respack1, *transform, tdash);
	}
	mat_free_unpkd(a, adash);
}
