#include "defs.h"
#include "ring.h"
#include "mat.h"
#include "q.h"
#include "error.e"

t_void
mat_fld_echelon WITH_10_ARGS(
	t_handle,		field,
	matrix,		a,
	integer_big *,	det1,
	t_int *, 		dim1,
	matrix *,	pb,
	Logical *,	unit,
	Logical,	istoreduce,
	matrix *,	red,
	Logical,	istoinverse,
	t_int,		start_row
)
/*
 * "field" is the coefficient field.
 * Given an m x n matrix "a", construct the reduced echelon
 * form "pb" of a. "start_row" is the row at which to start the echelonisation.
 *
 * The dimension of the row space of a is returned as "dim1".
 * The resulting vectors are returned in the matrix "red".
 *
 * For a square matrix, the determinant is returned as "*det1".
 * Note: For a non-square matrix, "*det1" will contain a value (ie occupied) but
 * the value will be garbage.
 *
 * In all cases, if an argument is a pointer, and it is 0 (the null pointer),
 * no assignment will be done.
 */
{
	register integer_small	m;
	register integer_small	n;
	register integer_small	i;
	register integer_small	k;
	register integer_small	l;
	register integer_small	p;
	register integer_small	below;
	register integer_small	pivot;
	register integer_small	dsquared;
	register integer_big	r;
	register integer_big	tem;
	integer_big	mul;
	register t_int	*ptr_a;
	register t_int	*ptr_b;
	matrix	mata;
	matrix	matb;
	matrix	matred;
	t_int		zero;
	t_int		one;
	integer_big		det;
	t_int		dim;
	t_int		j;
	t_int		last;
	integer_big	temp1;
	Logical	respack1;
	Logical	respack2;
	Logical	no_pb;
	Logical	no_red;
	integer_small	ring_t;
	t_pfl		ringeq;
	t_pfh		ringneg;
	t_pfh		ringmult;
	t_pfh		ringinv;

	ring_t = ring_type(field);
        
        ringeq   = ring_equality(field);
        ringneg  = ring_negation(field);
        ringmult = ring_multiplication(field);
        ringinv  = ring_inversion(field);

	mata = matb = matred = 0;
	if (start_row == 0)
		start_row = 1;
	m = mat_row(a);
	n = mat_col(a);
	DENY (istoinverse && m != n);
	zero = ring_zero(field);
	one = ring_one(field);

	no_pb = pb == 0;
	no_red = red == 0;
	if (! no_pb)
	{
		respack1 = mat_result_pkd(field, *pb);
		matb = 0;
		mat_alloc_result_unpkd(respack1, *pb, matb, m, n);
	}
	else
		matb = mat_buff_alloc(m, n);

	if (! no_red)
	{
		respack2 = mat_result_pkd(field, *red);
		matred = 0;
		mat_alloc_result_unpkd(respack2, *red, matred, m, m);
	}

	dsquared = m * n;

	if (no_pb || a != *pb)
	{
		mata = 0;
		mat_create_unpkd(field, a, mata, m, n);
		ptr_a = mat_elt0_ptr(mata);
		ptr_b = mat_elt0_ptr(matb);
		/*
		 * Copy mata to matb and increment reference counts.
		 */
		for (i = 1; i <= dsquared; i++)
		{
			ptr_b[i] = ptr_a[i];
			ring_elt_incref(field, ptr_a[i]);
		}
		mat_free_unpkd(a, mata);
	}
	if (! no_red)
		mat_ring_create_id_sub(field, matred);

	det = one;
	ring_elt_incref(field, one);
	dim = 0;
	pivot = start_row - 1;
	/*
	 * Run through the columns, reducing the lower half of the matrix
	 * to zeroes.
	 * l is the column.
	 */
	for (l = 1; l <= n; l++)
	{
		/*
		 * Point to the next pivot element
		 */
		pivot++;
		if (pivot > m)
		{
			break;
		}
		below = pivot + 1;
		/*
		 * is the pivot element zero?
		 */
		if ((*ringeq) (field, mat_elt(matb, pivot, l), zero))
		{
			/*
			 * it is so find a non-zero element below it.
			 */
			for (k = below; k <= m; k++)
				if (! (*ringeq) (field, mat_elt(matb, k, l), zero))
					break;
			if (k > m)
			{
				/*
				 * This column does not contain a pivot element,
				 * so prepare to move to the next column
				 */
				pivot--;
				continue;
			}
			/*
			 * swap the rows given by pivot and k.
			 */
			mat_ring_row_swap(field, matb, pivot, k, 1, n);
			if (! no_red)
				mat_ring_row_swap(field, matred, pivot, k, 1, m);
			/*
			 * Correct determinant.
			 */
			temp1 = det;
			det = (*ringneg) (field, temp1);
			ring_elt_delete(field, &temp1);
		}
		/*
		 * rescale to make pivot element the identity.
		 */
		dim++;
		tem = mat_elt(matb, pivot, l);
		/*
		 * if not equal to one
		 */
		if (! (*ringeq) (field, tem, one) )
		{
			temp1 = det;
			det = (*ringmult) (field, temp1, tem);
			ring_elt_delete(field, &temp1);
			mul = (*ringinv) (field, tem);
			/*
			 * divide throughout by pivot element.
			 */
			mat_ring_row_mult(field, matb, pivot, mul, l, n);
			if (! no_red)		 /* yes, below, the last number is a ONE */
				mat_ring_row_mult(field, matred, pivot, mul, 1, m);
			ring_elt_delete(field, &mul);
		}

		/*
		 * now subtract multiples of this row from those below it
		 */
		for (p = pivot + 1; p <= m; p++)
		{
			/*
			 * the particular multiple is given by matb[p, l]
			 */
			tem = mat_elt(matb, p, l);
			if ((*ringeq) (field, tem, zero))	/* if == zero */
				continue;
			mul = (*ringneg) (field, tem);
			/*
			 * calculate the reduced row.
			 */
			mat_ring_row_add(field, matb, pivot, p, mul, l, n);
			if (! no_red)		 /* yes, below, the last number is a ONE */
				mat_ring_row_add(field, matred, pivot, p, mul, 1, m);
			ring_elt_delete(field, &mul);
		}
	}

	/*
	 * matrix is now upper triangular.
	 * clear the upper triangle if required.
	 */
	if (istoreduce)
		for (p = dim; p >= start_row + 1; p--)
		{
			last = p - start_row + 1;
			while (last <= n && (*ringeq) (field, mat_elt(matb, p, last), zero))	/* if == zero */
				last++;
			if (last > n)
				error_internal("Bug in mat_fld_echelon");
			for (j = 1; j <= p - 1; j++)
			{
				/*
				 * the particular multiple is given by matb[j, last]
				 */
				tem = mat_elt(matb, j, last);
				if ((*ringeq) (field, tem, zero))	/* if == zero */
					continue;
				mul = (*ringneg) (field, tem);
				/*
				 * calculate the reduced row.
				 */
				if (! no_pb)
					mat_ring_row_add(field, matb, p, j, mul, last, n);
				if (! no_red)		 /* yes, below, the last number is a ONE */
					mat_ring_row_add(field, matred, p, j, mul, 1, m);
				ring_elt_delete(field, &mul);
			}
		}


	/*
	 * matrix is now in echelon form: return
	 */
	if (dim != m)
	{
		ring_elt_delete(field, &det);
		det = zero;
		ring_elt_incref(field, zero);
	}
	if (unit)
	{
		if (ring_type(field) == RING_MAT)
		{
			*unit = mat_fld_unit(field, det);
		}
		else
			*unit = ( ! (*ringeq) (field, det, zero) );	/* if != zero */
	}
	if (det1)
		*det1 = det;
	else
		ring_elt_delete(field, &det);

	DENY (istoinverse && dim != m);
	if (dim1)
		*dim1 = dim;
	if (! no_pb)
	{
		mat_create_result(field, respack1, *pb, matb);
	}
	else
	{
		mat_delete_entries(field, matb);
		mat_buff_free(&matb);
	}

	if (! no_red)
		mat_create_result(field, respack2, *red, matred);

	ring_elt_delete(field, &zero);
	ring_elt_delete(field, &one);

	return;
}
