#include "defs.h"
#include "dyn_arr.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"

extern Int	iuz[];

Void
i_medium_prime_divisor WITH_5_ARGS(
    integer_big,       aint,
    integer_big,       min,
    integer_big,       max,
    integer_big *,     prime,
    integer_big *,     quot
)
/*
** input:  aint - single or infinite-precision positive integer
** min, max - single or infinite-precision positive integers
** s.t. min <= max <= aint, and aint has no positive divisors
** less than min
** output: prime, quot - single or infinite-precision positive integers
** impds computes the medium prime divisor search.  if aint has a prime
** divisor in the interval [min, max], then prime is the least such prime
** and quot = aint / prime.  otherwise, prime = 1 and quot = aint.
*/
{
	block_declarations;

	t_int    r1;
	t_int    r2;
	t_int    rem;
	inthdl_handle    ahdl;
	inthdl_length    alen;
	inthdl_handle    qhdl;
	inthdl_handle    rhdl;
	inthdl_handle    phdl;
	t_int    uindex;
	t_int    pdash;
	t_int	i, ind, m, k, phi;
	dyn_arr_handle	uz;


	if (integer_is_single( aint ))
		ahdl = inthdl_alloc_1( aint );
	else
		ahdl = inthdl_big_to_handle( integer_incref( aint ));

	alen = intbig_curr_size( ahdl );
	qhdl = inthdl_alloc(alen);
	rhdl = inthdl_alloc(alen);

	/*
	 * determine first possible divisor.
	 */
	k = 210;
	phi = 48;
	m = integer_rem(aint, k);
	if ( integer_rem(m, 2) == 0 )
		k = k/2;
	if ( integer_rem(m, 3) == 0 )
	{
		k = k/3;
		phi = phi/2;
	}
	if ( integer_rem(m, 5) == 0 )
	{
		k = k/5;
		phi = phi/4;
	}
	if ( integer_rem(m, 7) == 0 )
	{
		k = k/7;
		phi = phi/6;
	}
	m = integer_rem(m, k);
	uz = dyn_arr_alloc( phi );


        ind = 0;
	if( k == 1)
	{
		dyn_arr_element( uz, ind ) = 0;
		ind++;
	}
	else
	{
		for( i = 1; i<= k; i++)
		{
			if( integer_gcd(k, i ) == 1 )
			{
				dyn_arr_element( uz, ind ) = i;
				ind++;
			}
		}
	}

	rem = integer_rem( min, k );
	uindex = 0;
	while (rem > dyn_arr_element(uz, uindex))
	{
		++uindex;
	}
	r1 = dyn_arr_element(uz, uindex);
	*prime = integer_add(min, r1 - rem);

	/*
	 * repeated trials of primes in the range [min, max]
	 * start of while loop, stop when next prime to be test > max
	 */
	while (integer_compare(*prime, max) <= 0)
	{
		if (integer_is_single(*prime))
			inthdl_quot_rem_beta( ahdl, *prime, qhdl, &rem );
		else
		{
			phdl = inthdl_big_to_handle( *prime );
			inthdl_quot_rem( ahdl, phdl, qhdl, rhdl );
			/*
			** need rem non-zero if and only if rhdl non-zero
			*/
			rem = rhdl;
		}

		/*
		 * case prime divides aint -- success
		 */
		if ( integer_sign( rem ) == 0)
		{
			*quot = inthdl_standardize( qhdl );
			inthdl_delref( ahdl );
			inthdl_delref( rhdl );
			dyn_arr_delete(&uz);
			return;
		}

		/*
		 * case doesn't divide -- continue by finding next prime
		 */
		++uindex;
		if (uindex > phi-1)
		{
			uindex = 0;
			r2 = r1 - k;
		}
		else
			r2 = r1;

		r1 = dyn_arr_element(uz, uindex);
		if (integer_is_single(*prime))
		{
			*prime = integer_add( *prime, r1 - r2 );
		}
		else
		{
			pdash = *prime;		/* transfer of reference */
			*prime = integer_add( pdash, r1 - r2 );
			integer_delref(pdash);
		}
	}

	dyn_arr_delete(&uz);
	/*
	 * end of while loop to test each prime in range [min, max]
	 * no divisors were found....
	 */
	integer_delref( *prime );
	*prime = 1;

	*quot = integer_incref( aint ) ;
	inthdl_delref( ahdl );
	inthdl_delref( qhdl );
	inthdl_delref( rhdl );
}
