/******************************************************************************
  order_reg_lbound.c
******************************************************************************/
#include "kant.h"

t_real
order_reg_lbound WITH_3_ARGS (
	order,		ord,
	integer_small,	k,
	t_real,		user_C

)
/******************************************************************************
 
Description:	Gives a lower bound for the regulator of an order.

		The program computes all units whose T2-norm is
		less than a bound C.

Calling sequence:
 
		l_bound = order_reg_lbound(ord,k,user_C)

		t_real		l_bound	: lower regulator bound
		order		ord	: t_handle of order
		integer_small   k	: to get a stronger estimate
					  the program will run (increasing C)
					  until k independent units 
					  are found.
					  If k = 0 the program automatically
					  sets k = 1.
		t_real		user_C	: starting value for C.
					  If user_C is not specified
					  C is set up to a default.

History:

	93-03-01 KW	using the units which are known already
	93-02-04 KW	simplified (50% faster)
	93-01-25 KW	simplified
	92-06-05 KW	written
 
******************************************************************************/
{
	block_declarations;

	anf_elt		alpha;
	dyn_arr_handle	unit,unitt2,unitlog;
	dyn_arr_handle	succ_min,tinu;
	t_handle	R;
	integer_small	i,j,m,mm,rk,rank;
	lat_elt		elt;
	lat_enum_env	env,lll_env;
	lattice		lat,lll_lat;
	matrix		gram,iula,iulb,iulc,iult,trans,invtrans;
	t_real		tempt2;
	t_real		tempp,tempq,tempr,temps,tempt,tempu;
	t_real		tempv,tempw,tempx,tempy,tempz;
	vector		v;

	if (anf_print_level > 0) printf("Compute lower regulator bound...\n");

	R = order_reals(ord);

	if (!k) k=1;
	k = (k > order_r(ord)) ? order_r(ord) : k;
	if (!k) return(ring_one(R));

/*
**	Compute LLL-reduced basis of ord.
*/
	order_lat(ord,&lat,&env);
	lll_lat = lat_lll_reduce(lat,&trans,&invtrans);
	lat_chol_calc(lll_lat);
	gram = lat_gram(lll_lat);
	rank = lat_rank(lll_lat);
	lll_env = lat_enum_create(lll_lat);
	lat_enum_request_set_next(lll_env);

/*
**	Set upper bound for enumeration.
*/
	if (user_C)
	{
		lat_enum_ubound(lll_env)= real_incref(user_C);
	}
	else
	{
		tempu	= (lat_gram_is_over_z(lll_lat))
			? conv_int_to_real(R,mat_elt(gram,rank,rank))
			: real_incref(mat_elt(gram,rank,rank));
		lat_enum_ubound(lll_env) = real_add(R,tempu,tempu);
		real_delete(&tempu);
	}

/*
**	Start enumeration.
*/
	unit    = dyn_arr_alloc(1);
	unitt2  = dyn_arr_alloc(1);
	unitlog = dyn_arr_alloc(1);
	m = mm = 0;
	iula = rk = 0;
	do
	{
		lat_enum_status_set_new(lll_env);
		if (anf_print_level > 0)
		{
			printf("T2 bound for enumeration is ");
			real_write(R,lat_enum_ubound(lll_env),20);
			printf(".\n");
		}
		while (lat_enum(lll_lat,lll_env))
		{
			elt = lat_elt_move(lll_lat,lat_enum_act_coefs(lll_env),trans);
			alpha = lat_elt_to_anf_elt(lat,elt,ord);
			lat_elt_delete(lat,&elt);

/*
**			Did we find an unit ?
*/
			if (anf_elt_is_unit(ord,alpha))
			{
				v = anf_elt_logs(ord,alpha);
/*
**				Is alpha a torsion unit ?
*/
				tempu = order_r1(ord) ? 0 : vec_ring_dot_product(R,v,v);

				if (tempu && (real_zero_eps(R,tempu,MEM_NH)))
				{
					anf_elt_factorize(ord,alpha);
				}
				else
				{
					m++;
					dyn_arr_assure_space_fun(unit   ,m,10);
					dyn_arr_assure_space_fun(unitt2 ,m,10);
					dyn_arr_assure_space_fun(unitlog,m,10);
					dyn_arr_element(unit   ,m-1) = anf_elt_incref(alpha);
					dyn_arr_element(unitt2 ,m-1) = real_incref(lat_enum_act_length(lll_env));
					dyn_arr_element(unitlog,m-1) = vec_incref(v);
				}
				vec_delete(R,&v);
				if (tempu) real_delete(&tempu);
			}
			anf_elt_delete(ord,&alpha);
		}
/*
**		Count independent units.
*/
		for (i=mm+1;(i<=m) && (rk < order_r(ord));i++)
		{
			v = dyn_arr_element(unitlog,i-1);
			if (mat_real_orthonormal_expand(R,iula,v,0,&iulb))
			{
				rk++;
				if (iula) mat_delref(R,&iula);
				iula = iulb;
			}
		}
		if (anf_print_level > 0)
			printf("order_reg_lbound: %d units found (%d independent).\n",m,rk);

/*
**		Increase bounds in enumeration enviroment.
*/
		if (lat_enum_lbound(lll_env)) real_delete(&lat_enum_lbound(lll_env));
		lat_enum_lbound(lll_env) = lat_enum_ubound(lll_env);
		lat_enum_ubound(lll_env) = real_add(R,lat_enum_ubound(lll_env),lat_enum_ubound(lll_env));
		mm = m;
	}
	while (rk < k);
	mat_delref(R,&iula);

/*
**	Compute higher successive minima.
**	First step: M*
*/
	succ_min = dyn_arr_alloc(order_r(ord));
	tinu = dyn_arr_alloc(rk);

	for (j=1;j<=rk;j++)
	{
		iulb = 0;
 		tempt2 = real_add(R,lat_enum_lbound(lll_env),lat_enum_lbound(lll_env));
		for (i=1;i<=m;i++)
		{
			if (!dyn_arr_element(unit,i-1)) continue;
			if (real_compare(R,dyn_arr_element(unitt2,i-1),tempt2) == -1)
			{
				if (mat_real_orthonormal_expand(R,iula,dyn_arr_element(unitlog,i-1),0,&iulc))
				{
					real_delete(&tempt2);
					tempt2 = real_incref(dyn_arr_element(unitt2,i-1));
					if (iulb) mat_delref(R,&iulb);
					iulb = iulc;
					anf_elt_delete(ord,&dyn_arr_element(tinu,j-1));
					dyn_arr_element(tinu,j-1) = anf_elt_incref(dyn_arr_element(unit,i-1));
				}
				else
				{
					anf_elt_delete(ord,&dyn_arr_element(unit,i-1));
				}
			}
		}
		dyn_arr_element(succ_min,j-1) = tempt2;
		if (iula) mat_delref(R,&iula);
		iula = iulb;
	}
	mat_delref(R,&iula); 

	if(order_units_count(ord))
	{
/*
**		Using the units which are known already.
*/
		for (j=1;j<=rk;j++)
		{
			anf_elt_factorize(ord,dyn_arr_element(tinu,j-1));
			anf_elt_delete(ord,&dyn_arr_element(tinu,j-1));
		}
		order_relation_eval_units(ord);
	}
	else
	{
/*
**		There are no units in ord.
*/
		order_units_count_set(ord,rk);
		for (j=1;j<=rk;j++) order_unit(ord,j) = dyn_arr_element(tinu,j-1);
	}
	dyn_arr_delete(&tinu);

	for (j=rk+1;j<=order_r(ord);j++)
		dyn_arr_element(succ_min,j-1) = real_incref(lat_enum_lbound(lll_env));

/*
**	Second step: M tilde.
*/
	tempp = conv_int_to_real(R,4);
	tempq = ring_one(R);
	tempr = conv_int_to_real(R,order_abs_degree(ord));
	for (j=1;(j<=rk+1) && (j <= order_r(ord));j++)
	{
		temps = real_divide(R,dyn_arr_element(succ_min,j-1),tempr);
		tempt = real_mult(R,temps,temps);
		tempu = real_subtract(R,tempt,tempq);
		tempv = real_sqrt(R,tempu);
		tempw = real_add(R,temps,tempv);
		tempx = real_log(R,tempw);
		tempy = real_mult(R,tempx,tempx);
		tempz = real_mult(R,tempy,tempr);
		real_delete(&dyn_arr_element(succ_min,j-1));
		dyn_arr_element(succ_min,j-1) = real_divide(R,tempz,tempp);
		real_delete(&temps);
		real_delete(&tempt);
		real_delete(&tempu);
		real_delete(&tempv);
		real_delete(&tempw);
		real_delete(&tempx);
		real_delete(&tempy);
		real_delete(&tempz);
	}
	real_delete(&tempp);
	real_delete(&tempq);
	real_delete(&tempr);

/*
**	Compute lower regulator bound
*/

	temps = conv_int_to_real(R,order_abs_degree(ord));
	tempt = real_hermite_const_ubound(R,order_r(ord));
	tempu = conv_int_to_real(R,2);
	tempv = real_power(R,tempu,order_r2(ord));        
	tempw = real_divide(R,tempv,temps);
	tempx = real_divide(R,tempw,tempt);

	for (j=1;j<=rk;j++)
	{
		tempy = tempx;
		tempx = real_mult(R,tempx,dyn_arr_element(succ_min,j-1));
		real_delete(&tempy);
	}

	if (rk < order_r(ord))
	{
		tempy = real_power(R,dyn_arr_element(succ_min,rk),order_r(ord)-rk);
		tempz = tempx;
		tempx = real_mult(R,tempx,tempy);
		real_delete(&tempy);
		real_delete(&tempz);
	}
	tempy = real_sqrt(R,tempx);

	real_delete(&temps);
	real_delete(&tempt);
	real_delete(&tempu);
	real_delete(&tempv);
	real_delete(&tempw);
	real_delete(&tempx);

/*
**	Delete...
*/
	for (j=0;j<order_r(ord);j++)
		real_delete(&dyn_arr_element(succ_min,j));
	dyn_arr_delete(&succ_min);

	for (j=0;j<m;j++)
	{
		anf_elt_delete(ord,&dyn_arr_element(unit,j));
		real_delete(&dyn_arr_element(unitt2,j));
		vec_delete(R,&dyn_arr_element(unitlog,j));
	}
	dyn_arr_delete(&unit);
	dyn_arr_delete(&unitt2);
	dyn_arr_delete(&unitlog);

	lat_enum_delete(lat,&env);
	lat_delete(&lat);
	lat_enum_delete(lll_lat,&lll_env);
	lat_delete(&lll_lat);
	mat_delref(structure_z,&trans);
	mat_delref(structure_z,&invtrans);

	if (anf_print_level > 0)
	{
		printf("Lower regulator bound is "); 
		real_write(R,tempy,20); printf("\n");
	}

	return(tempy);
}
