/*******************************************************************************
  anf_rep_mat_mod.c
********************************************************************************/


#include "kant.h"
#include "integer.e"
#include "mat.h"
#include "anf.h"


matrix
anf_rep_mat_mod WITH_3_ARGS(
    	    order,	  ord,
            integer_big,  modul,
	    anf_elt,	  alpha
)
/*******************************************************************************
Description:

	Representation matrix of an algebraic number modulo a integer modul.
	Careful! This routine does not care for any denominators, it assumes  
	that the denominator of alpha is one and actually computes the
	representation matrix of the numerator of alpha.
	The calculation is done straightforward using the multiplication table.
        
         
Calling sequence:


        mat = anf_rep_mat_mod (ord,modul,alpha);


            order        ord   : all operations are done in the order.
            integer_big  modul : since we are computing something modular
                                 we need a modul. 
            anf_elt      alpha : alg. number defined over ord.

            matrix       mat   : the representation matrix of alpha with 
                                 respect to the basis of ord.
 
History:         

        MD 92.03.23  order_must_be_over_z added.
        MD 91.12.10  first modular version.        
	JS 91.09.17  first non-modular version.
        
********************************************************************************/
{
	block_declarations;
 
	matrix		mat;
	order		ordcoef;
	integer_small	deg;
	anf_elt		accu, temp1, temp2;
	integer_small	i, j, k;
                                             
        
/*  
    Is the order equal Z?
*/
 
	if (ring_type(ord) == RING_Z) 
	{
		mat = mat_new(1,1);
		mat_elt(mat, 1, 1) = integer_rem (alpha,modul);
		return mat;
	}
            
/* All Comp. are done Modular! So we need the following restriction */
        order_must_be_over_z (ord);

	deg = order_rel_degree(ord);
        ordcoef = order_coef_order(ord);
 
 
/*
   Is alpha an integer? In this case the result is a diagonal matrix.
*/
 
	if (anf_elt_is_integer(alpha))
	{
		mat = mat_ring_create_zero(ordcoef, deg);
		for (i=1; i<=deg; ++i)
			mat_elt(mat,i,i) = integer_rem(alpha,modul);
		return mat;
	}
		
	
/*  
    Allocation
*/
	mat = mat_new(deg, deg);
 
 
/*
    Loop for matrix elements. The representation matrix is the scalar product
    of the coefficient vector of alpha times the array of multiplication
    matrices in the multiplication table.
*/
       	for (i=1; i<=deg; ++i)
	{
		for (j=1; j<=deg; ++j)
		{
			accu = 0;
			for (k=1; k<=deg; ++k)
			{             
		          temp1 = accu;
			  temp2 = anf_mult_mod(ordcoef, modul, anf_elt_coef(alpha,k),
     		                                               mat_elt(order_mult_matrix(ord,k), i, j));
			  accu  = anf_add_mod (ordcoef, modul, temp1, temp2);
			  anf_elt_delete(ordcoef, &temp1);
			  anf_elt_delete(ordcoef, &temp2);
			}
			mat_elt(mat, i, j) = accu;
		}
	}

 
  return mat;
}
