/******************************************************************************
  anf_elt_minpoly.c                                                           
 
  This file contains:
  anf_elt_minpoly
  anf_elt_minpoly_real
  complex_mult

******************************************************************************/
 
#include <stdio.h>
#include "kant.h"
#include "integer.e"
#include "z.e"
#include "real.e"
#include "conv.e"
#include "poly.h"
#include "vector.h"        
#include "mat.h"
#include "anf.h"

/* private function */
static void complex_mult P_((t_handle, t_handle, t_handle, t_handle, t_handle, t_handle *, t_handle *));

t_poly
anf_elt_minpoly WITH_2_ARGS(
        order,        ord,
        anf_elt,      beta)

/*********************************************************************************

Description: 
	
	Creates the minimal polynomial over Z to the element beta
	and takes care of the precision. The precision of the real values of
        ord might be changed.
 
        In case beta is given via conjugates and the rpecision is not sufficient
        errors might occur. In this case the conjugates should be recomputed
        in this routine using anf_con_to_anf_elt (which does not exist yet).

Calling sequence: 
	
	p = anf_elt_minpoly(ord, beta);

	order		ord   = t_handle to the order
	anf_elt		beta  = t_handle to algebraic number
        t_poly       p     = minimal polynomial of alpha

History:

	92-04-10 JS 	ipoly_gcd now used
	92-03-31 JS 	anf_con_elt if necessary
	92-03-24 JPS 	first version
*********************************************************************************/
{
	block_declarations;

	t_poly		pol;
	integer_small		real_prec, int_prec, i, nterms;	
        integer_big		maxx, old_max, abss;
        anf_elt			beta1;
        t_handle			hpol;

	real_prec = order_real_prec(ord );
/*
    calculating real values of ord if necessary
*/
	if (real_prec == 0)
		{ 
                order_reals_set(ord, 20);
		real_prec = 20;
		order_reals_create(ord );
		} 
/*
    first try
*/
	pol = anf_elt_minpoly_real(ord, beta);
	maxx = 0;
	hpol = m_poly_poly_to_handle(pol );
	nterms = m_poly_nterms(hpol );        
	for(i=0; i<nterms; ++i)
		{ 
		old_max = maxx;
		abss = integer_abs(m_poly_coefft(hpol, i) );
		maxx = integer_max(maxx, abss);
		integer_delete(&old_max );
		integer_delete(&abss );
               	}
	int_prec = integer_num_decimal_digits(maxx );
	integer_delete(&maxx );
/*
   if the precision was not sufficient we enlarge it
*/
        if (int_prec + 5 > real_prec)
		{
		order_reals_set(ord, int_prec+5);       
                order_reals_create(ord );
                m_poly_z_delref(structure_pring_z, pol );
                beta1 = anf_elt_is_con(beta)
                        ? anf_con_elt(ord, beta)
                        : anf_elt_incref(beta);  
		pol = anf_elt_minpoly_real(ord, beta1);
                anf_elt_delete(ord, &beta1);
		}
	return pol;            
}


t_poly
anf_elt_minpoly_real WITH_2_ARGS(
        order,        ord,
        anf_elt,      beta)

/*********************************************************************************

Description: 

 	Creates the minimal polynomial over Z to the element alpha, using
	Newton's relations.
	This routine should not be called by the user, please take
	anf_elt_minpoly which takes care of the precision! 
 
        In this version the polynomial which is computed might be a power
        of the minimal polynomial.

Calling sequence:

	p = anf_elt_minpoly_real(ord, beta);

	order		ord   = t_handle to the order
	anf_elt		beta  = t_handle to algebraic number		
        t_poly       p     = minimal polynomial of alpha

History:

	92-05-25 JS	anf_print_level
	92-03-24 JPS 	first version

*********************************************************************************/
{
	block_declarations;

	t_poly         pol, pol1, old_pol, gcd_pol;
	anf_elt		  alpha;
        integer_small     i, j, r1, r2, r12, deg, poldeg, k, s, l, pointer, m;

        t_handle		  Z, R, CR, CI, I;
	t_real		  tempr, tempi, tempr_old, tempi_old, temp, temp_old, sqrt2;
	integer_big	  koeff;
        t_logical           exit;    
	t_void	          complex_mult();

	Z = m_z_str_incref(structure_z);
 	deg = order_abs_degree(ord);
	R = order_reals(ord);

	if (anf_elt_is_integer(beta))
 
		{ 
		/*
		trivial case : beta is integer
		*/
 		m_poly_u_create_empty(&pol, deg+1);
		m_poly_coefft(pol, 1) = 1;
                m_poly_expt(pol, 1) = 1;
		m_poly_coefft(pol, 0) = integer_negate(beta);
		m_poly_expt(pol, 0) = 0;
		return poly_z_clean(structure_pring_z, m_poly_handle_to_poly(pol));
	        }
	else
		{ 
	        CR = vec_new(deg);
		CI = vec_new(deg);
		I = vec_new(deg); 

 		if (!anf_elt_is_con(beta))
			{                               
			alpha = anf_elt_con(ord, beta);
			}
		else 
			{
			alpha = anf_elt_incref(beta); 
			}                                                     
		r1  = order_r1(ord);
		r2  = order_r2(ord);
		r12 = r1 + r2;
		sqrt2 = order_sqrt_2(ord );
		m_poly_u_create_empty(&pol, deg+1);
		/*
		store conjugates in CR and CI
		*/
		for (i=1; i<=r1; ++i)
			{
	  		vec_entry(CR, i) = real_incref(anf_con(alpha, i));
                        vec_entry(CI, i) = ring_zero(R);
			}      
		for (i=r1+1; i<=r12; ++i)
			{
			vec_entry(CR, i)    = real_divide(R, anf_con(alpha, i), sqrt2);
			vec_entry(CR, i+r2) = real_divide(R, anf_con(alpha, i), sqrt2); 
			vec_entry(CI, i)    = real_divide(R, anf_con(alpha, i+r2), sqrt2);
			vec_entry(CI, i+r2) = real_negate(R, anf_con(alpha, i+r2));
                        temp = vec_entry(CI, i+r2);
			vec_entry(CI, i+r2) = real_divide(R, temp, sqrt2);
                        real_delete(&temp);
			}
		for (k=1; k<=deg; ++k)
			/*
			create all subsets of {1...n} 
			*/
			{
			temp = ring_zero(R);
			for (l=1; l<=k; ++l)
				vec_entry(I, l) = l;
			exit = FALSE;
			pointer = k;	                        
			do
				{
				tempr = conv_int_to_real(R, 1);
				tempi = ring_zero(R);   	
				for (j=1; j<=k; ++j)                            
					{
					tempr_old = tempr;
					tempi_old = tempi;
					complex_mult( R,
						      vec_entry(CR,
								vec_entry(I, j)),
						      vec_entry(CI,
								vec_entry(I, j)),		      	
       	        	                              tempr,
       	   					      tempi,
          					      &tempr,
       	   					      &tempi
       	   				   	    ); 
					real_delete(&tempr_old);
					real_delete(&tempi_old);
					}
				temp_old = temp;
				temp = real_add(R, temp, tempr);
    	   	   		real_delete(&temp_old);
				/*
				next subset of {1...n}
				*/
				m = vec_entry(I, pointer);
				++m;
				if (m>deg-k+pointer)
					{
					do        
						{
						--pointer;
						if (pointer == 0) exit = TRUE;	
						}
					while ( (vec_entry(I, pointer) >= deg-k+pointer) && (!exit));
          				++vec_entry(I, pointer);
          				for(l=1; l<=k-pointer; ++l)
          					vec_entry(I, l+pointer)=vec_entry(I, pointer) + l;
					pointer = k;
					}
				else	 
                                 	{
					vec_entry(I, pointer) = m;
					} 
				real_delete(&tempr);
				real_delete(&tempi);
                                }
			/*
			vorzeichen bestimmen
			*/
	   		while (!exit);
                        if (anf_print_level > 2)
                        {
                                printf("anf_elt_minpoly: Coefficient of %d: ",deg-k);
                                real_write(R, temp, 50);
                                puts("");
                        }
	                koeff = conv_real_to_int_round(R, temp);
			real_delete(&temp);
			s= (k&1) ? -1 : 1 ;
			m_poly_coefft(pol, deg-k) = integer_mult(koeff, s);
	                m_poly_expt(pol, deg-k) = deg-k; 
			integer_delete(&koeff);
			}
		/*
		polynom-koeffizienten belegen
		*/		
                m_poly_coefft(pol, deg) = 1;
		m_poly_expt(pol, deg) = deg;
		anf_elt_delete(ord, &alpha);             
		vec_delete(R, &CR);                     
	 	vec_delete(R, &CI);
		vec_delete(Z, &I);
		}
	old_pol = pol;
	pol = poly_z_clean(structure_pring_z, m_poly_handle_to_poly(pol));
        m_poly_z_delref(structure_pring_z, old_pol);
	/*
	ggt(pol, pol') berechnen, und falls nicht trivial : pol / ggt(pol, pol')
	*/	                                          
                                                                  
	pol1 = poly_z_deriv_princvar(structure_pring_z, pol );
	gcd_pol = poly_z_gcd(structure_pring_z, pol, pol1);
 
	if (m_poly_not_const(gcd_pol))
		{ 
		old_pol = pol;
		pol = poly_z_div(structure_pring_z, pol, gcd_pol);
                m_poly_z_delref(structure_pring_z, old_pol );
		}
 
	m_poly_z_delref(structure_pring_z, pol1);
	m_poly_z_delref(structure_pring_z, gcd_pol);
 
	return pol;   	
} 
 

static
void complex_mult WITH_7_ARGS( 

	t_handle,	      R,
	t_handle,	      r1,
	t_handle,	      i1,
 	t_handle,	      r2,
        t_handle,       i2,
	t_handle*,      r3,
	t_handle*,      i3     ) 

/*********************************************************************************

Description: 

	Complex multiplication

Calling sequence: 

    	t_handle	 	R   = reals
	t_handle		r1  = 1. number  RE
	t_handle	  	i1  = 1. number  IM
 	t_handle		r2  = 2. number  RE
        t_handle  	i2  = 2. number  IM
	t_handle*  	r3  = product    RE
	t_handle*    	i3  = product    IM

History:

	92-03-24 JPS written

*********************************************************************************/
{
	block_declarations;

	t_real		temp1, temp2;

        temp1 = real_mult(R, r1, r2);
	temp2 = real_mult(R, i1, i2);
        *r3   = real_subtract(R, temp1, temp2);

	real_delete(&temp1);
	real_delete(&temp2);

	temp1 = real_mult(R, r1, i2);
	temp2 = real_mult(R, r2, i1);
	*i3   = real_add(R, temp1, temp2);

	real_delete(&temp1);
	real_delete(&temp2);
} 

