/***************************************
  $Revision: 1.16 $

  SQL module (sq) - this is a MySQL implementation of the SQL module.

  Status: NOT REVUED, NOT TESTED

  ******************/ /******************
  Filename            : mysql_driver.c
  Author              : ottrey@ripe.net
  OSs Tested          : Solaris
  ******************/ /******************
  Copyright (c) 1999                              RIPE NCC
 
  All Rights Reserved
  
  Permission to use, copy, modify, and distribute this software and its
  documentation for any purpose and without fee is hereby granted,
  provided that the above copyright notice appear in all copies and that
  both that copyright notice and this permission notice appear in
  supporting documentation, and that the name of the author not be
  used in advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.
  
  THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
  AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
  DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
  ***************************************/
#include <stdlib.h>
#include <stdio.h>
#include <sys/timeb.h>
#include <strings.h>

#include "mysql_driver.h"
#include "constants.h"

/*+ String sizes +*/
#define STR_S   63
#define STR_M   255
#define STR_L   1023
#define STR_XL  4095
#define STR_XXL 16383


/* log_query() */
/*++++++++++++++++++++++++++++++++++++++
  Log the query.  This should/will get merged with a tracing module.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
static void log_query(const char *logfile, const char *query, struct timeb *start, struct timeb *stop) {
  FILE *logf;
  int seconds;
  int milliseconds;

  seconds = (int)(stop->time - start->time);
  milliseconds = (int)(stop->millitm - start->millitm);
  if (milliseconds < 0) {
    milliseconds += 1000;
    seconds--;
  }

  if (strcmp(logfile, "stdout") == 0) {
    printf("query=[%s] took %d sec %d msec\n", query, seconds, milliseconds);
  }
  else {
    logf = fopen(logfile, "a");
    fprintf(logf, "query=[%s] took %d sec %d msec\n", query, seconds, milliseconds);
    fclose(logf);
  }

} /* log_query() */

/* SQ_get_connection() */
/*++++++++++++++++++++++++++++++++++++++
  Get a connection to the database.

  const char *host
  
  unsigned int port

  const char *db
  
  const char *user
  
  const char *password
   
  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_init">mysql_init()</A>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_real_connect">mysql_real_connect()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
SQ_connection_t *SQ_get_connection(const char *host, unsigned int port, const char *db, const char *user, const char *password) {

  SQ_connection_t *sql_connection;

  sql_connection = mysql_init(NULL);
  if (!sql_connection) {
/* Check for errors */
	  printf("Connection init error\n");
  }

  sql_connection = mysql_real_connect(sql_connection, host, user, password, db, port, NULL, 0);
  if (!sql_connection) {
/* Check for errors */
	  printf("Connection error: Failed to connect to database.... %s\n", db);
    /* XXX Don't be so harsh!
    exit(-1);
    */
  }

  return sql_connection;

} /* SQ_get_connection() */

SQ_connection_t *SQ_get_connection2(void) {
  return SQ_get_connection(CO_get_host(),
                           CO_get_database_port(),
                           CO_get_database(),
                           CO_get_user(),
                           CO_get_password()
                          );
} /* SQ_get_connection() */

/* SQ_execute_query() */
/*++++++++++++++++++++++++++++++++++++++
  Execute the sql query.

  SQ_connection_t *sql_connection Connection to database.
  
  const char *query SQL query.
  
  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_query">mysql_query()</A>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_use_result">mysql_use_result()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
SQ_result_set_t *SQ_execute_query(int store_or_not,
				  SQ_connection_t *sql_connection, 
				  const char *query) {
  struct timeb *start_time;
  struct timeb *stop_time;
  int err;
  SQ_result_set_t *result;

  if (CO_get_query_logging() == 1) {
    start_time=(struct timeb *)calloc(1, sizeof(struct timeb)+1);
    stop_time=(struct timeb *)calloc(1, sizeof(struct timeb)+1);

    ftime(start_time);
    err = mysql_query(sql_connection, query);
    ftime(stop_time);

    log_query(CO_get_query_logfile(), query, start_time, stop_time);
    
    free(start_time);
    free(stop_time);
  }
  else {
    err = mysql_query(sql_connection, query);
  }

  if (err == 0) {
    result = mysql_store_result(sql_connection);
    if(store_or_not == SQ_NOSTORE) {
      mysql_free_result(result);
      result = NULL;
    }
  }
  else {
    result = NULL;
  }

  return result;

} /* SQ_execute_query() */

/* SQ_get_column_count() */
/*++++++++++++++++++++++++++++++++++++++
  Get the column count.

  SQ_result_set_t *result The results from the query.
  
  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_num_fields">mysql_num_fields()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
int SQ_get_column_count(SQ_result_set_t *result) {
  int cols;

  cols = mysql_num_fields(result);

  return cols;

} /* SQ_get_column_count() */

/* SQ_get_column_label() */
/*++++++++++++++++++++++++++++++++++++++
  Get the column label.

  SQ_result_set_t *result The results from the query.
  
  unsigned int column The column index.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_fetch_field_direct">mysql_fetch_field_direct()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
char *SQ_get_column_label(SQ_result_set_t *result, unsigned int column) {
  char *str;
/* MySQL decided to change their interface.  Doh! */
#ifdef OLDMYSQL
  MYSQL_FIELD field;

  field = mysql_fetch_field_direct(result, column);

  str = (char *)calloc(1, strlen(field.name)+1);
  strcpy(str, field.name);
#else
  MYSQL_FIELD *field;

  field = mysql_fetch_field_direct(result, column);

  str = (char *)calloc(1, strlen(field->name)+1);
  strcpy(str, field->name);
#endif

/*
  printf("column=%d\n", column);
  printf("field.name=%s\n", field.name);
  printf("field.table=%s\n", field.table);

  printf("field.def=%s\n", field.def);

  printf("field.type=%d\n", field.type);
  printf("field.length=%d\n", field.length);
  printf("field.max_length=%d\n", field.max_length);
  printf("field.flags=%d\n", field.flags);
  printf("field.decimals=%d\n", field.decimals);
*/

  return str;

} /* SQ_get_column_label() */

/* SQ_get_column_max_length() */
/*++++++++++++++++++++++++++++++++++++++
  Get the max length of the column.

  SQ_result_set_t *result The results from the query.
  
  unsigned int column The column index.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_fetch_field_direct">mysql_fetch_field_direct()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
unsigned int SQ_get_column_max_length(SQ_result_set_t *result, unsigned int column) {
/* MySQL decided to change their interface.  Doh! */
#ifdef OLDMYSQL
  MYSQL_FIELD field;

  field = mysql_fetch_field_direct(result, column);

  return field.length;
#else
  MYSQL_FIELD *field;

  field = mysql_fetch_field_direct(result, column);

  return field->length;
#endif

} /* SQ_get_column_max_length() */

/* SQ_row_next() */
/*++++++++++++++++++++++++++++++++++++++
  Get the next row.

  SQ_result_set_t *result The results from the query.
  
  unsigned int column The column index.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_fetch_row">mysql_fetch_row()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
SQ_row_t *SQ_row_next(SQ_result_set_t *result) {

  return (SQ_row_t *)mysql_fetch_row(result);

} /* SQ_row_next() */

/* SQ_get_column_string() */
/*++++++++++++++++++++++++++++++++++++++
  Get the column string.

  SQ_row_t *current_row The current row (obtained from a SQ_row_next() ).
  
  unsigned int column The column index.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
char *SQ_get_column_string(SQ_result_set_t *result, SQ_row_t *current_row, unsigned int column) {
  char *str=NULL;
  int length = mysql_fetch_lengths(result)[column];


  if (current_row != NULL && current_row[column] != NULL) {
    str = (char *)malloc(length + 1);
    if (str != NULL) {
      memcpy(str, current_row[column], length );
      str[length] = '\0';
    }
  }

  return str;
  
} /* SQ_get_column_string() */

/* SQ_get_column_strings() */
/*++++++++++++++++++++++++++++++++++++++
  Get the all the strings in one column.

  SQ_result_set_t *result The results.
  
  unsigned int column The column index.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
char *SQ_get_column_strings(SQ_result_set_t *result, unsigned int column) {
  MYSQL_ROW row;
  char str_buffer[STR_XXL];
  char str_buffer_tmp[STR_L];
  char *str;

  strcpy(str_buffer, "");

  while ((row = mysql_fetch_row(result)) != NULL) {
    if (row[column] != NULL) {
      sprintf(str_buffer_tmp, "%s\n", row[column]);
    }
    strcat(str_buffer, str_buffer_tmp);

    if (strlen(str_buffer) >= (STR_XXL - STR_XL) ) {
      strcat(str_buffer, "And some more stuff...\n");
      break;
    }
  }

  if (strcmp(str_buffer, "") != 0) {
    str = (char *)calloc(1, strlen(str_buffer)+1);
    strcpy(str, str_buffer);
  }
  else {
    str = NULL;
  }

  return str;

} /* SQ_get_column_strings() */

/* SQ_get_column_int() */
/*++++++++++++++++++++++++++++++++++++++
  Get an integer from the column.

  SQ_result_set_t *result The results.
  
  SQ_row_t *current_row The current row.

  unsigned int column The column index.

  This uses atoi.  So it may be advisable not to use it.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
int SQ_get_column_int(SQ_result_set_t *result, SQ_row_t *current_row, unsigned int column) {
  int ret_val=-1;

  if (*current_row[column] != NULL) {
    ret_val = atoi(*current_row[column]);
  }
  else {
    ;
  }

  return ret_val;
  
} /* SQ_get_column_int() */


/* SQ_result_to_string() */
/*++++++++++++++++++++++++++++++++++++++
  Convert the result set to a string.

  SQ_result_set_t *result The results.
  
  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
char *SQ_result_to_string(SQ_result_set_t *result) {
  MYSQL_ROW row;
  unsigned int no_cols;
  unsigned int i, j;
  char str_buffer[STR_XXL];
  char str_buffer_tmp[STR_L];
  char border[STR_L];
  char *str;

  char *label;

  unsigned int length[STR_S];

  strcpy(str_buffer, "");

  no_cols = mysql_num_fields(result);

  /* Determine the maximum column widths */
  /* XXX Surely MySQL should keep note of this for me! */
  strcpy(border, "");
  for (i=0; i < no_cols; i++) {
    length[i] = SQ_get_column_max_length(result, i);
    /* Make sure the lenghts don't get too long */
    if (length[i] > STR_M) {
      length[i] = STR_M;
    }
    strcat(border, "*");
    for (j=0; (j <= length[i]) && (j < STR_L); j++) {
      strcat(border, "-");
    }
  }
  strcat(border, "*\n");
  /*
  for (i=0; i < no_cols; i++) {
    printf("length[%d]=%d\n", i, length[i]);
  }
  */

  strcat(str_buffer, border);

  for (i=0; i < no_cols; i++) {
    label = SQ_get_column_label(result, i);
    if (label != NULL) {
      sprintf(str_buffer_tmp, "| %-*s", length[i], label);
      strcat(str_buffer, str_buffer_tmp);
    }
  }
  strcat(str_buffer, "|\n");
  
  strcat(str_buffer, border);


  while ((row = mysql_fetch_row(result)) != NULL) {
    for (i=0; i < no_cols; i++) {
      if (row[i] != NULL) {
        sprintf(str_buffer_tmp, "| %-*s", length[i], row[i]);
      }
      else {
        sprintf(str_buffer_tmp, "| %-*s", length[i], "NuLL");
      }
      strcat(str_buffer, str_buffer_tmp);
    }
    strcat(str_buffer, "|\n");

    if (strlen(str_buffer) >= (STR_XXL - STR_XL) ) {
      strcat(str_buffer, "And some more stuff...\n");
      break;
    }
  }

  strcat(str_buffer, border);
  
  str = (char *)calloc(1, strlen(str_buffer)+1);
  strcpy(str, str_buffer);

  return str;

} /* SQ_result_to_string() */

/* SQ_free_result() */
/*++++++++++++++++++++++++++++++++++++++
  Free the result set.

  SQ_result_set_t *result The results.
  
  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_free_result">mysql_free_result()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
void SQ_free_result(SQ_result_set_t *result) {
  mysql_free_result(result);
} /* SQ_free_result() */


/* SQ_close_connection() */
/*++++++++++++++++++++++++++++++++++++++
  Call this function to close a connection to the server

  SQ_connection_t *sql_connection The connection to the database.
  
  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_close">mysql_close()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
void SQ_close_connection(SQ_connection_t *sql_connection) {

  mysql_close(sql_connection);

}

/* SQ_num_rows() */
/*++++++++++++++++++++++++++++++++++++++
  Call this function to find out how many rows are in a query result

  SQ_result_set_t *result The results.
  
  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_num_rows">mysql_num_rows()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
int SQ_num_rows(SQ_result_set_t *result) {
  int rows=-1;

  if (result != NULL) {
    rows = mysql_num_rows(result);
  }

  return rows;
}

/* SQ_info_to_string() */
/*++++++++++++++++++++++++++++++++++++++
  Convert all available information about the sql server into a string.

  SQ_connection_t *sql_connection The connection to the database.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
char *SQ_info_to_string(SQ_connection_t *sql_connection) {
  char str_buffer[STR_XXL];
  char str_buffer_tmp[STR_L];
  char *str;
  char *str_tmp;

  strcpy(str_buffer, "");

  /* Makes the server dump debug information to the log. */
  sprintf(str_buffer_tmp, "mysql_dump_debug_info()=%d\n", mysql_dump_debug_info(sql_connection));
  strcat(str_buffer, str_buffer_tmp);

  /* Returns the error number from the last MySQL function. */
  sprintf(str_buffer_tmp, "mysql_errno()=%d\n", mysql_errno(sql_connection));
  strcat(str_buffer, str_buffer_tmp);

  /* Returns the error message from the last MySQL function. */
  sprintf(str_buffer_tmp, "mysql_error()=%s\n", mysql_error(sql_connection));
  strcat(str_buffer, str_buffer_tmp);

  /* Returns client version information. */
  sprintf(str_buffer_tmp, "mysql_get_client_info()=%s\n", mysql_get_client_info() );
  strcat(str_buffer, str_buffer_tmp);

  /* Returns a string describing the connection. */
  sprintf(str_buffer_tmp, "mysql_get_host_info()=%s\n", mysql_get_host_info(sql_connection));
  strcat(str_buffer, str_buffer_tmp);

  /* Returns the protocol version used by the connection. */
  sprintf(str_buffer_tmp, "mysql_get_proto_info()=%d\n", mysql_get_proto_info(sql_connection));
  strcat(str_buffer, str_buffer_tmp);

  /* Returns the server version number. */
  sprintf(str_buffer_tmp, "mysql_get_server_info()=%s\n", mysql_get_server_info(sql_connection));
  strcat(str_buffer, str_buffer_tmp);

  /* Information about the most recently executed query. */
  /* XXX Check for NULL */
  str_tmp = mysql_info(sql_connection);
  if (str_tmp != NULL) {
    sprintf(str_buffer_tmp, "mysql_info()=%s\n", str_tmp);
  }
  else {
    sprintf(str_buffer_tmp, "mysql_info()=%s\n", "NulL");
  }
  strcat(str_buffer, str_buffer_tmp);


  /* Returns a list of the current server threads. */
  sprintf(str_buffer_tmp, "mysql_list_processes()=%d\n", mysql_list_processes(sql_connection));
  strcat(str_buffer, str_buffer_tmp);

  /* Checks if the connection to the server is working. */
  sprintf(str_buffer_tmp, "mysql_ping()=%d\n", mysql_ping(sql_connection));
  strcat(str_buffer, str_buffer_tmp);

  /* Returns the server status as a string. */
  sprintf(str_buffer_tmp, "mysql_stat()=%s\n", mysql_stat(sql_connection));
  strcat(str_buffer, str_buffer_tmp);

  /* Returns the current thread id. */
  sprintf(str_buffer_tmp, "mysql_thread_id()=%d\n", mysql_thread_id(sql_connection));
  strcat(str_buffer, str_buffer_tmp);


  str = (char *)calloc(1, strlen(str_buffer)+1);
  strcpy(str, str_buffer);

  return str;

} /* SQ_info_to_string() */

/* SQ_error() */
/*++++++++++++++++++++++++++++++++++++++
  Get the error string for the last error.

  SQ_connection_t *sql_connection The connection to the database.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_error">mysql_error()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
char *SQ_error(SQ_connection_t *sql_connection) {

  return mysql_error(sql_connection);

} /* SQ_error() */

/* SQ_errno() */
/*++++++++++++++++++++++++++++++++++++++
  Get the error number for the last error.

  SQ_connection_t *sql_connection The connection to the database.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+     <LI><A HREF="http://www.tcx.se/Manual/manual.html#mysql_free_result">mysql_free_result()</A>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
int SQ_errno(SQ_connection_t *sql_connection) {

  return mysql_errno(sql_connection);

} /* SQ_errno() */

