/*
 *	Copyright (c) 1994, University of Kansas, All Rights Reserved
 *
 *	Include File:	capstdio.h
 *	Purpose:	Provide functions to capture output to stdio and
 *			stderr.
 *	Remarks/Portability/Dependencies/Restrictions:
 *		We do not include the capstdio header file because it would
 *		further screw up any of the stdio function names that we may
 *		call in the below code.
 *	Revision History:
 *		01-08-94	created
 */
#include"capalloc.h"
#include<stdio.h>
#include<stdarg.h>
#include<string.h>
#include<errno.h>

int capfprintf(FILE *Fp_stream, const char *cp_format, ...)	{
/*
 *	Purpose:	Check if output is directed at stdout or stderr and
 *			if so redirect to a user implemented function.
 *	Arguments:	Fp_stream	The destination file.
 *			cp_format	The format string describing the stack
 *			...		Variable number of arguments matching
 *					the stack.
 *	Return Value:	int	On error, EOF.
 *				On success, the number of bytes written.
 *	Remarks/Portability/Dependencies/Restrictions:
 *		We must limit the amount to be written to stdout and stderr
 *		by a set buffersize of 1024 bytes.
 *	Revision History:
 *		01-08-94	created
 */

	va_list argptr;
	int retval;

	/*
	 *	Set up our variable number of arguments.
	 */
	va_start(argptr, cp_format);

	/*
	 *	Do we capture the output?
	 */
	if(Fp_stream == stdout || Fp_stream == stderr)	{
		/*
		 * 	Capture.
		 */
		char ca_buffer[1024];
		retval = vsprintf(ca_buffer, cp_format, argptr);
		if(Fp_stream == stdout)
			cap_stdout(ca_buffer);
		else
			cap_stderr(ca_buffer);
	}
	else	{
		/*
		 *	Normal.
		 */
		retval = vfprintf(Fp_stream, cp_format, argptr);
	}

	va_end(argptr);
	return(retval);
}

int capfputs(const char *cp_s, FILE *Fp_stream)	{
/*
 *	Purpose:	Check if output is directed at stdout or stderr and
 *			if so redirect.
 *	Arguments:	cp_s		The string to output.
 *			Fp_stream	The file to write the output to.
 *	Return Value:	int	On error, EOF.
 *				On success, the last character written.
 *	Remarks/Portability/Dependencies/Restrictions:
 *	Revision History:
 *		01-08-94	created
 */
	if(EOF == capfprintf(Fp_stream, "%s", cp_s))
		return(EOF);
	else
		return(*(cp_s + strlen(cp_s) - 1));
}

size_t capfwrite(const void *vp, size_t st_size, size_t st_n,
	FILE *Fp_stream)	{
/*
 *	Purpose:	Check if output is directed at stdout or stderr and
 *			if so redirect.
 *	Arguments:	vp		A pointer to objects of any size.
 *			st_size		The size of the objects to write.
 *			st_n		The number of objects to write.
 *			Fp_stream	The output stream.
 *	Return Value:	size_t	The number of items written.
 *				On success, returns st_n.
 *				On failure, returns < st_n.
 *	Remarks/Portability/Dependencies/Restrictions:
 *	Revision History:
 *		01-08-94	created
 */

	size_t retval;
	char *cp_traverse = vp;

	/*
	 *	Do we capture the output?
	 */
	if(Fp_stream == stdout || Fp_stream == stderr)	{
		/*
		 *	capture
		 */
		size_t st_items;
		size_t st_bytes;

		/*
		 *	Loop through the items.
		 */
		for(st_items = (size_t)0; st_items < st_n; st_items++)	{
			/*
			 *	Loop through the bytes.
			 */
			for(st_bytes = (size_t)0; st_bytes < st_size;
				st_bytes++)	{
				/*
				 *	print one byte at a time.
				 */
				capfprintf(Fp_stream, "%c", *cp_traverse++);
			}
		}

		retval = st_n;
	}
	else	{
		/*
		 *	normal
		 */
		retval = fwrite(vp, st_size, st_n, Fp_stream);
	}

	return(retval);
}

void capperror(const char *cp_s)	{
/*
 *	Purpose:	Redirect stderr output.
 *	Arguments:	cp_s	A message.
 *	Return Value:	void
 *	Remarks/Portability/Dependencies/Restrictions:
 *	Revision History:
 *		01-08-94	created
 */

	capfprintf(stderr, "%s : %s", cp_s, strerror(errno));
}

int capprintf(const char *cp_format, ...)	{
/*
 *	Purpose:	Redirect stdout output.
 *	Arguments:	cp_format	The printf style format string.
 *			...		Variable arguments corresponding to
 *					the format string.
 *	Return Value:	int	On Error, returns EOF.
 *				On success, returns number of bytes written.
 *	Remarks/Portability/Dependencies/Restrictions:
 *		The amount written is limited to 1024 bytes.
 *	Revision History:
 *		01-08-94	created
 */

	int retval;
	char ca_buffer[1024];
	va_list argptr;

	/*
	 *	Handle variable number of arguments.
	 */
	va_start(argptr, cp_format);

	/*
	 *	redirect stdout
	 */
	retval = vsprintf(ca_buffer, cp_format, argptr);
	cap_stdout(ca_buffer);

	va_end(argptr);
	return(retval);
}

int capputc(int c, FILE *Fp_stream)	{
/*
 *	Purpose:	Captures output directed at stdio and stderr and
 *			redirects.
 *	Arguments:	c		The character to output.
 *			Fp_stream	The output stream.
 *	Return Value:	int	On success, c.
 *				On failure, EOF.
 *	Remarks/Portability/Dependencies/Restrictions:
 *	Revision History:
 *		01-08-94	created
 */

	return(capfprintf(Fp_stream, "%c", c) == EOF ? EOF : c);
}

#ifdef WATTCP
/*
 *	The following are to take over some wattcp assembly functions that
 *	write directly to the screen.
 *	Assume wattcp would only do this in the case of an error, so put
 *	to stderr, the above functions will handle the rest.
 */

void capoutch(char c)	{
/*
 *	Purpose:	Capture output meant for screen and redirect.
 *	Arguments:	c	The character to output.
 *	Return Value:	void
 *	Remarks/Portability/Dependencies/Restrictions:
 *	Revision History:
 *		01-08-94	created
 */

	capputc(c, stderr);
}

void capouts(char *cp)	{
/*
 *	Purpose:	Capture output meant for screen and redirect.
 *	Arguments:	cp	The string to output.
 *	Return Value:	void
 *	Remarks/Portability/Dependencies/Restrictions:
 *	Revision History:
 *		01-08-94	created
 */
	capfputs(cp, stderr);
}

void capouthex(char c)	{
/*
 *	Purpose:	Capture output meant for screen and redirect.
 *	Arguments:	c	The character to output in hex.
 *	Return Value:	void
 *	Remarks/Portability/Dependencies/Restrictions:
 *	Revision History:
 *		01-08-94	created
 */

	char c_temp;

	/*
	 *	Print the first hex digit.
	 */
	c_temp = c >> 4;
	c_temp = (c_temp < 10) ? '0' + c_temp : 'A' + c_temp - 10;

	/*
	 *	Printf the second hex digit.
	 */
	c_temp = c & 0x0F;
	c_temp = (c_temp < 10) ? '0' + c_temp : 'A' + c_temp - 10;
	capfprintf(stderr, "%c", c_temp);
}
#endif /* WATTCP */