/* $Header: tio.c,v 1.3 86/04/09 21:10:30 notes Exp $ */

/*
 * Miscellaneous Terminal IO and Signal Handling Routines
 */

#include "parms.h"
#include "structs.h"

#ifdef	SYSV
#include <termio.h>
#else
#include <sgtty.h>
#endif

#include <signal.h>
#include <errno.h>

static int modeset = 0;			/* == 1 if ttyflags messed with */
static char t_erase, t_kill;
short ospeed;				/* for tputs padding */
int replot;

#ifdef	SYSV
static struct termio tty, otty;
#else
static int oldmode;			/* prev mode bits */
static struct sgttyb tty;
#endif

/*
 * switch into character at a time mode
 */
ttystrt()
{
	static int initialized = 0;

	if (!initialized) {
#ifdef	SYSV
		x (ioctl(0, TCGETA, &tty) < 0, "ttystrt: gtty");
		otty = tty;
		t_erase = tty.c_cc[VERASE];
		t_kill = tty.c_cc[VKILL];
		ospeed = tty.c_cflag & CBAUD;
#else	SYSV
		x(ioctl(0, TIOCGETP, &tty) < 0, "ttystrt: gtty");
		oldmode = tty.sg_flags;
		t_erase = tty.sg_erase;
		t_kill = tty.sg_kill;
		ospeed = tty.sg_ospeed;
#endif	SYSV
		initialized = 1;
	}

	if (!modeset) {
#ifdef	SYSV
		tty.c_lflag &= ~(ECHO|ICANON);
		tty.c_cc[VEOF] = 1;
		tty.c_cc[VEOL] = 1;
		x (ioctl(0, TCSETA, &tty) < 0, "ttystrt: stty");
#else	SYSV
		tty.sg_flags |= CBREAK;
		tty.sg_flags &= ~ECHO;
		x(ioctl(0, TIOCSETN, &tty) < 0, "ttystrt: stty");
#endif	SYSV
		replot = 1;
		modeset = 1;
	}
}

/*
 * back to cooked mode
 */
ttystop()
{
	if (modeset) {
#ifndef	SYSV
		tty.sg_flags = oldmode;
		if (ioctl(0, TIOCSETN, &tty) < 0)
#else	SYSV
		if (ioctl(0, TCSETA, &otty) < 0)
#endif	SYSV
			printf("ttystop: stty");
			/* can't use x() cause it calls us */
		modeset = 0;
	}
}

#ifdef SYSV
static char pending = 0;
#endif

/*
 * return 1 if there is input from the terminal,
 * 0 otherwise.  systems without the appropriate
 * call should always return 0.
 */
isinput()
{
#ifndef	SYSV
	long retval;

	if (ioctl(0,FIONREAD,&retval) == 0)
		return(retval != 0);
#else
#include <fcntl.h>

	/* do a read with no wait, push back the char if we get one */

	register int ret;
	extern int errno;

	if (pending!=0)			/* anything waiting already? */
		return(1);		/* if so, return true */

	fcntl(0, F_SETFL, fcntl(0, F_GETFL) | O_NDELAY);  /* read no wait */

	while ((ret = read (0, &pending, 1)) == -1)
		/* try again only if interrupted */
		x (errno != EINTR, "isinput: read error");

	fcntl(0, F_SETFL, fcntl(0, F_GETFL) & ~O_NDELAY); /* read with wait */

	if (pending!=0)			/* anything waiting now? */
		return(1);		/* if so, return true */

#endif
	return(0);
}

/*
 * get the next character from the terminal,
 * stripping the parity bit and mapping \r to \n
 */
gchar(flush)
{
	char c;
	int ret;
	extern int errno;

	fflush (stdout);		/* get rid of what's there */

#ifdef SYSV
	/* is there a pending character pushed back from isinput? */
	if (pending!=0) {
		c = pending;		/* get the pending char */
		pending = 0;		/* not pending any more */
	}
	else 
#endif
	     {
		while ((ret = read (0, &c, 1)) <= 0)
			/* try again only if interrupted */
			x(ret == 0 || errno != EINTR, "gchar: read error");
	}
	c &= 0177;
	if (c == '\r')
		c = '\n';
	if (flush) flushout();		/* toss output in case user aborted */
	return(c);
}

/*
 * get a number from the user.
 * c is the first digit (already typed).
 */
getnum(c)
int c;
{
	int num, numin;
	int x, y;

	num = c - '0';
	curpos(&x, &y);
	numin = 1;
	putch(c);
	for (; ;) {
		c = gchar(0);
		if (c == '\n')
			return(num);			/* done */
		if (c == t_erase) {
			if (numin > 0) {
				num /= 10;
				numin--;
				putstr("\b \b");
			}
			continue;
		}
		if (c == t_kill) {
			num = 0;
			numin = 0;
			at(x, y);
			clear_eol();
			continue;
		}
		if ('0' <= c && c <= '9') {		/* valid character */
			putch(c);
			numin++;
			num = 10 * num + (c - '0');
		} else
			putch('\07');
	}
	/*NOT REACHED*/
}

/*
 * get a line from the terminal
 *
 * return the line in string p
 * p is null terminated and does not include the new line delimeter
 * at most max characters are returned, including the null.
 *	
 * do erase and kill processing.
 * The line is terminated by the user typing a <cr> or <nl>. This
 * character is converted to null and left on the end of the
 * string returned. The count of characters is returned.
 */
gline(p, max)
char *p;
{
	register int numin;
	register char *q;
	register int c;
	int x, y;

	q = p;	
	numin = 0;
	curpos(&x, &y);
	for (;;) {
		c = gchar(0);
		if (c == t_erase) {
			if (numin > 0) {
				numin--;
				q--;
				putstr("\b \b");
				if (*q < 040 || *q == 0177)
					putstr("\b \b");
			}
			continue;
		}
		if (c == t_kill) {
			at(x, y);
			clear_eol();
			q = p;
			numin = 0;
			continue;
		}
		switch (c) {
		case '\n': 
			*q = '\0';
			return(numin);
		case 'V'&037:			/* ^V */
		case '\\': 			/* escape character */
			putstr("^\b");		/* back space to it */
			c = gchar(0);		/* grab escaped character */
			/* FALL INTO ... */
		default:
			if (numin < max-1) {
				*q++ = c;
				numin++;
				mapch(c);
			} else
				putch('\007');
				/* show him I ignored char */
			break;
		}
	}
	/*NOT REACHED*/
}

/*
 * returns y or n to the asked question
 */
askyn()
{
	int c;
	int x, y;

	curpos(&x, &y);
	for (; ;) {
		c = gchar(0);
		if (c == 'y' || c == 'n') {
			putch(c);
#ifdef	UCB
			putstr("  (working)");
#endif
			fflush(stdout);
			return(c);
		}
		putchar('\007');
		putstr("  (y/n)");
		at(x, y);
	}
	/*NOT REACHED*/
}

wfchar()
{
	printf("--Hit any key to continue--");
	gchar(0);
}

/* VARARGS1 */
warn(s, a1, a2, a3)
char *s;
{
	char buf[BUFSIZ];

	at(0, 1);
	sprintf(buf, s, a1, a2, a3);
	standout(1);
	putstr(buf);
	standout(0);
	clear_eol();
}

/* VARARGS1 */
prompt(s, a1, a2, a3)
char *s;
{
	char buf[BUFSIZ];

	at(0, 1);
	sprintf(buf, s, a1, a2, a3);
	putstr(buf);
	clear_eol();
}

cmdprompt()
{
#ifdef notdef
	short set = LFLUSHO;

	fflush(stdout);		/* sigh... want a synchronous write here */
	ioctl(0, TIOCLBIC, &set);
#endif
	at(0, 1);
	putstr(prompt_str);
}

putstr(s)
char *s;
{
	while (*s)
		putch(*s++);
}

/*
 * a putchar with all the fancy control
 * character mapping
 */
mapch(c)
{
	if (c < 040 || c == 0177) {
		putch('^');
		if (c < 040)
			c |= 0100;
		else
			c = '?';
	}
	putch(c);
}

/*
 * signal handling routines
 */
static SIGHNDLR (*osigint)();
static SIGHNDLR (*osigquit)();			/* hold signal status */
#ifdef	SIGTSTP
static SIGHNDLR (*osigtstp)();			/* control-z job stop */
#endif

int ignsigs;
int contsigs;
struct notesenv curenv;

/*
 * the new fancy signal catcher.
 * for interrupts and quits.
 */
SIGHNDLR
catchint(signo)
{
	signal(signo, catchint);
	if (ignsigs)
		return;
	if (replot == 0)
		warn("Interrupt");
	longjmp(jenv, 1);
}

#ifdef	SIGTSTP

/*
 * the new fancy signal handler
 * for ^Z/SIGTSTP
 */
SIGHNDLR
catchz()
{
	int wasset;

	signal(SIGTSTP, catchz);
	if (ignsigs)
		return;
	if ((wasset = modeset) != 0) {		/* want assignment */
		ttystop();			/* fix it up for user */
		at(0, 1);
		fflush(stdout);
	}
#ifdef	BSD4_2
	(void)sigsetmask(0);
#endif
	signal(SIGTSTP, SIG_DFL);		/* make sure it nabs us */
	kill(0, SIGTSTP);			/* halt myself */
	signal(SIGTSTP, catchz);		/* ready to catch again */
	if (wasset)
		ttystrt();			/* fix his tty */
	if (!contsigs)
	    longjmp(jenv, 1);
}
#endif

catchem()
{
	osigint = signal(SIGINT, catchint);		/* interrupts */
	osigquit = signal(SIGQUIT, catchint);		/* quits */
#ifdef	SIGTSTP
	osigtstp = signal(SIGTSTP, catchz);		/* control Z */
#endif
}

uncatchem()
{
	signal(SIGINT, osigint);
	signal(SIGQUIT, osigquit);
#ifdef	SIGTSTP
	signal(SIGTSTP, osigtstp);
#endif
}
/*
 * Flush output queue.  This is done after an isinput is recognised
 * and also just after reading a command;  this prevents waiting for
 * output to drain after interrupting a repaint.
 */

flushout()
{

#ifdef SYSV
	x (ioctl(0, TCFLSH, 1) < 0, "flushout: flush");
#endif
}
